/*
 * The 3D Studio File Format Library
 * Copyright (C) 1996-2001 by J.E. Hoffmann <je-h@gmx.net>
 * All rights reserved.
 *
 * This program is  free  software;  you can redistribute it and/or modify it
 * under the terms of the  GNU Lesser General Public License  as published by 
 * the  Free Software Foundation;  either version 2.1 of the License,  or (at 
 * your option) any later version.
 *
 * This  program  is  distributed in  the  hope that it will  be useful,  but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or  FITNESS FOR A  PARTICULAR PURPOSE.  See the  GNU Lesser General Public  
 * License for more details.
 *
 * You should  have received  a copy of the GNU Lesser General Public License
 * along with  this program;  if not, write to the  Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: document.cpp,v 1.4 2001/01/15 10:56:12 jeh Exp $
 */
#include "document.h"
#include <lib3ds/file.h>
#include <lib3ds/camera.h>
#include <qtimer.h>


/*!
 *  
 */
Document::Document()
{
  d_file=0;
  d_timer=new QTimer(this);
  connect(d_timer, SIGNAL(timeout()), this, SLOT(timerSlot()));
}


/*!
 *  
 */
Document::~Document()
{
  if (d_file) {
    lib3ds_close(d_file);
    d_file=0;
  }
}


/*!
 *  
 */
bool 
Document::open(QString filename)
{
  if (d_file) {
    lib3ds_close(d_file);
    d_file=0;
  }
  d_file=lib3ds_open((const char*)filename);
  if (!d_file) {
    puts("***ERROR*** Loading 3DS file failed.");
    return(false);
  }
  if (!d_file->cameras) {
    puts("***ERROR*** No Camera found.");
    lib3ds_close(d_file);
    d_file=0;
    return(false);
  }
  d_camera=d_file->cameras->name;

  lib3ds_file_eval(d_file,d_file->current_frame);
  emit documentChanged();
  return(true);
}


/*!
 *  
 */
bool 
Document::save(QString filename)
{
  /* FIXME: */
  ASSERT(false);
  return(false);
}


/*!
 *  
 */
void 
Document::begin()
{
  d_file->current_frame=0;
  emit documentChanged();
}


/*!
 *  
 */
void 
Document::prev()
{
  --d_file->current_frame;
  if (d_file->current_frame<0) {
    d_file->current_frame=0;
  }
  emit documentChanged();
}


/*!
 *  
 */
void 
Document::play()
{
  d_timer->start(0,true);
}


/*!
 *  
 */
void 
Document::stop()
{
  d_timer->stop();
}


/*!
 *  
 */
void 
Document::next()
{
  ++d_file->current_frame;
  if (d_file->current_frame>d_file->frames) {
    d_file->current_frame=d_file->frames;
  }
  emit documentChanged();
}


/*!
 *  
 */
void 
Document::end()
{
  d_file->current_frame=d_file->frames;
  emit documentChanged();
}


/*!
 *  
 */
Lib3dsFile* 
Document::file()
{
  return(d_file);
}


/*!
 *  
 */
QString
Document::camera()
{
  return(d_camera);
}


/*!
 *  
 */
void 
Document::setCurrent(int current)
{
  if (!d_file) {
    return;
  }

  d_file->current_frame=current;
  if (d_file->current_frame>d_file->frames) {
    d_file->current_frame=d_file->frames;
  }
  lib3ds_file_eval(d_file, d_file->current_frame);
  emit documentChanged();
}


/*!
 *  
 */
void 
Document::setCamera(const QString& name)
{
  if (d_camera!=name) {
    d_camera=name;
    emit documentChanged();
  }
}


/*!
 *  
 */
void 
Document::timerSlot()
{
  ASSERT(d_timer);
  if (!d_file) {
    return;
  }

  ++d_file->current_frame;
  if (d_file->current_frame>d_file->frames) {
    d_file->current_frame=0;
  }
  lib3ds_file_eval(d_file, d_file->current_frame);

  emit documentChanged();
  d_timer->start(0,true);
}





