/*
                   Vertex - Plug-in Function Prototypes

	Prototypes for functions that Vertex checks for in its loaded
	plugins. Functions that are mandatory to exist on the plug-in
	will be denoted, all other prototyped functions already exist in
	Vertex.

	Prefix for Vertex Plug-ins API will be VPI (which stands for
	Vertex Plug-In).

	All Vertex plug-ins must #include this header file.
 */

#ifndef VPI_H
#define VPI_H

#include <sys/types.h>
#include <sys/stat.h>
#include "v3d.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


/* ***********************************************************************
 *                 PLUG-IN BASIC TYPES AND DEFINATIONS
 * ******************************************************************** */

/*
 *	VPI plug-in id:
 */
#define vpi_id		void

/*
 *	VPI code type (used in VPIGet*():
 */
#define vpi_code_t      int

/*
 *	VPI flag type:
 */
#define vpi_flag	u_int32_t

/*
 *	Other VPI types:
 */
#define vpi_icon_data	u_int8_t


/*
 *	True and false:
 */
#define VPI_FALSE	0
#define VPI_TRUE	1


/*
 *	VPI error codes:
 *
 *	Note any value that is non-zero can be considered an error.
 */
#define VPI_SUCCESS		0
#define VPI_ERROR		-1		/* Same as VPI_ERROR. */
#define VPI_ERROR_GENERAL	VPI_ERROR
#define VPI_ERROR_NOT_FOUND	-2
#define VPI_ERROR_SYSTEM	-3
#define VPI_ERROR_USER_ABORT	-4
#define VPI_ERROR_BAD_VALUE	-5
#define VPI_ERROR_NOT_SUPPORTED	-6

/*
 *	Message types (suggests icon to be used):
 */
#define VPI_MESSAGE_INFO	0
#define VPI_MESSAGE_WARNING	1
#define VPI_MESSAGE_ERROR	2
#define VPI_MESSAGE_QUESTION	3

#define VPI_MESSAGE_IMPORT	10
#define VPI_MESSAGE_EXPORT	11

#define VPI_MESSAGE_RENDER	20
#define VPI_MESSAGE_OPTIONS	30


/*
 *	Question response codes:
 */
#define VPI_RESPONSE_NOT_AVAILABLE	-1
#define VPI_RESPONSE_NO			0
#define VPI_RESPONSE_YES		1
#define VPI_RESPONSE_YES_TO_ALL		2
#define VPI_RESPONSE_CANCEL		3
#define VPI_RESPONSE_OK			4


/*
 *	Position codes:
 */
#define VPI_POSITION_FIRST	0
#define VPI_POSITION_LAST	1
#define VPI_POSITION_SELECTED	2


/*
 *	File extension type structure:
 *
 *	Note that this is not use for setting file type support.
 */
typedef struct {

	/* Defines which members of this structure are set. */
#define VPI_FTYPE_EXT		(1 << 0)
#define VPI_FTYPE_NAME		(1 << 1)
	vpi_flag flags;

	char *ext;	/* Extension, ie ".dxf". */
	char *name;	/* Descriptive name, ie "AutoCAD Drawing Exchange". */

} vpi_ftype_struct;

/*
 *	All purpose color structure:
 */
typedef struct {

	double r, g, b, a;	/* Coeff from 0.0 to 1.0. */

} vpi_color_struct;

/*
 *	Camera structure:
 */
typedef struct {

	/* Defines which members of this structure are set. */
#define VPI_CAMERA_POS		(1 << 0)
#define VPI_CAMERA_DIR		(1 << 1)
#define VPI_CAMERA_UP		(1 << 2)
#define VPI_CAMERA_FOV		(1 << 3)	/* Field of view. */
#define VPI_CAMERA_ASPECT	(1 << 4)
#define VPI_CAMERA_CLIP_NEAR	(1 << 5)
#define VPI_CAMERA_CLIP_FAR	(1 << 6)
	vpi_flag flags;

	/* Position. */
	double pos_x, pos_y, pos_z;

	/* Direction vector (unit length). */
	double dir_x, dir_y, dir_z;

	/* Up vector (unit length). */
	double up_x, up_y, up_z;

        /* Camera field of view (along z axis/about the x axis),
         * in radians.
         */
        double fov;

        /* Camera viewport aspect ratio (width/height). */
        double aspect;

        /* Camera near and far clip. */
        double clip_near, clip_far;

} vpi_camera_struct;

/*
 *      Light structure:
 */
typedef struct {

        /* Defines which members of this structure are set. */
#define VPI_LIGHT_POS           (1 << 0)
#define VPI_LIGHT_DIR           (1 << 1)	/* Not set implies omnidirectional. */
#define VPI_LIGHT_AMBIENT       (1 << 2)
#define VPI_LIGHT_DIFFUSE       (1 << 3)
#define VPI_LIGHT_SPECULAR      (1 << 4)
#define VPI_LIGHT_SPOT_CUTOFF   (1 << 5)	/* Not set implies omnidirectional. */
	vpi_flag flags;

        /* Note there is no enabled marker, when a light is defined
         * it is considered enabled.
         */

        double pos_x, pos_y, pos_z;
        double dir_x, dir_y, dir_z;
        vpi_color_struct ambient;
        vpi_color_struct diffuse;
        vpi_color_struct specular;
        double spot_cutoff;     /* In radians. */

} vpi_light_struct;


/* ***********************************************************************
 *     PLUG-IN SIDE INFO INITIALIZE MANAGE AND SHUTDOWN PROTOTYPES
 * ******************************************************************** */

/*	Structures defined here are for recieving of values from Vertex
 *	when Vertex calls your plug-in's info, initialize, manage, and
 *	shutdown functions.
 *
 *	Those functions are also prototyped here but commented out, since
 *	your plug-in must prototype them and exactly as shown farther
 *	below.
 */

/*	`Information request' value structure, used with function
 *	VPIInfo().
 */
typedef struct {

	/* Defines which members of this structure are set. */
#define VPI_FLAG_INFO_VERSION_MAJOR	(1 << 0)
#define VPI_FLAG_INFO_VERSION_MINOR	(1 << 1)
#define VPI_FLAG_INFO_VERSION_RELEASE	(1 << 2)
        vpi_flag flags;

	/* The version of Vertex (not your plug-in). */
	int version_major, version_minor, version_release;

} vpi_info_struct;

/*
 *	`Initialize' value structure, used with function VPIInit().
 */
typedef struct {

	/* Defines which members of this structure are set. */
#define VPI_FLAG_INIT_START_TYPE	(1 << 0)
#define VPI_FLAG_INIT_FILENAME		(1 << 1)
#define VPI_FLAG_INIT_ARGC		(1 << 2)
#define VPI_FLAG_INIT_ARGV		(1 << 3)
	vpi_flag flags;

#define VPI_INIT_START_TYPE_STANDARD	0
	int start_type;

	/* Full path to this plug-in binary. */
	char *filename;

	/* Startup arguments. */
	int argc;
	char **argv;

} vpi_init_struct;

/*
 *	`Manage' value structure, used with function VPIManage().
 */
typedef struct {

	/* Defines which members of this structure are set. */
#define VPI_FLAG_MANAGE_CLIENT_DATA	(1 << 0)
        vpi_flag flags;

	/* Client data (set by VPISetClientData()). */
	void *client_data;

} vpi_manage_struct;

/*
 *	`Shutdown' value structure, used with function VPIShutdown().
 */
typedef struct {

        /* Defines which members of this structure are set. */
#define VPI_FLAG_SHUTDOWN_REASON	(1 << 0)
#define VPI_FLAG_SHUTDOWN_CLIENT_DATA	(1 << 1)
        vpi_flag flags;

#define VPI_SHUTDOWN_REASON_NORMAL_REQUEST	0
#define VPI_SHUTDOWN_REASON_ERROR_GENERAL	1
	int reason;

        /* Client data (set by VPISetClientData()). */
        void *client_data;

} vpi_shutdown_struct;


/* These functions are standard, they should exist but are not required.
 *
 * Note that these are commented out, because they are defined on the
 * plug-in. The plug-in needs to prototype them exactly as shown below.
 *
VPI_PROTO_RETURN_INFO VPIInfo(VPI_PROTO_INPUT_INFO);
VPI_PROTO_RETURN_INIT VPIInit(VPI_PROTO_INPUT_INIT);
VPI_PROTO_RETURN_MANAGE VPIManage(VPI_PROTO_INPUT_MANAGE);
VPI_PROTO_RETURN_SHUTDOWN VPIShutdown(VPI_PROTO_INPUT_SHUTDOWN);
 *
 * Return value for VPIInfo(), VPIInit(), and VPIManage() can be
 * VPI_TRUE (1) for success or VPI_FALSE (0) for failure.
 * If VPI_FALSE is returned by any of those functions then VPIShutdown()
 * will be called and the plug-in will be unloaded. Detailed explaination
 * of each of these functions is as follows:
 *
 * VPIInfo() can be called at any time (even before initialization) to
 * query information about your plug-in. See demos on what information
 * you should respond with.
 *
 * VPIInit() is called to formally initialize your plug-in once per
 * instance/reference.
 *
 * VPIManage() is called one per cycle (which is very often) to allow
 * your plug-in to manage its own resources (if any).
 *
 * VPIShutdown() is called just before unloading of one instance/reference
 * of your plug-in. This is a good time to deallocate any resources that
 * your plug-in allocated since VPIInit().
 */

/* Some #define's for the names of the above mandatory functions. */
#define VPI_PROTO_NAME_INFO	"VPIInfo"
#define VPI_PROTO_INPUT_INFO	vpi_id *id, vpi_info_struct *v
#define VPI_PROTO_RETURN_INFO	int

#define VPI_PROTO_NAME_INIT	"VPIInit"
#define VPI_PROTO_INPUT_INIT	vpi_id *id, vpi_init_struct *v
#define VPI_PROTO_RETURN_INIT	int

#define VPI_PROTO_NAME_MANAGE	"VPIManage"
#define VPI_PROTO_INPUT_MANAGE	vpi_id *id, vpi_manage_struct *v
#define VPI_PROTO_RETURN_MANAGE	int

#define VPI_PROTO_NAME_SHUTDOWN		"VPIShutdown"
#define VPI_PROTO_INPUT_SHUTDOWN	vpi_id *id, vpi_shutdown_struct *v
#define VPI_PROTO_RETURN_SHUTDOWN	void


/*
 *	Sets the client data.
 */
extern void VPISetClientData(vpi_id *id, void *client_data);

/*
 *	Gets the client data.
 */
extern void *VPIGetClientData(vpi_id *id);


/* ***********************************************************************
 *                          VERSION SETTING
 * ******************************************************************** */

/*      Sets the plug-in version, if the given version is newer than
 *      the version of Vertex then run time issues (if any) will be
 *      taken into consideration.
 *
 *      Returns 0 if the plug-ins version is equal or older than Vertex's
 *      version, and 1 if the plug-in's version is newer than Vertex's.
 *
 *      Can also return -1 on error.
 */
extern int VPISetPluginVersion(
	vpi_id *id, int version_major, int version_minor
);


/* ***********************************************************************
 *                 DATA TYPE ALLOCATION & DEALLOCATION
 * ******************************************************************** */

/* Allocates and deallocates VPI data type structures.
 *
 * For all *New() functions here, return of NULL means error.
 *
 * For all *Delete() functions here, any allocated members on the given
 * structure will be deallocated.
 */
extern vpi_ftype_struct *VPIFTypeNew(void);
extern void VPIFTypeDelete(vpi_ftype_struct *buf);

extern vpi_color_struct *VPIColorNew(void);
extern void VPIColorDelete(vpi_color_struct *buf);

extern vpi_camera_struct *VPICameraNew(void);
extern void VPICameraDelete(vpi_camera_struct *buf);

extern vpi_light_struct *VPILightNew(void);
extern void VPILightDelete(vpi_light_struct *buf);


/* ***********************************************************************
 *                   MESSAGES AND USER RESPONSE IO
 * ******************************************************************** */

/*	Prints message and blocks execution untill user response.
 *
 *	Given plug-in id and editor id are optional.
 */
extern void VPIMessage(
        vpi_id *id, int editor_id,	/* Can be NULL and -1. */
        const char *title,
        const char *message,
        const char *details,
        int type                /* One of VPI_MESSAGE_*. */
);

/*
 *	Asks user for input in response to a question, returns one of
 *	VPI_RESPONSE_* depending on what flags were set by response_flags.
 *
 *	VPI_RESPONSE_NOT_AVAILABLE can be returned regardless of what is
 *	set in response_flags. The VPI_RESPONSE_NOT_AVAILABLE response
 *	code means that it was unable to query the user for a response (ie
 *	if another query was being made). You should always check for this
 *	response code.
 *
 *	Given plug-in id and editor id are optional.
 */
#define VPI_RESPONSE_FLAG_OK		(1 << 0)
#define VPI_RESPONSE_FLAG_YES		(1 << 1)
#define VPI_RESPONSE_FLAG_YES_TO_ALL	(1 << 2)
#define VPI_RESPONSE_FLAG_NO		(1 << 3)
#define VPI_RESPONSE_FLAG_CANCEL	(1 << 4)
#define VPI_RESPONSE_FLAG_IGNORE	(1 << 5)
#define VPI_RESPONSE_FLAG_RETRY		(1 << 6)
#define VPI_RESPONSE_FLAG_ABORT		(1 << 7)
extern int VPIQuestion(
        vpi_id *id, int editor_id,	/* Can be NULL and -1. */
        const char *title,
        const char *message,
        const char *details,
        int type,               /* One of VPI_MESSAGE_*. */
	vpi_flag response_flags,	/* Any or'ed set of VPI_RESPONSE_FLAG_*. */
	vpi_flag default_response	/* One of VPI_RESPONSE_FLAG_*. */
);

/*
 *	Simplified version of VPIQuestion(), this one just returns
 *	VPI_RESPONSE_YES, VPI_RESPONSE_NO, or VPI_RESPONSE_NOT_AVAILBLE.
 */
extern int VPIQuestionSimple(
        vpi_id *id, int editor_id,      /* Can be NULL and -1. */
        const char *message
);


/*        
 *      Maps the progress dialog (as needed) and sets the given values.
 *      If the dialog is already mapped then this will only update
 *      its values.
 * 
 *      A returns the number of stop counts, where 0 is no stops and
 *	1 or higher means there have been that many stop requests. 
 *	Basically it returns the number of times the user clicked on stop.
 *      The higher the return value, the greater the priority to stop.
 *
 *      Given plug-in id and editor id are optional.
 *
 *	Also by calling this function, you allow Vertex a chance to
 *	manage its own resources while the toplevel thread of execution is
 *	still in your plug-in's routines.
 */
extern int VPIProgress(
        vpi_id *id, int editor_id,
        const char *title,	/* Title of progress. */
        const char *message,	/* Message. */
        const char *stop_label,	/* Stop button label. */
	int type,		/* One of VPI_MESSAGE_*. */
	double progress,	/* 0.0 to 1.0 (or -1.0). */
	unsigned int nblocks    /* 0 for continuous. */
);

/*
 *      Unmaps the progress dialog.
 *
 *	Given plug-in id and editor id are optional.
 *
 *      Also by calling this function, you allow Vertex a chance to
 *      manage its own resources while the toplevel thread of execution is
 *      still in your plug-in's routines.
 */
extern void VPIProgressDone(vpi_id *id, int editor_id);


/*
 *	Maps the file browser and prompts the user to select a file.
 *
 *	The returned pointer values should not be deallocated and are
 *	only valid on the same control level as the calling function.
 *
 *	Returns VPI_TRUE if a file was selected or VPI_FALSE if
 *	user canceled/response not available.
 */
extern int VPIFileSelect(
        vpi_id *id, int editor_id,
        const char *title,		/* File browser title. */
        const char *ok_label,		/* OK button label. */
	const char *cancel_label,	/* Cancel button label. */
        const char *path,		/* Initial path, can be NULL. */
        vpi_ftype_struct **ftype, int total_ftypes,	/* File types list. */
        char ***path_rtn, int *total_path_rtns,		/* Returned paths list. */
        vpi_ftype_struct **ftype_rtn	/* Selected file type return. */
);
/*
 *	Simplified version of VPIFileSelect().
 *
 *	Returned path must not be deallocated and is only valid on the
 *	same control level as the calling function. Can return NULL if
 *	user canceled or response not available.
 */
extern char *VPIFileSelectSimple(
        vpi_id *id, int editor_id,
        const char *path                /* Initial path, can be NULL. */
);


/*
 *	Returns the pointer to the editor's toplevel GtkWidget, the
 *	returned pointed will be a GtkWindow *.
 */
extern void *VPIGetEditorToplevel(vpi_id *id, int editor_id);


/* ***********************************************************************
 *                  VALUE GET/FETCH ABSTRACTION LAYER
 * ******************************************************************** */

/*	For fetching global values from Vertex. The input code specifies
 *	which value to return. The calling function must know the 
 *	proper code to pass to the right function so the correct data
 *	type can be returned. For example if fetching for a string
 *	value, you must call VPIGetS() or VPIGetP(). Fetching a string
 *	with VPIGetI() will fail even though the code is valid. You
 *	can fetch a int value with VPIGetF() or VPIGetD() and vice versa.
 *
 *	All pointer returns point to read-only memory and should not
 *	be modified.
 *
 *	Valid inputs for code can be any of VPI_CODE_* as shown below.
 *
 *      Returns VPI_SUCCESS on success or VPI_ERROR_* on failure.
 */
extern int VPIGetP(vpi_id *id, vpi_code_t code, void **p);
extern int VPIGetS(vpi_id *id, vpi_code_t code, char **s);
extern int VPIGetI(vpi_id *id, vpi_code_t code, int *i);
extern int VPIGetUI(vpi_id *id, vpi_code_t code, unsigned int *ui);
extern int VPIGetL(vpi_id *id, vpi_code_t code, long *l);
extern int VPIGetUL(vpi_id *id, vpi_code_t code, unsigned long *ul);
extern int VPIGetF(vpi_id *id, vpi_code_t code, float *f);
extern int VPIGetD(vpi_id *id, vpi_code_t code, double *d);

/* Codes for VPIGet*():
 *
 *	Code defination				Code	Data return type
 */
#define VPI_CODE_LANGUAGE			0x0100	/* char * */
#define VPI_CODE_VERSION_MAJOR			0x0101	/* int */
#define VPI_CODE_VERSION_MINOR			0x0102	/* int */
#define VPI_CODE_VERSION_RELEASE		0x0103	/* int */
#define VPI_CODE_COMPILE_TIME			0x0108	/* unsigned long */
#define VPI_CODE_COMPILE_LOCATION		0x0109	/* char * */
#define VPI_CODE_COMPILE_USER			0x010a	/* char * */
#define VPI_CODE_COMPILE_COMPILER		0x010b	/* char * */
#define VPI_CODE_VENDOR_NAME			0x010e	/* char * */

#define VPI_CODE_MAX_LIGHTS			0x0208	/* int */
#define VPI_CODE_TOTAL_EDITORS			0x0210	/* int */
#define VPI_CODE_TOTAL_PLUGINS			0x0260	/* int */

#define VPI_CODE_TOOLBAR_STYLE			0x0300	/* int
							 *	2 = icon + label
							 *	1 = icon only
							 *	0 = label only
							 */
#define VPI_CODE_SHOW_TOOLTIPS			0x0301	/* int */
#define VPI_CODE_RECORD_LAST_POS_AND_SIZE	0x0302	/* int */
#define VPI_CODE_SHOW_TIPOFDAY			0x0303	/* int */

#define VPI_CODE_GLOBAL_DIR			0x0330	/* char * */
#define VPI_CODE_LOCAL_DIR			0x0331	/* char * */
#define VPI_CODE_TMP_DIR			0x0340	/* char * */
#define VPI_CODE_FONT_NAME_TEXT_EDITABLE	0x0480	/* char * */
#define VPI_CODE_FONT_NAME_TEXT_TERMINAL	0x0481	/* char * */

/* Codes from 0x1000 and up are for external resources. */
#define VPI_CODE_X_DISPLAY			0x1000	/* void * */
#define VPI_CODE_IMLIB_HANDLE			0x1001	/* void * */
#define VPI_CODE_IMLIB2_HANDLE			0x1002	/* void * */


/* ***********************************************************************
 *                      OPERATION SUPPORT QUERYING
 * ******************************************************************** */

/*	Checks if Vertex supports a certain plug-in operation by returning
 *	VPI_TRUE if it is supported or VPI_FALSE if it is not supported.
 *
 *	The given id can be NULL, however if it is some returns may
 *	be inaccurate.
 *
 *	The given op_code can be any of VPI_OP_* listed below.
 *
 *	Note that VPIQuery is for checking if an operation or capability
 *	is supported, it is not used to fetch values (see VPIGet*() for
 *	value fetching).
 */
extern int VPIQuery(vpi_id *id, int op_code);

/* Following are op_codes supported by VPIQuery(). */
/* Standard operations. */
#define VPI_OP_NULL			0
#define VPI_OP_IMPORT			10
#define VPI_OP_EXPORT			11
#define VPI_OP_RENDER			20
#define VPI_OP_CONFIGURE		30
#define VPI_OP_CONFIGURATION_CHANGED	31

/* Supported capabilities. */
#define VPI_SUPPORT_GTK		1000	/* GTK+ (must also #include <gtk/gtk.h>). */
#define VPI_SUPPORT_GTK_GLAREA	1001	/* GtkGLArea widget. */
#define VPI_SUPPORT_GDK_RGB_BUFFERS	1010	/* GDK rgb buffers. */


/* ***********************************************************************
 *                      OPERATION CALLBACK SETTING
 * ******************************************************************** */

/*	Tells Vertex which functions exist on the plug-in, and thus
 *	what this plug-in is capable of doing. Any callback function 
 *	pointer that is not NULL implies that the plug-in supports it.
 *
 *	These should be set when Vertex calls your plug-in initialize
 *	function VPIInit().
 *
 *	Each plug-in operation callback consists of the following:
 *
 *	1.	A structure named by vpi_op_*_struct where * is the name
 *		of the operation, see the list of typedef'ed structures
 *		below.
 *
 *	2.	A callback function on the plug-in's side that is 
 *		prototyped:
 *
 *		int (*func_cb)(vpi_id *, vpi_op_*_struct *, void *)
 *
 *		For example:
 *
 *		int MyExportCB(
 *			vpi_id *id, vpi_op_export_struct *v, void *client_data
 *		)
 *
 *		A return value of VPI_TRUE means success and VPI_FALSE
 *		means failure.
 */

/*
 *	Null operation callback value structure:
 */
typedef struct {

	/* Defines which members of this structure are set. */
	vpi_flag flags;

} vpi_op_null_struct;

/*
 *      Import operation callback value structure:
 */
typedef struct {

	/* Defines which members of this structure are set. */
#define VPI_OP_IMPORT_EDITOR_ID		(1 << 0)
#define VPI_OP_IMPORT_FILENAME		(1 << 1)
#define VPI_OP_IMPORT_HEADER		(1 << 2)
#define VPI_OP_IMPORT_MODELS		(1 << 3)
	vpi_flag flags;

	int editor_id;
	char *filename;

	/* Model file header items. */
	void **mh_item;
	int total_mh_items;

	/* Models. */
	v3d_model_struct **model;
	int total_models;

/* Need to add more members. */

} vpi_op_import_struct;

/*
 *	Export operation callback value structure:
 */
typedef struct {

	/* Defines which members of this structure are set. */
#define VPI_OP_EXPORT_EDITOR_ID		(1 << 0)
#define VPI_OP_EXPORT_FILENAME		(1 << 1)
#define VPI_OP_EXPORT_HEADER		(1 << 2)
#define VPI_OP_EXPORT_MODELS		(1 << 3)
	vpi_flag flags;

	int editor_id;
	char *filename;

	/* Model file header items. */
	void **mh_item;
        int total_mh_items;

        v3d_model_struct **model;
        int total_models;

/* Need to add more members. */

} vpi_op_export_struct;

/*
 *	Render operation callback value structure:
 */
typedef struct {

        /* Defines which members of this structure are set. */
#define VPI_OP_RENDER_EDITOR_ID		(1 << 0)
#define VPI_OP_RENDER_HEADER		(1 << 1)
#define VPI_OP_RENDER_MODELS		(1 << 2)
#define VPI_OP_RENDER_CAMERA		(1 << 3)
#define VPI_OP_RENDER_LIGHT		(1 << 4)
#define VPI_OP_RENDER_CULL_FACE		(1 << 5)
#define VPI_OP_RENDER_CULL_DIR		(1 << 6)
        vpi_flag flags;

        int editor_id;

        /* Model file header items. */
        void **mh_item;
        int total_mh_items;

	/* Models. */
        v3d_model_struct **model;
        int total_models;

	/* Camera. */
	vpi_camera_struct camera;

	/* Lights. */
	vpi_light_struct *light;
        int total_lights;

	/* Cull face (cull face is on if this value is true). */
	int cull_face;

	/* Cull direction (winding). */
#define VPI_CULL_DIRECTION_CW	0
#define VPI_CULL_DIRECTION_CCW	1
	int cull_dir;

} vpi_op_render_struct;

/*
 *	Configure operation callback value structure:
 */
typedef struct {

        /* Defines which members of this structure are set. */
        vpi_flag flags;

} vpi_op_configure_struct;

/*
 *	Configuration changed operation callback value structure:
 */
typedef struct {

        /* Defines which members of this structure are set. */
        vpi_flag flags;

} vpi_op_configuration_changed_struct;


/*
 *      Sets the callback function func_ptr for the operation specified
 *      by op_code.
 *
 *      The func_ptr must point to a function defined on the plug-in
 *      and prototyped as follows:
 *
 *      int (*func_cb)(vpi_id *, vpi_op_*_struct *, void *)
 *
 *      The first input is the given id, the second is one of the
 *      vpi_op_*_struct structure pointers, and the third argument
 *      is the given client_data.
 *
 *      Returns VPI_SUCCESS on success or VPI_ERROR_* on failure.
 */
extern int VPISetOperationCB(
	vpi_id *id,
	int op_code,			/* One of VPI_OP_*. */
	void *func_ptr,			/* Can be NULL to disable. */
	void *client_data
);

/*
 *	Unsets the operation callback, any previous function callback set
 *	by VPISetOperationCB() will no longer be referenced.
 */
extern void VPIUnsetOperationCB(
	vpi_id *id,
	int op_code			/* One of VPI_OP_*. */
);


/*
 *	Extended operation features: Import
 *
 *      Returns VPI_SUCCESS on success or VPI_ERROR_* on failure.
 */
extern int VPIImportSetFileType(vpi_id *id, vpi_ftype_struct *ftype);
extern void VPIImportClearFileTypes(vpi_id *id);
extern int VPIImportSetFCheckFunc(
        vpi_id *id,
        int (*import_fcheck)(
		vpi_id *,
		const char *,		/* Full path. */
		const char *,		/* Just the extension, ie ".txt". */
		void *			/* Client data. */
	),
        void *client_data
);

/*
 *      Extended operation features: Export
 *
 *      Returns VPI_SUCCESS on success or VPI_ERROR_* on failure.
 */
extern int VPIExportSetFileType(vpi_id *id, vpi_ftype_struct *ftype);
extern void VPIExportClearFileTypes(vpi_id *id);

/*
 *      Extended operation features: Render
 */
extern void VPIRenderSetLabel(
        vpi_id *id,
        const char *label	/* What will appear on the render menu as. */
);



/* ***********************************************************************
 *                      VERTEX PLUG-IN APPEARANCE IO
 * ******************************************************************** */

/*	Appearance is used for displaynig of icons and other asthetic
 *	attributes of the plug-in on Vertex.
 */
typedef struct {

	/* Defines which members of this structure are set. */
#define VPI_APPEARANCE_FLAG_TITLE		(1 << 0)
#define VPI_APPEARANCE_FLAG_DESCRIPTION		(1 << 1)
#define VPI_APPEARANCE_FLAG_ICON_STD		(1 << 2)
#define VPI_APPEARANCE_FLAG_ICON_LIST_STD	(1 << 3)
#define VPI_APPEARANCE_FLAG_ICON_LIST_DISABLED	(1 << 4)
	vpi_flag flags;

	/* Title/non-conical name of plug-in. */
	char *title;

	/* Friendly description of plug-in. */
	char *description;

	/* Note icon data pointers point to multiple lines of string
	 * that are of XPM format. Pointers will NOT be deallocated by
	 * VPIAppearanceDelete(), it is up to the calling function
	 * to deallocate (if needed).
	 */
	vpi_icon_data **icon_std;
	vpi_icon_data **icon_list_std;
	vpi_icon_data **icon_list_disabled;

} vpi_appearance_struct;

extern vpi_appearance_struct *VPIAppearanceNew(void);
extern void VPIAppearanceDelete(vpi_appearance_struct *v);
extern void VPIAppearanceSet(vpi_id *id, const vpi_appearance_struct *v);
extern vpi_appearance_struct *VPIAppearanceGet(vpi_id *id);



/* ***********************************************************************
 *                            DISK OBJECT IO
 * ******************************************************************** */

/*
 *	These functions work just like the POSIX stat() and SVr4 lstat()
 *	functions.
 */
extern int VPIStat(vpi_id *id, const char *path, struct stat *stat_buf);
extern int VPILStat(vpi_id *id, const char *path, struct stat *stat_buf);

/*	File IO, same as their ANSI - C and POSIX.1 counter-parts.
 *
 *	By using these functions instead, Vertex will be aware of your file
 *	activities. You don't have to use these if you know that system
 *	file IO capabilities are available on the platform.
 *
 *	You can mix and match these calls with their ANSI - C counter-parts,
 *	if they are availble with one exception: if you open a file with
 *	VPIFOpen() then it MUST be closed with VPIFClose().
 */
extern FILE *VPIFOpen(vpi_id *id, const char *path, const char *mode);
extern int VPIFGetC(vpi_id *id, FILE *fp);
extern char *VPIFGetS(vpi_id *id, char *s, int size, FILE *fp);
extern long VPIFRead(vpi_id *id, void *ptr, long size, long nmemb, FILE *fp);
extern int VPIFPutC(vpi_id *id, int c, FILE *fp);
extern int VPIFPutS(vpi_id *id, const char *s, FILE *fp);
extern long VPIFWrite(vpi_id *id, const void *ptr, long size, long nmemb, FILE *fp);
extern int VPIFSeek(vpi_id *id, FILE *fp, long offset, int whence);
extern long VPIFTell(vpi_id *id, FILE *fp);
extern void VPIRewind(vpi_id *id, FILE *fp);
extern void VPIFClose(vpi_id *id, FILE *fp);

/* Seeks to next line, escape sequences will be parsed. */
extern void VPIFSeekNextLine(vpi_id *id, FILE *fp);

/* Seeks fp past any blank characters (spaces and tabs). */
extern void VPIFSeekPastSpaces(vpi_id *id, FILE *fp);

/* Seeks fp past the first occurance of c or EOF. */
extern void VPIFSeekPastChar(vpi_id *id, FILE *fp, char c);

/* Returns a dynamically allocated string containing
 * the value as a string obtained from the file specified
 * by fp. Reads from the current position to the next new   
 * line character or EOF.
 * Escape sequences will be parsed and spaces will be stripped.
 * The fp is positioned after the new line or at the EOF.
 */
extern char *VPIFGetString(vpi_id *id, FILE *fp);

/* Works just like VPIFGetString() except the string is loaded
 * literally and the only escape sequence to be handled will
 * be the two characters '\\' '\n', if/when those two characters
 * are encountered the character '\n' will be saved into the return
 * string.
 * Spaces will not be striped, the fp will be positioned after the
 * newline or EOF (whichever is encountered first).
 */
extern char *VPIFGetStringLined(vpi_id *id, FILE *fp);

/* Works just like VPIFGetString() except the string is loaded
 * literally and no escape sequences parsed, that would be all
 * characters from the current given fp position to the first
 * encountered newline ('\n') character (escaped or not).
 * Spaces will not be striped, the fp will be positioned after the
 * newline or EOF (whichever is encountered first).
 */
extern char *VPIFGetStringLiteral(vpi_id *id, FILE *fp);



/* ***********************************************************************
 *                     V3D MODEL DATA & EDITOR IO
 * ******************************************************************** */

/*	These functions return pointers to shared resources (on the 
 *	specified editor when applicatable), they should never be modified
 *	or deallocated. They may also become invalid after the control
 *	of the calling function(s) when they return to Vertex's level of
 *	control.
 *
 *	Note that these pointers are no longer valid when the plug-in
 *	function that made this call reaches end of its execution.
 *
 *      Returns VPI_SUCCESS on success or VPI_ERROR_* on failure.
 */
extern int VPIEditorGetHeader(
	vpi_id *id, int editor_id,
	const char **filename,		/* Loaded file name, can return NULL. */
	void ***mh_item, int *total_mh_items
);
extern int VPIEditorGetModels(
	vpi_id *id, int editor_id,
	v3d_model_struct ***model, int *total_models
);
extern int VPIEditorGetCamera(
        vpi_id *id, int editor_id, vpi_camera_struct **cam
);
/* Can return VPI_ERROR_NOT_FOUND even if the light number is valid but
 * that light is not enabled. */
extern int VPIEditorGetLight(
        vpi_id *id, int editor_id,
	vpi_light_struct **light,	/* Pointer to light return. */
	int light_num			/* Get values from this light. */
);
extern int VPIEditorGetLights(
        vpi_id *id, int editor_id,
	vpi_light_struct **light,	/* Pointer to array of lights return. */
	int *total_lights		/* Total number of lights return. */
);
extern v3d_model_struct *VPIEditorGetModelByNumber(
	vpi_id *id, int editor_id, int model_num
);


/*
 *	Creates a new model at the specified position on the editor with
 *	the given type and name, returns the model pointer or NULL on
 *	failure.
 */
extern v3d_model_struct *VPIEditorModelNew(
        vpi_id *id, int editor_id,
        int create_position,    /* VPI_POSITION_*. */
        int *model_num,
        int type, const char *name
);
/*
 *	Deletes the specified model on the editor.
 */
extern void VPIEditorModelDelete(
        vpi_id *id, int editor_id,
        int model_num
);

/*
 *	Instructs the editor to (re)select the specified model if it is
 *	valid.  If model_num is -1 then the model is unselected.
 */
extern void VPIEditorModelSelect(
        vpi_id *id, int editor_id, int model_num
);
/*
 *	Returns the current selected model on the editor or -1.
 */
extern v3d_model_struct *VPIEditorModelGetSelected(
        vpi_id *id, int editor_id, int *model_num
);

/*
 *      Instructs the editor to (re)select the specified primitive if it
 *	is valid.  If pn is -1 then all primitives will be unselected.
 */
extern void VPIEditorPrimitiveSelect(
        vpi_id *id, int editor_id, int pn
);
/*
 *	Returns the curret selected primitives on the editor or NULL.
 *	The returned pointer must be free'ed by the calling function.
 */
extern int *VPIEditorPrimitiveGetSelected(
        vpi_id *id, int editor_id, int *total
);


/*
 *      Redraw the editor.
 *
 *      If refresh_lists is VPI_TRUE then the lists will be refreshed.
 */
extern void VPIEditorRedraw(vpi_id *id, int editor_id, int refresh_lists);

/*
 *      Returns VPI_TRUE if the editor's has_changes flag is set,
 *      otherwise returns VPI_FALSE.
 */
extern int VPIEditorGetHasChanges(vpi_id *id, int editor_id);

/*
 *      Sets the has_changes flag on the editor.
 */
extern void VPIEditorSetHasChanges(vpi_id *id, int editor_id, int has_changes);



/* ***********************************************************************
 *                           MISC FUNCTIONS
 * ******************************************************************** */

/*	In cases where both the VPI and OS version of a function is 
 *	available (ie Vertex's VPIMove() and ANSI C's rename()), it is
 *	slighly preffered to call Vertex's version so that Vertex is given
 *	a chance to be notified internally about the call.
 */

/*
 *	Returns a statically allocated string describing the given
 *	error (or success) code.
 */
extern const char *VPIErrorString(vpi_id *id, int error_code);

/*
 *      Creates a new tempory file and returns the FILE pointer opened
 *      for writing with the stream positioned at the beginning.
 *
 *      The name_prefix specifies the prefix for the name of the new
 *      tempory file, the limits on the prefix are defined by the
 *      platform's tempnam() function.
 * 
 *      The new file will be placed in Vertex's tempory files directory.
 *
 *      If (and only if) the return path new_name is not NULL then the
 *      path to the new file name will be returned. The returned value is
 *      dynamically allocated so the calling function must deallocate it.
 */
extern FILE *VPICreateTemporyFile(
        vpi_id *id,
        const char *name_prefix,
        char **new_path_rtn
);

/*
 *	Moves the object specified by old_path to the location specified
 *	by new_path. This works the same as the ANSI C rename(), so it
 *	can be used to rename objects as well.
 */
extern int VPIMove(vpi_id *id, const char *old_path, const char *new_path);
/* Alias VPIRename() that reffers to VPIMove(). */
#define VPIRename	VPIMove

/*
 *      Coppies the object specified by src_path to the location
 *	specified by tar_path.
 *
 *      The src_path and/or tar_path cannot reffer to a directory or a
 *      link that points to a directory.
 *
 *	If the target object exists then it will be atomitically
 *	overwritten.
 *
 *      Returns 0 on success or -1 on error.
 */
extern int VPICopy(vpi_id *id, const char *src_path, const char *tar_path);

/*
 *	Creates a new directory specified by the given path.
 *
 *      Returns VPI_SUCCESS on success or VPI_ERROR_* on failure (can
 *	return VPI_ERROR if the directory already exists).
 */
extern int VPIMkDir(
	vpi_id *id, const char *path, mode_t m, int make_parents_as_needed
);

/*
 *	Same as POSIX rmdir().
 */
extern int VPIRmDir(vpi_id *id, const char *path);

/*
 *	Returns a dynamically allocated array with pointers each
 *	pointing to dynamically allocated strings listing the directory
 *	contents of the given path. The returned array and strings must
 *	be deallocated by the calling function.
 *
 *	If nentries is not NULL then the number of entries in the
 *	returned array of strings will be set.
 *
 *	Can return NULL on error.
 */
extern char **VPIDirEnts(
	vpi_id *id, const char *path, int *nentries_rtn
);

/*
 *	Same as POSIX.1 unlink().
 */
extern int VPIUnlink(vpi_id *id, const char *path);

/*
 *      Same as POSIX symlink().
 */
extern int VPISymLink(vpi_id *id, const char *value, const char *path);

/*
 *      Same as POSIX link().
 */
extern int VPILink(vpi_id *id, const char *value, const char *path);

/*
 *      Returns a pointer to the value of the given link specified
 *      by path. The returned pointer may not be deallocated or
 *	modified.
 *
 *      Can return NULL on error.
 */
extern char *VPIGetLink(vpi_id *id, const char *path);


/*
 *	Executes the given command.
 *
 *	It does not use system(), it uses exec() for security reasons.
 *	If block is VPI_TRUE then the execution will block untill the
 *	process has exited, otherwise it returns immediatly.
 *
 *	For VPIExecO() and VPIExecOA(), the optional stdout_file and
 *	stderr_file paths can be given. The VPIExecO() will overwrite
 *	any existing output file while VPIExecOA() will appened to any
 *	existing file. Output files are immediatly available for opening
 *      on return.
 *
 *	Returns a pid on success or 0 on failure.
 */
extern int VPIExec(
	vpi_id *id, const char *command, int block
);
extern int VPIExecO(
	vpi_id *id, const char *command, int block,
	const char *stdout_file, const char *stderr_file
);
extern int VPIExecAO(
	vpi_id *id, const char *command, int block,
	const char *stdout_file, const char *stderr_file
);


/*
 *	Checks if the given pid is currently running.
 *
 *	Returns VPI_TRUE if the given pid is running or VPI_FALSE if it
 *	is no longer running.
 */
extern int VPIIsRunning(vpi_id *id, int pid);

/*
 *	Same as POSIX.1 kill(), input for s can be any signal defined
 *	in signal.h.
 */
extern int VPIKill(vpi_id *id, int pid, int s);



#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif	/* VPI_H */
