/*
 * ===========================
 * VDK Visual Development Kit
 * Version 2.0.0
 * February 2001
 * ===========================
 *
 * Copyright (C) 1998,199,2000,2001 Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
#include <vdk/vdkeditor.h>
#include <vdk/colors.h>
#include <cstdio>
#include <cstring>
#include <sys/stat.h>
static Tipwin* tip_window = NULL;
static char buff[1024];
static char floating_token[256];

/*
*/
VDKEditor::VDKEditor(VDKForm* owner, GtkSourceBuffer* buff):
  VDKObject(owner),
  Syntax("Syntax",this,true,&VDKEditor::SetSyntax),
  Pointer("Pointer",this,0,&VDKEditor::SetPointer,
	    &VDKEditor::GetPointer),
  Column("Column",this,0,&VDKEditor::SetColumn,
	   &VDKEditor::GetColumn),
  Line("Line",this,0,&VDKEditor::SetLine,
	 &VDKEditor::GetLine),
  Length("Length",this,0,&VDKEditor::GetLength),
  Editable("Editable",this,true,&VDKEditor::SetEditable,
	     &VDKEditor::GetEditable),
 TabStop("TabStop",this,5,&VDKEditor::SetTabStop,
	     &VDKEditor::GetTabStop),
  MaxUndo("MaxUndo",this,5),
  LineAutoSelect("LineAutoSelect",this,false), // dummy
  ShowLineNumbers("ShowLineNumbers",this,false,&VDKEditor::SetShowLineNumbers),
  FirstVisibleLine("FirstVisibleLine",this,0,&VDKEditor::GetFirstVisibleLine),
  LastVisibleLine("LastVisibleLine",this,0,&VDKEditor::GetLastVisibleLine),
  Changed("Changed",this,false)// ,&VDKEditor::SetChanged,&VDKEditor::GetChanged)

{
  // GtkTextTagTable *table;
  // initializes buffer
  if(!buff)
    {
      //table = gtk_text_tag_table_new();  
      buffer = GTK_SOURCE_BUFFER(gtk_source_buffer_new(NULL));
      //      g_object_ref(table);
      // GTK_TEXT_BUFFER(buffer)->tag_table = table;
      
    }
  else
    buffer = buff;
  widget = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (widget),
  				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);

  view = gtk_source_view_new_with_buffer (GTK_SOURCE_BUFFER(buffer));
  sigwid = view;
  gtk_container_add (GTK_CONTAINER (widget), view);
  gtk_widget_show(view);
  ConnectDefaultSignals();
  LocalConnect();
 }

void
VDKEditor::HandleRealize(GtkWidget* widget, gpointer gp)
{
  VDKEditor* editor = reinterpret_cast<VDKEditor*>(gp);
  if(editor)
    {
      editor->SignalEmit("realize");
      g_signal_connect_closure(G_OBJECT(editor->buffer),"changed",
			       g_cclosure_new( G_CALLBACK(VDKEditor::OnBufferChanged),
					       editor,NULL), FALSE);
    }
}

/*
 */
VDKEditor::~VDKEditor()
{

}
/*

*/
void 
VDKEditor::SetFont(VDKFont* font)
{
  VDKObject::_setFont_(sigwid,font);
}
/*
*/
void 
VDKEditor::SetBackground(VDKRgb rgb, GtkStateType state)
{
  VDKColor *color = new VDKColor(Owner(),rgb.red,rgb.green,rgb.blue);
  /*
  GdkWindow* win = gtk_text_view_get_window (GTK_TEXT_VIEW(view),GTK_TEXT_WINDOW_TEXT);
  if(win)
    {
      gdk_window_set_background (win,color->Color());
    }
  */
  gtk_widget_modify_base (GTK_WIDGET(view),state, color->Color());
}
/*
*/
void 
VDKEditor::SetForeground(VDKRgb rgb, GtkStateType state)
{
  VDKColor *color = new VDKColor(Owner(),rgb.red,rgb.green,rgb.blue);
  gtk_widget_modify_text (GTK_WIDGET(view), state, color->Color());
}
/*
*/
bool 
VDKEditor::LoadFromFile(const char* filename)
{
  GError *err = NULL;
  Clear();
  if(gtk_source_buffer_load (GTK_SOURCE_BUFFER(buffer),filename,&err))
    {
      Changed = false;
      return true;
    }
  else
    return false;
  
}

/*
 */
void SyntaxTableForEach(GtkTextTag *tag, gpointer data)
{
   VDKList<GtkTextTag> *list = reinterpret_cast<VDKList<GtkTextTag>* >(data);
   list->add(tag);
  
}

void
VDKEditor::ClearSyntaxTable()
{
  /*
  GtkTextTagTable* new_table;
  new_table = gtk_text_tag_table_new();
  if(GTK_TEXT_BUFFER(buffer)->tag_table)
    g_object_unref(GTK_TEXT_BUFFER(buffer)->tag_table);
  GTK_TEXT_BUFFER(buffer)->tag_table = new_table;
  g_object_ref(GTK_TEXT_BUFFER(buffer)->tag_table);
  */
  /*
  VDKList<GtkTextTag> list;
  GtkTextTagTable* table = gtk_text_buffer_get_tag_table (GTK_TEXT_BUFFER(buffer));
  if(table)
    {
      gtk_text_tag_table_foreach  (table,SyntaxTableForEach,&list);
      if(list.size())
	{
	  VDKListIterator<GtkTextTag> li(list);
	  for(;li;li++)
	    gtk_text_tag_table_remove (table,li.current());
	}
    }
  */
}
 
/*
*/
void 
VDKEditor::InstallSyntaxTable (VDKColor *key_color,
			       VDKFont  *key_font,
			       VDKColor *gtk_color,
   			       VDKFont  *gtk_font,
			       VDKColor *macro_color,
   			       VDKFont  *macro_font,
			       VDKColor *pp_color,
   			       VDKFont  *pp_font,
			       VDKColor *const_color,
   			       VDKFont  *const_font,
			       VDKColor *comment_color,
   			       VDKFont  *comment_font)
{

  GtkTextTag* tag;
  GtkTextTagTable* table = gtk_text_buffer_get_tag_table (GTK_TEXT_BUFFER(buffer));
  GList *list = NULL;
  // keywords tags
  if(key_color)
    {
      tag = gtk_text_tag_table_lookup  (table,"keywords");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", key_color->Color(),
		     key_font ? "font_desc" : NULL,
		     key_font ? key_font->AsPangoFontDescription(): NULL,
		     NULL);
      else
	{
	  tag = gtk_pattern_tag_new("keywords",
  "\\b\\(do\\|while\\|for\\|if\\|else\\|bool\\|template\\|continue"
  "\\|case\\|switch\\|break\\|new\\|delete\\|default\\|return"
  "\\|char\\|short\\|int\\|long\\|float\\|double\\|unsigned\\|true\\|false"
  "\\|this\\|void\\|struct\\|union\\|enum\\|const\\|static\\|typedef\\|virtual"
  "\\|class\\|public\\|protected\\|private\\|friend\\|dynamic_cast\\|reinterpret_cast"
  "\\|DEFINE_SIGNAL_MAP\\|ON_SIGNAL\\|END_SIGNAL_MAP\\|DECLARE_SIGNAL_MAP"
  "\\|DECLARE_SIGNAL_LIST\\|DEFINE_SIGNAL_LIST"
  "\\|DECLARE_EVENT_LIST\\|DEFINE_EVENT_LIST"
  "\\|SignalConnect\\|EventConnect\\)\\b");
	  g_object_set(G_OBJECT (tag),
                 "foreground_gdk", key_color->Color(),
                 key_font ? "font_desc" : NULL,
                 key_font ? key_font->AsPangoFontDescription(): NULL,
                 NULL);
	  // gtk_text_tag_table_add (table, tag);
	  list = g_list_append (list, (gpointer) tag);
	}

      tag = gtk_text_tag_table_lookup  (table,"C++-funcs");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", key_color->Color(),
		     key_font ? "font_desc" : NULL,
		     key_font ? key_font->AsPangoFontDescription(): NULL,
		     NULL);
      else
	{
	  tag = gtk_pattern_tag_new("C++-funcs",
				    "^\\(\\([a-zA-Z_][a-zA-Z0-9_]*\\)::\\([~a-zA-Z_][a-zA-Z0-9_]*\\)\\)[ \t]*(");
	  g_object_set(G_OBJECT (tag),
		       "foreground_gdk", key_color->Color(),
		       key_font ? "font_desc" : NULL,
		       key_font ? key_font->AsPangoFontDescription(): NULL,
		       NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	  list = g_list_append (list, (gpointer) tag);
	}
      }

  // gtk functions and members tag
  if(gtk_color)
    {
      tag = gtk_text_tag_table_lookup  (table,"gtk_functions");
      if(tag)
	g_object_set(G_OBJECT (tag),
                 "foreground_gdk", gtk_color->Color(),
                 gtk_font ? "font_desc" : NULL,
                 gtk_font ? gtk_font->AsPangoFontDescription(): NULL,
                 NULL);
      else
	{
	  tag = gtk_pattern_tag_new("gtk_functions",
				    "\\b\\(gtk\\|gdk\\|g\\|gnome\\)_[a-zA-Z0-9_]+");
	  g_object_set(G_OBJECT (tag),
		       "foreground_gdk", gtk_color->Color(),
		       gtk_font ? "font_desc" : NULL,
		       gtk_font ? gtk_font->AsPangoFontDescription(): NULL,
		       NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	list = g_list_append (list, (gpointer) tag);
	}

      tag = gtk_text_tag_table_lookup  (table,"gnu_typedef");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", gtk_color->Color(),
		     gtk_font ? "font_desc" : NULL,
		     gtk_font ? gtk_font->AsPangoFontDescription() : NULL,
		     NULL);
      else
	{
	  tag = gtk_pattern_tag_new("gnu_typedef","\\b\\(Gtk\\|Gdk\\|Gnome\\)[a-zA-Z0-9_]+");
	  g_object_set(G_OBJECT (tag),
		       "foreground_gdk", gtk_color->Color(),
		       gtk_font ? "font_desc" : NULL,
		       gtk_font ? gtk_font->AsPangoFontDescription() : NULL,
		       NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	list = g_list_append (list, (gpointer) tag);
	}

      tag = gtk_text_tag_table_lookup  (table,"vdk_functions");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", gtk_color->Color(),
		     gtk_font ? "font_desc" : NULL,
		     gtk_font ? gtk_font->AsPangoFontDescription(): NULL,
		     NULL);
      else
	{
	  tag = gtk_pattern_tag_new("vdk_functions", "\\b\\(VDK\\|vdk\\)[a-zA-Z0-9_]+");
	  g_object_set(G_OBJECT (tag),
		   "foreground_gdk", gtk_color->Color(),
		   gtk_font ? "font_desc" : NULL,
		   gtk_font ? gtk_font->AsPangoFontDescription(): NULL,
		   NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	list = g_list_append (list, (gpointer) tag);
	}
      }
  // macro tags
  if(macro_color)
    {
      tag = gtk_text_tag_table_lookup  (table,"macro");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", macro_color->Color(),
		     macro_font ? "font_desc" : NULL,
		     macro_font ? macro_font->AsPangoFontDescription() : NULL,
		     NULL);
      else
	{
	  tag = gtk_pattern_tag_new("macro","\\b[A-Z_][A-Z0-9_\\-]+\\b");
	  g_object_set(G_OBJECT (tag),
		       "foreground_gdk", macro_color->Color(),
		       macro_font ? "font_desc" : NULL,
		       macro_font ? macro_font->AsPangoFontDescription() : NULL,
		       NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	list = g_list_append (list, (gpointer) tag);
	}
    }

  // preprocessor directives tags
  if(pp_color)
    {
      tag = gtk_text_tag_table_lookup  (table,"defs");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", pp_color->Color(),
		     pp_font ? "font_desc" : NULL,
		     pp_font ? pp_font->AsPangoFontDescription(): NULL,
		     NULL);
      else
	{
	  tag = gtk_pattern_tag_new("defs",
			"^#[ \t]*\\(include\\|if\\|ifdef\\|ifndef\\|else\\|elif\\|define\\|endif\\|pragma\\)\\b");
	  g_object_set(G_OBJECT (tag),
		       "foreground_gdk", pp_color->Color(),
		       pp_font ? "font_desc" : NULL,
		       pp_font ? pp_font->AsPangoFontDescription(): NULL,
		       NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	list = g_list_append (list, (gpointer) tag);
	}
    }
  // constants tags
  if(const_color)
    {
      tag = gtk_text_tag_table_lookup  (table,"char_string");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", const_color->Color(),
		     const_font ? "font_desc" : NULL,
		     const_font ? const_font->AsPangoFontDescription(): NULL,
		     NULL);
      else
	{
	  tag = gtk_pattern_tag_new("char_string","'\\?[a-zA-Z0-9_\\()#@!$%^&*-=+\"{}<)]'");
	  g_object_set(G_OBJECT (tag),
		       "foreground_gdk", const_color->Color(),
		       const_font ? "font_desc" : NULL,
		       const_font ? const_font->AsPangoFontDescription(): NULL,
		       NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	list = g_list_append (list, (gpointer) tag);
	}
      
      tag = gtk_text_tag_table_lookup  (table,"numbers");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", const_color->Color(),
		     const_font ? "font_desc" : NULL,
		     const_font ? const_font->AsPangoFontDescription(): NULL,
		     NULL);
      else
	{
	  tag = gtk_pattern_tag_new("numbers","\\b[0-9]+\\.?\\b");
	  g_object_set(G_OBJECT (tag),
		       "foreground_gdk", const_color->Color(),
		       const_font ? "font_desc" : NULL,
		       const_font ? const_font->AsPangoFontDescription(): NULL,
		       NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	list = g_list_append (list, (gpointer) tag);
	}

      tag = gtk_text_tag_table_lookup  (table,"string");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", const_color->Color(),
		     const_font ? "font_desc" : NULL,
		     const_font ? const_font->AsPangoFontDescription(): NULL,
		     NULL);
      else
	{
	  tag = gtk_syntax_tag_new("string","\"","\"");
	  g_object_set(G_OBJECT (tag),
		       "foreground_gdk", const_color->Color(),
		       const_font ? "font_desc" : NULL,
		       const_font ? const_font->AsPangoFontDescription(): NULL,
		       NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	list = g_list_append (list, (gpointer) tag);
	}
    }
  //
  if(comment_color)
    {
      tag = gtk_text_tag_table_lookup  (table,"description");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", comment_color->Color(),
		     comment_font ? "font_desc": NULL,
		     comment_font ? comment_font->AsPangoFontDescription(): NULL,
		     NULL);
      else
	{
	  tag = gtk_syntax_tag_new("description","//","\n");
	  g_object_set(G_OBJECT (tag),
		       "foreground_gdk", comment_color->Color(),
		       comment_font ? "font_desc": NULL,
		       comment_font ? comment_font->AsPangoFontDescription(): NULL,
		       NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	list = g_list_append (list, (gpointer) tag);
	}
      
      tag = gtk_text_tag_table_lookup  (table,"description_multiline");
      if(tag)
	g_object_set(G_OBJECT (tag),
		     "foreground_gdk", comment_color->Color(),
		     comment_font ? "font_desc": NULL,
		     comment_font ? comment_font->AsPangoFontDescription(): NULL,
		     NULL);
      else
	{
	  tag = gtk_syntax_tag_new("description_multiline","/\\*","\\*/");
	  g_object_set(G_OBJECT (tag),
		       "foreground_gdk", comment_color->Color(),
		       comment_font ? "font_desc": NULL,
		       comment_font ? comment_font->AsPangoFontDescription(): NULL,
		       NULL);
	  //	  gtk_text_tag_table_add (table, tag);
	list = g_list_append (list, (gpointer) tag);
	}
      }
  if(list)
    {
     	gtk_source_buffer_install_regex_tags (GTK_SOURCE_BUFFER(buffer), list);
	g_list_free (list);
    }
}
/*
*/
void 
VDKEditor::Scroll (int pointer, int margin)
{
  if(pointer >= 0)
    Pointer = pointer;
  GtkTextMark* mark = gtk_text_buffer_get_mark(GTK_TEXT_BUFFER(buffer),
                            INSERT_MARK);
  if(mark)
    // more args on new gtk+ snapshoots
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(view),mark,margin,0,0,0);
}
/*
*/
void 
VDKEditor::Scroll(int line, int col, int margin)
{
  Line = line;
  Column = col;
  GtkTextMark* mark = gtk_text_buffer_get_mark(GTK_TEXT_BUFFER(buffer),
                            INSERT_MARK);
  if(mark)
    // more args on new gtk+ snapshoots
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(view),mark,margin,0,0,0);
}
/*
*/
void
VDKEditor::Clear()
{
  GtkTextIter start, end;
  gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER(buffer), &start, &end);
  gtk_text_buffer_delete(GTK_TEXT_BUFFER(buffer),&start,&end);
}
/*
*/
gchar* 
VDKEditor::GetChars(int start, int end)
{
  GtkTextIter first,last;
  gtk_text_buffer_get_iter_at_offset(GTK_TEXT_BUFFER(buffer),&first,start);
  if(end >= 0)
    gtk_text_buffer_get_iter_at_offset(GTK_TEXT_BUFFER(buffer),&last,end);
  else
    gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER(buffer),&last);
  // returned address should be gfree()'d by user or it will leak.
  return gtk_text_buffer_get_text(GTK_TEXT_BUFFER(buffer),&first,&last,FALSE);
}
/*
*/
bool 
VDKEditor::SaveToFile(const char* filename)
{
  GError *err = NULL;
  if(gtk_source_buffer_save (GTK_SOURCE_BUFFER(buffer),filename,&err))
    {
      Changed = false;
      return true;
    }
  else
    return false;
  /*
  FILE* fp;
  fp = fopen(filename,"w+b");
  if(fp)
    {
      int result = 0;
      char* p;
      p = GetChars(0,-1);
      if(p)
	{
	  result = fwrite(p,std::strlen(p),1,fp);
	  g_free(p);
	}
      fclose(fp);
      Changed= false;
      return result == 1;
  }
  else 
    return false;
  */
}
/*
*/
void 
VDKEditor::TextInsert(const char* txt, int nchar)
{
  gtk_text_buffer_insert_interactive_at_cursor (GTK_TEXT_BUFFER(buffer),txt,nchar,
						gtk_text_view_get_editable(GTK_TEXT_VIEW(view)));
}
/*
*/
void  
VDKEditor::ForwardDelete(int nchars)
{
  GtkTextIter  start, end;
  GtkTextMark* mark = gtk_text_buffer_get_mark(GTK_TEXT_BUFFER(buffer),INSERT_MARK);
  if(mark)
    {
      int offset = (int) Pointer + nchars;
      int len = Length;
      gtk_text_buffer_get_iter_at_mark(GTK_TEXT_BUFFER(buffer),&start,mark);
      if(offset < len)
        gtk_text_buffer_get_iter_at_offset(GTK_TEXT_BUFFER(buffer),&end,offset);
      else
        gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER(buffer),&end);
      gtk_text_buffer_delete(GTK_TEXT_BUFFER(buffer),&start,&end);
    }
}

/*
*/
void  
VDKEditor::BackwardDelete(int nchars)
{
  GtkTextIter  start, end;
  GtkTextMark* mark = gtk_text_buffer_get_mark(GTK_TEXT_BUFFER(buffer),INSERT_MARK);
  if(mark)
  { 
      int offset = (int) Pointer - nchars;
      offset = offset < 0 ? 0 : offset;
      gtk_text_buffer_get_iter_at_mark(GTK_TEXT_BUFFER(buffer),&end,mark);
      gtk_text_buffer_get_iter_at_offset(GTK_TEXT_BUFFER(buffer),&start,offset);
      gtk_text_buffer_delete(GTK_TEXT_BUFFER(buffer),&start,&end);
  }
}
int 
VDKEditor::GetLineAtOffset(int offset)
{
  int line = -1;
  GtkTextIter  iter;
  gtk_text_buffer_get_iter_at_offset(GTK_TEXT_BUFFER(buffer),&iter,offset);
  line  = gtk_text_iter_get_line(&iter);
  return line; 
}

//
/*
  gets the word currently at <pos> position,
  if pos < 0 gets the word at insertion point.
  some known c/c++ delimiters are used.
  Note:
  returned char* should be g_free()'d by user.
*/

static const int SAFE = 64;
// some c/c++ tokens delimiters
const char* delimiters = " \n\t()[];{}*&!:.-><+=/#";
//
bool isADelimiter(const char d)
{
  const char* p = delimiters;
  for(; *p ; p++)
    if(*p == d)
      return true;
  return false;
}
char* 
VDKEditor::GetWord(int pos)
{
  char* word = NULL;
  GtkTextIter cursor, *istart,*iend;
  int start,end;
  gunichar car;
  int safe;
  if(pos < 0)
    pos = Pointer;
  gtk_text_buffer_get_iter_at_offset(GTK_TEXT_BUFFER(buffer),&cursor,pos);
  istart = gtk_text_iter_copy(&cursor);
  iend = gtk_text_iter_copy(&cursor);
  // move backward until a delimiter or a safe range
  safe = 0;
  do 
    {
      // next gtk+ snapshoot
       gtk_text_iter_backward_char(istart);
      //      gtk_text_iter_prev_char(istart);
      car = gtk_text_iter_get_char(istart);
      if(isADelimiter(car))
        break;
      else
          safe++;
    } while(safe < SAFE);
  if(isADelimiter(car))
      // next gtk+ snapshoot
    gtk_text_iter_forward_char(istart);
  // gtk_text_iter_next_char(istart);
  start = gtk_text_iter_get_offset(istart);
  gtk_text_iter_free(istart);
  // move forward until a delimiter or a safe range
  safe = 0;
  while(safe < SAFE)
    {
     car = gtk_text_iter_get_char(iend);
     if( isADelimiter(car))
       break;
     else
       {
         // next gtk+ snapshoot
          gtk_text_iter_forward_char(iend);
         // gtk_text_iter_next_char(iend);
         safe++;
       }
    }
  end = gtk_text_iter_get_offset(iend);
  gtk_text_iter_free(iend);
  if(start < end)
    word = GetChars(start,end);
  return word;
}
/*
  Overlaps src to tgt and return where overlaps ends.
  Overlap must be complete, returns NULL on failure
 */
static char* overlap(char* tgt, const char* src)
{
  int t = 0;
  unsigned int z = std::strlen(src);
  if(z > std::strlen(tgt))
    return (char*) NULL;
  else
    {
      for(;src[t] && (src[t] == tgt[t]);t++) 
        	;
      return (unsigned int) t == z ? &tgt[t] : (char*) NULL;
    }
}
/*
*/
bool
VDKEditor::MakeCompletion(const char* word)
{
  TokenList local;
  TokenListIterator li(*tokenlist);
  for(;li;li++)
    {
      char* token = (char*) li.current();
      if(overlap(token,word))
	  local.add(li.current());
    }
  // one token found
  if(local.size() == 1)
    {
      char* token = (char*) local[0];
      char* p;
      if((std::strlen(word) < std::strlen(token)) &&
	 ( p = overlap(token,word)))
	  TextInsert(p);
    }
  // more than one token found
  else if(local.size() > 1)
    {

      sprintf(buff,"%2d more words:\n",local.size());
      for(TokenListIterator li(local);li;li++)
	{
	  std::strcat(buff,(char*) li.current());
	  std::strcat(buff,"\n");
	}
      ShowTipWindow(buff);
    }
  // no tokens found, will be added to tokenlist
  // if user hits ctrl-a
  else
    {
      sprintf(buff,"\"%s\" isn't in word completion list.\n\nHit ctrl-a to add it",word);
      ShowTipWindow(buff);
      // copies word on floating buffer;
      std::strcpy(floating_token,word);
    }
  return local.size() > 0;
}
/*
  add a new token to token list
 */
void
VDKEditor::AddToken()
{
  if(!*floating_token)
    {
      sprintf(buff,"Nothing to add to completion list");
      ShowTipWindow(buff);
    }
  else
    {
      VDKString s(floating_token);
      if(!tokenlist->find(s))
	{
	  tokenlist->add(s);
	  sprintf(buff,"Word: \"%s\" added to completion list",floating_token);
	  ShowTipWindow(buff);
	}
      else
        // should be never the case :-)
	{
	  sprintf(buff,"%s already on completion list",floating_token);
	  ShowTipWindow(buff);
	}
      *floating_token = '\0';
    }
}

/*
 */
bool
VDKEditor::Undo(void)
{
  /*
  if(sigwid && (gtk_source_buffer_get_undo_count(buffer)) > 0) 
     {
       gtk_source_buffer_undo(buffer);
       return true;
     }
   else
  */
     return false;
}
// SETTING PROPERTIES
/*
*/
void
VDKEditor::SetSyntax(bool set)
{
  gtk_source_buffer_set_highlight(buffer, set);
}
/*
*/
void 
VDKEditor::SetPointer(int char_offset)
{
  GtkTextIter iter;
  gtk_text_buffer_get_iter_at_offset (GTK_TEXT_BUFFER(buffer),&iter,char_offset);
  gtk_text_buffer_place_cursor(GTK_TEXT_BUFFER(buffer),&iter);
}
/*
*/
int
VDKEditor::GetPointer()
{
  int offset = -1;
  GtkTextIter  iter;
  GtkTextMark* mark = gtk_text_buffer_get_mark(GTK_TEXT_BUFFER(buffer),INSERT_MARK);
  if(mark)
    {
      gtk_text_buffer_get_iter_at_mark(GTK_TEXT_BUFFER(buffer),&iter,mark);
      offset = gtk_text_iter_get_offset(&iter);
    }
  return offset;
}
/*
*/
int
VDKEditor::GetColumn()
{
  int lineoffset = -1;
  GtkTextIter  iter;
  GtkTextMark* mark = gtk_text_buffer_get_mark(GTK_TEXT_BUFFER(buffer),INSERT_MARK);
  if(mark)
    {
      gtk_text_buffer_get_iter_at_mark(GTK_TEXT_BUFFER(buffer),&iter,mark);
      lineoffset = gtk_text_iter_get_line_offset (&iter);
    }
  return lineoffset;
}
/*
*/
void 
VDKEditor::SetColumn(int col)
{
  GtkTextIter iter;
  gtk_text_buffer_get_iter_at_line_offset(GTK_TEXT_BUFFER(buffer),&iter,Line,col);
  gtk_text_buffer_place_cursor(GTK_TEXT_BUFFER(buffer),&iter);
}
/*
*/
void 
VDKEditor::SetLine(int row)
{
  GtkTextIter iter;
  gtk_text_buffer_get_iter_at_line(GTK_TEXT_BUFFER(buffer),&iter,row);
  gtk_text_buffer_place_cursor(GTK_TEXT_BUFFER(buffer),&iter);
}
/*
*/
int
VDKEditor::GetLine()
{
  int line = -1;
  GtkTextIter  iter;
  GtkTextMark* mark = gtk_text_buffer_get_mark(GTK_TEXT_BUFFER(buffer),INSERT_MARK);
  if(mark)
    {
      gtk_text_buffer_get_iter_at_mark(GTK_TEXT_BUFFER(buffer),&iter,mark);
      line  = gtk_text_iter_get_line(&iter);
    }
  return line;
}
/*
 */
bool 
VDKEditor::AddMarkIcon(VDKPixbuf* icon, const char * key, bool overwrite)
{
  if(!icon)
    return false;
  else
    return  gtk_source_view_add_pixbuf(GTK_SOURCE_VIEW(view), 
				       key, icon->AsGdkPixbuf() , 
				       overwrite);
}
/*
 */
void 
VDKEditor::AddLineMark(int line, const char* icon_key)
{
    if(!gtk_source_buffer_line_has_markers (GTK_SOURCE_BUFFER(buffer),line))
  	gtk_source_buffer_line_add_marker (GTK_SOURCE_BUFFER(buffer),line,icon_key); 
}
/*
 */
void 
VDKEditor::RemoveLineMark(int line, const char* icon_key)
{
    if(!gtk_source_buffer_line_has_markers (GTK_SOURCE_BUFFER(buffer),line))
  	gtk_source_buffer_line_remove_marker (GTK_SOURCE_BUFFER(buffer),line,icon_key); 
}
/*
 */
void 
VDKEditor::RemoveLineMarks(int line)
{
 gtk_source_buffer_line_remove_markers (GTK_SOURCE_BUFFER(buffer),line); 
}
/*
 */
void 
VDKEditor::RemoveAllLineMarks()
{
int end = gtk_text_buffer_get_line_count (GTK_TEXT_BUFFER (buffer));
gtk_source_buffer_remove_all_markers (GTK_SOURCE_BUFFER(buffer),-1,end);
gtk_widget_queue_draw(GTK_WIDGET(WrappedWidget()));  
}
/*
*/
void 
VDKEditor::SetTabStop(int tab_stop)
{
  gtk_source_view_set_tab_stop(GTK_SOURCE_VIEW(view), tab_stop);
}
/*
*/
int
VDKEditor::GetTabStop()
{
  return gtk_source_view_get_tab_stop(GTK_SOURCE_VIEW(view));
}
/*
*/
unsigned int 
VDKEditor::GetLength() 
{
  return gtk_text_buffer_get_char_count(GTK_TEXT_BUFFER(buffer));
}
/*
*/
bool 
VDKEditor::GetEditable() 
{ 
return gtk_text_view_get_editable (GTK_TEXT_VIEW(view));
}
/*
*/
void 
VDKEditor::SetEditable(bool f)
{ 
gtk_text_view_set_editable (GTK_TEXT_VIEW(view),f);
}

/*
*/
void 
VDKEditor::SetShowLineNumbers(bool show)
{
  gtk_source_view_set_show_line_numbers(GTK_SOURCE_VIEW(view),show);
}
/*
*/
bool 
VDKEditor::GetShowLineNumbers(void)
{
  return gtk_source_view_get_show_line_numbers(GTK_SOURCE_VIEW(view));
}
/*
*/
int
VDKEditor::GetFirstVisibleLine()
{
    GdkRectangle  visible_rect;
    GtkTextIter iter;
    int line_top;
    gtk_text_view_get_visible_rect(GTK_TEXT_VIEW(view),&visible_rect);
    gtk_text_view_get_line_at_y(GTK_TEXT_VIEW(view),&iter,visible_rect.y,NULL);
    line_top = gtk_text_iter_get_line(&iter);
    return line_top;
}
/*
*/
int
VDKEditor::GetLastVisibleLine()
{
    GdkRectangle  visible_rect;
    GtkTextIter iter;
    int line_top;
    gtk_text_view_get_visible_rect(GTK_TEXT_VIEW(view),&visible_rect);
    gtk_text_view_get_line_at_y(GTK_TEXT_VIEW(view),&iter,
        visible_rect.y+visible_rect.height,NULL);
    line_top = gtk_text_iter_get_line(&iter);
    return line_top;
}

/*
*/

bool 
VDKEditor::GetChanged()
{
    return gtk_text_buffer_get_modified(GTK_TEXT_BUFFER(buffer));
}
/*
*/
void 
VDKEditor::SetChanged(bool f)
{
    gtk_text_buffer_set_modified(GTK_TEXT_BUFFER(buffer),f); 
}
/*
 */
void 
VDKEditor::SetMaxUndo(int m)
{
  /*
  if(sigwid)
    gtk_source_buffer_set_max_undo_stack(buffer, m);
  */
}

/*
      SIGNALS SECTION
*/
void 
VDKEditor::LocalConnect()
{

  // handles tab
  gtk_signal_connect (GTK_OBJECT (sigwid), "key_press_event", 
  			GTK_SIGNAL_FUNC (VDKEditor::TabHandler), this);
  // handles paren match
  // gtk_signal_connect (GTK_OBJECT (sigwid), "key_release_event", 
  // 			GTK_SIGNAL_FUNC (VDKEditor::OnKeyRelease), this);

  gtk_signal_connect(GTK_OBJECT(sigwid),"realize",
			 GTK_SIGNAL_FUNC(VDKEditor::HandleRealize), this);
}

/*
  update modified flag
 */
void 
VDKEditor::OnBufferChanged(GtkWidget* buf, gpointer gp)
{
  VDKEditor*   editor = reinterpret_cast<VDKEditor*>(gp);
  g_return_if_fail(editor != NULL);
  editor->Changed = true;
}
/*
  Originally was made to handle <tab> only,
  now handles more keystrokes, but the name 
  remained unchanged.
 */
/*
  some special key handling:
  - ctrl-tab or ctrl-a for words completion
  - ctrl-q for hints
  - ) or } for parenthesis matching
*/
static 
struct TIMERSTRUCT 
{ 
  VDKEditor* widget; 
  int match,pos; 
  char key;
  int timerid;
  bool insert;
} TimerStruct;

static bool timeron = false;

static int ParenMatch(VDKEditor* text, int pos, char paren);

static int HandleTimeOut(gpointer gp)
{
  g_return_val_if_fail (gp != NULL,FALSE);
  struct TIMERSTRUCT* ts = reinterpret_cast<struct TIMERSTRUCT*>(gp);
  char key[2];
  key[0] = ts->key;
  key[1] = '\0';
  ts->widget->UnselectText();
  ts->widget->Pointer = ts->pos;
  if(ts->insert)
    ts->widget->TextInsert(key);
  gtk_timeout_remove(ts->timerid);
  timeron = false;
  return FALSE;
}


/*
handle key release
for <alt> + key pad num strokes
*/
//static char ansibuffer[32] = {'\0'};
//static int ansicount = 0;
int
VDKEditor::OnKeyRelease (GtkWidget *widget,
                             GdkEvent *ev,
                             gpointer gp)
{
  VDKEditor* editor;
  GdkEventKey* event = NULL;
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (ev != NULL, FALSE);
  g_return_val_if_fail (gp != NULL, FALSE);
  editor = reinterpret_cast<VDKEditor*>(gp);
  event = (GdkEventKey*) ev;
  // bool isCtrl  = event->state & GDK_CONTROL_MASK;
  // bool isAlt = event->state & GDK_MOD1_MASK;
  // bool isShift = event->state & GDK_SHIFT_MASK;

/*
  bool isAlt = (event->keyval == GDK_Alt_L) &&
    (event->state & GDK_MOD1_MASK) ;
*/
  /*
  if(! isShift && (event->keyval == GDK_Left || event->keyval == GDK_Right))
    {
      int pos = editor->Pointer;
      int start = event->keyval == GDK_Left ? pos : pos-1;
      int end = event->keyval == GDK_Left ? pos+1 : pos;
      char* key = editor->GetChars(start,end);
      if(* key && 
        (key[0] == GDK_parenright || key[0] == GDK_braceright))
        editor->ShowParenMatch(start,key[0],widget,false,
                               event->keyval == GDK_Left ? -1 : pos);
      g_free(key);
    }
  */
  return FALSE; // TRUE;
}
/*
*/
int
VDKEditor::TabHandler (GtkWidget *widget,
                             GdkEvent *ev,
                             gpointer gp)
{
  VDKEditor* editor;
  char* word;
  GdkEventKey* event = NULL;
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (ev != NULL, FALSE);
  g_return_val_if_fail (gp != NULL, FALSE);
  editor = reinterpret_cast<VDKEditor*>(gp);
  event = (GdkEventKey*) ev;
  // destroy tip window if any
  if(tip_window)
    {
      	tip_window->Close();
      	tip_window->Destroy();
      	tip_window = NULL;
    }
  bool handled = false;
  bool isCtrl  = event->state & GDK_CONTROL_MASK;
  // bool isAlt = event->state & GDK_MOD1_MASK;
  // bool isShift = event->state & GDK_SHIFT_MASK;

  // handle word completion
  // provided a token list was assigned to this
 if((isCtrl)  &&  ((event->keyval == GDK_Tab ) || event->keyval == GDK_1) )
    {
      if(editor->tokenlist)
        {
          word = editor->GetWord();
          if(word)
          {
            editor->MakeCompletion(word);
            g_free(word);
          }
        }
      else
        {
          sprintf(buff,"No token list was provided");
          editor->ShowTipWindow(buff);
        }
      handled = true;
    }
 // handles explicit parent match
 else if(isCtrl && (event->keyval == GDK_p ))
    {
      int pos = editor->Pointer;
      char* key = editor->GetChars(pos,pos+1);
      if(* key &&  (key[0] == GDK_parenright || key[0] == GDK_braceright))
        editor->ShowParenMatch(pos,key[0],widget,false, pos);
      g_free(key);      
      handled = true;
    }
 // handles paren match for () and {}
 else if (event->keyval == GDK_parenright || event->keyval == GDK_braceright)
   {
     // shows paren match and insert keystroke (last arg = true)
     int pos = editor->Pointer;
     handled = editor->ShowParenMatch(pos,event->keyval,widget,true,-1);
   }
  // handles ctrl-a (adding new tokens to completion list)
  // provided a token list was assigned to this
  else if(editor->tokenlist && isCtrl && (event->keyval == GDK_a ))
    {
      editor->AddToken();
      handled = true;
    }
  // if keystroke handled stops signal emission
 /*
   if(handled)
   gtk_signal_emit_stop_by_name (GTK_OBJECT (widget),"key_press_event"); 
 */
  return handled;
}
/*
  search and display nearest nested parenthesis matching
  selecting it or if not visible prompting the user at which
  line the matching is.
  FIXME: fix to show also line content.
 */
int
VDKEditor::ShowParenMatch(int start,char keyval,
				GtkWidget* widget,
				bool insert, int restore)
{
  int pos = insert ? start : start-1;
  int match;
  int match_line = -1;
  // answers a match > 0 if a matching paren was found
  if ((match = ParenMatch(this,pos,keyval)) >= 0)
    {
      int visible_line = FirstVisibleLine;
      // stops the signal here
      // so won't reach underlying gtksourceview
      if(insert)
        gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), 
 				    "key_press_event");
      // check if matching position is visible or not
      match_line = GetLineAtOffset(match);
      // is visible so enligths matching position
      // for a short period using a timeout.
      // timeout handling function will insert keystroke as well
      if( match_line >= visible_line)
	{
       SelectText(match,match+1);
       // when timer expires will restore
       // cursor position and inserts matching char
       timeron = true;
       TimerStruct.widget = this;
       TimerStruct.match = match;
       if(restore < 0)
         TimerStruct.pos = insert ? pos : pos + 1;
       else
         TimerStruct.pos = restore;
       TimerStruct.key = keyval;
       TimerStruct.insert = insert;
       TimerStruct.timerid = gtk_timeout_add(100,HandleTimeOut,&TimerStruct);
	}
      // matching position is not visible, prompts user
      // with matching line number
      // FIXME: fix to show also line content.
      else
	{
       char key[2];
       key[0] = keyval;
       key[1] = '\0';
       if(insert)
         TextInsert(key,1);
	  sprintf(buff,"Match at line:%d",match_line);
  	  ShowTipWindow(buff);
	}
      return TRUE;
    }
  else
    // most probably a paren mismatch, warns user
    {
      sprintf(buff,"Humm.., probably a parenthesis mismatch");
      ShowTipWindow(buff);
      return FALSE; 
    }
}

/*
  search where (offset) is the nearest nested parenthesis.
  Uses a stack-like method to store nested position pushing when
  a closing paren and popping when an opening paren is
  found. Stack underflow means we reached the nearest nested
  parenthesis (returns pos >= 0), if buffer beginning is 
  encountered without having a stack underflow means a paren 
  mismatching (returns pos < 0)
 */
int ParenMatch(VDKEditor* text, int pos ,char paren)
{
  char match;
  if(paren == GDK_parenright)
    match = GDK_parenleft;
  else if (paren == GDK_braceright)
    match = GDK_braceleft;
  else
    return 0;
  int stack = 1;
  do
    {
      char* slice = text->GetChars(pos,pos+1);
      char key = slice[0];
      if(key == match)
        	stack--;
      else if (key == paren)
          stack++;
      if(stack > 0)
        	pos--;
      g_free(slice);
    } while((stack > 0)  && (pos >= 0));
  return pos;
}

/*
*/
void 
VDKEditor::SelectText(int start, int end)
{
GtkTextIter istart,iend;
gtk_text_buffer_get_iter_at_offset(GTK_TEXT_BUFFER(buffer),
                                        &istart,
                                        start);
gtk_text_buffer_get_iter_at_offset(GTK_TEXT_BUFFER(buffer),
                                        &iend,
                                        end);
gtk_text_buffer_place_cursor (GTK_TEXT_BUFFER(buffer),&istart);
gtk_text_buffer_move_mark (GTK_TEXT_BUFFER(buffer),
      gtk_text_buffer_get_mark (GTK_TEXT_BUFFER(buffer),"selection_bound"),&iend);
}
/*
*/
void 
VDKEditor::UnselectText()
{
  GtkTextIter insert;
  gtk_text_buffer_get_iter_at_mark (GTK_TEXT_BUFFER(buffer),
                                    &insert,
                                    gtk_text_buffer_get_mark (GTK_TEXT_BUFFER(buffer),
                                    "insert"));
  gtk_text_buffer_move_mark (GTK_TEXT_BUFFER(buffer),
                             gtk_text_buffer_get_mark (GTK_TEXT_BUFFER(buffer),
                             "selection_bound"),
                             &insert);
}


/*
  Scrolls to a pointer pos or (default) to current
  pointer position, leaving <margin> pixels free
*/
void 
VDKEditor::ScrollToPos (int pointer, int margin)
{
  if(pointer >= 0)
    Pointer = pointer;
  GtkTextMark* mark = gtk_text_buffer_get_mark(GTK_TEXT_BUFFER(buffer),
                            INSERT_MARK);
  if(mark)
    // more args on new gtk+ snapshoots
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(view),mark,margin,1,0.5,0.5);
}
/*
  Scrolls to a line,column leaving <margin> pixels free
*/
void 
VDKEditor::ScrollToLine(int line, int col, int margin)
{
  Line = line;
  Column = col;
  GtkTextMark* mark = gtk_text_buffer_get_mark(GTK_TEXT_BUFFER(buffer),
                            INSERT_MARK);
  if(mark)
    // more args on new gtk+ snapshoots
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(view),mark,margin,1,0.5,0.5);
}

/*
*/
TokenList*
VDKEditor::LoadTokens(const char* filename)
{
  FILE* fp;
  TokenList* local = NULL;
  fp = fopen(filename,"r");
  if(fp)
    {
      local = new TokenList;
      char token[256];
      while(fgets(token,256,fp))
      	{
	  int t = std::strlen(token);
	  if(t > 0)
	    {
	      token[t-1] = '\0';
	      if(*token)
		{
		  VDKString tk = token;
		  local->add(tk);
		}
	    }
	}
      fclose(fp);
    }
  return  local;
}

/*
        TIP WINDOW CLASS
 */
void 
Tipwin::Setup(void) 
{
  VDKEventBox *vbox = new VDKEventBox(this,v_box);
  Add(vbox,0,1,1,0); 
  vbox->NormalBackground = VDKRgb(255,255,255);
  label = new VDKLabel(this,tip);
  vbox->Add(label,0,1,1,0);
}

/*
 */
void
VDKEditor::ShowTipWindow(char* tip)
{
  if( tip && !tip_window)
    {
      int x, y;
      GdkRectangle location;
      GtkTextIter iter;
      int window_x,window_y;
      GtkTextView *text = GTK_TEXT_VIEW(sigwid);
      GdkWindow* win = gtk_text_view_get_window(text,GTK_TEXT_WINDOW_TEXT);
      gdk_window_get_deskrelative_origin(win,&x,&y);
      gtk_text_buffer_get_iter_at_mark (GTK_TEXT_BUFFER(buffer),
                                    &iter,
                                    gtk_text_buffer_get_mark (GTK_TEXT_BUFFER(buffer),
                                    "insert"));
      gtk_text_view_get_iter_location (text,&iter,&location);
      gtk_text_view_buffer_to_window_coords (text,
                                            GTK_TEXT_WINDOW_TEXT,
                                            location.x,
                                            location.y,
                                            &window_x,
                                            &window_y);
      x += window_x;
      y += window_y;
      tip_window = new Tipwin(Owner(),tip);
      tip_window->Setup();
      tip_window->Position = VDKPoint(x,y);
      tip_window->Show();
     }
}


