/* ==================================================== ======== ======= *
*
*  upane.hpp
*  Ubit Project [Elc][2003]
*  Author: Eric Lecolinet
*
*  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
*
*  (C) 1999-2003 Eric Lecolinet @ ENST Paris
*  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
*
* ***********************************************************************
* COPYRIGHT NOTICE :
* THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE
* IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
* YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU
* GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION;
* EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
* SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
* ***********************************************************************
*
* ==================================================== [Elc:03] ======= *
* ==================================================== ======== ======= */

#ifndef _upane_hpp_
#define	_upane_hpp_
//pragma ident "@(#)upane.hpp		ubit:03.06.03"
#include <ubit/uborder.hpp>
#include <ubit/uview.hpp>

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Pane gadget (for making Viewports, Cardboxes, Transparent layers).
*
*  There are 5 ways of using UPanes:
*
*  - as a Wiewport: the pane must have only one UBox child.
*    setScroll(), setXScroll(), setYScroll() are then used to control
*    which part of the child is visible.
*
*  - as a Scrollpane: by using subclass UScrollpane or by adding
*    UScrollbars to this UPane
*
*  - as a Cardbox: when several UBox children are added to the Pane
*    they are superimposed and have the same size (which is the
                                                   *    largest size of all children)
*
*  - as a Cardbox with transparent layers: by making all chidren
*    transparent or transluscent except the first one
*
*  - as a Wiewport/Scrollpane that is also a Cardbox: by adding
*    several children to the pane and using the setScroll() methods
*    or the associated scrollbars (the same children parts will
                                   *    be visible)
*/
class UPane: public UBox {
public:
  static UStyle *style;

  UPane(const UArgs& a = UArgs::none);
  ///< constructor: see shortcut creator upane() and UGroup::~UGroup()

  friend UPane& upane(const UArgs& a = UArgs::none);
  ///< shortcut creator: creates a new UPane

  virtual ~UPane();

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  float getXScroll();
  float getYScroll();
  ///< returns the current scroll values (float values between 0 and 100).

  void setXScroll(float _xscroll);
  void setYScroll(float _yscroll);
  void setScroll(float xscroll, float yscroll);
  ///< scrolls the pane (and the associated scrollbars when applicable) float values are between 0 and 100.

  void scrollImpl(float new_xscroll, float new_yscroll,
                  bool upd_x, bool upd_y, bool update_pane);
  ///< [impl] does not update the associated scrollbars.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UBox *getViewport();
  /**< returns the current Viewport (if any; null otherwise).
    * The <b>viewport</b> is the scrolled box in the center area.
    */

  UView *getViewportView(UView *pane_view);
  /**< returns the corresponding Viewport View.
    * Returns the UView of the current Viewport that is contained
    * in Pen view that is given as an argument
    */

  class UScrollbar* getVScrollbar();
  class UScrollbar* getHScrollbar();
  ///< return the Vertical and Horizontal Scrollbars (if any; null otherwise)

  //void setVScrollbar(UScrollbar*, bool add_to_pane = true);
  //void setHScrollbar(UScrollbar*, bool add_to_pane = true);
  /**<
    *  sets the Vertical and Horizontal Scrollbars.
    *  If 'add_to_pane' is true (default case), the scrollbar is added to
  *  the Pane child list and it will appear inside the Pane.
    *  (otherwise, the scrollbar is NOT added to the Pane and MUST be added
        *  to parent somewhere else)
    *  This makes it possible to CONTROL a Pane by a scrollbar
    *  that is not included in this Pane. */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

#ifndef NO_DOC
  friend class UPaneView;

  UBoxLink* getViewportLink();
  void setVScrollbarImpl(class UScrollbar* vs) {vscrollbar = vs;}
  void setHScrollbarImpl(class UScrollbar* hs) {hscrollbar = hs;}

protected:
    class UScrollbar *hscrollbar, *vscrollbar;
  float xscroll, yscroll;

  virtual void viewResized(UEvent&);
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** UScrollpane = UPane with scrollbars.
* see class UPane for details.
*/
class UScrollpane: public UPane {
public:
  UScrollpane(const UArgs& a = UArgs::none);
  /**< constructor: creates a pane with 2 scrollbars.
  * See also: UPane and shortcut creator uscrollpane() */

  UScrollpane(int v_scrollbar_mode, int h_scrollbar_mode,
              const UArgs& a = UArgs::none);
  /**< constructor: creates a pane with 0, 1 or 2 scrollbars.
    * the vertical (resp. horizontal) scrollbar is not created if
    * v_scrollbar_mode (resp. h_scrollbar_mode) is false.
    * See also: UPane and shortcut creator uscrollpane() */

  friend UScrollpane& uscrollpane(const UArgs& a = UArgs::none);
  ///< shortcut creator: creates a pane with 2 scrollbars

  friend UScrollpane& uscrollpane(int Vscrollbar_policy, int Hscrollbar_policy,
                                  const UArgs& a = UArgs::none);
  ///< shortcut creator: creates a pane with 0, 1 or 2 scrollbars.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

protected:
    void constructs(int vs_mode, int hs_mode, const UArgs&);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Pane View Renderer.
*  used by: UPane
*  - Implementation: clients usually don't need to use this class
*/
class UPaneView: public UView {
public:
  static  UViewStyle style;  // renderer
  virtual UViewStyle* getViewStyle() {return &style;}

  UPaneView(UBoxLink*, UView* parview, UWinGraph*);
  virtual ~UPaneView() {}

  static UView* makeView(UBoxLink*, UView* parview, UWinGraph*);
  ///< makeView() is a "static creator" used by UViewStyle to make a new view.

  UPane *getPane();
  ///< same as getBox() with an UPane cast.

  u_dim getXScroll() {return xscroll;}
  u_dim getYScroll() {return yscroll;}

  virtual void setXScroll(u_dim scr) {xscroll = scr;}
  virtual void setYScroll(u_dim scr) {yscroll = scr;}
  ///< note that using these functions is not a good idea if this pane is already controlled by scrollbars.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

#ifndef NO_DOC
  friend class UPane;
  UMargins margins;
  u_dim xscroll, yscroll;
#endif
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */

