/* ==================================================== ======== ======= *
 *
 *  ubox.hpp : Ubit Box containers.
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _ubox_hpp_
#define	_ubox_hpp_
//pragma ident	"@(#)ubox.hpp		ubit:03.06.00"
#include <ubit/ugroup.hpp>


/** Graphical "viewable" container.
 *
 *  This class is a lightweight container that is able to control one 
 *  or several Views (UView class) on the Screen. This class is the base
 *  for all interactive "gadgets" (UButton, etc.), visible containers 
 *  (UHbox, UVbox, UMenubar, etc.) and Windows (UWin, UDialog, UMenu...)
 * 
 *  Notes: 
 *  - gadgets can be made transparent or translucent (for instance
 *    for creating transparent glasses) by adding an UAlpha (for alpha
 *    blending) or UBgcolor::none (for full transparency) brick to them.
 *  - gadgets are visually replicated when they have several parents.
 *    This property is recursive so thta a complet subtree of gadgets
 *    can be automatically replicated.
 *
 *  See:
 *  - UGroup::~UGroup for <b>important notes about deletion</b>
 *  - UCtrl (the generic controller) and UGroup as most methods are 
 *    inherited from these classes
 *  - various gadgets and interactors (ULabel, UButton, UTextbox, etc.)
 *    that are derived from this class
 *  - UWin (base class for creating menus ans dialogs) that is also
 *    derived from this class
 *
 *  Geometry:
 *  - UBox, UHbox, UBar : horizontal lay out by default (add brick: 
 *    UOrient::vertical as a child for vertical lay out). These objects
 *    are "flexible" in the vertical direction: children are automatically 
 *    adjusted in this direction (same as uvflex())
 *
 *  - UVbox : vertical lay out by default. This object is "flexible" 
 *    in the horizontal direction: children are automatically adjusted 
 *    in this direction (same as uhflex())
 *
 *  - UFlowbox : "flow" lay out (similar to those of an HTML page)
 *    Typically used from creating text areas or hypermedia gadgets.
 *
 *  - Boxes are resized when their children are modified (ie. added,
 *    removed or resized). 
 *
 *  - UFlowbox implements a specific behavior: its height changes
 *    but its width remains constant (see this class for details)
 *
 *  - UFlowbox (and UTextbox and subclasses) should be used for 
 *    displaying textual messages that are dynamically changed.
 *
 * Examples:
 *  <pre><tt>
 *     UBox& r = uhbox( UFont::bold + "String" + ubutton("Button") + etc. );
 *     UBox* p = &uvbox( UColor::red + ubutton(UPix::folder + "Open") + etc. );
 *     uptr<UBox>smp = ubar( ubutton("Btn 1") + ubutton("Btn 2") + etc. );
 * </tt></pre>
 *
 * These 3 examples show how to create objects in the heap (the dynamic memory). 
 * uhbox(), uvbox(), etc. are <b>creator shortcuts</b> that create a new
 * object by calling primitive 'new':  uhbox(...)  equals  *new UHbox(...) 
 * They can have a list of child objects as an argument (these children being
 * separated by the + operator). 
 *
 * Notes:
 * - uptr<UBox> is a <b>Ubit a smart pointer</b> that is somewhat 
 *   similar to a Java reference. uptr(s) make it possible to handle objects 
 *   created in the heap (= by using new) in a safe way. This is not the case
 *   with usual C++ pointers and references (especially when objects are deleted).
 *   See: class UBrick and template 'uptr' for more info.
 *
 * - uhbox("String") is equivalent to: uhbox(ustr("String"))
 * - uhbox("String1" + "String2") is illegal (that's a C++ limitation).
 *   instead you shouls write:  uhbox(ustr("String1") + "String2")
 * - strings can be made changeable and/or editable as follows:
 *  <pre><tt>
 *     UStr& str = ustr("String")
 *     UBox& box = uhbox(uedit() + str);  // uedit() makes string editable
 *     ....
 *     cout << str + endl;   // prints the content of 'str'
 *     str.set("xxxxx");     // changes the content of 'str' and update 'box'
 */
class UBox: public UGroup {
public:
  static UStyle *style;

  UBox(const UArgs& a = UArgs::none);
  /**< 
   * constructor; see also class UGroup, destructor UGroup::~UGroup() and the <em>creator shortcut</em> <b>ubox()</b>.
   */

  friend UBox& ubox(const UArgs& a = UArgs::none) {return *new UBox(a);}
  /**< 
   * creator shortcut that is equivalent to: *new UBox().
   * Note: watch the case: UBox is a class while ubox is a function!
   */

  virtual ~UBox() {destructs();}
  ///< see <b>important notes</b> on <b>recursive deletion</b> : see UGroup::~UGroup()

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // methods (same as UGroup counterparts)

  UBox& addlist(const UArgs& a) {UGroup::addlist(a); return *this;}

  virtual void update();
  virtual void update(UUpdate upmode);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // view management

  virtual UView*  getView(int no) const;
  virtual int getViewCount() const;
  virtual UView** getViews() const;
  virtual UView** getViews(int &view_count) const;
  virtual int getViews(std::vector<UView*>&) const;
  /**<
   * get the views managed by this Box.
   * these functions work in the same way as UGroup::getChidren()/getChild()
   */

  UView* getViewContaining(UEvent&) const;
  ///< returns the Box's View that contains the Event location (null if none).

  UView* getViewContaining(UView* child_view) const;
  ///< returns the Box's View that contains this (direct or indirect) child view (null if none).

  UView* getFirstViewInside(UView* parent_view) const;
  /**< 
   * returns the first Box's View that is contained in this (direct) parent view.
   * Notes:
   * - 'parent_view' must be a DIRECT parent
   * - there will be several Box's views inside 'parent_view' if the Box has beed 
   *   multiply added to this parent
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // box events

  virtual UBox& onPostChildEvent(const UArgs&);
  /**<
   *  detect events that *have already* been received by children.
   * intercepts events that have already bee received by the direct 
   * or indirect children of this UBox (or subclass):
   * - events are first sent to children, then to this object
   * - events which are not processed by children are lost (for 
   *   instance a UOn::action condition can only be detected at this
   *   stage if the child has a UOn::action condition)
   * - all UEvent methods (such as getSource(), getX(), getY() ...)
   *   are relative to the CHILD
   *
   * See also:
   * -  the event condition: UOn::preChildEvent that detect
   *    events that *will* be received by children.
   *
   * Example:
   * <pre>
   *   box->onPostChildEvents( UOn::mmove :: ucall(arg1, func1) 
   *                          + UOn::mpress:: ucall(arg2, func2))
   * </pre>
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

#ifndef NO_DOC
public:
  friend class UWin;

  virtual class UBox* boxCast() {return this;}
  virtual ULink* makeLink();

  virtual void initView(ULink *selflink, UView *ancestor_view);
  virtual void initChildViews(UGroup*, ULink *childlink);
  virtual void deleteRelatedViews(class UView* parview, class ULink*);
  //NB: link can be null (but not parview) for UBox::deleteRelatedViews

  virtual void updateView(UEvent&, UView*, const UUpdate&);
  static void updateImpl(const class UUpdate& upmode, 
			 UView* winview, UView* layoutview, UView* showview,
			 bool impose_size, u_dim w, u_dim h); 
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** UVbox = vertical Box.
 *  Geometry: see class UBox for details.
 */
class UVbox: public UBox {
public:
  static UStyle *style;

  UVbox(const UArgs& a = UArgs::none) : UBox(a) {}
  ///< constructor; see also ~UGroup and the <em>creator shortcut</em> <b>uvbox()</b>

  friend UVbox& uvbox(const UArgs& a = UArgs::none) {return *new UVbox(a);}
  ///< creator shortcut that is equivalent to: *new UVbox()

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static const UStyle& makeStyle();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** UHbox = horizontal Box.
 *  Geometry: see class UBox for details.
 */
class UHbox: public UBox {
public:
  static UStyle *style;

  UHbox(const UArgs& a = UArgs::none) : UBox(a) {}
  ///< constructor; see also ~UGroup and the <em>creator shortcut</em> <b>uhbox()</b>

  friend UHbox& uhbox(const UArgs& a = UArgs::none) {return *new UHbox(a);}
  /// creator shortcut that is equivalent to: *new UHbox().

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static const  UStyle& makeStyle();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** UBar = horizontal Bar (for making toolbars).
 *  This class is similar to UHbox but with a specific ("toolbar like") decoration.
 *  Buttons included in this object will also be displayed in a specific
 *  way (their border won't appear, etc.)
 *
 *  Geometry: see class UBox for details.
 *
 *  See also: classes UMenu and UMenubar for creating menus and menu bars
 */
class UBar: public UBox {
public:
  static UStyle *style;

  UBar(const UArgs& a = UArgs::none): UBox(a) {}
  ///< constructor; see also ~UGroup and the <em>creator shortcut</em> <b>ubar()</b>

  friend UBar& ubar(const UArgs& a = UArgs::none) {return *new UBar(a);}
  ///< creator shortcut that is equivalent to: *new UBar()

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** UFlowbox = Box with a Flow Layout (similar to an HTML page).
 *  This class is typically used from creating text areas or hypermedia gadgets.
 *
 *  Geometry: this class implements a specific behavior that differs from
 *  other boxes: its height changes but its width remains constant when 
 *  children are modified (= these objects will keep their initial width).
 *  The width can be specified by adding an UWidth child. It can also
 *  be changed through user interaction if this object is included
 *  in a box that is "flexible" in the horizontal direction (see UBox).
 *
 *  UFlowbox (and UTextbox and subclasses) should be used for displaying 
 *  textual messages that are dynamically changed for optimization purpose.
 *
 *  an UFlowbox is an UBox with UFlowView renderer.
 */
class UFlowbox: public UBox {
public:
  static UStyle *style;

  UFlowbox(const UArgs& a = UArgs::none);

  friend UFlowbox& uflowbox(const UArgs& a = UArgs::none)
  {return *(new UFlowbox(a));}

  virtual const UStyle& getStyle(UContext*) const {return makeStyle();}
  static  const UStyle& makeStyle();
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
