/* ==================================================== ======== ======= *
 *
 *  uappli.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _uappli_hpp_
#define	_uappli_hpp_
//pragma ident	"@(#)uappli.hpp	ubit:03.05.06"
#include <ubit/udisp.hpp>
#include <ubit/ustr.hpp>
#include <ubit/ugroup.hpp>
#include <ubit/uconf.hpp>

/* ==================================================== ======== ======= */
/** Application Context. 
 * Notes:
 * - there should be ONLY ONE application context per program.
 *
 * - many methods are inherited from UDisp, the Display Context
 *
 * - applications can be configured through command line options
 *   or by a UConf object (see UAppli::UAppli and class UConf)
 *
 * - Ubit applications won't work properly on X servers that do not
 *   support True Colors. They are always launched in True color mode
 *   if this feature is supported by the X server (even if this is
 *   not the default behavior). See UDisp::setTrueColors() and
 *   UDisp::setPseudoColors() for details.
 *
 * - Timers and Input callbacks can be associated to the UAppli:
 *   see methods openTimer() and openInput().
 *   Timers with a 0 or 1 mS delay can be used as a way to postpone
 *   callback actions (they will be fired immediately when the event
 *   loop becomes idle).
 *
 * - an application can open windows on an arbitrary number of remote
 *   Displays (see method openDisp()). The --group command line
 *   option should be used in this case (the UConf argument can
 *   also be set to do this automatically)
 *
 * - several Event flows can be associated to an application
 *   (see inherited method UDisp::openFlow()). Alertnate event flows
 *   are automatically created if the UMS (Ubit Mouse Server) daemon
 *   is running (see directory ubit/ums and program umsd)
 *
 * - applications can send "umessage"s and pesudo-events to the
 *   UMS daemon and to other applications by opening a connection
 *   with the UMS (or with several UMSs located on different machines)
 *   see UAppli::openUMS() and class UMSclient.
 */
class UAppli: public UDisp {
public:

  UAppli(UConf& conf);
  UAppli(int* argc, char** argv);
  UAppli(int& argc, char** argv);
  /**<
   * constructors for creating the Application Context.
   * Arguments:
   * - 'argc' and 'argv' : the parameters of the main() function
   * - 'conf' : configuration of the UAppli (see UConf).
   *            this argument must *not* be deleted.
   * Exception:
   * - UError exception thrown if the X server can't be opened.
   *
   * Options on the command line:
   * - see class UConf or launch a Ubit appli with option: --help-ubit 
   *   on the command line to see all options
   * - some useful options:
   * <pre>
   *    --help-ubit, --help-x      : prints help message"
   *    --conf <option>            : sets configuration option
   *    --disp[lay] <displayname>  : starts the appli on this X-server display
   *    --doubleb[uffering], --dbf : double buffering (avoids flickering)
   *    --scale <int>              : scale: 0=default, > 0 larger, < 0 smaller
   *    --group[ware]              : groupware mode (--softwin and --tele)
   *    --soft[win]                : soft window mode
   *    --tele[pointers]           : tele pointer mode
   *    --no-transps[crollbars], --no-tsb: no transparent scrollbars"
   * debug:
   *    --no-grab                : disables mouse grabs
   *    --sync                   : synchronous mode
   * </pre>
   * - example:  myappli -doublebuf -scale -2 -display myserver:0 
   *
   * See also: classes UAppli and UDisp for important info.
   */

  virtual ~UAppli();
  ///< destructor.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  const class UConf& getConf() const      {return conf;}
  static const class UConf& getDefaults() {return default_appli->conf;}
  /**<
   * configuration of the Appli context.
   */

  static class UAppli* getApp();
  /**<
    * returns the default UAppli context.
    * the "default" UAppli is *the* UAppli (as there is only one UAppli for
    * a given program).
    *
    * Note: it's illegal to call this function before the UAppli has been
    * created (this will throw an UError exception)
    */
  
  const char* getCommandName() const;
  const char* getCommandPath() const;
  /**< returns the name or full path of the command that started the appli.
    *  Note: name = path without the directory
    */

  class UFrame* getMainFrame() const;
  /**< returns the Main Frame.
    * the main frame is the first UFrame that was added to the UAppli (if any)
    */

  void setTitle(const UStr &title);
  /**< changes the title of the Main Frame.
    * see also: UTitle
    */
  
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // main loop

  virtual int mainLoop();
  virtual int start() {return mainLoop();}
  /**< starts the event main loop of the application.
   * returns the status argument given as an arg of the quit() method
    */

  virtual void quitLoop(int status);
  static void quit(int status);
  /**< quits the main loop of the application.
   * Notes:
   * - this status is returned by the mainLoop() method (it should typically
   *   be 0 for normal termination and > 0 otherwise)
   * - this method can be redefined by clients for appropriate behaviors
   *   (eg. saving data before quitting etc.)
   */

  virtual void realize();
  /**< 
   * realizes this application (allocates the X resources).
   * this function does not need to be explicitely called except if you 
   * need to draw Graphics before calling the mainLoop 
   *
   * Exception:
   * - UError exception thrown if the application can't be realized
   */

  virtual bool isRealized() const;
  ///< is the X data initialized?.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Windows

  virtual void add(class UWin*);
  virtual void add(class UWin&);
  /**< 
   * add a window (UFrame, UDialog) to the application.
   * notes:
   * - at least one Window (typically an UFrame) should be added to
   *   the UAppli 
   * - the "Main Frame" is the first UFrame that is added to the UAppli
   * - windows are initially closed (including the Main Frame) :
   *   you must first add their subcomponents (using add(), addlist() etc.)
   *   THEN open them (when needed) by calling their UWin::show() method.
   * - by default their size won't change once their show() method has
   *   been called.
   * See classes: UWin, UFrame, UDialog.
   */ 

  virtual void remove(UWin& child, int remove_mode);
  virtual void remove(UWin* child, int remove_mode);
  ///< same as UGroup::remove().

  virtual void updateAll();
  virtual void updateAll(UUpdate upmode); 
  ///< updates all windows (see UWin::update()).

  class UGroup* getOpenedMenu();
  ///< returns the menu that is currently opened (null otherwise).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // File and socket Input

  static class UInput* openInput(int input_source);
  /**< this input source (file, socket) will call callback functions
   *   when it receives data.
   * Example:
   * <pre>
   *    uptr<UInput> in = appli.openInput(socket_desc);
   *    if (in) in->onAction( ucall(...) );
   * </pre>
   * Note: the UInput is automatically destroyed when the source (file, 
   * socket) is closed except if it is still pointed by an 'uptr'.
   */

  static void closeInput(class UInput*);
  /**< closes this input source.
   * Note: the UInput is destroyed except if is still pointed by an 'uptr'
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Timers

  static UTimer* openTimer(u_time delay, int ntimes = 1);
  /**< opens a new Timer that calls callback functions.
   * Args:
   * - 'delay' is in milliseconds
   * - 'ntimes' specifies how many times the callbacks are called
   *   (each time after the specified 'delay'). -1 means always, 0 never. 
   * Example:
   * <pre>
   *    uptr<UTimer> t = appli.openTimer(1000, 10); // 10 times separed by 1s
   *    if (t) t->onAction( ucall(...) );
   * </pre>
   * Notes:
   * - the UTimer is automatically destroyed when over except if it
   *   is still pointed by an 'uptr'.
   * - if 'delay' = 0, the callback is immediately called when the main loop 
   *   becomes idle. This is useful for postponing callbacks that take a long
   *   time to execute. if 'ntimes' = 1 the functions are aclled once, if
   *   'ntimes' = -1 they are called until program termination.
   */

  static void closeTimer(class UTimer*);
  /**< closes this Timer.
   * Note: the UTimer is destroyed except if is still pointed by an 'uptr'
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Multiple Displays.

  static class UDisp* openDisp(const UStr& display_name);
  /**< opens a connection on a new X display.
   * 'display_name' name of the X Window server:
    * - syntax: hostname[:screen_number]
    * - hostname:0 is the default if [:screen_number] is omitted
   */

  static void closeDisp(class UDisp*);
  ///< [unstable].

  static class UDisp& getDefaultDisp()  {return *displist[0];}
  static class UDisp* getDisp(int disp_id);

  static unsigned int getDispCount();
  static const UDispList& getDisps()  {return displist;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Multiple Event Flows.
  // inherited from UDisp

  // class UFlow* openFlow(int flow_id);
  // void closeFlow(class UFlow*);
  // class UFlow& getDefaultFlow() const {return *flowlist[0];}
  // class UFlow* getFlow(int flow_id) const;
  // unsigned int getFlowCount() const;
  // const UFlowList& getFlows() const {return flowlist;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // connection to an UMS server.

  static class UMSclient* openUMS(const UStr& ums_hostname, int ums_port = 0);
  /**<
   * opens a connection with the UMS (Ubit Multiple Mouse/Message Server).
   * This makes it possible to control the mouse pointer(s) and to send
   * events and messages to X applications on the remote display where
   * the UMS is running.
   *  Args:
   *  - ums_hostname: the host where the UMS is running
   *  - ums_port: the port used by the UMS (uses UMS_DEFAULT_PORT if 0)
   *  Note: 
   *  - the 'umsd' program must already be running
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // variable database

  static const char* getVar(const char *name, bool get_from_shell = true);
  /**< retreives a value from a variable name.
   * searches variable in the Application variable database, then, if not
   * found and 'get_from_shell' is true, in the Unix SHELL environment.
   * - note: when a variable is retrieved from the SHELL, it is copied in
   *   the application database and is never searched again in the SHELL */

  static const char* getImaPath();
  /**< gets the value of the UIMA_PATH variable (the DEFAULT location of images).
   * see: setImaPath() and getVar() */

  static const char* setVar(const char *name, const char *value);
  /**< adds a variable/value pair to the Application variable database.
   * Notes:
   * - the 'name' and 'value' strings are internally duplicated.
   * - this fct. returns the duplicated value (null if memory is exhausted)
   * - variables can be used for defining image file prefixes:
   *   uima("$MYDIR/my_image.gif") => MYDIR is expanded to its actual value
   * - see: setImaPath() (and special variable UIMA_PATH) */

  static const char* setImaPath(const UStr& value);
  static const char* setImaPath(const char* value);
  /**< sets the value of the UIMA_PATH variable (the DEFAULT location of images).
   * the value of the UIMA_PATH variable is prefixed to image file names 
   * that do not start with / or . or $
   * <br>Examples:
   * - uima("my_image.gif")        will load:  $UIMA_PATH/my_image.gif
   * - uima("./my_image.gif")      will load:  ./my_image.gif
   * - uima("$MYDIR/my_image.gif") will load:  $MYDIR/my_image.gif 
   * where $XXX is replaced by the value of variable XXX */

  static char *makeImaPath(const char* filename);
  /**< creates a full image file path name.
   * prefixes 'filename' with UIMA_PATH value if not starting by / . or $
   * and expands 'filename' if starting by $  */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

#ifndef NO_DOC
  friend class UDisp;
  friend class UFlow;
  friend class UNatAppli;
  friend class UInput;
  friend class UTimer;

  virtual void constructs();
  ///< [impl] internal constructor.

  virtual int subLoop();
  ///< [impl] subloop lauched by modal dialogs.

  void addModalwin(class UWin*);
  void removeModalwin(class UWin*);
  int  getModalStatus() const  {return modal_status;}
  void setModalStatus(int _s)  {modal_status = _s;}

  static void safeDeleteRequest(void*);
  static void processSafeDeleteRequests();

  static void deleteNotifyAll(class UView*  deleted_view);
  static void deleteNotifyAll(class UGroup* deleted_group);
  static void deleteNotifyAll(class UDisp*  deleted_disp);
  ///< [impl] notifies all the displays of the appli that a view, a widget
  ///< or a display is being destroyed.

  void closeRelatedMenus(class UMenu*);

  //class UNatAppli* getNatAppli() const {return natappli;}
  ///< [impl] returns the Native Appli object (implementation dependent data).
  
private:  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  // only one UAppli should be created
  static UAppli* default_appli;
  
  // the display list must be static to avoid seg faults if the UAppli
  // is distroyed before the widgets by the client program (nb: this is
  // OK as there is only one UAppli)
  static UDispList displist;

  static std::vector<void*> safe_delete_list;

  class UFrame* main_frame;	               // the main frame of the appli
  static class UGroup *inputs, *timers;        // input and timer callback lists
  class UNatAppli* natappli;                   // native appli data
  static const char *pixmapPath, *bitmapPath;  // image paths
  static struct UAPP_VAR* varDB;	       // variable DataBase array
  static int varDB_count;                      // ... and its cardinal

  // current modal dialog in the Display
  // (null if no modal window currently activated)
  UChain modalwin_list;
  class UWin* modalwin;

  // value is init. to LOOP_IDLE by UAppli(), then to LOOP_RUN
  // by mainLoop() and a value >= LOOP_QUIT quits the mainLoop
  enum {LOOP_STOP = -2, LOOP_RUN = -1, LOOP_QUIT = 0};
  int main_status;    // status of the event loop of the UAppli
  int modal_status;   // status of the inner loop the current modal dialog
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** a UInput object activates callbacks when a file or a socket gets data.
 *  Usage: see UAppli::openInput() and UAppli::closeInput()
 */
class UInput : public UBrick {
public:
  UInput(UAppli&, int source);
  ///< constructor (use UAppli::openInput() instead).

  virtual ~UInput();

  void onAction(UCall&);
  ///< add a callback function that is called when data is received.

  void onInput(UCall& c) {onAction(c);}
  ///< obsolete synonym

  virtual void close();
  int getSource() const {return source;}

protected:
  virtual bool verifies(const UContext*, const UCtrl*) const {return false;}

private:
  friend class UNatAppli;
  UAppli& appli;
  int source;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** a UTimer object activates callbacks when a timeout is reached 
 *  (or when the main loop become idle).
 *  Usage: see UAppli::openTimer() and UAppli::closeTimer()
 */
class UTimer : public UBrick {
public:
  UTimer(UAppli&, u_time delay, int ntimes);
  ///< constructor (use UAppli::openTimer() instead).

  virtual ~UTimer();

  void onAction(UCall&);
  ///< add a callback function that is called after the delay.

  void onTimeout(UCall& c) {onAction(c);}
  ///< obsolete synonym

  virtual void close();
  virtual void reset(u_time delay, int ntimes);

  u_time getDelay() const {return delay;}
  int getTimes() const {return ntimes;}

protected:
  virtual bool verifies(const UContext*, const UCtrl*) const {return false;}

private:
  friend class UAppli;
  friend class UNatAppli;
  UAppli& appli;
  struct timeval* timeout;
  u_time delay;
  int  ntimes;
  bool must_rearm;
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
