#ifndef VHDLKERNEL_HH
#define VHDLKERNEL_HH
//---------------------------------------------------------------------------
// Copyright (c) 1995-2001 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.
//---------------------------------------------------------------------------
//
// $Id: VHDLKernel.hh,v 1.1 2003/12/04 19:13:19 dmartin Exp $
//
//---------------------------------------------------------------------------

// Standard C and C++ includes
#include <cassert>
#include <vector>

// Project includes
#include "tyvis/vhdl.hh"
#include "tyvis/Wait.hh"
#include "tyvis/CallStack.hh"
#include "tyvis/VHDLKernel_state.hh"
#include "tyvis/VHDLVTime.hh"

using std::vector;
using std::istream;

// Class prototypes
class RecordType;
class ScalarType;
class _savant_entity_elab;
class VHDLType;
class SharedFileType;
class Event;
class SigEvent;
class WaitEvent;

/** This class includes the methods and data required by a VHDL Object to
   interface to warped.  It is intended to be the glue between a VHDL
   process and Warped.  */

class VHDLKernel;
extern vector<VHDLKernel*> listOfProcesses;

class VHDLKernel : public SimulationObject {
public:
  /** Get a unique global object identifier. Will produce an integer number
      which is ensured to be globally unique and thus may be used as an
      identifier of the object.  Global id with increment each time this is
      called. */
  static int getGlobalObjectId();

  virtual void initialize();
  void finalize();
  virtual void executeProcess();           

  virtual void deallocateState(State *state);
  virtual void reclaimEvent(const Event* event);
  virtual void reclaimSerializedInstance(SerializedInstance *instance);

  
  void cancelSingleMessage(Event*);
  void sendNegEvent(Event*);

  // Some of the new methods that is required by warped 2.0 API...
  void reportError( const string &, const SEVERITY );


  virtual void executeVHDL( VHDLKernel_state & ){ abort(); }
  
  const VHDLVTime &getTimeNow() const;
  
  /** Implicitly assumes that a variable called "stack" is defined in the
      user-defined state of type CallStack. */
  CallStack* getCallStack() {
    return &getVHDLState()->stack;
  }

  const string &getName() const { return processName; }
  int getVHDLProcessId() const { return vhdlProcessId; }

  static const string &getName( const OBJECT_ID& id );
  static void initializeGlobalObjectId();

  /** Handle this "disconnect driver" event. */
  void disconnectDriver( SigEvent * );
  
  /** Defer the disconnect to our superclass */
  virtual void disconnectDriver( SignalBase *, int ){ abort(); }

  void setSharedData( const string &newData ){ sharedData = newData; }

  /** Called by the SharedFile event to see if there's a shared read
      pending. */
  bool hasSharedReadPending(){
    return getVHDLState()->sharedReadPending;
  }

  /** The following Function updates the driver of the signal corresponding
      to its sig id. Also adds this signal to the list of signals that
      should be updated in the current simulation cycle. */
  void updateDriver( SigEvent * );
  
  /**
     The actual driver updation is deferred to the VHDLProcess.
  */
  virtual void updateDriver( SignalBase *signal, 
			     int sigSrc,
			     int senderSigId,
			     const VHDLData *newData,
			     ArrayInfo *sourceInfo,
			     ArrayInfo *destInfo );

  /**
     Handles a wait event for us.
  */
  void updateWait( WaitEvent * );

  //@{
  /**
     Wait updation is deferred to the VHDLProcess.
  */
  virtual void updateWait( const VHDLVTime & );
  virtual bool resumeWait(int, const EnumerationType& = SAVANT_BOOLEAN_TRUE);
  virtual void executeWait(int, const PhysicalType& = SAVANT_INFINITY_TIME);
  virtual void setProcedureWait( Wait *myWait );
  //@}

  /** The following are the methods that operate over signals.  They are
      all upcalls to superclasses. */
  //@{
  /** Okay, immaterial of the original type of the signal we are now
      assiging values to a scalar signal.  Note that assigning a value to a
      signal does not change the value of the signal, but merely generates
      transactions on the signal.  The transactions ultimately get
      converted to events and get shipped out to the various (VHDL)
      processes that are sensitive to this signal.
      
      The process of posting transactions on signals also involves a phase
      called marking.  Each scalar signal does its own marking and
      generates the transactions to be sent out.  The actual events are
      sent out by the process.  */
  virtual void assignSignal( VHDLType &dest, 
			     VHDLKernel *srcId, 
			     const VHDLType &src,
			     const PhysicalType &delay,
			     const PhysicalType &rejTime = SAVANT_ZERO_TIME, 
			     const ArrayInfo &dinfo = defaultInfo,
			     const ArrayInfo &sinfo = defaultInfo );

  /** Okay, immaterial of the original type of the signal we are now
      assiging values to a scalar signal.  Note that assigning a value to a
      signal does not change the value of the signal, but merely generates
      transactions on the signal.  The transactions ultimately get
      converted to events and get shipped out to the various (VHDL)
      processes that are sensitive to this signal.
      
      The process of posting transactions on signals also involves a phase
      called marking.  Each scalar signal does its own marking and
      generates the transactions to be sent out.  The actual events are
      sent out by the process.  */
  virtual void assignSignal( SignalBase &dest, 
			     VHDLKernel *srcId, 
			     const VHDLData &src,
			     const VHDLVTime &delay, 
			     const VHDLVTime &rejTime = VHDLVTime::getVHDLVTimeZero(),
			     const ArrayInfo &dinfo = defaultInfo, 
			     const ArrayInfo &sinfo = defaultInfo );

  /** The following function calls the resolution function and the type
      conversion functions if any for a Composite resolved signal and
      updates the values of its dependent signals and attributes.

      This method is public because the type code defers to this method
      sometimes.
 */
  virtual void updateSignal(VHDLType*);

  /**  The following function calls the resolution function and The type
       conversion functions if any for a signal and updates the values of
       its dependent signals and attributes. 

      This method is public because the type code defers to this method
      sometimes.
  */
  virtual void updateSignal( SignalBase*, bool initializingSimulation );

  virtual int getWaitLabel(){ abort(); }

  virtual EnumerationType locateQuietAttribute( const SignalBase *, 
						const VHDLVTime = VHDLVTime::getVHDLVTimeZero() );

  virtual EnumerationType locateQuietAttribute( const VHDLType *sig, 
						const VHDLVTime = VHDLVTime::getVHDLVTimeZero());

  virtual EnumerationType locateQuietAttribute( const ScalarType *sig, 
						const VHDLVTime = VHDLVTime::getVHDLVTimeZero());

  virtual EnumerationType locateQuietAttribute( const RecordType *sig,
						const VHDLVTime = VHDLVTime::getVHDLVTimeZero());

  virtual EnumerationType locateQuietAttribute( const ArrayType *sig,
						const VHDLVTime = VHDLVTime::getVHDLVTimeZero());

  virtual EnumerationType locateEventAttribute( const VHDLType *sig );
  virtual EnumerationType locateEventAttribute(SignalBase *);
  virtual EnumerationType locateEventAttribute( const ScalarType *sig);
  virtual EnumerationType locateEventAttribute( const RecordType *sig);
  virtual EnumerationType locateEventAttribute( const ArrayType *sig);
  
  virtual EnumerationType locateActiveAttribute( const VHDLType *sig);
  virtual EnumerationType locateActiveAttribute( const SignalBase *);
  virtual EnumerationType locateActiveAttribute( const ScalarType *sig);
  virtual EnumerationType locateActiveAttribute( const RecordType *sig);
  virtual EnumerationType locateActiveAttribute( const ArrayType *sig);
  virtual EnumerationType locateStableAttribute( const SignalBase *, 
						 const VHDLVTime = VHDLVTime::getVHDLVTimeZero());
  virtual EnumerationType locateStableAttribute( const VHDLType *sig, 
						 const VHDLVTime = VHDLVTime::getVHDLVTimeZero());
  virtual EnumerationType locateStableAttribute( const ScalarType *sig, 
						 const VHDLVTime = VHDLVTime::getVHDLVTimeZero());
  virtual EnumerationType locateStableAttribute( const RecordType *sig, 
						 const VHDLVTime = VHDLVTime::getVHDLVTimeZero());
  virtual EnumerationType locateStableAttribute( const ArrayType *sig, 
						 const VHDLVTime = VHDLVTime::getVHDLVTimeZero());

  virtual EnumerationType *locateTransactionAttribute( const VHDLType *sig);
  virtual EnumerationType *locateTransactionAttribute( const SignalBase *);
  virtual EnumerationType *locateTransactionAttribute( const ScalarType *sig);
  virtual EnumerationType *locateTransactionAttribute( const RecordType *sig);
  virtual EnumerationType *locateTransactionAttribute( const ArrayType *sig);

  PhysicalType locateLastActiveAttribute( const VHDLType *sig);
  PhysicalType locateLastActiveAttribute( const SignalBase *);
  PhysicalType locateLastActiveAttribute( const ScalarType *sig);
  PhysicalType locateLastActiveAttribute( const RecordType *sig);
  PhysicalType locateLastActiveAttribute( const ArrayType *sig);

  const PhysicalType locateLastEventAttribute( const VHDLType *sig);
  const PhysicalType locateLastEventAttribute( const SignalBase *);
  const PhysicalType locateLastEventAttribute( const ScalarType *sig);
  const PhysicalType locateLastEventAttribute( const RecordType *sig);
  const PhysicalType locateLastEventAttribute( const ArrayType *sig);

  VHDLType *locateLastValueAttribute( const VHDLType *);
  VHDLType *locateLastValueAttribute( const SignalBase *);
  VHDLType *locateLastValueAttribute( const ScalarType *);
  VHDLType *locateLastValueAttribute( const RecordType *);
  VHDLType *locateLastValueAttribute( const ArrayType *);

  //@}
      
  
  /** lib STANDARD and TEXTIO functions and procedures
      bool savantendfile(int); */
  //@{
  EnumerationType savantendfile_boolean( VHDLKernel *, int );
  EnumerationType savantendfile_boolean( VHDLKernel *, FileType& );
  EnumerationType savantendfile( VHDLKernel *, int );
  EnumerationType savantendfile( VHDLKernel *, FileType& );
  const VHDLVTime &_savant_now() const { return getTimeNow(); }

  int savantreadline( VHDLKernel *, int, AccessVariable &);

  int savantwriteline( VHDLKernel *, int, AccessVariable &);
  int savantwriteline( VHDLKernel *, int, AccessType &);
  int savantreadline( VHDLKernel *, int, AccessType &);
  
  int savantwriteline( VHDLKernel *, FileType&, AccessType &);
  int savantreadline( VHDLKernel *, const FileType&, AccessType &);

  int savantwriteline( VHDLKernel *, SharedFileType *, AccessType &);
  int savantreadline( VHDLKernel *, SharedFileType *, AccessType &);
  
  int savantreadline( VHDLKernel *, int, AccessType&, IntegerType&);
  //@}

  /** Wrappers to the warped i/o interface. */
  //@{
  int openFile( const string &,
		const _savant_file_open_kind &, 
		ios::openmode = ios::out );
  
  void closeFile( const _savant_file_open_kind &, int );
  //@}

  void setProc( _savant_entity_elab *newProc ){ proc = newProc; }
  _savant_entity_elab *getProc(){ return proc; }

  const string &getProcessName(){ return processName; }

  void deallocateState( const State *toDeallocate ){ delete toDeallocate; }
  
protected:
  /**
     Schedules a wait for the waitId at the specified time.
  */
  void scheduleWait( int waitId, const VHDLVTime &waitTime );

  /** This method wraps warped's getState() called and does the appropriate
      casting for us to return a VHDLKernel_state */
  VHDLKernel_state *getVHDLState(){ return dynamic_cast<VHDLKernel_state *>(getState()); }

  /** This method first tries to fit the file line into a static buffer.
      If the line fits, this value is reused.  If the buffer is too small,
      the file is rewound to its original position, and the line is reread
      into a dynamically allocated buffer twice the size of the static one.
      If the line still doesn't fit, the process is repeated, doubling the
      buffer size each time, until the line if successfully read in its
      entirety. */
  char *getline( istream &, int & );
  
  VHDLKernel( const string &name, _savant_entity_elab* = NULL);
  virtual ~VHDLKernel();

  // XXX - Should be private.
  vector <SimulationStream*> fileHandles;

private:
  bool postponed;
  int numFiles;

  bool eatwhite( AccessVariable & );
  virtual void updateGuard( VHDLType * ){ abort(); }

  /**
     This method processes the events that warped has passed us.  It
     handles all of the cancellations for us, and only gives back a set
     of events which we actually need to execute.     
  */
  Block *getEventsToExecute();

  virtual VHDLType *getGuard() { return NULL; };
  virtual VHDLType *getGuardExpression() { return NULL;}
  int   vhdlProcessId;
  string processName;
  string sharedData;
  _savant_entity_elab *proc;
};

#endif
