#include "ArrayTypeInfo.hh"
#include "ArrayType.hh"

// Member functions of ArrayTypeInfo

//ArrayTypeInfo constructors
ArrayTypeInfo::ArrayTypeInfo() :
  elementTypeInfo( 0 ),
  unconstrainedType( true ),
  compositeResolvedType( false ),
  resolutionFunctionId( -1 ){}

ArrayTypeInfo::ArrayTypeInfo( int dim,
			      TypeInfo *elementTI,
			      bool unconstrained,
			      bool compositeResolvedFlag,
			      ResolutionFnId_t resolveFn,
			      TypeInfo* r1Info, ...) :
  elementTypeInfo( elementTI ),
  unconstrainedType( unconstrained ),
  compositeResolvedType( compositeResolvedFlag ),
  resolutionFunctionId( resolveFn ){
  ASSERT ( dim > 0 );
  ranges.push_back( r1Info );

  if (dim > 1) {
    va_list ap;
    va_start(ap, r1Info);

    for( int i = 1; i < dim; i++ ){
      TypeInfo *temp  = va_arg(ap, TypeInfo *);
      ranges.push_back( temp );
    }
    va_end(ap);
  }
}

ArrayTypeInfo::ArrayTypeInfo( TypeInfo *elementTI,
			      bool unconstrained,
			      bool compositeResolvedFlag,
			      ResolutionFnId_t resolveFn,
			      const vector<TypeInfo *> &initRanges ) :
  elementTypeInfo( elementTI ),
  unconstrainedType( unconstrained ),
  compositeResolvedType( compositeResolvedFlag ),
  resolutionFunctionId( resolveFn ){
  ranges.insert( ranges.begin(), initRanges.begin(), initRanges.end() );
}

ArrayTypeInfo::ArrayTypeInfo( const ArrayTypeInfo& aInfo ) :
  TypeInfo(),
  ranges( aInfo.ranges ),
  elementTypeInfo( aInfo.elementTypeInfo ),
  unconstrainedType( aInfo.isUnconstrainedArrayType() ),
  compositeResolvedType( aInfo.isCompositeResolvedType() ),
  resolutionFunctionId( aInfo.getResolutionFunctionId() ){
}

TypeInfo*
ArrayTypeInfo::getRangeInfo(const int dimension) const {
  TypeInfo *retval = 0;
  if( dimension <= (int)ranges.size() - 1 ){
    retval = ranges[dimension];
  }
  return retval;
}

int
ArrayTypeInfo::get_number_of_elements(const int dimension) const {
  int retval = 0;
  if( dimension <= (int)ranges.size() - 1 ){
    retval = ranges[dimension]->get_range();
  }
  return retval;
}

ArrayTypeInfo &
ArrayTypeInfo::operator=( const ArrayTypeInfo &rhs ){
  if( !ranges.empty() )  {
    for(int i = 0; i < get_dimensions(); i++) {
      delete ranges[i];
    }
    ranges.clear();
  }
  
  for(int i = 0; i < rhs.get_dimensions(); i++) {
    ranges.push_back( rhs.getRangeInfo(i) );
  }
  
  elementTypeInfo       = rhs.getElementTypeInfo();
  unconstrainedType     = rhs.isUnconstrainedArrayType();
  compositeResolvedType = rhs.isCompositeResolvedType();
  resolutionFunctionId  = rhs.getResolutionFunctionId();
  
  return *this;
}

VHDLType *
ArrayTypeInfo::createObject(ObjectBase::ObjectType objType) const {
  return new ArrayType(objType, *this, -1);
}

ArrayInfo
ArrayTypeInfo::getBounds(int dimension) const {
  if (ranges[dimension]->getKind() == RANGE_INFO) {
    IntegerTypeInfo *iti = dynamic_cast<IntegerTypeInfo *>(ranges[dimension]);
    return ArrayInfo(iti->get_left(), iti->get_direction(), iti->get_right());
  }
  else if (ranges[dimension]->getKind() == ENUM_INFO) {
    EnumerationTypeInfo *eti = dynamic_cast<EnumerationTypeInfo *>(ranges[dimension]);
    return ArrayInfo(eti->get_left(), eti->get_direction(), eti->get_right());
  }

  cerr << "Error - ArrayTypeInfo::getBound(int) const encountered an "
       << "unknown/unhandled type_info structure\n";
  
  return nullInfo;
}
