#!/bin/sh
# the next line restarts using wish \
                exec wish "$0" "$@"

#!/usr/local/bin/wish

## This is an API wrapper for the tkchooser
## plugins which allows them to run individually.
## Note that this wrapper does NOT provide ANY
## network modules as all. This is mainly for
## the Enscript plugin
## Ethan Gold <etgold@cs.columbia.edu> updated 3/10/99

## standalone plugin globals
set currplug ""
set plugfilename "Enscript.tcl"

########## plugin API globals ############
#set libdir "/usr/local/lib/tkchooser2"
## /usr/local/etc for finding enscript.cfg
set libdir "/usr/local"
set userconfdir "$env(HOME)/.tkchooser"
set plugindir "plugins"
set propfont "Helvetical -12 bold"
set fixedfont "fixed"

set plug_globals(dummy) dum
unset plug_globals(dummy)
set curr_pluglist_item ""
set errno 1
set debug 0

######### plugin API widgets ##########
set pluglist ".top.pluglist"
set plugframe ".bottom"
set pluglistlabel ".top.pluglabel"

############## build top frame
## a frame for the top right half of the window
frame .top

## a listbox to be made available to plugins
## as defined in the global section
label $pluglistlabel -text "Entities:" -border 2
listbox $pluglist -width 25 -height 12 -exportselection 0 \
	-yscrollcommand [list $pluglist"scroll" set]
scrollbar $pluglist"scroll" -command "$pluglist yview" \
	-takefocus 0

pack $pluglistlabel -side top
pack $pluglist -side left -fill both
pack $pluglist"scroll" -side left -fill y

## an empty frame for use by plugins
## on the bottom right section of the main window
frame $plugframe -width 200 -height 195 -relief groove -border 2

## a status widget in the plugin frame
message $plugframe.status -width 200 -text "ready."
pack $plugframe.status -side bottom -anchor s -fill x

## pack top and bottom halves of rightside
pack .top -side top -anchor n
pack $plugframe -side bottom -fill both

update

#########################################################
## plugin API functions
## functions to return the names of GUI
## elements that the plugins are allowed to manipulate
proc plug_frame {} {
    global plugframe
    return $plugframe
}
proc plug_list {} {
    global pluglist
    return $pluglist
}
proc plug_list_label {} {
    global pluglistlabel
    return $pluglistlabel
}

proc get_curr_item {} {
    global curr_pluglist_item
    return $curr_pluglist_item
}

proc get_glob {index} {
    global plug_globals
    if {[info exists plug_globals($index)]} {
	return $plug_globals($index)
    } else {
	return ""
    }
}

proc set_glob {index value} {
    global plug_globals
    set plug_globals($index) $value
}

proc get_propfont {} {
    global propfont
    return $propfont
}

proc get_fixedfont {} {
    global fixedfont
    return $fixedfont
}

proc debug {} {
    global debug
    return $debug
}

## function to do tilde completion on a filename
proc tilde {name} {
    global env
    ## do tilde expansion
    regsub "~/" $name "$env(HOME)/" newfile
    ## substitute - assume bash variables if USER isn't there
    if {[info exists env(USER)]} {
	regsub $env(USER) $env(HOME) "" tophome
    } else {
	regsub $env(LOGNAME) $env(HOME) "" tophome
    }
    regsub "~" $newfile $tophome file_name
    return $file_name
}

## display an error message
proc error {caller errormessage} {
    global errno
    toplevel .error_$errno -class Dialog
    label .error_$errno.label -relief groove -text "$caller error:"
    message .error_$errno.message -width 300 -text "$errormessage"
    button .error_$errno.ok -text "Ok" -command [list destroy .error_$errno]
    pack .error_$errno.label -fill x
    pack .error_$errno.message -fill x
    pack .error_$errno.ok
    wm title . "tkchooser error #$errno"
    incr errno
}

proc register_plugin {major plugname plugpubname} {
    ## save the plugname for generic starting
    global currplug
    set currplug $plugname
}

################# Internal tkchooser procedures ##################

proc doubleclick_currplug {} {
    global currplug
    $currplug.doubleclick
}

################ UI Event Bindings ###################
## plugin listbox
bind $pluglist <ButtonRelease> {
    if {[debug]} {
	puts "[$pluglist curselection]"
    }
    if {[string compare [$pluglist curselection] ""] != 0} {
	set curr_pluglist_item [$pluglist get [$pluglist curselection]]
    }
}

bind $pluglist <Double-1> {
    #set curr_pluglist_item [selection get]
    if {[string compare [$pluglist curselection] ""] != 0} {
	set curr_pluglist_item [$pluglist get [$pluglist curselection]]
    }
    doubleclick_currplug
}

bind $pluglist <Return> {
    if {[string compare [$pluglist curselection] ""] != 0} {
	set curr_pluglist_item [$pluglist get [$pluglist curselection]]
    }
    doubleclick_currplug    
}

################# Begin Main Code ##################

## only necessary if we're taking an outside module
#if {[file readable $libdir/$plugindir/$plugfilename]} {
#    source $libdir/$plugindir/$plugfilename
#} else {
#    puts "could not file $libdir/$plugindir/$plugfilename"
#    exit
#}


################### PLUGIN CODE HERE ########################
## This is a plugin to provide some kind of printing
## pre-filtering for printing utilities using GNU Enscript.
## It builds an enscript postscript formatter config file
## and relies on the printfilter plugin to choose
## the printfilter that calls enscript.
##
## This plugin uses a comment structure inside enscript config
## files to parse everything for the GUI
##
## Ethan Gold <etgold@cs.columbia.edu> 3/27/98
##

if {[debug]} { puts "loading Enscript plugin..." }

## register the plugin with the main chooser
register_plugin misc enscript Enscript

#### required functions to return plugin information
#### use these instead of globals
proc enscript.geticon {} { return "enscript.ppm" }
proc enscript.getpubname {} { return "Enscript" }
proc enscript.getprotocol {} { return "misc" }
proc enscript.getfilterfile {} { return ".enscriptplug.filter" }
proc enscript.getconfigfile {} { return ".enscriptplug.config" }

####### required functions that actually do stuff (maybe) #########

## start function
proc enscript.start {} {
    ## read in config file and set options as global
    if {[debug]} {puts "enscript: starting enscript plugin"}
    set pluglabel [plug_list_label]
    $pluglabel configure -text "Categories:"
    update
    enscript.widgets "start"
    enscript.readcfg
    enscript.showsections
}

## stop function
proc enscript.stop {} {
    ## global data area will automatically be cleared
    if {[debug]} {puts "enscript: stopping enscript plugin"}
    enscript.widgets "stop"
}

## doubleclick callback
proc enscript.doubleclick {} {
    ## get the value of the double-clicked item
    ## and pass it to the configurator
    enscript.configsection [get_curr_item]
}

## callback for new zone selection
proc enscript.newzone {} {
    ## do nothing
}

####### end required functions ########

proc enscript.widgets {command} {
    set plugframe [plug_frame]

    if {[string compare $command "start"] == 0} {
	button $plugframe.save -text "save" -command {enscript.doSave ""}
	button $plugframe.saveas -text "save as..." -command {enscript.doSaveAs}

	pack $plugframe.save
	pack $plugframe.saveas
	update
    } else {
	destroy $plugframe.save
	destroy $plugframe.saveas
	update
    }
}

## procedure to read in the configuration
## information in the chosen config file
proc enscript.readcfg {} {
    ## Check for global config in tkchooser directory
    ## and for personal config in ~/.enscriptrc. If
    ## the personal one doesn't contain the properly
    ## formatted options, make a note of that or have
    ## and option to use the global instead.

    global plug_globals env

    set plugframe [plug_frame]
    $plugframe.status configure -text "reading enscript config..."
    update

    set sections ""
    set currsection ""

    set configpath [enscript.choosecfg]
    #vwait configpath

    if {[debug]} {puts "enscript: using $configpath"}

    set configrc [open $configpath r]
    gets $configrc line
    if {![regexp {.*tkchooser valid.*} $line]} {
	set currsection "Everything"
	set sections "$sections $currsection"
    }
    
    ## read thru the whole config file
    while {[gets $configrc line] != -1} {
	if {[regexp {^\# tkchooser help.*} $line] } {
	    set directive [lindex $line 3]
	    set mesg [lindex $line 4]
	    set plug_globals($currsection.$directive.help) $mesg
	} elseif {[regexp {^\# tkchooser options.*} $line] } {
	    set directive [lindex $line 3]
	    set options [lrange $line 4 end]
	    set plug_globals($currsection.$directive.options) $options
	    if {[debug]} {puts "enscript: found options \"$options\" for \"$directive\""}
	} elseif {[regexp {^\# tkchooser comment.*} $line] } {
	    set directive [lindex $line 3]
	    set directive [string trim $directive :]
	    set comment [lrange $line 4 end]
	    set plug_globals($currsection.$directive.comment) $comment
	    if {[debug]} {puts "enscript: found comment"}
	} elseif {[regexp {^\# tkchooser section .*} $line] } {
	    set currsection [lindex $line 3]
	    set sections "$sections $currsection"
	    if {[debug]} {puts "enscript: entering section $currsection"}
	} elseif {[regexp {^\#.+} $line] } {
	    ## ignore all other comments - do nothing
	} elseif {[regexp {^$} $line] } {
	    ## ignore blank lines
	} else {
	    ## else we have an enscript directive
	    set directive [string trim [lindex $line 0] :]
	    set value [lrange $line 1 end]
	    if {[string compare $directive "Media"] == 0} {
		set value \"[lrange $line 1 end]\"
		
	    }
	    if {[debug]} {
		puts "enscript: found directive \"$directive\" with value \"$value\""
	    }
	    if {[info exists plug_globals($currsection.$directive)]} {
		lappend plug_globals($currsection.$directive) $value
	    } else {
		set plug_globals($currsection.$directive) $value
	    }
	}	
    }
    set plug_globals(sections) $sections

    $plugframe.status configure -text "ready."
    update
}

## pick the config file to used based
## on contents and location and return
## its path.
proc enscript.choosecfg {} {
    ## The first line must be "# tkchooser valid"

    global libdir plug_globals env
    
    set configpath "$libdir/enscript.cfg"
    ## first look for a personal .enscriptrc and see if
    ## it's valid
    
    if {[file readable $env(HOME)/.enscriptrc]} {
	set homerc [open $env(HOME)/.enscriptrc r]
	gets $homerc line
	close $homerc
	if {[regexp {.*tkchooser valid.*} $line]} {
	    set configpath $env(HOME)/.enscriptrc
	} else {
	    ## Personal .enscriptrc is in the wrong format.
	    ## Use it anyway?
	    if {[enscript.formatquery]} {
		set configpath $env(HOME)/.enscriptrc
	    }
	}	
    }
    return $configpath
}

## procedure to query user if bad
## .enscriptrc is found
proc enscript.formatquery {} {
    set w .enscriptquery

    set homeflag -1

    toplevel $w -class dialog
    message $w.message -width 150 \
	    -text "Your ~/.enscriptrc is not in the enscript \
	    plugin format. Do you want to use it anyway?"
    button $w.yes -text "Yes, use it" -command {set homeflag 1}
    button $w.no -text "No, use default" -command {set homeflag 0}
    pack $w.message
    pack $w.no -side left
    pack $w.yes -side right
    
    vwait homeflag
    destroy $w
    if {[debug]} {puts "enscript: homeflag = $homeflag"}
    return $homeflag

}

## procedure that builds and runs the configuration
## windows for each section.
proc enscript.configsection {section} {
    global plug_globals

    if {[debug]} { puts "enscript: configuring $section" }
    set w ".a$section"
    catch {destroy $w}
    
    set directivelist ""

    foreach item [array names plug_globals "$section.*"] {
	## make sure we only grab the directive entry
	if {![regexp "$section\..*\.options" $item] \
		&& ![regexp "$section\..*\.help" $item]} {
	    if {[debug]} {puts "enscript: found $item"}
	    regsub "$section\." $item "" item
	    set directivelist "$directivelist $item"
	}
    }

    set directivelist [lsort -dictionary $directivelist]
    if {[debug]} {puts "enscript: directivelist is: $directivelist"}
    
    ## create the main window and then
    ## build all the items into it, adding
    ## option lists where available
    toplevel $w -class Dialog
    set f $w.listwin
    frame $w.listwin -relief groove
    canvas $f.canvas -width 325 -height 300 -relief ridge \
	    -yscrollcommand "$f.fscroll set"
    scrollbar $f.fscroll -takefocus 0\
	    -command "$f.canvas yview"
    
    frame $f.canvas.f
    $f.canvas create window 0 0 -anchor nw -window $f.canvas.f

    button $w.close -text "Ok" -command "destroy $w" -default active
    bind $w.close <Return> "$w.close invoke"

    pack $f.canvas -side left -anchor n -pady 4 -padx 4 -fill y -expand on
    pack $f.fscroll -side right -anchor n -fill y -expand on
    pack $f -padx 2 -pady 6 -fill y -expand on
    pack $w.close -side bottom -fill x -expand on
    

    ## each entry will consist of a frame which
    ## contains a label, entry or option list, and help
    
    foreach directive $directivelist {
	set ff $f.canvas.f.frame$directive
	frame $ff -relief groove -width 300
	
	## build help
	button $ff.help$directive -text "help" -takefocus 0\
		-command "enscript.showhelp $section $directive"
	pack $ff.help$directive -side left

	## build label
	label $ff.label$directive -text "$directive"
	pack $ff.label$directive -side left

	## build entry or options
	
	## special cases:
	if {[string compare $directive "DefaultMedia"] == 0} {
	    if {![info exists plug_globals($section.$directive.options)]} {
		foreach item $plug_globals($section.Media) {
		    set medianame [lindex $item 0]
		    if {[debug]} {puts "enscript: adding $medianame to list"}
		    if {[lsearch plug_globals($section.$directive.options) $medianame] != -1} {
			lappend plug_globals($section.$directive.options) $medianame
		    }
		}
	    }
	}
	if {[string compare $directive "Printer"] == 0} {
	    ## call readprintcap to get list of printers
	    foreach item [enscript.getprinters] {
		lappend plug_globals($section.$directive.options) $item
	    }
	}
	
	## normal
	if {[info exists plug_globals($section.$directive.options)]} {
	    menubutton $ff.menu$directive -relief sunken \
		    -text $plug_globals($section.$directive)
	    menu $ff.menu$directive.menu
	    foreach item $plug_globals($section.$directive.options) {
		$ff.menu$directive.menu add command -label $item \
			-command "set \
			plug_globals($section.$directive) $item;\
			$ff.menu$directive configure -text $item"
	    }
	    $ff.menu$directive configure -menu $ff.menu$directive.menu
	    pack $ff.menu$directive -side left
	} else {
	    entry $ff.entry$directive \
		    -textvariable plug_globals($section.$directive)
	    pack $ff.entry$directive -side left -fill x -expand on
	}
	
	pack $ff -anchor w -fill x
    }

    set optionheight [winfo reqheight $ff.help$directive]
    set listheight [expr [llength $directivelist] * $optionheight]
    $f.canvas configure -scrollregion "-1 -1 400 $listheight"
    wm title $w $section
}

## procedure to parse printernames from /etc/printcap
proc enscript.getprinters {} {
    if {[debug]} {puts "enscript: in getprinters"}
    set printerlist ""
    if {[file readable "/etc/printcap"]} {
	set fd [open "/etc/printcap" r]
	while {[gets $fd line] != -1} {
	    if {[regexp {^\#.*} $line]} {
		## do nothing - ignore comments
	    } elseif {[regexp {^.+} $line]} {
		set printer [lindex $line 0]
		if {![regexp {^:.*} $printer]} {
		    set printer [lindex [split $printer |] 0]
		    if {[debug]} {puts "enscript: found printer $printer"}
		    set printerlist "$printerlist $printer"
		}
	    }
	}
    }
    return [lsort -dictionary $printerlist]
}

## procedure to show help window
## for directives
proc enscript.showhelp {section directive} {
    global plug_globals

    if {[info exists plug_globals($section.$directive.help)]} {
	set mesg $plug_globals($section.$directive.help)
    } else {
	set mesg "Sorry, no help available for this item."
    }
    
    set w .help$section$directive
    catch {destroy $w}
    toplevel $w -class Dialog
    message $w.mesg -width 200 -text $mesg -relief groove
    button $w.close -text "Close" -command "destroy $w"
    pack $w.mesg -pady 2 -padx 2
    pack $w.close -fill x

    wm title $w "Help for $directive"

    bind $w <Return> "$w.close invoke"
    bind $w <Control-c> "$w.close invoke"
    bind $w <Control-w> "$w.close invoke"
    bind $w <Meta-w> "$w.close invoke"
}


## query for a file to save the configuration in
proc enscript.doSaveAs {} {
    set w .saveas
    catch {destroy $w}
    
    toplevel $w -class Dialog
    label $w.label -text "Save config as: "
    entry $w.entry -textvariable filename
    button $w.cancel -text "Cancel" -command "destroy $w"
    button $w.save -text "Save" \
	    -command "enscript.doSave \$filename; destroy $w" \
	    -default active

    bind $w <Return> "$w.save invoke"
    
    pack $w.label
    pack $w.entry
    pack $w.cancel -side left -anchor s -padx 4 -pady 4
    pack $w.save -side right -anchor s
}

## Save the enscript configuration to
## ~/.enscriptrc
proc enscript.doSave {homefile} {
    global plug_globals env
    if {[string compare $homefile ""] == 0} {
	set homefile "$env(HOME)/.enscriptrc"
    }

    set homefile [tilde $homefile]

    if {[debug]} {puts "enscript: doSave called with $homefile"}


    if {[file readable $homefile]} {
	file delete $homefile.bak
	file copy $homefile $homefile.bak
	file delete $homefile
    }

    set fd [open $homefile w]

    set validflag "# tkchooser valid"
    set sectionflag "# tkchooser section"
    set helpflag "# tkchooser help"
    set optionflag "# tkchooser options"
    set commentflag "# tkchooser comment"
    set colon ":"

    ## write out valid enscript plugin flag line
    puts $fd "$validflag"
    puts $fd "## This file is automatically generated! Edit with care!!"
    puts $fd ""

    foreach section $plug_globals(sections) {
	puts $fd ""
	puts $fd "#############################################"
	puts $fd "$sectionflag $section"
	puts $fd "#############################################"
	puts $fd ""

	## extract list of directives
	set directivelist ""
	foreach item [array names plug_globals "$section.*"] {
	    ## make sure we only grab the directive entry
	    if {![regexp "$section\..*\.options" $item] \
		    && ![regexp "$section\..*\.help" $item]} {
		if {[debug]} {puts "enscript: wrote $item"}
		regsub "$section\." $item "" item
		set directivelist "$directivelist $item"
	    }
	}
	
	foreach directive $directivelist {
	    puts $fd ""
	    if {[string compare $directive "Media"] == 0} {
		foreach type $plug_globals($section.$directive) {
		    if {[llength $type] > 1} {
			if {[debug]} {puts "enscript: Media: $type"}
			puts $fd "$directive$colon $type"
		    }
		}
		continue
	    }
	    
	    if {[info exists plug_globals($section.$directive.help)]} {
		puts $fd "$helpflag $directive \
			\"$plug_globals($section.$directive.help)\""
	    }
	    if {[info exists plug_globals($section.$directive.options)]} {
		puts $fd "$optionflag $directive \
			$plug_globals($section.$directive.options)"
	    }
	    if {[info exists plug_globals($section.$directive.comment)]} {
		puts $fd "$commentflag $directive$colon \
			$plug_globals($section.$directive.comment)"
	    }
	    puts $fd "$directive$colon $plug_globals($section.$directive)"
	}
    }   
    close $fd
}

## show the section headings in
## the listbox window and bind them
proc enscript.showsections {} {
    global plug_globals

    enscript.clearsections

    set pluglist [plug_list]
    set sections [lsort -dictionary $plug_globals(sections)]
    foreach section $sections {
	$pluglist insert end $section
    }
}

## delete the stuff in the listbox
proc enscript.clearsections {} {
    set pluglist [plug_list]
    $pluglist delete 0 end
}

if {[debug]} { puts "finished loading Enscript plugin." }


################### END PLUGIN CODE  ########################

## now that we've got the plugin's name, start it up
if {$argc == 1} {
    
}


$currplug.start



