/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#include <sys/stat.h>

#include <cerrno>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <string>

#include <gtk/gtkrc.h>

#include <gtk--/main.h>

#include "thoughttracker.h"
#include "app.h"
#include "util.h"

#define DATA_DIR "/.thoughttracker"  // to be appended to getenv("HOME")

string data_dir = "";

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
#ifdef readlink
/* reads the link <filename>, dynamically allocating memory as required;
   the resulting string has to be delete()d by the caller */
char*
readlink_dynamic(const char *filename)
{
  for (int size = 128; ; size *= 2) {
    char *buf = new char[size];
    int n = readlink(filename, buf, size);
    if (n < size) return buf;
    delete buf;
  }
}
#endif

//.............................................................................
/* checks for the presence of the data directory and creates it, if
   necessary */
void
check_environment()
{
  struct stat buf;
  int i;

  if (!data_dir.length()) data_dir = string(getenv("HOME")) + DATA_DIR;
  i = stat(data_dir.c_str(), &buf);
  if (i) {  // stat() failed
    if (errno == ENOENT) {
#ifdef ENABLE_DEBUG
      cerr << "check_environment(): creating directory " << data_dir << endl;
#endif
      if (mkdir(data_dir.c_str(), 0700)) {
        cerr << _("Failed to create data directory ") << data_dir << '.' <<
          endl;
        exit(EXIT_FAILURE);
      }
    } else {
      cerr << _("stat() failed in check_environment().") << endl;
      exit(EXIT_FAILURE);
    }
#if defined(S_ISLNK) && defined(readlink)
  } else {  // stat() succeeded
    for (char *s = (char*) data_dir.c_str(); S_ISLNK(buf.st_mode); ) {
      s = readlink_dynamic(s);
      if (stat(s, &buf)) {
        cerr << _("stat() failed in check_environment().") << endl;
        exit(EXIT_FAILURE);
      }
      delete s;
    }
    if (!S_ISDIR(buf.st_mode)) {
      cerr << _("Data dir link doesn't point to a directory.") << endl;
      exit(EXIT_FAILURE);
    }
#endif
  }
}

//.............................................................................
// prints the command line help screen and exits with <retval>
void
print_help(int retval)
{
  cerr << "ThoughtTracker v" VERSION << endl <<
    endl <<
    COPYRIGHT << ", " << MAILADDR << '.' << endl <<
    _("This program is distributed under the terms of the GNU GPL2, a copy "
      "of\nwhich is included with the source package of this program.") <<
     endl <<
    endl <<
    _("usage: ") << "thoughttracker" << _(" [options] [DIRECTORY]") << endl <<
    endl <<
    _("options:") << endl <<
    "  --display DISPLAY    " << _("use display DISPLAY") << endl <<
    "  --help, -h           " << _("print this help") << endl <<
    "  --version, -v        " << _("print just the version") << endl <<
    endl <<
    _("For more information on how to use this program, read its man page,")
      << endl <<
    endl <<
    "  `man thoughttracker`." << endl <<
    endl <<
    _("For the newest release, visit ThoughtTracker's home page at") <<
      endl <<
    HOMEPAGE << '.' << endl;
  exit(retval);
}

//.............................................................................
/* parses the list of command line arguments before GTK's routine */
void
parse_argv_before_gtk(int &argc, char **&argv)
{
  
  for (int i = 1; i < argc; i++) {
    if (!strcmp(argv[i], "--help") || !strcmp(argv[i], "-h"))
      print_help(EXIT_SUCCESS);
    else if (!strcmp(argv[i], "--version") || !strcmp(argv[i], "-v")) {
      cerr << "ThoughtTracker v" VERSION << endl;
      exit(EXIT_SUCCESS);
    }
  }
}

//.............................................................................
// parses command line arguments after GTK's init has removed those understood
// by it
void
parse_argv_after_gtk(int &argc, char **&argv)
{
  for (int i = 1; i < argc; i++) {
    if (*argv[i] != '-' && !data_dir.length())  // alternative data dir
      data_dir = argv[i];
    // check if handled by parse_argv_before_gtk()
    else if (strcmp(argv[i], "--help") && strcmp(argv[i], "--version"))
      print_help(EXIT_FAILURE);  // else exit
  }
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
int
main(int argc, char **argv)
{
#ifdef ENABLE_NLS
  gtk_set_locale();
  bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
  textdomain(PACKAGE);
#endif

  parse_argv_before_gtk(argc, argv);
#ifdef ENABLE_NLS
  Gtk::Main gtk(argc, argv, true);
#else
  Gtk::Main gtk(argc, argv, false);
#endif
  parse_argv_after_gtk(argc, argv);
  check_environment();
  gtk_rc_parse((data_dir + "/gtkrc").c_str());

  TTApplication app(data_dir);
  app.show();
  gtk.run();

  return EXIT_SUCCESS;
}

