#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif


#include "lib.h"
#include "stardict.h"
#include <sys/stat.h>
#include "string.h"
#include "zlib.h"
#ifdef HAVE_MMAP
#include <sys/types.h>
#include <fcntl.h>
#include <sys/mman.h>
#endif

// Notice: read src/tools/DICTFILE_FORMAT for the dictionary file's format information!

cacheItem::cacheItem()
{
	data= NULL;
}

cacheItem::~cacheItem()
{
	if (data)
		g_free(data);
}


//===================================================================
DictBase::DictBase()
{
	sametypesequence = NULL;
	dictfile = NULL;
	dictdzfile = NULL;
	cache_cur =0;
}

DictBase::~DictBase()
{
	g_free(sametypesequence);
	if (dictfile)
		fclose(dictfile);
	if (dictdzfile)
		dict_data_close(dictdzfile);
}

gchar* DictBase::GetWordData(glong idxitem_offset, glong idxitem_size)
{
	for (int i=0;i<WORDDATA_CACHE_NUM;i++)
	{
		if ((cache[i].data) && (cache[i].offset == idxitem_offset))
		{
			return cache[i].data;
		}
	}
	if (dictfile) {
		fseek(dictfile, idxitem_offset, SEEK_SET);
	}
	gchar *data;
	if (sametypesequence) {
		gchar *origin_data;
		origin_data = (gchar *)g_malloc(idxitem_size);
		
		if (dictfile)
			fread(origin_data,idxitem_size,1,dictfile);
		else
			dict_data_read (dictdzfile, origin_data, idxitem_offset, idxitem_size);
	
		glong data_size;
		gint sametypesequence_len;
		sametypesequence_len = strlen(sametypesequence);
		//there have sametypesequence_len char being omitted.
		data_size = idxitem_size + sizeof(glong) + sametypesequence_len;
		//if the last item's size is determined by the end up '\0',then +=sizeof(gchar);
		//if the last item's size is determined by the head glong type data,then +=sizeof(glong);
		switch (sametypesequence[sametypesequence_len-1]) {
			case 'm':
			case 't':
			case 'y':
			case 'o':
				data_size += sizeof(gchar);
				break;
			case 'W':
			case 'P':
				data_size += sizeof(glong);				
				break;
		}			
		data = (gchar *)g_malloc(data_size);
		gchar *p1,*p2;
		p1 = data + sizeof(glong);
		p2 = origin_data;
		glong sec_size;
		//copy the head items.
		for (int i=0;i< sametypesequence_len-1;i++) {
			memcpy(p1, &(sametypesequence[i]), sizeof(gchar));
			p1+= sizeof(gchar);
			switch (sametypesequence[i]) {
				case 'm':
				case 't':
				case 'y':
				case 'o':
					sec_size = strlen(p2)+1;
					memcpy(p1, p2, sec_size);
					p1+= sec_size;
					p2+= sec_size;
					break;
				case 'W':
				case 'P':
					memcpy(&sec_size, p2, sizeof(glong));
					sec_size += sizeof(glong);
					memcpy(p1, p2, sec_size);
					p1+= sec_size;
					p2+= sec_size;
					break;
			}							
		}	
		//calculate the last item 's size.
		sec_size = idxitem_size - (p2-origin_data);
		memcpy(p1, &(sametypesequence[sametypesequence_len-1]), sizeof(gchar));
		p1+= sizeof(gchar);
		switch (sametypesequence[sametypesequence_len-1]) {
			case 'm':
			case 't':
			case 'y':
			case 'o':
				memcpy(p1, p2, sec_size);
				p1 += sec_size;				
				memcpy(p1, "", sizeof(gchar)); //add the end up '\0';
				break;
			case 'W':
			case 'P':
				memcpy(p1,&(sec_size), sizeof(glong)); //add the head glong size data.
				p1 += sizeof(glong);
				memcpy(p1, p2, sec_size);
				break;
		}		
		g_free(origin_data);		
	}
	else {		
		data = (gchar *)g_malloc(idxitem_size + sizeof(glong));
		if (dictfile)
			fread(data+sizeof(glong),idxitem_size,1,dictfile);		
		else
			dict_data_read (dictdzfile, data+sizeof(glong), idxitem_offset, idxitem_size);
	}	
	memcpy(data,&(idxitem_size),sizeof(glong));
	if (cache[cache_cur].data)
	{
		g_free(cache[cache_cur].data);
	}
	cache[cache_cur].data = data;
	cache[cache_cur].offset = idxitem_offset;
	cache_cur++;
	if (cache_cur==WORDDATA_CACHE_NUM)
		cache_cur =0;
    return data;
}


//===================================================================
Lib::Lib()
{
    wordcount=0;
	bookname = NULL;
	idxfile = NULL;
	wordlist = NULL;
#ifdef HAVE_MMAP
	mmap_fd = -1;
	mmap_idxmap_size = 0;
#endif
	idxdatabuffer = NULL;
}

Lib::~Lib()
{
	if (bookname)
		g_free(bookname);
	if (idxfile) {
		fclose(idxfile);
		if (wordoffset)
			g_free(wordoffset);
	}
	else {
		if (wordlist)
			g_free(wordlist);

#ifdef HAVE_MMAP
		if (mmap_fd >= 0) {
			if (mmap_idxmap_size)
				munmap(idxdatabuffer, mmap_idxmap_size);
			close(mmap_fd);
		}
		else {
			if (idxdatabuffer)
				g_free(idxdatabuffer);
		}
#else
		if (idxdatabuffer)
			g_free(idxdatabuffer);
#endif
	}
}

gboolean Lib::load(const char *ifofilename)
{	
	gulong idxfilesize;
	if (!load_ifofile(ifofilename, &idxfilesize))
		return false;

	gchar fullfilename[256];
	
	strcpy(fullfilename, ifofilename);
	strcpy(fullfilename+strlen(fullfilename)-sizeof("ifo") +1, "dict.dz");	
	
	if (g_file_test(fullfilename, G_FILE_TEST_EXISTS)) {
		dictdzfile = dict_data_open(fullfilename, 0);
		if (!dictdzfile)
		{
			//g_print("open file %s failed!\n",fullfilename);
			return false;
		}
	}
	else {
		fullfilename[strlen(fullfilename)-3] = '\0';
		dictfile = fopen(fullfilename,"rb");
		if (!dictfile)
		{
			//g_print("open file %s failed!\n",fullfilename);
			return false;
		}
	}

	strcpy(fullfilename, ifofilename);
	strcpy(fullfilename+strlen(fullfilename)-sizeof("ifo") +1, "idx.gz");	
	
	if (g_file_test(fullfilename, G_FILE_TEST_EXISTS)) {
		gzFile in;	
		in = gzopen(fullfilename,"rb");
		if (in == NULL) {
			//g_print("Open file %s failed!\n",fullfilename);
			return false;
		}
		
		idxdatabuffer = (gchar *)g_malloc(idxfilesize);
		
		gulong len;
		len = gzread(in, idxdatabuffer, idxfilesize);
		if (len < 0)
			return false;
		gzclose(in);
		if (len != idxfilesize)
			return false;
	}
	else {
		fullfilename[strlen(fullfilename)-3] = '\0';

#ifdef HAVE_MMAP
		if ((mmap_fd = open(fullfilename, O_RDONLY )) < 0) {
			//g_print("Open file %s failed!\n",fullfilename);
			return false;
		}
		idxdatabuffer = (gchar *)mmap( NULL, idxfilesize, PROT_READ, MAP_SHARED, mmap_fd, 0);		
		if ((void *)idxdatabuffer == (void *)(-1)) {
			//g_print("mmap file %s failed!\n",idxfilename);
			return false;
		}
		mmap_idxmap_size = idxfilesize;
#else
  #ifdef _WIN32
		HANDLE hFile = CreateFile(fullfilename, GENERIC_READ, 0, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0);
		HANDLE hFileMap = CreateFileMapping(hFile, NULL, PAGE_READONLY, 0,  idxfilesize, NULL);
		idxdatabuffer = (gchar *)MapViewOfFile(hFileMap, FILE_MAP_READ, 0, 0, idxfilesize);
  #else
		FILE *file;
		if (!(file = fopen (fullfilename, "rb"))) {
			//g_print("Open file %s failed!\n",fullfilename);
			return false;
		}
		idxdatabuffer = (gchar *)g_malloc(idxfilesize);
		gint read_len;
		read_len = fread (idxdatabuffer, 1, idxfilesize, file);
		fclose (file);
		if (read_len!=idxfilesize)
			return false;		
  #endif
#endif
		if (true) {
			loadwordoffset();
#ifdef HAVE_MMAP
			munmap(idxdatabuffer, mmap_idxmap_size);
			idxdatabuffer = NULL;
			mmap_idxmap_size = 0;
			close(mmap_fd);
			mmap_fd = -1;
#else
  #ifdef _WIN32
			UnmapViewOfFile(idxdatabuffer);
			idxdatabuffer = NULL;
			CloseHandle(hFileMap);
			CloseHandle(hFile);
  #else
			g_free(idxdatabuffer);
			idxdatabuffer = NULL;
  #endif
#endif			
			if (!(idxfile = fopen (fullfilename, "rb"))) {
				if (wordoffset) {
					g_free(wordoffset);
					wordoffset = NULL;
				}
				return false;
			}
			cur_wordindex = -2;	// so it is always invalid in GetWord();
			g_print("bookname: %s , wordcount %ld\n",bookname, wordcount);
			return true;
		}
		else {
			// The .idx file will load into memory. Who need this opinion?
		}
	}

	loadwordlist();
	g_print("bookname: %s , wordcount %ld\n",bookname, wordcount);
	return true;
}

gboolean Lib::load_ifofile(const char *ifofilename, gulong *idxfilesize)
{
	struct stat stats;		
	if (stat (ifofilename, &stats) == -1) {
		//g_print("File: %s don't exist!\n",idxfilename);
		return false;
	}
		
	FILE *file;
	if (!(file = fopen (ifofilename, "rb"))) {
		//g_print("Open file %s failed!\n",idxfilename);
		return false;
	}
	gchar *buffer = (gchar *)g_malloc (stats.st_size + 1);
	fread (buffer, 1, stats.st_size, file);
	buffer[stats.st_size] = '\0';
	fclose (file);
	
	if (!g_str_has_prefix(buffer, "StarDict's dict ifo file\nversion=2.4.2\n")) {
		g_print("Bad dict ifo file %s, skiped!\n", ifofilename);
		g_free(buffer);
		return false;
	}
	gchar *p1= buffer + sizeof("StarDict's dict ifo file\nversion=2.4.2\n")-1 -1;
	
	gchar *p2,*p3;

	p2 = strstr(p1,"\nidxfilesize=");
	if (!p2) {
		g_free(buffer);
		return false;
	}
	p3 = strchr(p2+ sizeof("\nidxfilesize=")-1,'\n');
	gchar *tmpstr = (gchar *)g_memdup(p2+sizeof("\nidxfilesize=")-1, p3-(p2+sizeof("\nidxfilesize=")-1)+1);
	tmpstr[p3-(p2+sizeof("\nidxfilesize=")-1)] = '\0';
	*idxfilesize = atol(tmpstr);
	g_free(tmpstr);

	p2 = strstr(p1,"\nwordcount=");
	if (!p2) {
		g_free(buffer);
		return false;
	}
	p3 = strchr(p2+ sizeof("\nwordcount=")-1,'\n');
	tmpstr = (gchar *)g_memdup(p2+sizeof("\nwordcount=")-1, p3-(p2+sizeof("\nwordcount=")-1)+1);
	tmpstr[p3-(p2+sizeof("\nwordcount=")-1)] = '\0';
	wordcount = atol(tmpstr);
	g_free(tmpstr);

	p2 = strstr(p1,"\nbookname=");
	if (!p2) {
		g_free(buffer);
		return false;
	}
	p3 = strchr(p2+ sizeof("\nbookname=")-1,'\n');
	bookname = (gchar *)g_memdup(p2+sizeof("\nbookname=")-1, p3-(p2+sizeof("\nbookname=")-1)+1);
	bookname[p3-(p2+sizeof("\nbookname=")-1)] = '\0';

	p2 = strstr(p1,"\nsametypesequence=");
	if (p2) {		
		p3 = strchr(p2+sizeof("\nsametypesequence=")-1,'\n');
		sametypesequence = (gchar *)g_memdup(p2+sizeof("\nsametypesequence=")-1, p3-(p2+sizeof("\nsametypesequence=")-1)+1);
		sametypesequence[p3-(p2+sizeof("\nsametypesequence=")-1)] = '\0';
	}

	g_free(buffer);
	return true;
}

void Lib::loadwordlist()
{
	wordlist = (gchar **)g_malloc((wordcount+1) * sizeof(gchar *));	
	gchar *p1 = idxdatabuffer;
	for (int i=0;i<wordcount;i++) {		
		wordlist[i] = p1;
		p1 += strlen(p1) +1 + 2*sizeof(glong);
	}
	wordlist[wordcount] = p1;
}

void Lib::loadwordoffset()
{
	// we can save the offsets in a file too, then just read it into the wordoffset array, or mmap it, this will make the loading be very fast.
	// If you want to develop a command line version of StarDict, you can try this technique.
	wordoffset = (glong *)g_malloc((wordcount+1) * sizeof(glong));	
	gchar *p1 = idxdatabuffer;
	for (int i=0;i<wordcount;i++) {		
		wordoffset[i] = p1 - idxdatabuffer;
		p1 += strlen(p1) +1 + 2*sizeof(glong);
		// We can check the word len < 256 here.
		// or we can save the max word length, then wordentry_buf=g_malloc(max_wordlen);
		// but wordentry_buf[256] should be enough for most case. Get the max length will slow down the loading a little.
	}
	wordoffset[wordcount] = p1 - idxdatabuffer;
}

//the next function is too slow as it call fgetc() frequently.
/*void Lib::loadwordoffset()
{
	wordoffset = (glong *)g_malloc((wordcount+1) * sizeof(glong));	

	glong p1 = 0;
	for (int i=0;i<wordcount;i++) {		
		wordoffset[i] = p1;
		while (fgetc(idxfile)!=0)
			p1++;
		p1 = p1 + 1 + 2*sizeof(glong);
		fseek(idxfile, p1, SEEK_SET);
	}
	wordoffset[wordcount] = p1;
}*/

gboolean Lib::Lookup(const char* sWord,glong *pIndex)
{
    gboolean bFound=false;		
    glong iTo=length()-1;
	if (stardict_strcmp(sWord, GetWord(0))<0) {
		*pIndex = 0;
	}
	else if (stardict_strcmp(sWord, GetWord(iTo)) >0 ) {
		*pIndex = INVALID_INDEX;
	}
	else {
	    glong iThisIndex=0;
    	glong iFrom=0;

		int cmpint;
    	while( !bFound && iFrom<=iTo )
    	{
        	iThisIndex=(iFrom+iTo)/2;
			cmpint = stardict_strcmp(sWord, GetWord(iThisIndex));
			//g_print("lookup %s %d\n",GetWord(iThisIndex),cmpint);
			if (cmpint == 0)
			{
				bFound=true;
			}
			else if (cmpint > 0)
			{
				iFrom=iThisIndex+1;
			}
			else
			{
				iTo=iThisIndex-1;
			}
    	}
	    if (!bFound)
	    {
			/*glong len = g_utf8_strlen(sWord, -1);
			gchar *last_str = g_utf8_offset_to_pointer(sWord, len-1);
			gunichar last = g_utf8_get_char(last_str);
			if (((g_unichar_isspace(last) || g_unichar_ispunct(last)) || g_unichar_isdigit(last))
				&& (g_ascii_strncasecmp(sWord, GetWord(iTo), (last_str - sWord))==0))
				*pIndex = iTo;      //previous
			else 
            	*pIndex = iFrom;    //next
			*/
			*pIndex = iFrom;    //next
    	}
		else
			*pIndex = iThisIndex;		
	}		
    return(bFound);
}

gboolean Lib::LookupWithRule(GPatternSpec *pspec,glong *aIndex,int iBuffLen)
{
    int iIndexCount=0;
    glong i;
    for(i=0;i<length() && iIndexCount<iBuffLen-1;i++)
    {
        if(g_pattern_match_string(pspec, GetWord(i)))
        {
            aIndex[iIndexCount++]=i;
        }
    }
    aIndex[iIndexCount]= -1; // -1 is the end.	
	
    return(iIndexCount>0);
}

gchar *
Lib::GetWord(glong index)
{
	if (idxfile) {
		if (index == cur_wordindex +1) {
			//needn't fseek().
		}
		// (index == cur_wordindex) seldom happen, so don't determine this here.
		else {
			fseek(idxfile, wordoffset[index], SEEK_SET);
		}
		cur_wordindex = index;
		
		fread(wordentry_buf, wordoffset[index+1] - wordoffset[index] - 2*sizeof(glong), 1, idxfile);
		//g_print("%s\n", wordentry_buf);
		fread(&wordentry_offset, sizeof(glong), 1, idxfile);
		wordentry_offset = g_ntohl(wordentry_offset);
		fread(&wordentry_size, sizeof(glong), 1, idxfile);
		wordentry_size = g_ntohl(wordentry_size);
		return wordentry_buf;
	}
	else {
		return wordlist[index];
	}
}

gchar *
Lib::GetWordData(glong index)
{
	if (idxfile) {
		if (index == cur_wordindex) {
			// wordentry_offset and wordentry_size are already cached by GetWord();
		}
		else {
			cur_wordindex = index;
			fseek(idxfile, wordoffset[index+1] - 2*sizeof(glong), SEEK_SET);
			fread(&wordentry_offset, sizeof(glong), 1, idxfile);
			wordentry_offset = g_ntohl(wordentry_offset);
			fread(&wordentry_size, sizeof(glong), 1, idxfile);
			wordentry_size = g_ntohl(wordentry_size);			
		}		
		return DictBase::GetWordData(wordentry_offset, wordentry_size);
	}
	else {
		gchar *p1 = wordlist[index+1] - 2*sizeof(glong);
		glong offset, size;
		memcpy(&offset,p1,sizeof(glong));
		offset = g_ntohl(offset);
		p1 = p1 + sizeof(glong);
		memcpy(&size, p1, sizeof(glong));
		size = g_ntohl(size);
		return DictBase::GetWordData(offset, size);
	}
}

//===================================================================
Libs::Libs()
{
	libcount =0;
	oLib = NULL;
}

Libs::~Libs()
{
	if (libcount)
	{
		for (int i=0;i< libcount;i++)
		{
			delete oLib[i];
		}	
		g_free(oLib);
	}
}

/********************************************************************/
glong Libs::iLength(int iLib)
{
    return (oLib[iLib])->length();
}

gchar* Libs::GetBookname(int iLib)
{
    return (oLib[iLib])->GetBookname();
}

/********************************************************************/
gboolean Libs::LookdupWordsWithRule(GPatternSpec *pspec,glong* aiIndexes,int iLen,int iLib)
{
	return (oLib[iLib]->LookupWithRule(pspec,aiIndexes,iLen));
}

void Libs::LoadDir(gchar *dirname, GSList *order_list, GSList *disable_list)
{	
	GDir *dir = g_dir_open(dirname, 0, NULL);	
	if (dir)
	{
		const gchar *filename;	
		gchar fullfilename[256];
		Lib *lib;
		gboolean loaded;
		GSList *tmplist1,*tmplist2;
		gboolean disabled;
		while ((filename = g_dir_read_name(dir))!=NULL)
		{	
			sprintf(fullfilename, "%s/%s", dirname, filename);
			if (g_file_test(fullfilename, G_FILE_TEST_IS_DIR)) {
				LoadDir(fullfilename, order_list, disable_list);
			}
			else if (g_str_has_suffix(filename,".ifo")) {				
				tmplist1 = order_list;
				loaded = false;
				while (tmplist1) {
					if (strcmp((gchar *)(tmplist1->data), fullfilename) == 0) {
						loaded = true;
						break;
					}
					tmplist1 = g_slist_next(tmplist1);
				}
				if (loaded)
					continue;
				
				tmplist2 = disable_list;
				disabled = false;
				while (tmplist2) {
					if (strcmp((gchar *)(tmplist2->data), fullfilename) == 0) {
						disabled = true;
						break;
					}
					tmplist2 = g_slist_next(tmplist2);
				}
				if (disabled)
					continue;
				
				lib = new Lib;
				if (lib->load(fullfilename))
				{
					libcount++;
					oLib = (Lib **)g_realloc(oLib,libcount * sizeof(Lib *));
					oLib[libcount-1] = lib;
				}
				else
				{
					delete lib;
				}				
			}
		}		
		g_dir_close(dir);
	}	
}

void Libs::Load()
{
	GSList *order_list, *disable_list;
#ifdef _WIN32
	rw_cfg_read_strlist (usercfgfile, "manage_dictionaries", "dict_order_list", &order_list);
	rw_cfg_read_strlist (usercfgfile, "manage_dictionaries", "dict_disable_list", &disable_list);
#else
	gpAppFrame->oAppConf.read_list("/apps/stardict/manage_dictionaries/dict_order_list", GCONF_VALUE_STRING, &order_list);
	gpAppFrame->oAppConf.read_list("/apps/stardict/manage_dictionaries/dict_disable_list", GCONF_VALUE_STRING, &disable_list);
#endif
	gchar *idxfilename;
	GSList *tmplist1,*tmplist2;
	gboolean disabled;
	Lib *lib;
	tmplist1 = order_list;	
	while (tmplist1) {
		idxfilename = (gchar *)(tmplist1->data);
		tmplist1 = g_slist_next(tmplist1);
		tmplist2 = disable_list;
		disabled = false;
		while (tmplist2) {
			if (strcmp((gchar *)(tmplist2->data), idxfilename) == 0) {
				disabled = true;
				break;
			}
			tmplist2 = g_slist_next(tmplist2);
		}
		if (disabled)
			continue;

		lib = new Lib;
		if (lib->load(idxfilename)) {
			libcount++;
			oLib = (Lib **)g_realloc(oLib,libcount * sizeof(Lib *));
			oLib[libcount-1] = lib;
		}
		else {
			delete lib;
		}					
	}

#ifdef _WIN32
	gchar *filename = g_build_filename(stardict_data_dir, "dic", NULL);
	LoadDir(filename, order_list, disable_list);
	g_free(filename);
#else
	gchar home_dir[256];
	sprintf(home_dir, "%s/.stardict/dic", g_get_home_dir());
	LoadDir(home_dir, order_list, disable_list);
	LoadDir(STARDICT_DATA_DIR "/dic", order_list, disable_list);
#endif

	g_slist_foreach (order_list, (GFunc)g_free, NULL);
	g_slist_free(order_list);
	
	g_slist_foreach (disable_list, (GFunc)g_free, NULL);
	g_slist_free(disable_list);	
}

gchar *
Libs::poGetWord(glong iIndex,int iLib)
{
    return( oLib[iLib]->GetWord(iIndex) );
}

gchar *
Libs::poGetWordData(glong iIndex,int iLib)
{
	if (iIndex==INVALID_INDEX)
		return NULL;
    return( oLib[iLib]->GetWordData(iIndex) );
}

gchar *
Libs::poGetCurrentWord(glong * iCurrent)
{
	gchar * poCurrentWord = NULL;
	gchar *word;
    for (int iLib=0;iLib<libcount;iLib++) {
		if (iCurrent[iLib]==INVALID_INDEX)
            continue;
		if ( iCurrent[iLib]>=iLength(iLib) || iCurrent[iLib]<0)
			continue;
		if ( poCurrentWord == NULL ) {
			poCurrentWord = poGetWord(iCurrent[iLib],iLib);
		}
		else {
			word = poGetWord(iCurrent[iLib],iLib);
			if (stardict_strcmp(poCurrentWord, word) > 0 )
				poCurrentWord = word;
		}
	}
    return poCurrentWord;
}

gchar *
Libs::poGetNextWord(const gchar *sWord, glong * iCurrent)
{
	// the input can be:
	// (word,iCurrent),read word,write iNext to iCurrent,and return next word. used by TopWin::NextCallback();
	// (NULL,iCurrent),read iCurrent,write iNext to iCurrent,and return next word. used by AppCore::ListWords();
    gchar * poCurrentWord = NULL;
    gint iCurrentLib=0;

	gchar *word;
    for (int iLib=0;iLib<libcount;iLib++) {
		if (sWord)
			oLib[iLib]->Lookup(sWord, &(iCurrent[iLib]));
        if (iCurrent[iLib]==INVALID_INDEX)
            continue;
		if ( iCurrent[iLib]>=iLength(iLib) || iCurrent[iLib]<0)
			continue;
		if ( poCurrentWord == NULL ) {
			poCurrentWord = poGetWord(iCurrent[iLib],iLib);
			iCurrentLib = iLib;
		}
		else {
			word = poGetWord(iCurrent[iLib],iLib);
			if (  stardict_strcmp(poCurrentWord, word) > 0 ) {
				poCurrentWord = word;
				iCurrentLib = iLib;
			}
        }
    }
    if (poCurrentWord) {
        iCurrent[iCurrentLib]++;
		for (int iLib=0;iLib<libcount;iLib++) {
			if (iLib == iCurrentLib)
				continue;
			if (iCurrent[iLib]==INVALID_INDEX)
	            continue;
			if ( iCurrent[iLib]>=iLength(iLib) || iCurrent[iLib]<0)
				continue;
			if (strcmp(poCurrentWord, poGetWord(iCurrent[iLib],iLib)) == 0 )
				iCurrent[iLib]++;
		}
		poCurrentWord = poGetCurrentWord(iCurrent);
	}
    return poCurrentWord;
}


gchar *
Libs::poGetPreWord(glong * iCurrent)
{
	// used by TopWin::PreviousCallback(); the iCurrent is cached by AppCore::TopWinWordChange();
    gchar * poCurrentWord = NULL;
    int iCurrentLib=0;

	gchar *word;
    for (int iLib=0;iLib<libcount;iLib++) {
		if (iCurrent[iLib]==INVALID_INDEX)
            continue;
		if ( iCurrent[iLib]>=iLength(iLib) || iCurrent[iLib]<=0)
			continue;
		if ( poCurrentWord == NULL ) {
			poCurrentWord = poGetWord(iCurrent[iLib]-1,iLib);
			iCurrentLib = iLib;
		}
		else {
			word = poGetWord(iCurrent[iLib]-1,iLib);
			if (stardict_strcmp(poCurrentWord, word) < 0 ) {
				poCurrentWord = word;
				iCurrentLib = iLib;
			}
		}
	}
	
    if (poCurrentWord) {
        iCurrent[iCurrentLib]--;
		for (int iLib=0;iLib<libcount;iLib++) {
			if (iLib == iCurrentLib)
				continue;
			if (iCurrent[iLib]==INVALID_INDEX)
	            continue;
			if ( iCurrent[iLib]>=iLength(iLib) || iCurrent[iLib]<=0)
				continue;
			if (strcmp(poCurrentWord, poGetWord(iCurrent[iLib]-1,iLib)) == 0 )
				iCurrent[iLib]--;
		}
	}
    return poCurrentWord;
}

gboolean Libs::LookupWord(const gchar* sWord,glong& iWordIndex,int iLib)
{
	return (oLib[iLib]->Lookup(sWord, &iWordIndex));
}

gboolean Libs::LookupSimilarWord(const gchar* sWord,glong& iWordIndex,int iLib)
{
    glong iIndex;
    gboolean bFound=false;
	gchar *casestr;

	if (!bFound) {
		// to lower case.
		casestr = g_utf8_strdown(sWord, -1);
		if (strcmp(casestr, sWord)) {
			if(oLib[iLib]->Lookup(casestr,&iIndex))
				bFound=true;
		}
		g_free(casestr);
		// to upper case.
		if (!bFound) {
			casestr = g_utf8_strup(sWord, -1);
			if (strcmp(casestr, sWord)) {
				if(oLib[iLib]->Lookup(casestr,&iIndex))
					bFound=true;
			}
			g_free(casestr);
		}	
		// to upper the first character.
		if (!bFound) {
			gchar *nextchar = g_utf8_next_char(sWord);
			gchar *firstchar = g_utf8_strup(sWord, nextchar - sWord);
			casestr = g_strdup_printf("%s%s", firstchar, nextchar);
			g_free(firstchar);
			if (strcmp(casestr, sWord)) {
				if(oLib[iLib]->Lookup(casestr,&iIndex))
					bFound=true;
			}
			g_free(casestr);
		}	
	}
	
    if (bIsPureEnglish(sWord))
    {		
        // If not Found , try other status of sWord.
        int iWordLen=strlen(sWord);
		gboolean isupcase;
		
		gchar *sNewWord = (gchar *)g_malloc(iWordLen + 1);

        //cut one char "s" or "d"
        if(!bFound && iWordLen>1) {
			isupcase = (sWord[iWordLen-1]=='S' || (!strncmp(&sWord[iWordLen-2],"ED",2)));
			if (isupcase || sWord[iWordLen-1]=='s' || (!strncmp(&sWord[iWordLen-2],"ed",2))) {
	            strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-1]='\0'; // cut "s" or "d"
    	        if(oLib[iLib]->Lookup(sNewWord,&iIndex))
        	        bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
			}
        }
        
        //cut "ly"
        if(!bFound && iWordLen>2) {
			isupcase = !strncmp(&sWord[iWordLen-2],"LY",2);
			if (isupcase || (!strncmp(&sWord[iWordLen-2],"ly",2))) {
	            strcpy(sNewWord,sWord);
    	        sNewWord[iWordLen-2]='\0';  // cut "ly"
        	    if ( iWordLen>5 && (sNewWord[iWordLen-3]==sNewWord[iWordLen-4])
            	     && !bIsVowel(sNewWord[iWordLen-4]) && bIsVowel(sNewWord[iWordLen-5]) )   //doubled
	            {
    	            sNewWord[iWordLen-3]='\0';
        	        if( oLib[iLib]->Lookup(sNewWord,&iIndex) )
            	        bFound=true;
                	else {
						if (isupcase || g_ascii_isupper(sWord[0])) {
							casestr = g_ascii_strdown(sNewWord, -1);
							if (strcmp(casestr, sNewWord)) {
								if(oLib[iLib]->Lookup(casestr,&iIndex))
									bFound=true;
							}
							g_free(casestr);
						}
						if (!bFound)
							sNewWord[iWordLen-3]=sNewWord[iWordLen-4];  //restore
					}					                    	
	            }
    	        if (!bFound) {
					if (oLib[iLib]->Lookup(sNewWord,&iIndex))
        	        	bFound=true;
					else if (isupcase || g_ascii_isupper(sWord[0])) {
						casestr = g_ascii_strdown(sNewWord, -1);
						if (strcmp(casestr, sNewWord)) {
							if(oLib[iLib]->Lookup(casestr,&iIndex))
								bFound=true;
						}
						g_free(casestr);
					}
        		}
			}
		}
        
        //cut "ing"
        if(!bFound && iWordLen>3) {
			isupcase = !strncmp(&sWord[iWordLen-3],"ING",3);
			if (isupcase || !strncmp(&sWord[iWordLen-3],"ing",3) ) {
	            strcpy(sNewWord,sWord);
    	        sNewWord[iWordLen-3]='\0';
        	    if ( iWordLen>6 && (sNewWord[iWordLen-4]==sNewWord[iWordLen-5])
            	     && !bIsVowel(sNewWord[iWordLen-5]) && bIsVowel(sNewWord[iWordLen-6]) )   //doubled
	            {
    	            sNewWord[iWordLen-4]='\0';
        	        if (oLib[iLib]->Lookup(sNewWord,&iIndex))
            	        bFound=true;
                	else {
						if (isupcase || g_ascii_isupper(sWord[0])) {
							casestr = g_ascii_strdown(sNewWord, -1);
							if (strcmp(casestr, sNewWord)) {
								if(oLib[iLib]->Lookup(casestr,&iIndex))
									bFound=true;
							}
							g_free(casestr);
						}
						if (!bFound)
                    		sNewWord[iWordLen-4]=sNewWord[iWordLen-5];  //restore
					}
	            }
    	        if( !bFound ) {
					if (oLib[iLib]->Lookup(sNewWord,&iIndex))
        	        	bFound=true;
					else if (isupcase || g_ascii_isupper(sWord[0])) {
						casestr = g_ascii_strdown(sNewWord, -1);
						if (strcmp(casestr, sNewWord)) {
							if(oLib[iLib]->Lookup(casestr,&iIndex))
								bFound=true;
						}
						g_free(casestr);
					}						
				}
            	if(!bFound)
	            {
					if (isupcase)
						strcat(sNewWord,"E"); // add a char "E"
					else
    	            	strcat(sNewWord,"e"); // add a char "e"
        	        if(oLib[iLib]->Lookup(sNewWord,&iIndex))
            	        bFound=true;
					else if (isupcase || g_ascii_isupper(sWord[0])) {
						casestr = g_ascii_strdown(sNewWord, -1);
						if (strcmp(casestr, sNewWord)) {
							if(oLib[iLib]->Lookup(casestr,&iIndex))
								bFound=true;
						}
						g_free(casestr);
					}						
            	}
			}
        }

        //cut two char "es"
        if(!bFound && iWordLen>3) {
			isupcase = (!strncmp(&sWord[iWordLen-2],"ES",2) && 
				(sWord[iWordLen-3] == 'S' || sWord[iWordLen-3] == 'X' || sWord[iWordLen-3] == 'O'
				|| (iWordLen >4 && sWord[iWordLen-3] == 'H' && (sWord[iWordLen-4] == 'C' || sWord[iWordLen-4] == 'S'))));
			if (isupcase || 
				(!strncmp(&sWord[iWordLen-2],"es",2) && 
				(sWord[iWordLen-3] == 's' || sWord[iWordLen-3] == 'x' || sWord[iWordLen-3] == 'o'
				|| (iWordLen >4 && sWord[iWordLen-3] == 'h' && (sWord[iWordLen-4] == 'c' || sWord[iWordLen-4] == 's')))))
        	{
            	strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-2]='\0';
    	        if(oLib[iLib]->Lookup(sNewWord,&iIndex))
        	        bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
	        }
		}

        //cut "ed"
        if( !bFound && iWordLen>3) {
			isupcase = !strncmp(&sWord[iWordLen-2],"ED",2);
			if (isupcase || !strncmp(&sWord[iWordLen-2],"ed",2) )
        	{
	            strcpy(sNewWord,sWord);
    	        sNewWord[iWordLen-2]='\0';
        	    if ( iWordLen>5 && (sNewWord[iWordLen-3]==sNewWord[iWordLen-4])
            	     && !bIsVowel(sNewWord[iWordLen-4]) && bIsVowel(sNewWord[iWordLen-5]) )   //doubled
	            {
    	            sNewWord[iWordLen-3]='\0';
        	        if( oLib[iLib]->Lookup(sNewWord,&iIndex) )
            	        bFound=true;
                	else {
						if (isupcase || g_ascii_isupper(sWord[0])) {
							casestr = g_ascii_strdown(sNewWord, -1);
							if (strcmp(casestr, sNewWord)) {
								if(oLib[iLib]->Lookup(casestr,&iIndex))
									bFound=true;
							}
							g_free(casestr);
						}
						if (!bFound)
                    		sNewWord[iWordLen-3]=sNewWord[iWordLen-4];  //restore
					}
            	}
            	if( !bFound ) {
					if (oLib[iLib]->Lookup(sNewWord,&iIndex))
                		bFound=true;
					else if (isupcase || g_ascii_isupper(sWord[0])) {
						casestr = g_ascii_strdown(sNewWord, -1);
						if (strcmp(casestr, sNewWord)) {
							if(oLib[iLib]->Lookup(casestr,&iIndex))
								bFound=true;
						}
						g_free(casestr);
					}
				}
			}
        }

        // cut "ied" , add "y".
        if(!bFound && iWordLen>3) {
			isupcase = !strncmp(&sWord[iWordLen-3],"IED",3);
			if (isupcase || (!strncmp(&sWord[iWordLen-3],"ied",3)))
    	    {
				strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-3]='\0';
				if (isupcase)
					strcat(sNewWord,"Y"); // add a char "Y"
				else
    	        	strcat(sNewWord,"y"); // add a char "y"
        	    if(oLib[iLib]->Lookup(sNewWord,&iIndex))
            	    bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
			}
        }
        
		// cut "ies" , add "y".
        if(!bFound && iWordLen>3) {
			isupcase = !strncmp(&sWord[iWordLen-3],"IES",3);
			if (isupcase || (!strncmp(&sWord[iWordLen-3],"ies",3)))
    	    {
				strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-3]='\0';
				if (isupcase)
					strcat(sNewWord,"Y"); // add a char "Y"
				else
    	        	strcat(sNewWord,"y"); // add a char "y"
        	    if(oLib[iLib]->Lookup(sNewWord,&iIndex))
            	    bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
			}
        }

		// cut "er".
        if(!bFound && iWordLen>2) {
			isupcase = !strncmp(&sWord[iWordLen-2],"ER",2);
			if (isupcase || (!strncmp(&sWord[iWordLen-2],"er",2)))
    	    {
				strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-2]='\0';
        	    if(oLib[iLib]->Lookup(sNewWord,&iIndex))
            	    bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
			}
        }

		// cut "est".
        if(!bFound && iWordLen>3) {
			isupcase = !strncmp(&sWord[iWordLen-3], "EST", 3);
			if (isupcase || (!strncmp(&sWord[iWordLen-3],"est", 3)))
    	    {
				strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-3]='\0';
        	    if(oLib[iLib]->Lookup(sNewWord,&iIndex))
            	    bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
			}
        }
		
		g_free(sNewWord);
    }	
	
	if (bFound)
    	iWordIndex = iIndex;
	else {
		//don't change iWordIndex here.
		//when LookupSimilarWord all failed too, we want to use the old LookupWord index to list words.
		//iWordIndex = INVALID_INDEX;
	}

    return(bFound);	
}

gboolean Libs::SimpleLookupWord(const gchar* sWord,glong& iWordIndex,int iLib)
{
	gboolean bFound = oLib[iLib]->Lookup(sWord, &iWordIndex);
	if (!bFound)
		bFound = LookupSimilarWord(sWord, iWordIndex, iLib);
	return bFound;
}

inline gboolean bIsVowel(gchar inputchar)
{
    gchar ch = g_ascii_toupper(inputchar);
    return( ch=='A' || ch=='E' || ch=='I' || ch=='O' || ch=='U' );
}



/**************************************************/
gboolean TreeDict::load(const char *ifofilename, GtkTreeStore *model)
{
	gulong tdxfilesize;
	if (!load_ifofile(ifofilename, &tdxfilesize))
		return false;

	gchar fullfilename[256];
	
	strcpy(fullfilename, ifofilename);
	strcpy(fullfilename+strlen(fullfilename)-sizeof("ifo") +1, "dict.dz");	
	
	if (g_file_test(fullfilename, G_FILE_TEST_EXISTS)) {
		dictdzfile = dict_data_open(fullfilename, 0);
		if (!dictdzfile) {
			//g_print("open file %s failed!\n",fullfilename);
			return false;
		}
	}
	else {
		fullfilename[strlen(fullfilename)-3] = '\0';
		dictfile = fopen(fullfilename,"rb");
		if (!dictfile) {
			//g_print("open file %s failed!\n",fullfilename);
			return false;
		}
	}

	strcpy(fullfilename, ifofilename);
	strcpy(fullfilename+strlen(fullfilename)-sizeof("ifo") +1, "tdx.gz");	
	
	gchar *buffer= NULL;
	if (g_file_test(fullfilename, G_FILE_TEST_EXISTS)) {
		gzFile in;	
		in = gzopen(fullfilename,"rb");
		if (in == NULL) {
			//g_print("Open file %s failed!\n",idxfilename);
			return false;
		}
		
		buffer = (gchar *)g_malloc(tdxfilesize);
		
		gulong len;
		len = gzread(in, buffer, tdxfilesize);
		if (len < 0) {
			g_free(buffer);
			return false;
		}
		gzclose(in);
		if (len != tdxfilesize) {
			g_free(buffer);
			return false;
		}
	}
	else {
		fullfilename[strlen(fullfilename)-3] = '\0';
		FILE *file;
		if (!(file = fopen (fullfilename, "rb"))) {
			//g_print("Open file %s failed!\n",fullfilename);
			return false;
		}
		buffer = (gchar *)g_malloc(tdxfilesize);
		gulong read_len;
		read_len = fread (buffer, 1, tdxfilesize, file);
		fclose (file);
		if (read_len!=tdxfilesize) {
			g_free(buffer);
			return false;
		}
	}
	
	gchar *tmp_buffer = buffer;
	load_model(&tmp_buffer, model, NULL, 1); // tmp_buffer will be changed.
	g_free(buffer);
	return true;
}

gboolean TreeDict::load_ifofile(const char *ifofilename, gulong *tdxfilesize)
{
	struct stat stats;
	if (stat (ifofilename, &stats) == -1) {
		//g_print("File: %s don't exist!\n",idxfilename);
		return false;
	}
		
	FILE *file;
	if (!(file = fopen (ifofilename, "rb"))) {
		//g_print("Open file %s failed!\n",idxfilename);
		return false;
	}
	gchar *buffer = (gchar *)g_malloc (stats.st_size + 1);
	fread (buffer, 1, stats.st_size, file);
	buffer[stats.st_size] = '\0';
	fclose (file);
	
	if (!g_str_has_prefix(buffer, "StarDict's treedict ifo file\nversion=2.4.2\n")) {
		g_print("Bad treedict ifo file %s, skiped!\n", ifofilename);
		g_free(buffer);
		return false;
	}
	gchar *p1= buffer + sizeof("StarDict's treedict ifo file\nversion=2.4.2\n")-1 -1;
	
	gchar *p2,*p3;

	p2 = strstr(p1,"\ntdxfilesize=");
	if (!p2) {
		g_free(buffer);
		return false;
	}
	p3 = strchr(p2+ sizeof("\ntdxfilesize=")-1,'\n');
	gchar *tmpstr = (gchar *)g_memdup(p2+sizeof("\ntdxfilesize=")-1, p3-(p2+sizeof("\ntdxfilesize=")-1)+1);
	tmpstr[p3-(p2+sizeof("\ntdxfilesize=")-1)] = '\0';
	*tdxfilesize = atol(tmpstr);
	g_free(tmpstr);

	p2 = strstr(p1,"\nsametypesequence=");
	if (p2) {		
		p3 = strchr(p2+sizeof("\nsametypesequence=")-1,'\n');
		sametypesequence = (gchar *)g_memdup(p2+sizeof("\nsametypesequence=")-1, p3-(p2+sizeof("\nsametypesequence=")-1)+1);
		sametypesequence[p3-(p2+sizeof("\nsametypesequence=")-1)] = '\0';
	}	
	
	g_free(buffer);	
	return true;
}

void TreeDict::load_model(gchar **buffer, GtkTreeStore *model, GtkTreeIter *parent, gint count)
{
	GtkTreeIter iter;
	gchar *p1;
	glong tmpglong;
	glong offset, size, subentry_count;
	
	for (int i=0; i< count; i++) {
		p1 = *buffer + strlen(*buffer) +1;
		memcpy(&(tmpglong),p1,sizeof(glong));
		offset = g_ntohl(tmpglong);
		p1 += sizeof(glong);
		memcpy(&(tmpglong),p1,sizeof(glong));
		size = g_ntohl(tmpglong);
		p1 += sizeof(glong);
		memcpy(&(tmpglong),p1,sizeof(glong));
		subentry_count = g_ntohl(tmpglong);
		p1 += sizeof(glong);
		gtk_tree_store_append(model, &iter, parent);
		gtk_tree_store_set(model, &iter, 0, *buffer, 1, offset, 2, size, -1);
		*buffer = p1;
		if (subentry_count)
			load_model(buffer, model, &iter, subentry_count);
	}
}


/**************************************************/
TreeDicts::TreeDicts()
{
	treedictcount =0;
	oTreeDict = NULL;
}

TreeDicts::~TreeDicts()
{
	if (treedictcount)
	{
		for (int i=0;i< treedictcount;i++)
		{
			delete oTreeDict[i];
		}	
		g_free(oTreeDict);
	}
}

GtkTreeStore* TreeDicts::Load()
{
	GtkTreeStore *model = gtk_tree_store_new (3, G_TYPE_STRING, G_TYPE_LONG, G_TYPE_LONG); //word, offset, size
	
	GSList *order_list, *disable_list;
#ifdef _WIN32
	rw_cfg_read_strlist (usercfgfile, "manage_dictionaries", "treedict_order_list", &order_list);
	rw_cfg_read_strlist (usercfgfile, "manage_dictionaries", "treedict_disable_list", &disable_list);
#else
	gpAppFrame->oAppConf.read_list("/apps/stardict/manage_dictionaries/treedict_order_list", GCONF_VALUE_STRING, &order_list);
	gpAppFrame->oAppConf.read_list("/apps/stardict/manage_dictionaries/treedict_disable_list", GCONF_VALUE_STRING, &disable_list);
#endif
	gchar *ifofilename;
	GSList *tmplist1,*tmplist2;
	gboolean disabled;
	TreeDict *lib;
	tmplist1 = order_list;	
	while (tmplist1) {
		ifofilename = (gchar *)(tmplist1->data);
		tmplist1 = g_slist_next(tmplist1);
		tmplist2 = disable_list;
		disabled = false;
		while (tmplist2) {
			if (strcmp((gchar *)(tmplist2->data), ifofilename) == 0) {
				disabled = true;
				break;
			}
			tmplist2 = g_slist_next(tmplist2);
		}
		if (disabled)
			continue;

		lib = new TreeDict;
		if (lib->load(ifofilename, model)) {
			treedictcount++;
			oTreeDict = (TreeDict **)g_realloc(oTreeDict,treedictcount * sizeof(TreeDict *));
			oTreeDict[treedictcount-1] = lib;
		}
		else {
			delete lib;
		}					
	}

#ifdef _WIN32	
	gchar *filename = g_build_filename(stardict_data_dir, "treedict", NULL);
	LoadDir(filename, order_list, disable_list, model);
	g_free(filename);
#else
	gchar home_dir[256];
	sprintf(home_dir, "%s/.stardict/treedict", g_get_home_dir());
	LoadDir(home_dir, order_list, disable_list, model);
	LoadDir(STARDICT_DATA_DIR "/treedict", order_list, disable_list, model);
#endif
	
	g_slist_foreach (order_list, (GFunc)g_free, NULL);
	g_slist_free(order_list);
	
	g_slist_foreach (disable_list, (GFunc)g_free, NULL);
	g_slist_free(disable_list);	

	return model;
}

void TreeDicts::LoadDir(gchar *dirname, GSList *order_list, GSList *disable_list, GtkTreeStore *model)
{
	GDir *dir = g_dir_open(dirname, 0, NULL);	
	if (dir)
	{
		const gchar *filename;	
		gchar fullfilename[256];
		TreeDict *lib;
		gboolean loaded;
		GSList *tmplist1,*tmplist2;
		gboolean disabled;
		while ((filename = g_dir_read_name(dir))!=NULL)
		{	
			sprintf(fullfilename, "%s/%s", dirname, filename);
			if (g_file_test(fullfilename, G_FILE_TEST_IS_DIR)) {
				LoadDir(fullfilename, order_list, disable_list, model);
			}
			else if (g_str_has_suffix(filename,".ifo")) {				
				tmplist1 = order_list;
				loaded = false;
				while (tmplist1) {
					if (strcmp((gchar *)(tmplist1->data), fullfilename) == 0) {
						loaded = true;
						break;
					}
					tmplist1 = g_slist_next(tmplist1);
				}
				if (loaded)
					continue;
				
				tmplist2 = disable_list;
				disabled = false;
				while (tmplist2) {
					if (strcmp((gchar *)(tmplist2->data), fullfilename) == 0) {
						disabled = true;
						break;
					}
					tmplist2 = g_slist_next(tmplist2);
				}
				if (disabled)
					continue;
				
				lib = new TreeDict;
				if (lib->load(fullfilename, model))
				{
					treedictcount++;
					oTreeDict = (TreeDict **)g_realloc(oTreeDict,treedictcount * sizeof(TreeDict *));
					oTreeDict[treedictcount-1] = lib;
				}
				else
				{
					delete lib;
				}				
			}
		}		
		g_dir_close(dir);
	}	
}

gchar* TreeDicts::poGetWordData(glong offset, glong size, int iTreeDict)
{
	return( oTreeDict[iTreeDict]->GetWordData(offset, size) );
}
