/***************************************************************************
    smb4kscanner.h  -  The network scan core class of Smb4K.
                             -------------------
    begin                : Sam Mai 31 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMB4KSCANNER_H
#define SMB4KSCANNER_H

#ifndef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qstringlist.h>
#include <qtimer.h>
#include <qptrqueue.h>

// KDE includes
#include <kprocess.h>
#include <kconfig.h>

// application specific includes
#include "smb4kdefs.h"
#include "smb4knetworkitems.h"

/**
 * This is the core class, that communicates with the network. All look-up
 * stuff is done here.
 */

class Smb4KScanner : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor.
     */
    Smb4KScanner( QObject *parent = 0, const char *name = 0 );
    /**
     * The destructor.
     */
    ~Smb4KScanner();
    /**
     * Aborts the network scan.
     */
    void abort();
    /**
     * Reads the options.
     */
    void readOptions();
    /**
     * Scan for the shares on a selected host.
     */
    void getShares( const QString &workgroup, const QString &host, const QString &ip );
    /**
     * Scans for workgroup members.
     */
    void getWorkgroupMembers( const QString &workgroup, const QString &master, const QString &ip );
    /**
     * Get more info about a share (i.e. server and OS string, etc.).
     */
    void getInfo( const QString &workgroup, const QString &host, const QString &ip );
    /**
     * Searches for a given host.
     */
    void makeSearch( const QString &host );
    /**
     * Get the preview of a share.
     */
    void getPreview( const QString &workgroup, const QString &host, const QString &ip, const QString &share, const QString &path );
    /** 
     * This function returns an Smb4KWorkgroupItem, if the the workgroup
     * exists in the list, or 0, if it does not.
     */
    Smb4KWorkgroupItem *getWorkgroup( const QString &workgroup );    
    /**
     * This function takes an Smb4KWorkgroupItem and appends it to the list, if the represented
     * workgroup isn't already in it.
     */
    void appendWorkgroup( Smb4KWorkgroupItem *item );

    
  signals:
    /**
     * This signal is emitted, when the scanner starts and ends.
     */
    void running( int mode, bool on );
    /**
     * This signal is emitted, when the workgroup list has been updated.
     */
    void workgroups( const QValueList<Smb4KWorkgroupItem *> &list );
    /**
     * Emits the list of workgroup members.
     */
    void members( const QValueList<Smb4KHostItem *> &list );
    /**
     * Emits the list of shares.
     */
    void shares( const QValueList<Smb4KShareItem *> &list );
    /**
     * This signal provides info about a certain host. It passes the server
     * and the OS string.
     */   
    void info( Smb4KHostItem *host );
    /**
     * This signal is emitted, if the process was killed.
     */
    void killed();
    /**
     * This signal emits the directory list of the share that is to be 
     * previewed or 0 if an error occurred.
     */
    void previewResult( const QValueList<Smb4KPreviewItem *> &list );
    /**
     * Is emitted, when the results of a network search are to be passed.
     */
    void searchResult( Smb4KHostItem *host );
    /**
     * This signal emits the error code and the error message.
     */
    void error( int error_code, const QString &error_message );

  protected:
    /**
     * Starts the scan process.
     */
    void startProcess( int state );
    /**
     * Is called, when the process ends.
     */
    void endProcess();
    /**
     * Processes the output of the network group scan.
     */
    void processWorkgroups();
    /**
     * Processes the output of the master host scan.
     */
    void processHosts();
    /**
     * Processes the output of the host scan.
     */
    void processShares();
    /**
     * Processes the output of the scan for the OS and Server
     * string.
     */
    void processInfo();
    /**
     * Processes the output you get after querying the host
     * to retrieve the browse list.
     */
    void processQueryHost();
    /**
     * Processes the output of the preview request.
     */
    void processPreview();
    /**
     * Processes the output of a search request.
     */
    void processSearch();

  protected slots:
    /**
     * Is called, if something is received via stdout.
     */
    void slotReceivedStdout( KProcess *, char *buf, int len );
    /**
     * Is called, when the KProcess exited.
     */
    void slotProcessExited( KProcess * );
    /**
     * Is called, if something is received via stderr.
     */
    void slotReceivedStderr( KProcess *, char  *buf, int len );
    /**
     * Starts processing the queue.
     */
    void start();
    /**
     * This slot is called, if the network search should be reinitiated.
     */
    void slotInit();

  private:
    /**
     * Inits the network (re-)scan.
     */
    void m_init();
    /**
     * Scans the group master for the group members.
     */
    void m_scanForWorkgroupMembers( const QString &workgroup, const QString &master, const QString &ip );
    /**
     * Scans the chosen host for its shares.
     */
    void m_scanForShares( const QString &workgroup, const QString &host, const QString &ip );
    /**
     * Scans a given host for its OS and Server string.
     */
    void m_scanForInfo( const QString &workgroup, const QString &host, const QString &ip );
    /**
     * Searches for a given host.
     */
    void m_searchForHost( const QString &host );
    /**
     * Gets a preview of a share.
     */
    void m_previewShare( const QString &workgroup, const QString &host, const QString &ip, const QString &share, const QString &path );
    /**
     * This function returns the advanced options for nmblookup, that the
     * user chose to use via the configuration dialog.
     */
    const QString getNmblookupOptions();
    /**
     * This function returns the advanced options for smbclient, that the
     * user chose to use via the configuration dialog.
     */
    const QString getSmbclientOptions();
    /**
     * The KConfig object.
     */
    KConfig *m_config;
    /**
     * Internal enumeration.
     */
    enum TODO{ Groups, QueryHost, Hosts, Shares, Info, Preview, Search, Init, Idle };
    /**
     * The KProcess object.
     */
    KProcess *m_proc;
    /**
     * Determines the state the process is in.
     */
    int m_state;
    /**
     * The buffer.
     */
    QString m_buffer;
    /**
     * Holds the current workgroup, that was passed by Smb4KBrowserWidget.
     */
    QString m_workgroup;
    /**
     * Holds the current host, that was passed by Smb4KBrowserWidget.
     */
    QString m_host;
    /**
     * Holds the current share.
     */
    QString m_share;
    /**
     * Holds the current path (for preview).
     */
    QString m_path;
    /**
     * Holds the IP address of the current host.
     */
    QString m_ip;
    /**
     * This queue stores the incoming requests until they are processed.
     */
    QPtrQueue<QString> m_queue;
    /**
     * The internal timer for the queue.
     */
    QTimer *m_timer;
    /**
     * Is set to true, if the scanner is running.
     */
    bool m_working;
    /**
     * The list of workgroups including the master browser and their
     * IP addresses.
     */
    QValueList<Smb4KWorkgroupItem *> m_workgroupList;
};

#endif
