/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_new_data_dialog.h"
#include "sg.h"
#include "sg_dialogs.h"
#include "sg_dataset.h"

#define DATASET_NAME_LENGTH 100

static SGlayer *the_layer;
static SGdialog active_dialog = COLUMNS_DIALOG;

static gchar *dialog_names_2d[] = {"Worksheet",
                                   "Matrix",
                                   "Functions",
                                   "Python variable/expression",
                                   NULL};

static gchar *dialog_names_3d[] = {"Worksheet",
                                   "Matrix",
                                   "Python variable/expression",
                                    NULL};


static SGdataset *dataset = NULL;
static GtkWidget *window = NULL;
static GtkWidget *main_table;
static GtkWidget *worksheet_combo, *goto_combo;
static SGplot *plot;
static GtkWidget* dialog = NULL;
static gboolean return_ok;

static GtkWidget *sg_pick_dialog                        (SGlayer *layer);

static gint goto_dialog                                 (SGlayer *layer,
                                                         SGdialog dialog);
static void goto_pressed                                (GtkWidget *widget,
                                                         gpointer data);

static gboolean
mw_destroy(GtkWidget *widget)
{
  /* This is needed to get out of gtk_main */

  sg_dialog_kill(widget);

  gtk_main_quit ();

  return FALSE;
}

static gboolean
cancel_pressed(GtkWidget *widget)
{
  gtk_widget_destroy(window);

  dataset = NULL;
  return FALSE;
}

static gboolean
ok_pressed(GtkWidget *widget)
{
  SGdataset *new_dataset = NULL;
  gboolean veto;
  gchar column_text[9][100];
  gint i;
  gchar name[DATASET_NAME_LENGTH] = "";


  switch(active_dialog){

   case COLUMNS_DIALOG:
    dataset = sg_edit_columns_update();
    if(!dataset) return FALSE;

    new_dataset = sg_project_new_dataset(dataset->style);
    gtk_widget_show(GTK_WIDGET(new_dataset->real_data));

    sg_dataset_set_worksheet(new_dataset, 
                             dataset->worksheet,
                             dataset->x,
                             dataset->y,
                             dataset->z,
                             dataset->a,
                             dataset->dx,
                             dataset->dy,
                             dataset->dz,
                             dataset->da,
                             dataset->l);

    break;
   case MATRIX_DIALOG:
    dataset = sg_edit_3d_update();
    if(!dataset) return FALSE;

    new_dataset = sg_project_new_dataset(dataset->style);
    gtk_widget_show(GTK_WIDGET(new_dataset->real_data));

    new_dataset->real_data->iterator_mask = dataset->real_data->iterator_mask;
    sg_dataset_set_matrix(new_dataset, dataset->worksheet);

    break;
  case FUNCTIONS_DIALOG:
    dataset = sg_edit_function_update();
    if(!dataset) return FALSE;

    new_dataset = sg_project_new_function(dataset->exp);
    gtk_widget_show(GTK_WIDGET(new_dataset->real_data));

    break;
  case PYTHON_DIALOG:

    dataset = sg_edit_exp_update();
    if(!dataset) return FALSE;

    new_dataset = sg_project_new_expression(dataset->style, dataset->p_exp);
    gtk_widget_show(GTK_WIDGET(new_dataset->real_data));

    for(i = 0; i < 9; i++){
      if(new_dataset->p_exp[i] && strlen(new_dataset->p_exp[i]) > 0)
      {
         if (i>0)
             g_snprintf(column_text[i], 100,",%s", new_dataset->p_exp[i]);
         else
             g_snprintf(column_text[i], 100,"%s", new_dataset->p_exp[i]);
      }
      else
      {
         sprintf(column_text[i],"");
      }
    }

    g_snprintf(name,100,"EXPR->(%s%s%s%s%s%s%s)", column_text[0],
                                                  column_text[1],
                                                  column_text[2],
                                                  column_text[3],
                                                  column_text[4],
                                                  column_text[5],
                                                  column_text[6]);

    sg_dataset_set_name(new_dataset, name);
    break;
  }

  new_dataset->real_data->line = dataset->real_data->line;
  new_dataset->real_data->symbol = dataset->real_data->symbol;
  new_dataset->real_data->line_connector = dataset->real_data->line_connector;
  if(GTK_IS_PLOT_SURFACE(new_dataset->real_data)){
    GTK_PLOT_SURFACE(new_dataset->real_data)->height_gradient = 
                       GTK_PLOT_SURFACE(dataset->real_data)->height_gradient;
  }

  gtk_object_destroy(GTK_OBJECT(dataset));
  dataset = new_dataset;
  return_ok = TRUE;
  gtk_widget_destroy(widget);

  return FALSE;
}


SGdataset *
sg_new_data_dialog (SGlayer *layer)
{
  GtkWidget *frame;
  GtkWidget *main_box;
  GtkWidget *box;
  GtkWidget *ok_button, *cancel_button;
  GtkWidget *action_area;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *worksheet_combo;
  GList *list;
  SGplot *plot;
  gchar text[DATASET_NAME_LENGTH];
  gint i;

  return_ok = FALSE;
  active_dialog = COLUMNS_DIALOG;
  dataset = NULL;

  plot = (SGplot *)layer->parent;
  the_layer = layer;

  dialog = NULL;

  window=gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);

  sg_dialog_new(window);

  gtk_window_set_title (GTK_WINDOW(window),"New dataset");

  gtk_window_set_modal (GTK_WINDOW(window),TRUE);

  /* Create widgets */

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_add(GTK_CONTAINER (window), frame);

/**********************************************************/
  main_table = gtk_table_new(3, 1, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 10);
  gtk_container_add(GTK_CONTAINER(frame), main_table);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);

  main_box = gtk_vbox_new (FALSE, 5);
  gtk_table_attach_defaults(GTK_TABLE(main_table), main_box, 0, 1, 3, 4);

  goto_dialog(layer, active_dialog);

  gtk_table_attach_defaults(GTK_TABLE(main_table), sg_pick_dialog (layer), 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(main_table), gtk_hseparator_new(), 0, 1, 2, 3);

/**********************************************************/

  /* Action Area */
  action_area = gtk_hbutton_box_new ();
  gtk_container_set_border_width(GTK_CONTAINER(action_area), 5);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_SPREAD);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_box_pack_end (GTK_BOX (main_box), action_area, FALSE, FALSE, 0);
  gtk_widget_show (action_area);

  ok_button = sg_stock_button ("Button_Ok");
  GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), ok_button, TRUE, TRUE, 0);
  gtk_widget_grab_default (ok_button);
  gtk_widget_show (ok_button);

  cancel_button = sg_stock_button ("Button_Cancel");
  GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), cancel_button, TRUE, TRUE, 0);
  gtk_widget_show (cancel_button);

  /* connect signals */

  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
                             GTK_SIGNAL_FUNC (cancel_pressed),
                             GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
                             GTK_SIGNAL_FUNC (ok_pressed),
                             GTK_OBJECT (window));

  gtk_signal_connect (GTK_OBJECT (window), "destroy",
                      GTK_SIGNAL_FUNC (mw_destroy),NULL);

  /* Show widgets */

  gtk_widget_show_all (window);

  /* wait until dialog get destroyed */
  gtk_main();

  if(!return_ok){
     if(dataset) gtk_object_destroy(GTK_OBJECT(dataset));
     dataset = NULL;
  }
  return dataset;
}

static GtkWidget *
sg_pick_dialog (SGlayer *layer)
{
  GtkWidget *hbox,*vbox;

  hbox = gtk_hbox_new(FALSE, 5);
  vbox = gtk_vbox_new(FALSE, 5);
  
  gtk_box_pack_start (GTK_BOX (vbox), hbox,
                      TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), gtk_hseparator_new(),
                      TRUE, TRUE, 0);
  
  gtk_box_pack_start (GTK_BOX (hbox), gtk_label_new("Get data from"),
                      TRUE, TRUE, 0);
  
  goto_combo = gtk_combo_new();
  gtk_box_pack_start (GTK_BOX (hbox), goto_combo, TRUE, TRUE, 0);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(goto_combo)->entry), FALSE);

  if(the_layer->type == SG_LAYER_3D)
    sg_combo_set_items(GTK_COMBO(goto_combo), dialog_names_3d);
  else
    sg_combo_set_items(GTK_COMBO(goto_combo), dialog_names_2d);


  switch(active_dialog){
    case COLUMNS_DIALOG:
      gtk_list_select_item(GTK_LIST(GTK_COMBO(goto_combo)->list), 0);
      break;
    case MATRIX_DIALOG:
    case FUNCTIONS_DIALOG:
      gtk_list_select_item(GTK_LIST(GTK_COMBO(goto_combo)->list), 1);
      break;
    case PYTHON_DIALOG:
      gtk_list_select_item(GTK_LIST(GTK_COMBO(goto_combo)->list), 2);
      break;
  }

  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(goto_combo)->entry),
                     "changed",
                     (GtkSignalFunc) goto_pressed, (gpointer)layer);

  gtk_widget_show_all(hbox);
  return(vbox);
}

static void
goto_pressed(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  gint i;
  gchar *text;

  text=gtk_entry_get_text(GTK_ENTRY(widget));

  if (strcmp(text, "Worksheet") == 0) active_dialog = COLUMNS_DIALOG;
  else if (strcmp(text, "Matrix") == 0) active_dialog = MATRIX_DIALOG;
  else if (strcmp(text, "Python variable/expression") == 0) active_dialog = PYTHON_DIALOG;
  else if (strcmp(text, "Functions") == 0) active_dialog = FUNCTIONS_DIALOG;

  goto_dialog((SGlayer *)data, active_dialog);
}

static gint
goto_dialog(SGlayer *layer, SGdialog new_dialog)
{
  gchar name[DATASET_NAME_LENGTH];

  if(GTK_IS_WIDGET(dialog)) gtk_container_remove(GTK_CONTAINER(main_table), dialog);

  if(dataset) gtk_object_destroy(GTK_OBJECT(dataset));

  switch(new_dialog){
    case COLUMNS_DIALOG:
      dialog = sg_edit_columns_dialog(NULL, layer->type);
      break;
    case MATRIX_DIALOG:
      dialog = sg_edit_3d_dialog(NULL, layer->type);
      break;
    case FUNCTIONS_DIALOG:
      dataset = sg_dataset_new_function(NULL);
      sprintf(name, "f%d(x)", last_function+1);
      sg_dataset_set_name(dataset, name);
      dialog = sg_edit_function_dialog(dataset);
      break;
    case PYTHON_DIALOG:
      dialog = sg_edit_exp_dialog(NULL, layer->type);
      break;
  }

  gtk_table_attach_defaults(GTK_TABLE(main_table), dialog, 0, 1, 1, 2);

  gtk_widget_show_all(dialog);
  return TRUE;
}

