
// Copyright (c) 1995-2003 The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Narayanan Thondugulam 
//          Radharamanan Radhakrishnan  

#include "scram.hh"
#include "savant.hh"
#include "symbol_table.hh"
#include <FlexLexer.h>
#include "IIR_DesignFile.hh"
#include "VHDLLexer.hh"
#include "VHDLParser.h"
#include "error_func.hh"
#include "sstream-wrap.hh"

extern bool verbose_output;
extern bool standard_fir;

scram::scram( bool init_need_to_write_libraries, IIR_LibraryDeclaration *work_library ) : 
  my_symbol_table( new symbol_table ),
  need_to_write_libraries( init_need_to_write_libraries ),
  my_work_library( work_library ){
}

scram::scram( bool init_need_to_write_libraries, 
	      const string &work_library_name ) : 
  my_symbol_table( new symbol_table ),
  need_to_write_libraries( init_need_to_write_libraries ),
  my_work_library( 0 ){
  my_work_library = library_manager::instance()->find_or_create_library( work_library_name );
  ASSERT( my_work_library != 0 );
  my_symbol_table->add_declaration( my_work_library );
}

scram::~scram(){
  delete my_symbol_table;
  my_symbol_table = NULL;
}

IIR_DesignFileList *
scram::parse_files( const vector<string> &fileList ){
  for( vector<string>::const_iterator currentFile = fileList.begin();
       currentFile < fileList.end();
       currentFile++ ){
    cerr << "Processing file " << *currentFile << endl;
    
    IIR_DesignFile *parsed_file = parse_file( *currentFile,
					      my_work_library );
    IIR_DesignFile::get_design_files().append( parsed_file );
  }
  
  return &IIR_DesignFile::get_design_files();
}

IIR_DesignFile *
scram::parse_file( const string &filename, IIR_LibraryDeclaration *lib ){
  ifstream yyin( filename.c_str() );

  if( verbose_output == true ){
    cerr << "Parse_file parsing " << filename << endl;
  }

  if (yyin == NULL) {
    ostringstream err;
    err << "Error opening file " << filename;
    report_error( err.str() );
    exit( 1 );
  } 

  IIR_DesignFile *vhdl_design_file = new IIR_DesignFile();
  vhdl_design_file->_set_parser( this );
  vhdl_design_file->_set_file_name( filename );
  if( my_work_library->_get_design_file() == 0 ){
    my_work_library->_set_design_file( vhdl_design_file );
  }

  VHDLLexer scan( &yyin , &vhdl_design_file->comments);
  ANTLRTokenBuffer pipe(&scan);
  scan.resetLexerFileLocators(filename);

  VHDLParser parser(&pipe);
  parser.init();
  parser.setEofToken(ENDFILE);

  parser.design_file( vhdl_design_file, my_symbol_table, lib, need_to_write_libraries );

  yyin.close();

  return vhdl_design_file;
}

IIR_LibraryDeclaration *
scram::get_work_library() const {
  return my_work_library;
}
