// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu

//---------------------------------------------------------------------------
#include "IIRScram_UseClause.hh"
#include "IIR_Name.hh"
#include "IIR_PackageDeclaration.hh"
#include "IIR_Identifier.hh"
#include "IIR_SelectedName.hh"
#include "IIR_SimpleName.hh"
#include "IIR_LibraryDeclaration.hh"
#include "symbol_table.hh"
#include "set.hh"
#include "error_func.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"


extern bool debug_symbol_table;

IIRScram_UseClause::IIRScram_UseClause(){
  static const string tmp_declarator = "use clause declarator";
  static IIR_Identifier *use_clause_declarator = IIR_Identifier::get( tmp_declarator );
  set_declarator( use_clause_declarator );
}

IIRScram_UseClause::~IIRScram_UseClause() {}


void 
IIRScram_UseClause::_publish_vhdl(ostream &_vhdl_out) {
  ASSERT(get_selected_name() != NULL);
  ASSERT(get_selected_name()->_is_resolved());

  IIR_LibraryDeclaration *library = get_selected_name()->_get_library_declaration();
  if( library != NULL ){
    if( library == _get_work_library() && 
	IIR_TextLiteral::_cmp( library->get_declarator(), "work" ) != 0 ){
      // Then this is the work library, but it's not called work...  We need to
      // emit a library declaration.
      _vhdl_out << "library ";
      library->_publish_vhdl( _vhdl_out );
      _vhdl_out << "; -- inserted by analyzer due to library renaming.\n";
    }
  }

  _vhdl_out << "use ";
  get_selected_name()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ";\n";
}


void
IIRScram_UseClause::_publish_vhdl_decl(ostream &_vhdl_out) {
  ASSERT(get_selected_name() != NULL);
  ASSERT(get_selected_name()->_is_resolved());

  _publish_vhdl(_vhdl_out);
}

IIR_Declaration*
IIRScram_UseClause::_get_package_declaration() {
  IIR_Declaration *retval = 0;
  if( dynamic_cast<IIR_SelectedName *>( get_selected_name() ) ){
    retval = dynamic_cast<IIR_SelectedName *>(get_selected_name())->_get_package_declaration();
  }
  return retval;
}

void
IIRScram_UseClause::_publish_cc_headers( published_file &_cc_out ) {
  if( _get_package_declaration() && !_get_package_declaration()->_is_standard() ){
    //The standard libraries are included by default and so
    //no need to include them
    ASSERT(_get_package_declaration()->get_kind() == IIR_PACKAGE_DECLARATION);
    _get_package_declaration()->_publish_cc_include( _cc_out );
  }
}

void
IIRScram_UseClause::_publish_cc_include( published_file &_cc_out ) {
  _publish_cc_headers(_cc_out);
}

void
IIRScram_UseClause::_publish_cc_package_name( published_file &_cc_out ){
  ASSERT(get_selected_name() != NULL);
  ASSERT(get_selected_name()->_is_resolved());
  ASSERT(get_selected_name()->get_kind() == IIR_SELECTED_NAME);

  IIR_Declaration* pkg_name = _get_package_declaration();
  if(pkg_name != NULL) {
    ASSERT( pkg_name->get_kind() == IIR_PACKAGE_DECLARATION );
    ((IIR_PackageDeclaration *)pkg_name)->_publish_cc_package_name(_cc_out.get_stream());
  }
}

void
IIRScram_UseClause::_type_check(){
  set<IIR_Declaration> *use_clause_decls = get_selected_name()->_symbol_lookup();

  if( use_clause_decls == NULL ){
    ostringstream err;
    err << "Unable to locate logical library |" << *get_selected_name() << "|."
	<< " Perhaps this library hasn't been successfully analyzed yet.";
    report_error( this, err.str() );
    return;
  }

  IIR_Name *tmp_name =  get_selected_name();
  ASSERT( tmp_name->get_kind() == IIR_SELECTED_NAME );

  IIR_Declaration *decl = use_clause_decls->get_element();
  while( decl != NULL ){
    tmp_name->_decl_to_decl( decl );
    
    if( debug_symbol_table == true ){
      cerr << _get_symbol_table() << " - processing use clause - making |" 
	   << *tmp_name << "| visible!\n";
    }
    if( decl->_is_iir_package_declaration() == TRUE ){
      _make_interface_visible( _get_symbol_table() );      
    }
    else{
      _get_symbol_table()->make_visible( decl );
    }
    decl =  use_clause_decls->get_next_element();    
  }

  delete use_clause_decls;

  ASSERT( get_selected_name()->_is_resolved() == TRUE );
}

void 
IIRScram_UseClause::_make_interface_visible( symbol_table *sym_tab ){
  ASSERT( get_selected_name()->_is_resolved() == TRUE );
  set<IIR_Declaration> *decls = get_selected_name()->_symbol_lookup();
  if( decls != 0 ){
    ASSERT( decls->num_elements() == 1 );
    IIR_Declaration *decl = decls->get_element();
    if( sym_tab->in_use_list( decl ) == FALSE ){
      sym_tab->add_to_use_list( decl );
      decl->_make_interface_visible( sym_tab );
    }
  }
  delete decls;
}

IIR *
IIRScram_UseClause::_clone() {
  return this;
}

visitor_return_type *IIRScram_UseClause::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_UseClause(this, arg);
};
