
#ifndef IIRSCRAM_HH
#define IIRSCRAM_HH

// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer
//          Malolan Chetlur
//          Krishnan Subramani
//          Umesh Kumar V. Rajasekaran
//          Narayanan Thondugulam
//          Radharamanan Radhakrishnan
//          Ardhendu Singh
//          Swaminathan Subramanian
//	    Magnus Danielson	cfmd@swipnet.se

#include <iostream>
#include <string>
#include "IIRBase.hh"
#include "node_visitor.hh"
#include "dl_list.hh"

using std::string;

extern bool no_mangling;

class symbol_table;
class include_manager;
class published_file;
class break_handler;
template <class type> class set;

class IIR;
class IIR_List;
class IIR_Declaration;
class IIR_TypeDefinition;
class IIR_DesignatorList;
class IIR_Identifier;
class IIR_TextLiteral;
class IIR_LibraryUnit;
class IIR_LibraryDeclaration;
class IIR_ArchitectureDeclaration;
class IIR_DeclarationList;
class IIR_DesignFile;
class IIR_ConcurrentStatement;
class IIR_Statement;
class IIR_AttributeSpecification;
class IIR_AttributeSpecificationList;
class IIR_FunctionDeclaration;
class IIR_AssociationList;
class IIR_Designator;
class IIR_EntityDeclaration;
class IIR_Label;
class IIR_PackageDeclaration;
class IIR_ArchitectureStatementList;
class IIR_ProcessStatement;
class IIR_GenericList;
class IIR_PortList;
class IIR_TypeDeclaration;
class IIR_IntegerLiteral;
class IIR_ConstantDeclaration;
class IIR_NatureDefinition;
class IIR_NatureDeclaration;
class IIR_SimultaneousStatement;
class IIR_ReferenceAttribute;
class IIR_ContributionAttribute;
class IIR_DotAttribute;
class IIR_InterfaceList;
class IIR_AboveAttribute;

/** This class is the base of the SAVANT extensions to the IIR.  Due to the
    design of the IIR, lots of things have been put in here that seem like
    they might be better suited somewhere else.  Unfortunately, it is often
    the case that IIR is the closest common ancestor that one can find to
    insert a virtual method into. */
class IIRScram : public IIRBase {

public:
  /** Accepts a visitor for this node.

      Pure virtual method that allows visitors access to the intermediate
      form.  The visitor pattern implemented in this intermediate provides
      support for parameter transmission by a visitor.  This capability is
      achieved by defining the visitor pattern with a return type of void*
      and with an input parameter of type void*.

      @return a void* parameter, allowing parameter transmission within the
      visitor that is not predetermined by the core intermediate.

      @param visitor a pointer to the visitor object.
      @param arg location for a visitor specific pointer to be passed
      into the visitor methods.

      @see node_visitor */

  // PHIL, MUST REPLACE THIS WHEN VISITORS ARE FULLY DEFINED.
  // virtual visitor_return_type *_accept_visitor(node_visitor* visitor, visitor_argument_type* arg) = 0;
  virtual visitor_return_type *_accept_visitor( node_visitor *, 
						visitor_argument_type *){ return 0; }

  /** @name VHDL publishing methods 
      Methods related to publishing of VHDL.
  */
  //@{
  /** Publishes VHDL for the node that the method is called on.  In
    general, the VHDL that is generated by calling this method will be
    appropriate for a reference to a node rather than defining the node.
    For instance, a declaration node can appear both in a declarative
    region or in a statement/use context.  If "_publish_vhdl" is called
    on a declaration, the name of the declaration will be the output.
    To publish the full declaration, one would call _publish_vhdl_decl.
     
     @see IIRScram#_publish_vhdl_decl */
  virtual void _publish_vhdl(ostream &);
  
  /** Publishes VHDL for the node that the method is called on. */
  virtual void _publish_vhdl_range(ostream &);
  virtual void _publish_vhdl_decl(ostream &);
  virtual void _publish_vhdl_type_decl(ostream &);
  virtual void _publish_vhdl_subtype_decl(ostream &);
  virtual void _publish_vhdl_operator(ostream &);
  //@}

  //@{
  /** Publish C++ appropriate for the lvalue of an assignment.  Often this
      will be a name of some sort. */
  virtual void _publish_cc_lvalue( published_file &_cc_out );
  
  /** Publish C++ appropriate for getting the rvalue of an assignment. */ 
  virtual void _publish_cc_rvalue( published_file &_cc_out );

  virtual void _publish_cc_init( published_file &_cc_out );
  virtual void _publish_cc_init_for_ams(published_file &);

  /** Publish the type of the object, as required by the type constructor.  */
  virtual const string _get_cc_object_type();

  /** Publish the name of this object as would be accessed from it's
      context. */
  virtual void _publish_cc_object_name( published_file &_cc_out );

  virtual void _publish_cc_wait_data( published_file &_cc_out );
  virtual void _publish_cc_data( published_file &_cc_out );
  virtual void _publish_cc_decl( published_file &_cc_out );
  virtual void _publish_cc_bounds( published_file &_cc_out );

  /** This function publishes the range of the type if it is scalar type
      and publishes the range of indices, if it is an array type */
  virtual void _publish_cc_range( published_file &_cc_out );

  virtual void _publish_cc_value( published_file &_cc_out );
  virtual void _publish_cc_universal_value( published_file &_cc_out );

  /** This method is defined for all the classes in the system that have an
      elaboration class associated with them.  This class is a reminisent of the
      days before we ever did any form of name mangling in the system and hence
      the useage "binding_name".  The class publishes necessary association for
      the elaboration classes (such as <architecture>_<entity>,
      <block>_<architecture>, <generate_statement>_<architecture> etc.) in the
      system.  This method is overloaded by the following child classes:
      IIRScram_ArchitectureDeclaration, IIRScram_ComponentDeclaration,
      IIRScram_ComponentInstantiationStatement, IIRScram_ConcurrentStatement,
      IIRScram_ConfigurationDelcaration, IIRScram_EntityDeclaration,
      IIRScram_ProcessStatement.  These methods also add predefined prefixes to
      the binding-names (such as SEA, SC, SG, SCFG etc.) to improve quick
      recognition of the type of elaboration class in the generated code.

      This method is also overloaded by the following classes to ease/assist
      code-generation: IIRScram_FunctionDeclaration, IIRScram_Label,
      IIRScram_Name, IIRScram_PackageBodyDeclaration,
      IIRScram_PackageDeclaration.
      
      @param  outstream  The output stream into which the binding-name
      needs to be published (_cc_out by default) */
  virtual void _publish_cc_binding_name( ostream &outstream );

  /** This method returns the c++ class name used for the elaboration of
      the node it is called on.  It relies on _publish_cc_binding_name for
      most of it's work. */
  virtual const string _get_cc_elaboration_class_name();

  /** For the "get_instantiated_unit()" of an
      IIR_ComponentInstantiationStatment, return the architecture (or
      null.)

      @see IIR_ComponentInstantiationStatement#get_instantiated_unit */
  virtual IIR_ArchitectureDeclaration *_get_cc_instantiated_architecture();

  /** For the "get_instantiated_unit()" of an
      IIR_ComponentInstantiationStatment, return the entity (or null.)

      @see IIR_ComponentInstantiationStatement#get_instantiated_unit */
  virtual IIR_EntityDeclaration *_get_cc_instantiated_entity();

  /** This method publishes an include appropriate for a generated file to
      include whatever this node is.  If this method isn't overridden in
      the leaf node calling this will generate a runtime error.  */
  virtual void _publish_cc_include( published_file &_cc_out );

  /** This method publishes an include appropriate for a generated file to
      include whatever this node is.  If this method isn't overridden in
      the leaf node calling this will generate a runtime error.  */
  virtual void _publish_cc_include_decls( published_file &_cc_out );

  /** This method publishes an include for whatever this node is.  (If it's
      defined for this node, generates a runtime error if not.  */
  virtual void _publish_cc_include_elab( published_file &_cc_out );

  /** Generate an include for the VHDL file name passed in.
     
      @param os The stream to publish the include to.
      @param file_to_include the file to include
      @param system_include - is this a system include that needs angle
      braces, or a user include that needs quotes? */
  virtual void _publish_cc_include( published_file &os,
				    const string &file_to_include,
				    bool system_include = false );

  /** This method is/can be used to publish the full name of the C++ class
      associated with a concurrent statements (Note: Only concurrent statements
      in the system have a unique class associated with them).  This method uses
      IIRScram::_publish_cc_binding_name( published_file &_cc_out ) to
      publish the class name. 

      @param The output stream to which the class name needs to be published
      into (it is _cc_out by default).  */
  virtual void _publish_cc_class_name( ostream &os );
  virtual const string _get_cc_type_name();
 
  virtual void _publish_cc_subprogram_arg( published_file &_cc_out );
  virtual void _publish_cc_constructor_args( published_file &_cc_out );
  virtual void _publish_cc_sigdest( published_file &_cc_out );
  virtual void _publish_cc_declarator( published_file &_cc_out );
  virtual void _publish_cc_for_index( published_file &_cc_out );
  virtual void _publish_cc_condition( published_file &_cc_out );
  virtual void _publish_cc_headers( published_file &_cc_out );
  virtual void _publish_cc_ams_function(published_file &);
  virtual void _publish_cc_ams_function_call_in_simult_stmt(published_file & ){
    // do nothing
  }

  /** This is the default behaviour for _publish_cc_headerfiles_for_cc( published_file &_cc_out ).
      It has to be called explicitly.  It's defined here as both
      declarations and statements call it. */
  void _publish_cc_headerfiles_for_cc_default( published_file &_cc_out );

  virtual void _publish_cc_direction( published_file &_cc_out );

  //Works for only one case. These two functions need a lot of work
  virtual void _publish_cc_name_elaborate( published_file &_cc_out );

  /** This function publishes the constructor of signals in the constructor
      of the state. */
  virtual void _publish_cc_state_object_init( published_file & ){ }

  /** This method publishes code for this node to act as an initializer for
      a constant or variable.  By default, it calls "_publish_cc". */
  virtual void _publish_cc_initialization_value( published_file &_cc_out );

  /** This function specifies the type of object instatiated in simulation
      kernel. */
  virtual void _publish_cc_universal_type( published_file &_cc_out );

  /** Publishes c++ for kernel types.  Right now this is only for scalar
      types. 
      @param os Optional stream to publish into.  Defaults to _cc_out. */
  virtual const string _get_cc_kernel_type();
  virtual void _publish_cc_kernel_type( ostream &os );
  virtual void _publish_cc_kernel_type( published_file &pf );

  /** This function is implemented recursively in different nodes to handle
      type conversion in procedure calls. The actual can be nest function
      calls and the actual nested with them is extracted and published. */
  virtual void _publish_cc_first_objectParameter( published_file &_cc_out ); 

  virtual void _publish_cc_copy_generics_to_globals( published_file &_cc_out );

  /** The following function is used for code generation that does runtime 
      elaboration. */
  virtual void _publish_cc_elaborate( published_file &_cc_out );

  virtual void _publish_cc_addChild( published_file &_cc_out );
  void _publish_cc_prefix_string( published_file &_cc_out );

  virtual void _publish_cc_scoping_prefix( ostream &os, IIR *, IIR * );

  /** This function provides debugging dumps. */
  void _publish_cc_dump( published_file &_cc_out );
 
  /** This method is used to obtain the declarator corresponding to this IIR
      node.  This method ALWAYS returns only the mangled declarator .ie. the
      declarator whose name hase been prefixed with the declarators of its
      enclosing scope.  This method is overloaded in the following derived
      classes: IIRScram_Declaration, IIRScram_Name, IIRScram_SimpleName,
      IIRScram_Statement, and IIRScram_TypeDefinition.  By default this method
      reports "undefined_function" when invoked on a IIR hierarchy that does
      not overload this virtual method.

      @return a IIR_TextLiteral that points to the mangled declarator.  The
      user MUST NOT delete or modify the pointer returned my this method.  */
  virtual IIR_TextLiteral *_get_declarator();

  //@}

  //@{
  /** These methods only apply to entities, components, etc.  (Or names
      representing them, like resolved selected names, etc.) */
  virtual IIR_GenericList *_get_generic_list();
  virtual IIR_PortList *_get_port_list();
  //@}

  /** This method has been introduced to remove unneccessary type
      casting.  If this method is overridden, it will call the derived
      method.  If not it simply returns NULL. */
  virtual IIR* get_value();

  /** This method returns any statement region found in the node. */
  virtual IIR_List *_get_statement_list();

  /** This method has been introduced to remove unneccessary type
      castings. */
  virtual IIR_Mode _get_mode();

  /** This method tells use whether or not this is <name>.all */
  virtual IIR_Boolean _is_by_all(){ return FALSE; }

  /** This method tells us whether this node has been resolved or not. */
  virtual IIR_Boolean _is_resolved(){ return FALSE; }

  virtual IIR_Boolean _is_literal(){ return FALSE; }

  /** Does this node represent an "object".  (An object, according to
      the LRM, is a constant, signal, variable, or a file.)  This method
      is overriden by things like IIR_SelectedName, and
      IIR_AliasDeclaration.  A "TRUE" from this method does NOT mean
      that it's safe to cast to IIR_ObjectDeclaration. */
  virtual IIR_Boolean _is_object(){ return FALSE; }

  virtual IIR_Boolean _is_ascending_range();
  virtual IIR_Boolean _is_name(){ return FALSE; }
  virtual IIR_Boolean _is_signal(){ return FALSE; }
  virtual IIR_Boolean _is_resolved_signal(){ return FALSE; }
  virtual IIR_Boolean _is_variable(){ return FALSE; }
  virtual IIR_Boolean _is_constant(){ return FALSE; }
  virtual IIR_Boolean _is_text_literal(){ return FALSE; }
  virtual IIR_Boolean _is_type(){ return FALSE; }
  virtual IIR_Boolean _is_incomplete_type_declaration(){ return FALSE; }
  virtual IIR_Boolean _is_subtype(){ return FALSE; }
  virtual IIR_Boolean _is_discrete_type(){ return FALSE; }
  virtual IIR_Boolean _is_scalar_type(){ return FALSE; }
  virtual IIR_Boolean _is_record_type(){ return FALSE; }
  virtual IIR_Boolean _is_array_type(){ return FALSE; }
  virtual IIR_Boolean _is_access_type(){ return FALSE; }
  virtual IIR_Boolean _is_integer_type(){ return FALSE; }
  virtual IIR_Boolean _is_floating_type(){ return FALSE; }
  virtual IIR_Boolean _is_character_type(){ return FALSE; }
  virtual IIR_Boolean _is_integer_literal(){ return FALSE; }
  virtual IIR_Boolean _is_floating_literal(){ return FALSE; }
  virtual IIR_Boolean _is_string_literal(){ return FALSE; }
  virtual IIR_Boolean _is_association(){ return FALSE; }
  virtual IIR_Boolean _is_sequential_signal_assignment(){ return FALSE; }
  virtual IIR_Boolean _is_designator(){ return FALSE; }
  virtual IIR_Boolean _is_interface(){ return FALSE; }
  virtual IIR_Boolean _is_enumeration_type(){ return FALSE; }
  virtual IIR_Boolean _is_enumeration_literal(){ return FALSE; }
  virtual IIR_Boolean _is_subprogram(){ return FALSE; }
  virtual IIR_Boolean _is_numeric_literal();
  virtual IIR_Boolean _is_numeric_type(){ return FALSE; }
  virtual IIR_Boolean _is_physical_type(){ return FALSE; }
  virtual IIR_Boolean _is_attribute();
  virtual IIR_Boolean _is_static_expression() { return FALSE;}
  virtual IIR_Boolean _is_longest_static_prefix();
  virtual IIR_Boolean _is_aggregate_expression() { return FALSE; }
  virtual IIR_Boolean _is_composite_resolved_signal() { return FALSE; }
  virtual IIR_Boolean _is_textio();
  virtual IIR_Boolean _is_standard();
  virtual IIR_Boolean _is_relational_operator();
  virtual IIR_Boolean _is_primary_unit(){ return FALSE; }
  virtual IIR_Boolean _is_secondary_unit(){ return FALSE; }

  virtual IIR_Boolean _is_component_decl() { return FALSE; }
  virtual IIR_Boolean _is_entity_decl() { return FALSE; }
  virtual IIR_Boolean _is_configuration_decl() { return FALSE; }
  virtual IIR_Boolean _is_kernel_type() { return FALSE;}
  virtual IIR_Boolean _is_simultaneous_statement() { return FALSE; }
  virtual IIR_Boolean _is_monadic_operator(){ return FALSE; }
  virtual IIR_Boolean _is_above_attribute_found() { return FALSE; }

  /** Warning!!!!!!  It is not always right to typecast to
      IIR_BlockStatement if this function returns true.  The reason being,
      this will return true, if it is a block statement or if the label is
      that of a block statement. */
  virtual IIR_Boolean _is_block_statement() { return FALSE;}

  /** Warning!!!!!!  It is not always right to typecast to
      IIR_ConcurrentGenerate*Statement if this function returns true.  The
      reason being, this will return true, if it is a concurrent
      generate*statement or if the label is that of a block statement. */
  virtual IIR_Boolean _is_concurrent_generate_statement() { return FALSE;}

  /** _is_resolved_type : TRUE if the type defined contain resolution function */
  virtual IIR_Boolean _is_resolved_type();

  /** This returns TRUE, if the type definition, or subtype definition is
      part of subtype declaration. */
  virtual IIR_Boolean _is_subtype_decl() { return FALSE; }

  virtual IIR_Boolean _is_operator() { return FALSE; }

  /** @name Node Type Queries
      The Intent of these operators is that if one of them returns true,
      it is absolutely safe to cast straight to the class implied by the
      name.  This is in direct contrast with some methods like
      "_is_signal", which will tell you if something is a signal but it
      doesn't mean the node is an IIR_SignalDeclaration.  

      The other method to perform these type of operations is via the
      get_kind method.  However, this technique doesn't allow a query of
      an abstract type - it only works for leaf nodes in the inheritence
      tree.  
  */

  //@{
  /// Is this node an IIR_ArchitectureDeclaration?
  virtual IIR_Boolean _is_iir_architecture_declaration(){ return FALSE; }
  /// Is this node an IIR_AttributeDeclaration?
  virtual IIR_Boolean _is_iir_attribute_declaration(){ return FALSE; }
  /// Is this node an IIR_Attribute?
  virtual IIR_Boolean _is_iir_attribute(){ return FALSE; }
  /// Is this node an IIR_ConcurrentStatement?
  virtual IIR_Boolean _is_iir_concurrent_statement(){ return FALSE; }
  /// Is this node an IIR_ConfigurationItem?
  virtual IIR_Boolean _is_iir_configuration_item(){ return FALSE; }
  /// Is this node an IIR_Declaration?
  virtual IIR_Boolean _is_iir_declaration(){ return FALSE; }
  /// Is this node an IIR_DyadicOperator?
  virtual IIR_Boolean _is_iir_dyadic_operator(){ return FALSE; }
  /// Is this node an IIR_EntityDeclaration?
  virtual IIR_Boolean _is_iir_entity_declaration(){ return FALSE; }
  /// Is this node an IIR_Expression?
  virtual IIR_Boolean _is_iir_expression(){ return FALSE; }
  /// Is this node an IIR_ConfigurationDeclaration?
  virtual IIR_Boolean _is_iir_configuration_declaration(){ return FALSE; }
  /// Is this node an IIR_ProcedureDeclaration?
  virtual IIR_Boolean _is_iir_procedure_declaration(){ return FALSE; }
  /// Is this node an IIR_FunctionDeclaration?
  virtual IIR_Boolean _is_iir_function_declaration(){ return FALSE; }
  /// Is this node an IIR_ObjectDeclaration?
  virtual IIR_Boolean _is_iir_object_declaration(){ return FALSE; }
  /// Is this node an IIR_PackageDeclaration?
  virtual IIR_Boolean _is_iir_package_declaration(){ return FALSE; }
  /// Is this node an IIR_PackageBodyDeclaration?
  virtual IIR_Boolean _is_iir_package_body_declaration(){ return FALSE; }
  /// Is this node an IIR_List?
  virtual IIR_Boolean _is_iir_list(){ return FALSE; }
  /// Is this node an IIR_TypeDeclaration?
  virtual IIR_Boolean _is_iir_type_declaration(){ return FALSE; }
  /// Is this node an IIR_SignalDeclaration?
  virtual IIR_Boolean _is_iir_signal_declaration(){ return FALSE; }
  /// Is this node an IIR_SubtypeDeclaration?
  virtual IIR_Boolean _is_iir_subtype_declaration(){ return FALSE; }
  /// Is this node an IIR_ComponentDeclaration?
  virtual IIR_Boolean _is_iir_component_declaration(){ return FALSE; }
  /// Is this node an IIR_GroupDeclaration?
  virtual IIR_Boolean _is_iir_group_declaration(){ return FALSE; }
  /// Is this node an IIR_InterfaceDeclaration?
  virtual IIR_Boolean _is_iir_interface_declaration(){ return FALSE; }
  /// Is this node an IIR_ElementDeclaration?
  virtual IIR_Boolean _is_iir_element_declaration(){ return FALSE; }
  /// Is this node an IIR_PhysicalUnit?
  virtual IIR_Boolean _is_iir_physical_unit(){ return FALSE; }
  /// Is this node an IIR_PhysicaTypeSubtypeDefinition?
  virtual IIR_Boolean _is_iir_physical_subtype_definition(){ return FALSE; }
  /// Is this node an IIR_SequentialStatement?
  virtual IIR_Boolean _is_iir_sequential_statement(){ return FALSE; }
  /// Is this node an IIR_TypeDefinition?
  virtual IIR_Boolean _is_iir_type_definition(){ return FALSE; }
  /// Is this node an IIR_RecordTypeDefinition?
  virtual IIR_Boolean _is_iir_record_type_definition(){ return FALSE; }
  /// Is this node an IIR_FloatingTypeDefinition?
  virtual IIR_Boolean _is_iir_floating_type_definition(){ return FALSE; }
  /// Is this node an IIR_IntegerTypeDefinition?
  virtual IIR_Boolean _is_iir_integer_type_definition(){ return FALSE; }
  /// Is this node an IIR_AccessTypeDefinition?
  virtual IIR_Boolean _is_iir_access_type_definition(){ return FALSE; }
  /// Is this node an IIR_ScalarTypeDefinition?
  virtual IIR_Boolean _is_iir_scalar_type_definition(){ return FALSE; }
  /// Is this node an IIR_EnumerationTypeDefinition?
  virtual IIR_Boolean _is_iir_enumeration_type_definition(){ return FALSE; }
  /// Is this node an IIR_PhysicalTypeDefinition?
  virtual IIR_Boolean _is_iir_physical_type_definition(){ return FALSE; }
  /// Is this node an IIR_ArrayTypeDefinition?
  virtual IIR_Boolean _is_iir_array_type_definition(){ return FALSE; }
  /// Is this node an IIR_CaseStatementAlternative?
  virtual IIR_Boolean _is_iir_case_statement_alternative(){ return FALSE; }
  /// Is this node an IIR_Label?
  virtual IIR_Boolean _is_iir_label(){ return FALSE; }
  /// Is this node an IIR_SubprogramDeclaration?
  virtual IIR_Boolean _is_iir_subprogram_declaration(){ return FALSE; }
  /// Is this node an IIR_ConfigurationSpecification?
  virtual IIR_Boolean _is_iir_configuration_specification(){ return FALSE; }
  /// Is this node an IIR_AttributeSpecification?
  virtual IIR_Boolean _is_iir_attribute_specification(){ return FALSE; }
  /// Is this node an IIR_DisconnectSpecification?
  virtual IIR_Boolean _is_iir_disconnect_specification(){ return FALSE; }
  /// Is this node an IIR_EnumerationLiteral?
  virtual IIR_Boolean _is_iir_enumeration_literal(){ return FALSE; }
  /// Is this node an IIR_TextLiteral?
  virtual IIR_Boolean _is_iir_text_literal(){ return FALSE; }
  /// Is this node an IIR_Name?
  virtual IIR_Boolean _is_iir_name(){ return FALSE; }
  /// Is this node an IIR_Statement?
  virtual IIR_Boolean _is_iir_statement(){ return FALSE; }
  /// Is this node an IIR_LibraryUnit?
  virtual IIR_Boolean _is_iir_library_unit(){ return FALSE; }
  /// Is this node an IIR_LibraryDeclaration?
  virtual IIR_Boolean _is_iir_library_declaration(){ return FALSE; }
  //@}

  /** Returns true if static analysis says this expression is readable,
     in a variable or signal assignment context, else returns false. */
  virtual IIR_Boolean _is_readable();

  /** Returns true if static analysis says this expression is writable,
     in a variable or signal assignment context, else returns false. */
  virtual IIR_Boolean _is_writable();
  
  virtual void _set_passed_through_out_port(IIR_Boolean val);
  

  //@{

  /** These methods tell us whether an expression is locally (or globally
      static) as defined by the LRM in section 7.4.1.  If
      _is_locally_static() isn't overridden, then being a locally static
      primary is sufficient to say we're locally static.  These methods
      also apply to names, as the rules given in LRM section 6.1.  The to
      find out if an IIR meets the criteria for a "static name", the method
      "_is_globally_static" should be called where a name is appropriate.
      To find out if an IIR meets the criteria for a "locally static name",
      the method "_is_locally_static" should be called where a name is
      appropriate.  Note too, that after resoultion simple names will be
      replaced with their declaration...*/

  virtual IIR_Boolean _is_locally_static(){ return _is_locally_static_primary(); }
  virtual IIR_Boolean _is_globally_static(){ return _is_globally_static_primary(); }
  virtual IIR_Boolean _is_locally_static_primary();
  virtual IIR_Boolean _is_globally_static_primary();
  //@}
  
#ifdef PROCESS_COMBINATION
  virtual void _static_elaborate(IIR_ArchitectureDeclaration*, 
				 IIR_DeclarationList*, char*);
#endif

  virtual IIR *_clone();
  virtual void _clone( IIR *);

  virtual IIR  *_transmute();

  virtual void _get_list_of_input_signals( set<IIR> *list );
  virtual void _get_signal_source_info( set<IIR> *siginfo );
  virtual IIR_AttributeSpecification *_get_attribute_specification(IIR*);

  virtual IIR_AttributeSpecificationList *_get_attribute_specification_list( );

  /** Needed for guard signals */
  virtual IIR_SignalKind _get_signal_kind();
  virtual IIR_Boolean _is_guard_signal();

  /** This method returns the declaration of an indexed name.  If the
      prefix is a declaration, returns it, or else, gets the declaration of
      the prefix recursively. */
  virtual IIR_Declaration* _get_prefix_declaration();

  /** This is used on a selected name to get the package name in it. */
  virtual IIR_Declaration* _get_package_declaration();

  /** The following function returns the type of the IIR_Name object that
      is referred to. */
  virtual IIR_TypeDefinition* _get_name_type();

  /** The following function is used for building the sensitivity list
      while doing transmogrification. */
  virtual void _build_sensitivity_list(IIR_DesignatorList* sensitivity_list);

  /** The following function is used for building the generic parameter
      set when generic statements occur inside Simultaneous Statements */
  virtual void _build_generic_parameter_set(set<IIR_Declaration> *);

  /** The following function is used for building the set of above attributes
      in an architecture. */
  virtual void _build_above_attribute_set(set<IIR_AboveAttribute> *);

  /** The following function is used to pick up the resolution function
      for a subtype/type definition. This function is needed as all the
      vhdl publishing is now done in scalar type.  This function is
      overloaded in the corresponding nodes that have resolution
      function defined.  */
  virtual IIR_FunctionDeclaration *_get_resolution_function();

  virtual IIR *_get_enclosing_scope();
  
  IIR_LibraryDeclaration *_get_work_library( );

  /** These methods give derived classes a chance to generate implicit
      declarations and so forth. */
  virtual void _come_into_scope( symbol_table * ){}
  virtual void _come_out_of_scope( symbol_table * ){}

  /** This function copies the identifiers declared in outer scope and accessed
      somewhere within nested scope(s) to outer scopes till it reaches the
      scope where the object was declared. */
  void _copy_symbols_defined_in_enclosing_scope(symbol_table *current, symbol_table *outer);
  
  /** The code generator needs to know what kind of unit is being
      published.  This is acheived by the following static variable and the
      corresponding wrapper methods. */
  enum PublishedUnit {ENTITY_DECL, ARCHITECTURE_DECL, BLOCK, FUNCTION, 
		      PROCEDURE, PACKAGE_PUB, PACKAGE_BODY, PROCESS, 
		      PROCESS_STATE, TYPE, GENERIC_MAP, GENERATE_FOR, 
		      GENERATE_IF, DUMMY_ENTITY_DECL, CONFIGURATION_DECLARATION,
		      CASE_STATEMENT, NONE_PARSING, SIMULTANEOUS_STATEMENT, 
		      SIMULTANEOUS_IF, BREAK_STATEMENT, ABOVE_ATTRIBUTE, NONE};
    
  IIR_Boolean _is_currently_publishing_subprogram();
  IIR_Boolean _is_currently_publishing_generate_for();

  void _report_undefined_scram_fn(char *);

  // These methods have to do with semantic checks and such.
  
  /** Symbol lookup methods are intended to be used on names. The method
      accepting an argument essentially narrows the search to be within the
      context of a set of declarations.  I.e. find the "foo.bars" within
      the context of the declarations of "baz"es. */
  virtual set<IIR_Declaration> *_symbol_lookup();
  virtual set<IIR_Declaration> *_symbol_lookup( IIR_Declaration *);
  virtual set<IIR_Declaration> *_symbol_lookup( set<IIR_Declaration> *);
  virtual set<IIR_Declaration> *_symbol_lookup(IIR_Boolean(IIR::*constraint_function)());

  
  IIR_Label *_lookup_label( IIR_Boolean complain_on_error );
  
  /** These methods calculate a node's rval_set.  Either will return a
      a set with a bunch of type definitions, or will return NULL if it
      doesn't have a valid rval.  (Like for an undefined symbol, or a
      library declaration or something).  The optional argument is a
      method declared in IIR that returns a boolean value.  This method
      will be called on declarations of things when being considered as
      possbilities for symbols.  If the method returns false, the
      declaration will not be considered as a possibility.  For
      instance, if there is a type "FOO" in the current scope, and a
      variable "FOO", and we're looking for the target of a variable
      assignment statement, only the variable assignment will be
      considered. */
  virtual set<IIR_TypeDefinition> *_get_rval_set(IIR_Boolean (IIR::*constraint_function)() = 0);

  virtual set<IIR_TypeDefinition> *_get_rval_set( set<IIR_TypeDefinition> *,
						  IIR_Boolean (IIR::*constraint_function)() = 0 );
  virtual set<IIR_TypeDefinition> *_get_rval_set( set<IIR_Declaration> *,
						  IIR_Boolean (IIR::*constraint_function)() = 0 );

  /** This method takes the a set of return values that are context,
      and a known return type, and returns the rval from the
      context rvals that is correct.  For instance, if we have:
      foo( 1 to  3 ).bar( 1 ) and we know the type of the whole expression,
      and we have a set of types for foo( 1 to 3 ), this method will tell
      us which is correct... */
  virtual IIR_TypeDefinition *_determine_rval_in_set( set<IIR_TypeDefinition> *,
						      IIR_TypeDefinition * );

  /** Same as previous method, but in cases like: work.foo( 1 ), where the
      prefix has a declaration, but not a type... */
  virtual IIR_Declaration *_determine_decl_in_set( set<IIR_Declaration> *,
						      IIR_TypeDefinition * );  

  virtual IIR_Declaration *_find_in_implicit_list( const string );
  virtual set<IIR_Declaration> *_get_implicit_declarations( );
  virtual void _set_implicit_declarations( set<IIR_Declaration> * );

  /** This method looks at this node as having been resolved as a formal in
      an association, and returns the declaration associated with it. */
  virtual IIR_Declaration *_find_formal_declaration();

  /** This method should only be called on resolved "things".  For
      instance, declarations, resolved names (indexed names with
      declaration prefixes, etc.) */
  virtual IIR_TypeDefinition *get_subtype();

  /** This method should only be called on resolved "things".  For
      instance, declarations, resolved names (indexed names with
      declaration prefixes, etc.)  It should only be called in objects that
      are access types (otherwise, it returns NULL). */
  virtual IIR_TypeDefinition *_get_rval_pointed_at();

  /** This method calculates the locally static value of the expression
      it's called on.  If this is impossible (i.e. the expression isn't
      static, or it's not integer valued, it will return NULL.)  If
      possible, it _allocates_ an IIR_IntegerLiteral with the result
      represented in it. */
  virtual IIR_IntegerLiteral *_get_integer_static_value();
  
  /** This method should only be called on resolved "things".  For
      instance, declarations, resolved names (indexed names with
      declaration prefixes, etc.)  Or, on type definitions.  This method
      returns the element subtype of an array for the index passed in.  If
      the array is multidimensional, the returned subtype will be another
      array type definition.  If the number passed in is higher than the
      number of indexes of the array, or if the object the method is called
      on isn't an array, the return value is NULL.  */
  virtual IIR_TypeDefinition *_get_type_of_element( int );

  /** This method returns the number of indexes an array object has. */
  virtual IIR_Int32 _get_num_indexes( );

  /** This method returns the type of an objects port in a particular
      position: port( x : in bit; y : out integer; ) 0 = bit
      typedefinition, 1 = integertypedefinition */
  virtual IIR_TypeDefinition *_get_port_type( int );
  

  /** Given our (resolved) r value, plus knowing what type of node we are,
      transform ourselves into the correct declaration... */
  virtual IIR *_rval_to_decl( IIR_TypeDefinition * );

  /** Given a prefix of type (param 1), return the declaration for a suffix
      with type (param 2) */
  virtual IIR *_rval_to_decl( IIR_TypeDefinition *, IIR_TypeDefinition * );

  /** Given a prefix declaration (param 1), return the declaration for a suffix with
      type (param 2) */
  virtual IIR *_rval_to_decl( IIR_Declaration *, IIR_TypeDefinition * );

  /** This method takes a declaration, and returns a resolved IIR * that
      has had the declaration applied to it.  For instance, a
      IIR_SimpleName returns the declaration itself.  An IIR_SelectedName
      resolves the prefix and the suffix, and returns itself. */
  virtual IIR *_decl_to_decl( IIR_Declaration * );

  /** This is the main call into the semantic processing routines.  The set
      that gets passed in is the list of possible l-values of the node
      being type-checked. */
  virtual void _type_check( set<IIR_TypeDefinition> * );
  virtual void _type_check( IIR_TypeDefinition * );

  /** The following methods are used to type check component
      configurations, and configuration specifications.  Unfortunately, IIR
      is their closest direct ancestor.
    
      Note the that int is actually an 
      IIRScram_ConfigurationSpecification::type_check_mode, but we can't easily
      #include that here.*/
  void _type_check_configuration( IIR_AssociationList &port_map_aspect,
				  IIR_AssociationList &generic_map_aspect,
				  int mode );

  /** Used by _type_check_configuration to resolve the port/generic map
      aspects.
     
      @see IIRScram#_type_check_configuration  */
  void _resolve_map( IIR_InterfaceList *entity_interface_list,
		     IIR_InterfaceList *component_interface_list,
		     IIR_AssociationList &map,
		     int tmp_mode );

  /** This method type checks the iteration scheme of IIR_ForLoopStatement
      and IIR_ConcurrentGenerateForLoop statement.  Again, IIR is the
      closest ancestor.  It takes the iteration scheme as param, and
      returns the correctly resolved subtype for it. */
  IIR_TypeDefinition *_type_check_iteration_scheme( IIR_ConstantDeclaration *iteration_scheme );

  virtual IIR *_get_component_name( );
  virtual void _set_component_name( IIR * );

  /** Calls "get_entity_aspect" on nodes that define it.  Generates an
      error for any other node. */
  virtual IIR_LibraryUnit *_get_entity_aspect();

  /** This method implements the rules found in section 5.2.2 of the '93
      LRM.  It looks for an entity with the same simple name as the
      component passed in.  Nodes that can be passed into this method
      include IIR_SimpleName, and IIR_ComponentDeclaration.  Anything else
      will generate an error.  If no matching entity is found, NULL is
      returned. */
  IIR_LibraryUnit *_find_default_binding( IIR *component_name );

  /** This virtual method allows semantic transformations to be made as
      part of type_checking.  These transformations include things like the
      conversion of indexed_names into function calls, and removing some of
      the bizarreness that occurs in parsing.  The argument to this
      function is the context that the node is seen in. (It's possible
      rvalues).  Most nodes don't have transformations - therefore the
      default behavior of this method, when not overridden, is simply to
      return "this".  NOTE: This method is intended to be called from
      _type_check of the _containing_ node. I.e. this method is not
      intended to be "recursively descending". */
  virtual IIR *_semantic_transform( set<IIR_TypeDefinition> * );
  virtual IIR *_semantic_transform( IIR_TypeDefinition * );

  /** The "print" method is intended to print out ONLY the raw data found
      in the IF.  This means it should not be made to print any formatting
      or "labels", like "NAME: foo".  An exception might be for lists that
      it would be alright to put commas between elements, although I'm not
      sure why anyone would want to print a list in this manner.  This
      method is NOT meant to be a way to publish VHDL source code.  It is
      intended more for debuging and/or the generation of error messages.
      Also, note that every overloaded _print method should call the print
      method of it's direct decendent. */
  virtual ostream &_print( ostream & );

  /** Returns a STL string representing this node, if defined.  Relies on
      the "_print" method being defined for the node. */
  const string _to_string( );

  /** This method, defined within the name class, IIR_Identifier class, and
      IIR_Identifier class, converts a name into a string suitable for use
      in a unix-type file system. */
  virtual const string _convert_to_library_name();
  
  /** This method gets the prefix of a name, and returns the string version
      of it. */
  virtual IIR_TextLiteral *_get_prefix_string();

  /* This gets the symbol table of the parser that parsed this file. */
  symbol_table *_get_symbol_table( );

  /** The following methods help to avoid publish the same signal attribute
      multiple times.  Okay, these do not make sense here, but virtual
      functions are better than having type casts in the code. */
  enum SignalAttribute {LAST_EVENT = 0, EVENT, ACTIVE, LAST_ACTIVE, LAST_VALUE,
			DRIVING, DRIVING_VALUE, QUIET, STABLE, TRANSACTION,
			INVALID};
  virtual IIR_Boolean _is_published_attribute_in_state(SignalAttribute);
  virtual void _add_published_attribute_in_state(SignalAttribute);
  virtual IIR_Boolean _is_published_attribute_in_constructor(SignalAttribute);
  virtual void _add_published_attribute_in_constructor(SignalAttribute);
  virtual IIR_Boolean _is_published_attribute_in_initstate(SignalAttribute);
  virtual void _add_published_attribute_in_initstate(SignalAttribute);

  virtual IIR_Boolean _is_quantity();
  virtual IIR_Boolean _is_terminal();

  // Is it a Branch Quantity
  virtual IIR_Boolean _is_branchQ();
  virtual void _set_stmt_node_index(IIR_Int32 *, bool, bool&);
  virtual void _set_stmt_qty_index(IIR_Int32 *, set<IIR_Declaration> *); 
  virtual IIR_Int32 _get_stmt_qty_index();
  virtual IIR_Int32 _get_stmt_signal_index();  

  /** Sometimes a declaration that goes out of scope needs to have it's
      interface made visible.  For instance, in
      component_instantiation_statements, the interface of the entity,
      component, or configuration instantiated needs to be visible at the
      time of type checking. */
  virtual void _make_interface_visible( symbol_table *add_declarations_into );
  virtual void _make_interface_visible( );

  IIR *_get_current_declarative_region();

  /** The following function adds the constant declarations that Initialize
      variables or signals into code generators symbol table. */
  virtual void _add_decl_into_cgen_symbol_table();

  /** This method will return precedence levels for any operator and
      literals.  The precedence levels are given in the LRM.  There are 8
      levels of precedence level 8 are the integer literals and all
      function calls which return value. All others take precedence values
      according to LRM. The higher the number , the higher the
      precedence. */
  enum OperatorPrecedenceLevel {LOGICAL_OPERATOR, RELATIONAL_OPERATOR, SHIFT_OPERATOR,
				ADDING_OPERATOR, SIGN_OPERATOR, MULTIPLYING_OPERATOR,
				MISCELLANEOUS_OPERATOR, OTHERS};

  virtual OperatorPrecedenceLevel _get_operator_precedence();

  bool _get_mangling_flag() const;

  IIR_DesignFile *_get_design_file() const {
    return _my_design_file;
  }

  void _set_design_file( IIR_DesignFile *new_design_file ){
    _my_design_file = new_design_file;
  }

  /** This function copies the line number information from one node into another. */
  static void copy_location ( const IIRScram *source, IIRScram *dest );

  /** Like strdup but allocates the memory with "new".  If no length > 0 is
      passed / in then strlen is called on "to_copy". */
  static char *_strdup( const char * const to_copy, unsigned int size = 0 );

  /** This method copies the information passed into this node, and any
      nodes instantiated directly in this one (should only be lists). */
  virtual void _set_base_info( IIR_DesignFile *, int, int, int );

  /** The following methods get called by the parser to hook specifications
      to IIR nodes that need them.  For instance, if there is a variable
      declaration foo and it has attribute bar, and in the VHDL exists
      "attribute bar of foo is 7;", this attribute specification will be
      attached to foo.  Similarly, disconnection specifications will be
      attached to guarded signals as required. */
  IIR_Boolean _attach_attribute_specification( IIR_AttributeSpecification * );
  IIR_Boolean _attach_disconnection_specification( IIR_AttributeSpecification * );

  virtual IIR_DeclarationList* _get_declaration_list();

  void _group_component_instantiations(IIR_ArchitectureStatementList* conc_stmt_list, int blockSize);
  IIR_Boolean get_is_currently_publishing_simultaneous_stmt();
  IIR_Boolean get_is_currently_publishing_simultaneous_if_stmt();

  void set_is_currently_publishing_simultaneous_stmt( bool );
  void set_is_currently_publishing_simultaneous_if_stmt( bool );

  virtual void _flush_stmt_index();
  virtual void _set_stmt_signal_index(IIR_Int32 *, set<IIR_Declaration> *);

  virtual IIR_Boolean _reference_quantity_found();
  virtual IIR_Boolean _contribution_quantity_found();
  virtual IIR_Boolean _differential_quantity_found();

  virtual void _build_reference_quantity_list(dl_list<IIR_ReferenceAttribute> *);
  virtual void _build_contribution_quantity_list(dl_list<IIR_ContributionAttribute> *);
  virtual void _build_differential_quantity_list(dl_list<IIR_DotAttribute> *);

  /** Called on expressions to publish array indexes - what goes between
      the [] in the C++. */
  virtual void _publish_cc_array_index( published_file &_cc_out );

  /**
     This node is the procedure_name in an IIR_ProcedureCall or
     IIR_FunctionCall statement.  Publish the c++ function name that needs
     to be called.
  */
  virtual void _publish_cc_subprogram_call_name( published_file &_cc_out );

protected:
  IIRScram();
  virtual ~IIRScram() = 0;

  void _publish_cc_class_includes( published_file &_cc_out, 
				   IIR_ArchitectureStatementList* conc_stmt_list);

  void _publish_cc_signals( published_file &_cc_out, IIR_DeclarationList* decl_list);

  void _publish_cc_object_pointers( published_file &_cc_out, IIR_ArchitectureStatementList* conc_stmt_list);
  virtual void _publish_cc_object_pointers_init( published_file &_cc_out,
						 IIR_ArchitectureStatementList* conc_stmt_list,
						 IIR_DeclarationList* decl_list);

  void _publish_cc_destructor( published_file &_cc_out, IIR_ArchitectureStatementList* conc_stmt_list);

  void _publish_cc_anonymous_drivers( published_file &_cc_out, IIR_DeclarationList *);
  void _publish_cc_reference_info( published_file &_cc_out, char *method_info, int line_number );
  void _publish_make_reference_info( published_file &_cc_out, char *method_info, int line_number );
  /** Redefine the savantnow macro to use a process pointer to access the
      time. */
  void _publish_cc_savantnow_process_ptr( published_file &_cc_out );
  /** Redefine the savantnow macro to not use a process pointer to access
      the time. */
  void _publish_cc_savantnow_no_process_ptr( published_file &_cc_out );

  
private:

  IIR_Boolean _currently_publishing_simultaneous_stmt;
  IIR_Boolean _currently_publishing_simultaneous_if_stmt;
  /** Do the actual publishing of the savantnow macro. */
  void _publish_cc_savantnow( published_file &_cc_out, const string &prefix );

  /** Get the singleton instance of the include manager. */
  static include_manager *get_include_manager();
};

inline 
ostream &
operator<<(ostream &os, IIRScram &is ){
  return is._print( os );
}

published_file &
operator<<(published_file &pf, IIRScram &is );

//@{
/** global functions - deprecated */

string &_get_current_publish_name();
void _set_current_publish_name( const string );

IIR *_get_current_publish_node();
void _set_current_publish_node( IIR * );

string &_get_current_configuration_name();
void _set_current_configuration_name( const string );

string &_get_current_architecture_name();
void _set_current_architecture_name( const string );

string &_get_current_entity_name();
void _set_current_entity_name( const string );

string &_get_current_elab_name();
void _set_current_elab_name( const string );

IIR_Int32 _get_number_of_processes();
void _set_number_of_processes( IIR_Int32 );

IIR_Int32 &_get_number_of_interface_signals();
dl_list<break_handler> &_get_breakList();

string &_get_current_package_name();
void _set_current_package_name( const string );

string &_get_current_package_body_name();
void _set_current_package_body_name( const string );

string &_get_current_suffix_name();
void _set_current_suffix_name( const string );

/** While Publishing Multidimensonal Arrays/Composite types, index level
    has to be kept track of,across IF nodes and hence the following
    declaration. */
IIR_Int32 _get_index_level();
void _set_index_level( IIR_Int32 );

/** In case of resolution function and type conversion function publishing,
    certain special handling is required for "return" statements.  This
    static value is used to indicate that a possible type conversion
    function or a resolution function is being published. */
IIR_Boolean _get_allocate_return_value();
void _set_allocate_return_value( IIR_Boolean );

string &_get_current_another_name();
void _set_current_another_name( const string );

/** This is a string that will be prepended to ALL declarations while hey
    are published.  This will help us in publishing different perfixes in
    different contxts.  The wrapper methods for this variable follow. */
string &_get_publish_prefix_string();
void _set_publish_prefix_string( const string );

IIR_TypeDefinition *_get_aggregate_iterator_subtype();
void _set_aggregate_iterator_subtype( IIR_TypeDefinition * );

int _get_aggregate_iterator_counter();
void _set_aggregate_iterator_counter( int );

IIRScram::PublishedUnit _get_currently_publishing_unit();
void _set_currently_publishing_unit( IIRScram::PublishedUnit );

IIRScram::PublishedUnit _get_currently_publishing_vhdl_unit();
void _set_currently_publishing_vhdl_unit( IIRScram::PublishedUnit );

/** The following variable is set either true or false.  If the variable is
    set true, the objects are published without the string
    "state.current->" as prefix. This has been done to reduce The number of
    functions to be written in all the nodes, one publishing with
    "state.current->" as prefix and one without it. */
IIR_Boolean _get_publish_object_without_prefix();
void _set_publish_object_without_prefix( IIR_Boolean );

IIR_ProcessStatement *_get_current_process_statement();
void _set_current_process_statement( IIR_ProcessStatement * );
//@}

/** SCRAM_CC_REF

    Description: Will conditionally dump scram source code reference to the
    code generator section involved in publish-cc code generation.

    Parameters:
    char *	f
     	Should hold a string containing the IIRScram subclass and
     	function name in class::function type of format.
	
	Generates reference according to: \
	<IIRScram_class>::<function>(<line#>) on <userfile>(userline#)	*/

#ifndef SCRAM_CC_REF

#if (defined(DEVELOPER_ASSERTIONS) && defined(GENERATE_CC_REFERENCES))
#define	SCRAM_CC_REF( file, string ) _publish_cc_reference_info( file, string, __LINE__ )
#else /* SCRAM_CC_REF */
#define SCRAM_CC_REF( file, string )
#endif
 
#endif /* SCRAM_CC_REF */

#ifndef SCRAM_MAKE_REF

#if (defined(DEVELOPER_ASSERTIONS) && defined(GENERATE_CC_REFERENCES))
#define	SCRAM_MAKE_REF( file, string ) _publish_make_reference_info( file, string, __LINE__ )
#else /* SCRAM_MAKE_REF */
#define SCRAM_MAKE_REF( file, string )
#endif

#endif /* SCRAM_MAKE_REF */
 
#endif
