/*
 *  dumpcommon.S
 *
 *  Common routines for dump records
 *    Copyright (C) 2001 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *    Author(s): Michael Holzheu  <holzheu@de.ibm.com>
 *
 * Uses extern functions:
 *  - _dump_mem (device dependent function to write dump)
 *
 * Functions:
 *  - _enable_device
 *  - _ssch
 *  - _wait4de
 *  - _panik
 *  - _take_dump
 *  - _store_status
 */


/* General defines */

#define PAGE_SIZE        0x1000                /* 4096 */
#define HEADER_SIZE      0x1000                /* 4096 */
#define END_MARKER_SIZE  0x10
#define DUMP_END_MARKER  0x44554d50,0x5f454e44 /* DUMP_END */
#define IPL_SC           0xb8                  /* Address of ipl subchannel */
#define S390_DUMP_MAGIC  0xa8190173,0x618f23fd /* magic number */
#define ARCH_S390_ID     0x1                   /* arch flag for s390  */
#define ARCH_S390X_ID    0x2                   /* arch flag for s390x */
 
/* Error codes */

#define OK               0x00000000  /* Dump completed successfully */
#define EMEM             0x00000001  /* Device too small for dump */
#define EDEV_INVAL       0x00000002  /* Device not supported */
#define EENABLE_DEV      0x00000100  /* enable device failed */
#define EDISABLE_DEV     0x00000101  /* disable device failed */
#define EDSSCH           0x00000102  /* start subchannel failed */

#define __LC_ARCH_MODE_ID 163        /* here is the arch flag in the lowcore */
#define PAGES_INVALID_TEXT 0x978995a5,0x81938984 /* pinvalid */





################################################################################
# MACRO: dump_header
################################################################################

.macro dump_header:

.align 8
#
# The Dump header
#
.Ldh_dumpheader:
.Ldh_magic_number:.long S390_DUMP_MAGIC
.Ldh_version:     .long 0x00000003
.Ldh_header_size: .long HEADER_SIZE
.Ldh_dump_level:  .long 0x00000004              # DUMP_ALL
.Ldh_page_size:   .long PAGE_SIZE
.Ldh_mem_size:    .long 0x00000000,0x00000000
.Ldh_mem_start:   .long 0x00000000,0x00000000
.Ldh_mem_end:     .long 0x00000000,0x00000000
.Ldh_num_pages:   .long 0x00000000
.Ldh_pad:         .long 0x00000000
.Ldh_time:        .long 0x00000000,0x00000000
.Ldh_cpuid:       .long 0x00000000,0x00000000
.Ldh_arch:        .long ARCH_S390_ID
.Ldh_vol_nr:      .long 0x00000000
#if defined(__s390x__)
.Ldh_build_arch:  .long ARCH_S390X_ID
#else
.Ldh_build_arch:  .long ARCH_S390_ID
#endif
.Ldh_real_mem_size: .long 0x00000000,0x00000000

#
# Dump End Marker
#
.align 8
.Ld_endmarker:    .long DUMP_END_MARKER
.Ld_end_time:     .long 0x00000000,0x00000000

.endm  /* dump_header */




#if defined(__s390x__)


/******************************** 64 BIT **************************************/


################################################################################
# MACRO: dump_io_subroutines_64
# - _enable_device_64
# - _ssch_64
# - _wait4de_64
# - _panik_64
################################################################################

.macro dump_io_subroutines_64

################################################################################
# Enable I/O on the ipl device.
#   -r2 : device subchannel id
################################################################################

_enable_device_64:
        stmg   %r6,%r15,48(%r15)
        basr   %r13,0                   # base register
0:      aghi   %r15,-200
        lgr    %r1,%r2
        lghi   %r2,EENABLE_DEV          # set panik code early
        stsch  1f-0b(%r13)
        oi     1f-0b+5(%r13),0x84       # enable ssch and multipath mode
        msch   1f-0b(%r13)
        bnz    _panik_64-0b(%r13)       # subchannel busy or in error ?
        lctl   %c6,%c6,3f-0b(%r13)      # enable all interrupt classes
        lmg    %r6,%r15,248(%r15)
        br     %r14
        .align 8
1:      .fill  64,1,0
3:      .long  0xff000000               # CR6 initial value

################################################################################
# Start I/O
#  -r2: device subchannel id
#  -r3: address of orb
#  -r4: address of irb
#  -r5: retry count
################################################################################

_ssch_64:
        stmg   %r6,%r15,48(%r15)
        basr   %r13,0                    # base register
0:      aghi   %r15,-200                 # create stack frame
        lgr    %r12,%r2                  # save subchannel id
        lgr    %r11,%r3                  # save orb
        lgr    %r10,%r4                  # save irb
        lgr    %r9,%r5                   # save retry count
1:      lgr    %r1,%r12
        ssch   0(%r11)                   # go
        bnz    5f-0b(%r13)               # houston, we have a problem
2:      lgr    %r2,%r12                  # call _wait4de with subchannel id
        lgr    %r3,%r10                  # and irb address as parameters
        bas    %r14,_wait4de_64-0b(%r13) # wait until DE or error
        tm     9(%r10),0xff              # test channel status
        bnz    5f-0b(%r13)
        tm     8(%r10),0xd2              # test device status
        bz     4f-0b(%r13)
        bct    %r9,1b-0b(%r13)           # something went wrong, retry.
4:      lmg    %r6,%r15,248(%r15)
        br     %r14

5:      lghi   %r2,EDSSCH
        bas    %r4,_panik_64-0b(%r13)

################################################################################
# Wait for interrupt subroutine
#  -r2: device subchannel id
#  -r3: address of irb
################################################################################

_wait4de_64:
        stmg   %r6,%r15,48(%r15)
        basr   %r13,0                    # base register
0:      aghi   %r15,-200                 # create stack frame
        lr     %r1,%r2
        mvc    496(16),6f-0b(%r13)       # set i/o new psw
1:      lpswe  5f-0b(%r13)
2:      stnsm  .Lpsw-0b(%r13),0xfd       # disable io interrupts
        c      %r1,0xb8                  # compare subchannel id
        bne    1b-0b(%r13)               # unequal -> continue waiting
        tsch   0(%r3)
        tm     9(%r3),0xff               # test channel status
        bnz    4f-0b(%r13)

3:      tm     8(%r3),0x02               # got unit check ?
        bnz    4f-0b(%r13)

        tm     8(%r3),0x04               # got device end ?
        bz     1b-0b(%r13)               # still busy -> continue waiting

4:      lmg   %r6,%r15,248(%r15)
        br     %r14
        .align 16
5:      .long  0x02020001,0x80000000,0x00000000,0x00000000+1b
6:      .long  0x00000001,0x80000000,0x00000000,0x00000000+2b
.Lpsw:  .long  0x0

################################################################################
# Panik routine. Loads a disabled wait psw
#   -r2 : panik code
################################################################################

_panik_64:
        basr   %r1,0
0:      stg    %r2,1f-0b+8(%r1)         # store code in address part of psw
        lpswe  1f-0b(%r1)
        .align 8
1:      .long  0x00020000,0x80000000,0x00000000,0x00000000

.endm








################################################################################
# MACRO: dump_common_fn_64
# - _take_dump_64
# - _count_mem_64
# - _store_status_64
# - _copy_lowcore_64
################################################################################

.macro dump_common_fn_64:

################################################################################
# Take the dump
#  - no parameters
################################################################################

_take_dump_64:
        stmg  %r6,%r15,48(%r15)
        basr  %r13,0
.Lbase: aghi  %r15,-200                          # create stack frame
        bas   %r14,_count_mem_64-.Lbase(%r13)    # count memory
        llgf  %r14,.Ldump_mem_64-.Lbase(%r13)
        basr  %r14,%r14                          # dump memory
        lghi  %r2,OK
        bas   %r14,_panik_64-.Lbase(%r13)        # everything ok: stop now

################################################################################
# Find out memory size:
# When accessing a page which is not there, we get a program check
# If the kernel was 32 bit we must take care not to dump more than 2GB
# Under 64 bit native we have to deal with the possible HSA storage hole
#  - no parameters
################################################################################


_count_mem_64:
        stmg  %r6,%r15,48(%r15)
        basr  %r13,0                          # base register
0:      aghi  %r15,-200                       # create stack frame
        lghi  %r9,0                           # base register for zero page
        lghi  %r7,0                           # HSA start
        mvc   464(16,%r9),.Lcount_mem_psw-0b(%r13) # setup program check new

        lghi  %r10,0
        lghi  %r11,1
        sll   %r11,20                         # 1 MB
.Lloop0:
        l     %r12,0(%r10)                    # test page
        agr   %r10,%r11                       # add 1M
        bnm   .Lloop0-0b(%r13)                # -> loop
.Lchkmem0:
        n     %r10,.L4malign0-0b(%r13)        # align to multiples of 4M

        llgf  %r8,.Ldh_arch-0b(%r13)          # get arch
        cghi  %r8,1                           #   1: ESA (S390) mode
        bnz   1f-0b(%r13)                     #   2: ESAME (S390X) mode

        /* ESA */

        llgf  %r8,.Ltwogb-0b(%r13)
        clr   %r10,%r8                        # if ESA & mem > 2GB then
        bl    3f-0b(%r13)                     # mem = 2GB
        llgf  %r10,.Ltwogb-0b(%r13)
        b     3f-0b(%r13)

1:      /* ESAME */

        cghi  %r7,0                           # first loop ?
        bne   2f-0b(%r13)
       
        /* check if we have an HSA under 2GB (only native systems)    */ 
        /* If we have one, and if we have more than 2GB main storage, */
        /* then there is a memory hole (HSA-Start - 2GB)              */

        llgf  %r8,.Ltwogb-0b(%r13)
        cgr   %r10,%r8
        bc    2,3f-0b(%r13)                   # if we already have > 2GB -> exit
        lr    %r7,%r10
        llgf  %r10,.Ltwogb-0b(%r13)           # check if there is memory > 2GB
        b     .Lloop0-0b(%r13)
2:      
        /* second loop */

        llgf  %r8,.Ltwogb-0b(%r13)
        cgr   %r10,%r8                        # equal ?
        bne   3f-0b(%r13)                     # jump if we have found an HSA
               
        lr    %r10,%r7                        # no HSA --> set saved size
3:     
        stg   %r10,.Ldh_real_mem_size-0b(%r13)
        lg    %r6,.Lmem_upper_limit-0b(%r13)  # check if we have an upper limit
        clr   %r10,%r6
        bl    4f-0b(%r13)
        lgr   %r10,%r6                        # upper mem limit set -> use it!
4:  
        stg   %r10,.Ldh_mem_size-0b(%r13)     # store memory size
        stg   %r10,.Ldh_mem_end-0b(%r13)      # store memory end
        srlg  %r12,%r10,12                    # calculate page count (/ 4096)
        st    %r12,.Ldh_num_pages-0b(%r13)    # store page count
        mvc   432(64,%r9),.Lnew_psws-0b(%r13)
        lmg   %r6,%r15,248(%r15)
        br    %r14
.Lcount_mem_psw:
.long 0x00000001,0x80000000,0x00000000,0x00000000 + .Lchkmem0
.Lnew_psws:
.long 0x00020000,0x80000000,0x00000000,0x00000432  # external new psw
.long 0x00020000,0x80000000,0x00000000,0x00000448  # svc new psw
.long 0x00020000,0x80000000,0x00000000,0x00000464  # program check new psw
.long 0x00020000,0x80000000,0x00000000,0x00000480  # machine check new psw
.L4malign0:
.long 0xffc00000
.Ltwogb:
.long 0x80000000


################################################################################
# store status of all cpus in their lowcores
#  - no parameters
################################################################################

_store_status_64:
        stmg  %r6,%r15,48(%r15)
        basr  %r13,0                          # base register
0:      aghi  %r15,-200
        lghi  %r7,0x0                         # base register for 0 page

        ######## move lowcore info (assume user has made store  ########
        ######## status) to prefix-page                         ########

        bas   %r14,_copy_lowcore_64-0b(%r13)

        ######## stop all cpus and store status in prefix pages ########

        lghi  %r8,0                           # first cpu
        stap  .Lcurrent_cpu_64+2-0b(%r13)     # store current cpu address

1:
        cl    %r8,.Lcurrent_cpu_64-0b(%r13)   # is ipl cpu ?
        be    4f-0b(%r13)                     # if yes get next cpu
2:
        lgr  %r9,%r7
        sigp %r9,%r8,0x9                      # store status of cpu
        bc   8,3f-0b(%r13)                    # accepted
        bc   4,4f-0b(%r13)                    # status stored: next cpu
        bc   2,2b-0b(%r13)                    # busy:          try again
        bc   1,4f-0b(%r13)                    # not op:        next cpu
3:
        bas   %r14,_copy_lowcore_64-0b(%r13)
4:
        aghi  %r8,1                           # next cpu (r8 +=1)
        cl    %r8,.Llast_cpu_64-0b(%r13)      # is last possible cpu ?
        bl    1b-0b(%r13)                     # jump if not last cpu

        lmg   %r6,%r15,248(%r15)
        br    %r14                            # return to caller
.Lcurrent_cpu_64:
        .long 0x0
.Llast_cpu_64:
        .long 0x0000ffff

################################################################################
# copy lowcore 64
#  - no parameters
################################################################################

_copy_lowcore_64:
        stmg   %r6,%r15,48(%r15)
        basr   %r13,0                         # base register
0:      aghi   %r15,-200

        lghi  %r2,0x1000                      # offset for first page
        llgf  %r3,792(%r2)                    # get prefix page of current cpu

        ###### check if lowcore address looks valid ######

        cl    %r3,.Llinux_start_64-0b(%r13)   # looks valid ?
        bl    .Lcpy_locore_exit_64-0b(%r13)   # if < linux-start addr
        llgf  %r6,.Lpage_align_64-0b(%r13)    # check page alignment
        nr    %r3,%r6
        cl    %r3,792(%r2)                    # 4888
        bnz   .Lcpy_locore_exit_64-0b(%r13)   # if not page aligned

        ###### copy lowcore                         ######

        llgf  %r3,792(%r2)                    # get prefix page of current cpu
        lghi  %r5,0x1000                      # first page 
        agr   %r3,%r5                         # get base register for second
                                              # page of prefix pages

        # |-----------------------------------------------------------|
        # | Decimal |  Length   | Data                                |
        # | Address |  in Bytes |                                     |
        # |_________|___________|_____________________________________|
        # | 163     | 1         | Architectural Mode ID               |
        # | 4608    | 128       | Fl-pt registers 0-15                |
        # | 4736    | 128       | General registers 0-15              |
        # | 4864    | 16        | Current PSW                         |
        # | 4888    | 4         | Prefix register                     |
        # | 4892    | 4         | Fl-pt control register              |
        # | 4900    | 4         | TOD programmable register           |
        # | 4904    | 8         | CPU timer                           |
        # | 4912    | 1         | Zeros                               |
        # | 4913    | 7         | Bits 0-55 of clock comparator       |
        # | 4928    | 64        | Access registers 0-15               |
        # | 4992    | 128       | Control registers 0-15              |
        # |_________|___________|_____________________________________|

        mvc   512(256,%r3),512(%r2)      # 4608
        mvc   768(16,%r3),768(%r2)
        mvc   792(8,%r3),792(%r2)        # 4888
        mvc   804(20,%r3),804(%r2)       # 4900
        mvc   832(192,%r3),832(%r2)      # 4928

.Lcpy_locore_exit_64:
        lmg   %r6,%r15,248(%r15)
        br    %r14                            # return to caller
.Lpage_align_64:
        .long -4096
.Llinux_start_64:
        .long  0x10000

.align 4
.Ldump_mem_64:    .long _dump_mem_64          # address of function

.endm /* dump_common_fn_64 */




#else /* __s390x__ */



/******************************** 32 BIT **************************************/



################################################################################
# MACRO: dump_io_subroutines_32
# - _enable_device_32
# - _ssch_32
# - _wait4de_32
################################################################################

.macro dump_io_subroutines_32

################################################################################
# Enable I/O on the ipl device.
#   -r2 : device subchannel id
################################################################################

_enable_device_32:
        stm    %r6,%r15,24(%r15)
        basr   %r13,0                   # base register
0:      s      %r15,1f-0b(%r13)
        lr     %r1,%r2
        l      %r2,4f-0b(%r13)          # set panik code early
        stsch  2f-0b(%r13)
        oi     2f-0b+5(%r13),0x84       # enable ssch and multipath mode
        msch   2f-0b(%r13)
        bnz    _panik_32-0b(%r13)       # subchannel busy or in error ?
        lctl   %c6,%c6,3f-0b(%r13)      # enable all interrupt classes
        lm     %r6,%r15,120(%r15)
        br     %r14
1:      .long  96
        .align 8
2:      .fill  64,1,0
3:      .long  0xff000000               # CR6 initial value
4:      .long  EENABLE_DEV

################################################################################
# Start I/O
#  -r2: device subchannel id
#  -r3: address of orb
#  -r4: address of irb
#  -r5: retry count
################################################################################

_ssch_32:
        stm    %r6,%r15,24(%r15)
        basr   %r13,0                    # base register
0:      s      %r15,.Lc96-0b(%r13)       # create stack frame
        lr     %r12,%r2                  # save subchannel id
        lr     %r11,%r3                  # save orb
        lr     %r10,%r4                  # save irb
        lr     %r9,%r5                   # save retry count
1:      lr     %r1,%r12
        stnsm  .Lpsw-0b(%r13),0xfd       # disable io interrupts
        ssch   0(%r11)                   # go
        bnz    5f-0b(%r13)               # houston, we have a problem

2:      lr     %r2,%r12                  # call _wait4de with subchannel id
        lr     %r3,%r10                  # and irb address as parameters
        bas    %r14,_wait4de_32-0b(%r13) # wait until DE or error
        tm     9(%r10),0xff              # test channel status
        bnz    5f-0b(%r13)
        tm     8(%r10),0xd2              # f3 test device status
        bz     4f-0b(%r13)
        bct    %r9,1b-0b(%r13)           # something went wrong, retry.

4:      lm     %r6,%r15,120(%r15)
        br     %r14

5:      l      %r2,6f-0b(%r13)
        bas    %r4,_panik_32-0b(%r13)
6:      .long  EDSSCH

################################################################################
# Wait for interrupt subroutine
#  -r2: device subchannel id
#  -r3: address of irb
################################################################################

_wait4de_32:
        stm    %r6,%r15,24(%r15)
        basr   %r13,0                   # base register
0:      s      %r15,.Lc96-0b(%r13)      # create stack frame
        lr     %r1,%r2
        mvc    0x78(8),6f-0b(%r13)      # set i/o new psw
1:      lpsw   5f-0b(%r13)
2:      stnsm  .Lpsw-0b(%r13),0xfd      # disable io interrupts
        c      %r1,0xb8                 # compare subchannel id
        bne    1b-0b(%r13)              # unequal -> continue waiting
        tsch   0(%r3)
        tm     9(%r3),0xff              # test channel status
        bnz    4f-0b(%r13)

3:      tm     8(%r3),0x02              # got unit check ?
        bnz    4f-0b(%r13)

        tm     8(%r3),0x04              # got device end ?
        bz     1b-0b(%r13)              # still busy -> continue waiting

4:      lm     %r6,%r15,120(%r15)
        br     %r14
        .align 8
5:      .long  0x020a0000,0x80000000+1b
6:      .long  0x00080000,0x80000000+2b # io new psw
.Lpsw:  .long  0x0

################################################################################
# Panik routine. Loads a disabled wait psw
#   -r2 : panik code
################################################################################

_panik_32:
        basr   %r1,0
0:      st     %r2,1f-0b+4(%r1)         # store code in address part of psw
        lpsw   1f-0b(%r1)
        .align 8
1:      .long  0x000a0000,0x00000000

.endm /* dump_io_subroutines_32 */







################################################################################
# MACRO: dump_common_fn_32
# - _take_dump_32
# - _count_mem_32
# - _wait4de_32
# - _store_status_32
# - _copy_lowcore_32
################################################################################

.macro dump_common_fn_32:


################################################################################
# Take the dump
#  - no parameters
################################################################################

_take_dump_32:
        stm   %r6,%r15,24(%r15)
        basr  %r13,0
.Lbase: s     %r15,.Lc96-.Lbase(%r13)            # create stack frame
        bas   %r14,_store_status_32-.Lbase(%r13) # store status
        bas   %r14,_count_mem_32-.Lbase(%r13)    # count memory 
        l     %r14,.Ldump_mem_32-.Lbase(%r13)
        basr  %r14,%r14                          # dump memory
        la    %r2,OK                         
        bas   %r14,_panik_32-.Lbase(%r13)        # everything ok: stop now

################################################################################
# Find out memory size:
# When accessing a page which is not there, we get a program check
#  - no parameters
################################################################################

_count_mem_32:
        stm    %r6,%r15,24(%r15)
        basr   %r13,0                         # base register
0:      s      %r15,.Lc96-0b(%r13)            # create stack frame
        slr    %r9,%r9                        # base register for zero page
        mvc   104(8,%r9),.Lcount_mem_psw-0b(%r13) # setup program check new psw
        slr   %r10,%r10
        la    %r11,1
        sll   %r11,20                         # 1 MB
.Lloop0:
        l     %r12,0(%r10)                    # test page
        ar    %r10,%r11                       # add 1M
        bnm   .Lloop0-0b(%r13)                # r10 < 0x80000000 -> loop
.Lchkmem0:
        n     %r10,.L4malign0-0b(%r13)        # align to multiples of 4M

        st    %r10,.Ldh_real_mem_size+4-0b(%r13)
        cl    %r10,.Lmem_upper_limit+4-0b(%r13)# check if we have an upper limit
        bl    1f-0b(%r13)
        l     %r10,.Lmem_upper_limit+4-0b(%r13)# upper mem limit set -> use it!
1:
        st    %r10,.Ldh_mem_size+4-0b(%r13)   # store memory size
        st    %r10,.Ldh_mem_end+4-0b(%r13)    # store memory end
        srl   %r10,12                         # calculate page count (/ 4096)
        st    %r10,.Ldh_num_pages-0b(%r13)    # store page count
        mvc   88(32,%r9),.Lnew_psws-0b(%r13)  # restore disabled wait new psws
        lm    %r6,%r15,120(%r15)
        br    %r14
.Lcount_mem_psw:  .long 0x00080000,0x80000000 + .Lchkmem0
.Lnew_psws:
.long  0x000a0000,0x00000058                  # external new psw
.long  0x000a0000,0x00000060                  # svc new psw
.long  0x000a0000,0x00000068                  # program check new psw
.long  0x000a0000,0x00000070                  # machine check new psw
.L4malign0:       
.long 0xffc00000

# expand Macros

   dump_common_store_status_32

# data 

.align 4
.Ldump_mem_32:       .long _dump_mem_32      # address of function

.endm    /* dump_common_fn_32 */



#endif /* __s390x__ */




################################################################################
# MACRO: dump_common_store_status_32
# - _store_status_32
# - _copy_lowcore_32
################################################################################


.macro dump_common_store_status_32:

################################################################################
# store status of all cpus in their lowcores
#  - no parameters
################################################################################

_store_status_32:
        stm   %r6,%r15,24(%r15)
        basr  %r13,0                          # base register
0:      s     %r15,.Lc96-0b(%r13)
        la    %r7,0x0                         # base register for 0 page

        ######## move lowcore info (assume user has made store  ########
        ######## status) to prefix-page                         ########

        bas   %r14,_copy_lowcore_32-0b(%r13)

        ######## stop all cpus and store status in prefix pages ########

        la    %r8,0                           # first cpu
        stap  .Lcurrent_cpu+2-0b(%r13)        # store current cpu address

1:
        cl    %r8,.Lcurrent_cpu-0b(%r13)      # is ipl cpu ?
        be    4f-0b(%r13)                     # if yes get next cpu
2:
        lr    %r9,%r7
        sigp  %r9,%r8,0x9                     # stop & store status of cpu
        bc   8,3f-0b(%r13)                    # accepted
        bc   4,4f-0b(%r13)                    # status stored: next cpu
        bc   2,2b-0b(%r13)                    # busy:          try again
        bc   1,4f-0b(%r13)                    # not op:        next cpu
3:
        bas   %r14,_copy_lowcore_32-0b(%r13)
4:
        la    %r8,1(%r8)                      # next cpu (r8 +=1)
        cl    %r8,.Llast_cpu-0b(%r13)         # is last possible cpu ?
        bl    1b-0b(%r13)                     # jump if not last cpu
.Lstore_status_exit:
        lm      %r6,%r15,120(%r15)
        br      %r14                          # return to caller
.Lcurrent_cpu:
        .long 0x0
.Llast_cpu:
        .long 0x0000ffff

################################################################################
# copy lowcore to prefix page
#  - no parameters
################################################################################

_copy_lowcore_32:
        stm    %r6,%r15,24(%r15)
        basr   %r13,0                         # base register
0:      s      %r15,.Lc96-0b(%r13)

        la     %r2,0                          # base register for lowcore
        l      %r3,0x108(%r2)                 # get prefix page from lowcore

        ###### check if lowcore address looks valid ######

        cl    %r3,.Llinux_start-0b(%r13)      # looks valid ?
        bl    .Lcpy_locore_exit-0b(%r13)      # if < linux-start addr
        l     %r6,.Lpage_align-0b(%r13)       # check page alignment
        nr    %r3,%r6
        cl    %r3,0x108(%r2)
        bnz   .Lcpy_locore_exit-0b(%r13)      # if not page aligned

        ###### copy lowcore                         ######

        # |-----------------------------------------------------------|
        # | Decimal |  Length   | Data                                |
        # | Address |  in Bytes |                                     |
        # |_________|___________|_____________________________________|
        # | 212     | 4         | Extended save area address          |
        # | 216     | 8         | CPU timer                           |
        # | 224     | 8         | Clock comparator                    |
        # | 256     | 8         | Current PSW                         |
        # | 264     | 4         | Prefix register                     |
        # | 288     | 64        | Access registers 0 through 15       |
        # | 352     | 32        | Floating-point registers 0 through 6|
        # | 384     | 64        | General registers 0 through 15      |
        # | 448     | 64        | Control registers 0 through 15      |
        # |_________|___________|_____________________________________|

        mvc   212(20,%r3),212(%r2)
        mvc   256(12,%r3),256(%r2)
        mvc   288(224,%r3),288(%r2)

.Lcpy_locore_exit:
        lm    %r6,%r15,120(%r15)
        br    %r14                            # return to caller
.Lpage_align:
        .long -4096
.Llinux_start:
        .long  0x10000

.endm /* dump_common_store_status_32 */
