/*
 *
 *   magick.c - Ruby interface for ImageMagick
 *
 *   Copyright (C) 2001 Ryuichi Tamura(tam@kais.kyoto-u.ac.jp)
 *
 *   $Date: 2001/08/02 21:24:38 $
 *   $Revision: 1.35 $
 *
 */

#include "magick.h"

VALUE mMagick;
VALUE cImage;
VALUE eImageError;

EXTERN void Init_MagickPrivate _((void));
EXTERN void Init_enums _((void));
EXTERN VALUE mgk_do_font_metrics _((VALUE, VALUE));





static void
free_Image(Image *image)
{
    if (image){
        if (image->next){
            DestroyImages(image);
        }
        else {
            DestroyImage(image);
        }
        image = (Image *)NULL;
    }
}

static void
free_ImageInfo(ImageInfo *info)
{
    if (info){
        DestroyImageInfo(info);
        info = (ImageInfo *)NULL;
    }
}

static void
free_MgkImage(MgkImage *im)
{
    free_Image(im->ref->image);
    free_ImageInfo(im->ref->image_info);
    im->ptr = (Image *)NULL;
    xfree(im->ref);
    im->ref = (ImageRef *)NULL;
    xfree(im);
    im = (MgkImage *)NULL;
}

static ImageRef*
init_ImageRef(Image *im, ImageInfo *info)
{
    ImageRef *imref;
    imref = ALLOC(ImageRef);

    imref->image = im;
    if (im){
        imref->image->next = (Image *)NULL;
        imref->image->previous = (Image *)NULL;
    }
    imref->image_info = CloneImageInfo(info);
    return imref;
}

static MgkImage*
init_MgkImage(Image *image, ImageInfo *info)
{
    MgkImage *mgk;
    ImageRef *imref;

    imref = init_ImageRef(image, info);

    mgk = ALLOC(MgkImage);
    mgk->ref = imref;
    mgk->ptr = (Image *)NULL;

    return mgk;
}





/*
 *
 *    warning/error handler routines
 *
 */

void
mgk_warn(const ExceptionType etype, const char *reason, const char *desc)
{
    rb_warn("%s: %s (%d)", reason, desc, etype);
}

void
mgk_raise(const ExceptionType etype, const char *reason, const char *desc)
{
    rb_raise(eImageError, "%s: %s (%d)", reason, desc, etype);
}

/* throws an exception, or only issues a warning message */
void
check_exception(Image *image, ExceptionInfo *e)
{
    if (!image){
        if (e->severity >= ResourceLimitWarning &&
            e->severity <= StreamWarning) {
            MagickWarning(e->severity, e->reason, e->description);
        }
        else if(e->severity >= FatalException &&
                 e->severity <= StreamError) {
            ThrowException(e, e->severity, e->reason, e->description);
        }
        else {
            rb_raise(rb_eRuntimeError, "unknown error occured");
        }
    }
}





static void
read_from_files(MgkImage *mgk, VALUE files)
{
    ImageInfo *image_info;
    Image *next;
    int i, index;
    ExceptionInfo e;

    GetInfoStruct(mgk, image_info);
    index = 1;
    for (i = 0; i < RARRAY(files)->len; i++){
        char *filename;
        filename = STR2CSTR(*(RARRAY(files)->ptr+i));
        strncpy(image_info->filename, filename, strlen(filename)+1);
        next = ReadImage(image_info, &e);
        check_exception(next, &e);
        if (!next)
            continue;
        next->scene = index;
        /* create the head image */
        if (mgk->ref->image == (Image *)NULL) {
            mgk->ref->image = next;
            mgk->ref->image->previous = (Image *)NULL;
        }
        /* create the linked image */
        else {
            register Image *q;
            for (q = mgk->ref->image; q->next != (Image *)NULL; q = q->next)
                ; /* do nothing */
            q->next = next;
            next->previous = q;
            next->next = (Image *)NULL;
        }
        index++;
    }
    mgk->ptr = mgk->ref->image;
}





/*
 *
 *  Class Method
 *
 */

static VALUE
mgk_s_image_ping(VALUE klass, VALUE fname)
{
    ExceptionInfo e;
    ImageInfo *image_info;
    Image *ping_image;
    VALUE ret_ary[4];

    Check_Type(fname, T_STRING);
    image_info = CloneImageInfo(NULL);
    GetExceptionInfo(&e);

    strncpy(image_info->filename, RSTRING(fname)->ptr, RSTRING(fname)->len+1);
    ping_image = PingImage(image_info, &e);

    ret_ary[0] = INT2FIX(ping_image->columns);
    ret_ary[1] = INT2FIX(ping_image->rows);
    ret_ary[2] = INT2NUM(ping_image->filesize);
    ret_ary[3] = rb_str_new2(ping_image->magick);

    DestroyImageInfo(image_info);
    DestroyImage(ping_image);

    return rb_ary_new4(4, ret_ary);
}


/*
 *
 *  Initialization
 *
 */

static VALUE
mgk_image_s_new(int argc, VALUE *argv, VALUE klass)
{
    MgkImage *mgk;
    VALUE obj;

    /* empty image */
    mgk = init_MgkImage(NULL, NULL);

    obj = Data_Wrap_Struct(klass, 0, free_MgkImage, mgk);
    rb_obj_call_init(obj, argc, argv);
    return obj;
}

static VALUE
mgk_image_initialize(int argc, VALUE *argv,  VALUE obj)
{
    return Qnil;
}

static VALUE
mgk_image_do_read_from_files(VALUE obj, VALUE io)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    read_from_files(mgk, io);

    return obj;
}

/*
 *
 *  S E T  /  G E T  A T T R I B U T E S
 *
 */

static VALUE
mgk_image_set(VALUE obj, VALUE hash)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);

    Check_Type(hash, T_HASH);
    mgk_set_attr(mgk, hash);

    return obj;
}

static VALUE
mgk_image_get(VALUE obj, VALUE args)
{
    MgkImage *mgk;
    int i;
    VALUE ret = Qnil;

    Get_MgkImage(obj, mgk);
    Check_Type(args, T_ARRAY);
    if (RARRAY(args)->len != 1){
        ret = rb_ary_new();
        for (i = 0; i < RARRAY(args)->len; i++)
            rb_ary_push(ret, mgk_get_attr(mgk, *(RARRAY(args)->ptr+i)));
    }
    else {
        ret = mgk_get_attr(mgk, *(RARRAY(args)->ptr));
    }

    return ret;
}





/*
 *
 *   I M A G E  R E F E R E N C E
 *
 */

static VALUE
mgk_image_at(VALUE obj, VALUE ind)
{
    MgkImage *mgk;
    Image *p;
    int ref, j;

    Get_MgkImage(obj, mgk);
    ref = FIX2INT(ind);
    if (GetNumberScenes(mgk->ref->image) - 1 < ref)
        return Qnil;
    for (p=mgk->ref->image, j=0; p!=(Image *)NULL; p=p->next, j++){
        if (j == ref) {
            /* now points to the refered image */
            mgk->ptr = p;
            break;
        }
    }
    rb_yield(obj);

    mgk->ptr = mgk->ref->image;
    return obj;
}

static VALUE
mgk_image_aref(VALUE obj, VALUE ind)
{
    MgkImage *mgk;
    Image *p;
    int ref, j;

    Get_MgkImage(obj, mgk);
    ref = FIX2INT(ind);
    if (GetNumberScenes(mgk->ref->image) - 1 < ref)
        return Qnil;
    for (p=mgk->ref->image, j=0; p!=(Image *)NULL; p=p->next, j++){
        if (j == ref) {
            /* now points to the refered image */
            mgk->ptr = p;
            break;
        }
    }
    return obj;
}

static VALUE
mgk_image_aset(VALUE obj, VALUE ind, VALUE other)
{
    MgkImage *mgk, *mgk_other;
    Image *p, *other_dup;
    ExceptionInfo e;
    int at, j;

    Get_MgkImage(obj, mgk);
    IsKindOf(other, cImage);
    Get_MgkImage(other, mgk_other);
    GetExceptionInfo(&e);
    other_dup = CloneImage(mgk_other->ptr, 0, 0, 0, &e);

    at = FIX2INT(ind);
    if (GetNumberScenes(mgk->ref->image) - 1 < at)
        rb_raise(rb_eRangeError, "index out of range");
    for (p=mgk->ref->image, j=0; p!=(Image *)NULL; p=p->next, j++){
        if (j == at){
            if (p->previous){
                other_dup->previous = p->previous;
                p->previous->next = other_dup;
                other_dup->next = p->next;
            }
            else {
                mgk->ref->image = other_dup;
                other_dup->next = p->next;
            }
            if (mgk->ptr == p)
                mgk->ptr = other_dup;
            DestroyImage(p);
            break;
        }
    }
    return obj;
}


static VALUE
mgk_image_each(VALUE obj)
{
    MgkImage *mgk;
    Image *p;

    Get_MgkImage(obj, mgk);
    for ( p = mgk->ref->image; p != (Image *)NULL; p = p->next){
        mgk->ptr = p;
        rb_yield(obj);
        /* mgk->ptr = p;*/
    }
    mgk->ptr = mgk->ref->image;
    return obj;
}

static VALUE
mgk_image_push(VALUE obj, VALUE other)
{
    MgkImage *mgk, *mgk_other;
    Image *clone_image, *p;
    ExceptionInfo e;

    IsKindOf(other, cImage);
    Get_MgkImage(obj, mgk);
    Get_MgkImage(other, mgk_other);
    GetExceptionInfo(&e);
    /* clone an image as orphant */
    clone_image = CloneImage(mgk_other->ptr, 0, 0, 1, &e);

    if (mgk->ref->image) { /* not empty image? */
        for (p = mgk->ref->image; p->next != (Image *)NULL; p = p->next)
             ; /* do nothing */
        p->next = clone_image;
        clone_image->previous = p;
    }
    else {
        clone_image->next = (Image *)NULL;
        clone_image->previous = (Image *)NULL;
        mgk->ref->image = clone_image;
        mgk->ptr = mgk->ref->image;
    }
        return obj;
}





static void
replace_image(MgkImage *mgk, Image *new_im)
{
    if (!new_im)
        return;

    if (mgk->ptr == mgk->ref->image){
        Image *orig_next;
        orig_next = mgk->ref->image->next;
        DestroyImage(mgk->ref->image); /* now orig_next is the head image */
        new_im->next = orig_next;
        if (orig_next)
            orig_next->previous = new_im;
        mgk->ptr = mgk->ref->image = new_im;
    }
    else {
        Image *orig_next, *orig_previous;
        orig_next = mgk->ptr->next;
        orig_previous = mgk->ptr->previous;
        DestroyImage(mgk->ptr);
        if (orig_next){
            new_im->next =  orig_next;
            orig_next->previous = new_im;
        }
        else {
            new_im->next = (Image *)NULL;
        }
        orig_previous->next = new_im;
        new_im->previous = orig_previous;
        mgk->ptr = new_im;
    }

}

/*
 *
 *  W R I T E    I M A G E
 *
 */

static VALUE
mgk_image_write_image(int argc, VALUE *argv, VALUE obj)
{
    MgkImage *mgk;
    Image *p;
    ImageInfo *info;
    VALUE opt_hash;
    int status, scene;

    rb_scan_args(argc, argv, "01", &opt_hash);
    Get_MgkImage(obj, mgk);
    GetInfoStruct(mgk, info);
    if (!NIL_P(opt_hash))
        mgk_set_attr(mgk,  opt_hash);

    scene = 0;
    for (p = mgk->ref->image; p != (Image *)NULL; p = p->next){
        strcpy(p->filename, info->filename);
        p->scene = scene;
        scene++;
    }
#if MagickLibVersion >=0x0534
    SetImageInfo(info, 1, &mgk->ref->image->exception);
#else
    SetImageInfo(info, 1);
#endif /* MagickLibVersion >= 0x0534 */
    for (p = mgk->ref->image; p != (Image *)NULL; p = p->next){
        status = WriteImage(info, p);
        if (!status)
            rb_warn("operation not completed: WriteImage()");
        if (info->adjoin)
            break;
    }
    return obj;
}


/*
 *
 *  I M A G E    E F F E C T S   M E T H O D S
 *
 */

static VALUE
mgk_image_add_noise(VALUE obj, VALUE noise_type)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);

    Get_MgkImage(obj, mgk);
    new_image = AddNoiseImage(mgk->ptr, FIX2UINT(noise_type), &e);
    check_exception(new_image, &e);

    replace_image(mgk, new_image);
    return obj;
}

static VALUE
mgk_image_blur(VALUE obj, VALUE radius, VALUE sigma)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);
    new_image = BlurImage(mgk->ptr,
                          NUM2DBL(radius),
                          NUM2DBL(sigma),
                          &e);
    check_exception(new_image, &e);

    replace_image(mgk, new_image);
    return obj;
}


static VALUE
mgk_image_colorize(VALUE obj, VALUE opacity_str, VALUE color)
{
    MgkImage *mgk;
    Image *new_image;
    PixelPacket target;
    ExceptionInfo e;

    Check_Type(opacity_str, T_STRING);
    Check_Type(color, T_STRING);

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    QueryColorDatabase(RSTRING(color)->ptr, &target);
    new_image = ColorizeImage(mgk->ptr, RSTRING(opacity_str)->ptr,
                              target, &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_convolve(VALUE obj, VALUE kern)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;
    double *kernel;
    int i, order;
    SetWarningHandler(mgk_raise);

    Check_Type(kern, T_ARRAY);
    order = RARRAY(kern)->len;
    kernel = ALLOC_N(double, order);

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);
    for (i = 1; i < order; i++)
        kernel[i] = *(RARRAY(kern)->ptr+i);

    new_image = ConvolveImage(mgk->ptr,
                              order, kernel,
                              &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);
    xfree(kernel);
    return obj;
}

static VALUE
mgk_image_despeckle(VALUE obj)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = DespeckleImage(mgk->ptr, &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_edge(VALUE obj, VALUE radius)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = EdgeImage(mgk->ptr, NUM2DBL(radius), &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_emboss(VALUE obj, VALUE radius, VALUE sigma)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = EmbossImage(mgk->ptr,
                          NUM2DBL(radius), NUM2DBL(sigma),
                          &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_enhance(VALUE obj)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = EnhanceImage(mgk->ptr,&e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_gaussian_blur(VALUE obj, VALUE radius, VALUE sigma)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = GaussianBlurImage(mgk->ptr,
                                  NUM2DBL(radius), NUM2DBL(sigma),
                                  &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_implode(VALUE obj, VALUE factor)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = ImplodeImage(mgk->ptr, NUM2DBL(factor), &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_median_filter(VALUE obj, VALUE radius)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = MedianFilterImage(mgk->ptr, NUM2DBL(radius), &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_morph(VALUE obj, VALUE number_frames)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = MorphImages(mgk->ptr,
                           FIX2UINT(number_frames), &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_motion_blur(VALUE obj, VALUE radius, VALUE sigma, VALUE amount)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = MotionBlurImage(mgk->ptr,
                                 NUM2DBL(radius), NUM2DBL(sigma),
                                 NUM2DBL(amount), &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}



static VALUE
mgk_image_oilpaint(VALUE obj, VALUE radius)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = OilPaintImage(mgk->ptr, NUM2DBL(radius), &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_plasma(VALUE obj, VALUE segment, VALUE attenuate, VALUE depth)
{
    MgkImage *mgk;
    SegmentInfo segment_info;
    int status;

    Check_Type(segment, T_ARRAY);
    srand(time(0));
    if (RARRAY(segment)->len != 4)
        rb_raise(rb_eArgError, "must specify [x1, y1, x2, y2]");

    Get_MgkImage(obj, mgk);
    segment_info.x1 = NUM2DBL(*(RARRAY(segment)->ptr));
    segment_info.y1 = NUM2DBL(*(RARRAY(segment)->ptr+1));
    segment_info.x2 = NUM2DBL(*(RARRAY(segment)->ptr+2));
    segment_info.y2 = NUM2DBL(*(RARRAY(segment)->ptr+3));

    status = PlasmaImage(mgk->ptr, &segment_info, FIX2INT(attenuate), FIX2INT(depth));
    if (!status)
        rb_warn("operation not completed: PlasmaImage()");

    return obj;
}

static VALUE
mgk_image_reduce_noise(VALUE obj, VALUE radius)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = ReduceNoiseImage(mgk->ptr, NUM2DBL(radius), &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_shade(VALUE obj, VALUE color_shading, VALUE azimuth, VALUE elevation)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = ShadeImage(mgk->ptr,
                           FIX2UINT(color_shading),
                           NUM2DBL(azimuth),
                           NUM2DBL(elevation),
                           &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_sharpen(VALUE obj, VALUE radius, VALUE sigma)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = SharpenImage(mgk->ptr,
                             NUM2DBL(radius), NUM2DBL(sigma),
                             &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_solarize(VALUE obj, VALUE factor)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    SolarizeImage(mgk->ptr, NUM2DBL(factor));

    return obj;
}

static VALUE
mgk_image_spread(VALUE obj, VALUE amount)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = SpreadImage(mgk->ptr, FIX2UINT(amount), &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_stegano(VALUE obj, VALUE watermark)
{
    MgkImage *mgk, *mgk_watermark;
    Image *new_image;
    ExceptionInfo e;

    IsKindOf(watermark, cImage);
    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);
    Get_MgkImage(obj, mgk_watermark);

    new_image = SteganoImage(mgk->ptr, mgk_watermark->ptr, &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_stereo(VALUE obj, VALUE offset_image)
{
    MgkImage *mgk, *mgk_offset;
    Image *new_image;
    ExceptionInfo e;

    IsKindOf(offset_image, cImage);
    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);
    Get_MgkImage(offset_image, mgk_offset);

    new_image = StereoImage(mgk->ptr, mgk_offset->ptr, &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_swirl(VALUE obj, VALUE degrees)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = SwirlImage(mgk->ptr, NUM2DBL(degrees), &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_threshold(VALUE obj, VALUE threshold)
{
    MgkImage *mgk;
    unsigned int th;

    Get_MgkImage(obj, mgk);
    th = ThresholdImage(mgk->ptr, NUM2DBL(threshold));

    return obj;
}

static VALUE
mgk_image_unsharp_mask(VALUE obj, VALUE radius, VALUE sigma, VALUE amount,
                       VALUE threshold)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = UnsharpMaskImage(mgk->ptr,
                                 NUM2DBL(radius),
                                 NUM2DBL(sigma),
                                 NUM2DBL(amount),
                                 NUM2DBL(threshold),
                                 &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_wave(VALUE obj, VALUE amplitude, VALUE wave_length)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = WaveImage(mgk->ptr,
                          NUM2DBL(amplitude),
                          NUM2DBL(wave_length),
                          &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}





/*
 *
 *  I M A G E    T R A N S F O R M   M E T H O D S
 *
 */

static void ary2rectinfo(VALUE, RectangleInfo*); /* forward decl */

static VALUE
mgk_image_chop(VALUE obj, VALUE rect)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;
    RectangleInfo rect_info;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    ary2rectinfo(rect, &rect_info);
    new_image = ChopImage(mgk->ptr, &rect_info, &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_coalesce(VALUE obj)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = CoalesceImages(mgk->ptr, &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_crop(VALUE obj, VALUE rect)
{
    MgkImage *mgk;
    Image *new_image;
    RectangleInfo rect_info;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);
    ary2rectinfo(rect, &rect_info);
    new_image = CropImage(mgk->ptr, &rect_info, &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_deconstruct(VALUE obj)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = DeconstructImages(mgk->ptr, &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_flip(VALUE obj)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = FlipImage(mgk->ptr, &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_flop(VALUE obj)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = FlopImage(mgk->ptr, &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_profile(VALUE obj, VALUE profname, VALUE filename)
{
    MgkImage *mgk;
    int status;
    char *prof, *fname;

    Check_Type(profname, T_STRING);
    Check_Type(filename, T_STRING);
    Get_MgkImage(obj, mgk);

    prof = NIL_P(profname) ? (char *)NULL : RSTRING(profname)->ptr;
    fname = NIL_P(filename) ? (char *)NULL : RSTRING(filename)->ptr;
    status = ProfileImage(mgk->ptr, prof, fname);
    if (!status)
        rb_warn("operation not completed: ProfileImage()");

    return obj;
}

static VALUE
mgk_image_roll(VALUE obj, VALUE offset_x, VALUE offset_y)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = RollImage(mgk->ptr,
                          FIX2UINT(offset_x), FIX2UINT(offset_y),
                          &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_transform(VALUE obj, VALUE geom, VALUE crop_geom)
{
    MgkImage *mgk;
    Image *new;
    ExceptionInfo e;

    Check_Type(crop_geom, T_STRING);
    Check_Type(geom, T_STRING);
    Get_MgkImage(obj, mgk);
    GetExceptionInfo(&e);

    new = CloneImage(mgk->ptr, 0, 0, 1, &e);

    TransformImage(&new, RSTRING(crop_geom)->ptr, RSTRING(geom)->ptr);
    check_exception(new, &e);

    replace_image(mgk, new);
    return obj;
}






/*
 *
 *  I M A G E   M A G N I F Y   M E T H O D S
 *
 */


static VALUE
mgk_image_magnify(VALUE obj)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    Get_MgkImage(obj, mgk);
    GetExceptionInfo(&e);

    new_image = MagnifyImage(mgk->ptr, &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_minify(VALUE obj)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    Get_MgkImage(obj, mgk);
    GetExceptionInfo(&e);

    new_image = MinifyImage(mgk->ptr, &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_resize(VALUE obj, VALUE rect,
                 VALUE filter_type, VALUE blur)
{
    MgkImage *mgk;
    Image *new_image;
    RectangleInfo rect_info;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    ary2rectinfo(rect, &rect_info);
    new_image = ResizeImage(mgk->ptr,
                            rect_info.width, rect_info.height,
                            FIX2UINT(filter_type), NUM2DBL(blur),
                            &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_sample(VALUE obj, VALUE rect)
{
    MgkImage *mgk;
    Image *new_image;
    RectangleInfo rect_info;
    ExceptionInfo e;

    Get_MgkImage(obj, mgk);
    GetExceptionInfo(&e);

    ary2rectinfo(rect, &rect_info);
    new_image = SampleImage(mgk->ptr,
                            rect_info.width, rect_info.height,
                            &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_scale(VALUE obj, VALUE rect)
{
    MgkImage *mgk;
    Image *new_image;
    RectangleInfo rect_info;
    ExceptionInfo e;

    Get_MgkImage(obj, mgk);
    GetExceptionInfo(&e);

    ary2rectinfo(rect, &rect_info);
    new_image = ScaleImage(mgk->ptr,
                           rect_info.width, rect_info.height,
                           &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_zoom(VALUE obj, VALUE rect)
{
    MgkImage *mgk;
    Image *new_image;
    RectangleInfo rect_info;
    ExceptionInfo e;

    Get_MgkImage(obj, mgk);
    GetExceptionInfo(&e);

    ary2rectinfo(rect, &rect_info);
    new_image = ZoomImage(mgk->ptr,
                          rect_info.width, rect_info.height,
                          &e);
    check_exception(mgk->ptr, &e);
    replace_image(mgk, new_image);

    return obj;
}





/*
 *
 *  I M A G E   A N N O T A T E   M E T H O D S
 *
 */


static VALUE
mgk_image_do_annotate(VALUE obj, VALUE dinfo)
{
    MgkImage *mgk;
    DrawInfo *draw_info;
    int status;

    IsKindOf(dinfo, cDrawInfo);
    Get_MgkImage(obj, mgk);
    Get_DrawInfo(dinfo, draw_info);

    status = AnnotateImage(mgk->ptr, draw_info);
    if (!status)
        rb_warn("operation not completed: AnnotateImage()");

    return obj;
}

static VALUE
mgk_image_do_draw(VALUE obj, VALUE dinfo)
{
    MgkImage *mgk;
    DrawInfo *draw_info;
    int status;

    Get_MgkImage(obj, mgk);
    Get_DrawInfo(dinfo, draw_info);

    status = DrawImage(mgk->ptr, draw_info);
    if (!status)
        rb_warn("operation not completed: DrawImage()");

    return obj;
}

static VALUE
mgk_image_colorfloodfill(VALUE obj, VALUE dinfo, VALUE color, VALUE rect, VALUE paint_method)
{
    MgkImage *mgk;
    RectangleInfo rect_info;
    DrawInfo *draw_info;
    PixelPacket target_color;
    int status;

    Check_Type(color, T_STRING);
    Check_Type(rect, T_ARRAY);
    Get_MgkImage(obj, mgk);
    Get_DrawInfo(dinfo, draw_info);
    ary2rectinfo(rect, &rect_info);
    QueryColorDatabase(RSTRING(color)->ptr, &target_color);
    status = ColorFloodfillImage(mgk->ptr,draw_info,target_color,
                                 rect_info.x, rect_info.y, FIX2INT(paint_method));
    if (!status)
        rb_warn("operation not completed: MatteFloodfillImage()");
    return obj;
}

static VALUE
mgk_image_mattefloodfill(VALUE obj, VALUE color, VALUE amount, VALUE rect, VALUE paint_method)
{
    MgkImage *mgk;
    RectangleInfo rect_info;
    PixelPacket target_color;
    int status;

    Check_Type(color, T_STRING);
    Check_Type(rect, T_ARRAY);
    Get_MgkImage(obj, mgk);
    ary2rectinfo(rect, &rect_info);
    QueryColorDatabase(RSTRING(color)->ptr, &target_color);
    status = MatteFloodfillImage(mgk->ptr,target_color,FIX2INT(amount),
                                 rect_info.x, rect_info.y, FIX2INT(paint_method));
    if (!status)
        rb_warn("operation not completed: MatteFloodfillImage()");
    return obj;
}

static VALUE
mgk_image_opaque(VALUE obj, VALUE opaque_color, VALUE pen_color)
{
    MgkImage *mgk;
    PixelPacket opaque_target, pen_target;
    int status;

    Check_Type(opaque_color, T_STRING);
    Check_Type(pen_color, T_STRING);

    Get_MgkImage(obj, mgk);
    QueryColorDatabase(RSTRING(opaque_color)->ptr,&opaque_target);
    QueryColorDatabase(RSTRING(pen_color)->ptr,&pen_target);

    status = OpaqueImage(mgk->ptr, opaque_target, pen_target);
    if (!status)
        rb_warn("operation not completed: OpaqueImage()");
    return obj;
}

static VALUE
mgk_image_transparent(int argc, VALUE *argv, VALUE obj)
{
    MgkImage *mgk;
    PixelPacket target;
    int status;
    VALUE color;
    VALUE opacity = Qnil;

    rb_scan_args(argc, argv, "11", &color, &opacity);
    Check_Type(color, T_STRING);
    Get_MgkImage(obj, mgk);
    QueryColorDatabase(RSTRING(color)->ptr, &target);

#if MagickLibVersion >= 0x0531
    opacity = TransparentOpacity;
    status = TransparentImage(mgk->ptr,target,
                              NIL_P(opacity) ? TransparentOpacity : FIX2INT(opacity));
#else
    status = TransparentImage(mgk->ptr,target);
#endif
    if (!status)
        rb_warn("operation not completed: TransparentImage()");
    return obj;
}


/*
 *
 *  Montage
 *
 */

static VALUE
mgk_image_do_montage(VALUE obj, VALUE minfo)
{
    MgkImage *mgk, *new_mgk;
    Image *new_im, *tmp;
    ImageInfo *image_info;
    MontageInfo *montage_info;
    ExceptionInfo e;

    Get_MgkImage(obj, mgk)
    GetInfoStruct(mgk, image_info);
    GetExceptionInfo(&e);

    Get_MontageInfo(minfo, montage_info);
    tmp = MontageImages(mgk->ref->image, montage_info, &e);
    check_exception(tmp, &e);
    new_im = CloneImage(tmp, 0, 0, 1, &e);
    check_exception(new_im, &e);
    DestroyImages(tmp);
    new_mgk = init_MgkImage(new_im, NULL);
    strcpy(new_mgk->ref->image_info->filename, montage_info->filename);
    strcpy(new_mgk->ref->image->magick, mgk->ref->image->magick);
    new_mgk->ptr = new_mgk->ref->image;
    return Data_Wrap_Struct(cImage, 0, free_MgkImage, new_mgk);
}





/*
 *
 *  C O U N T I N G   C O L O R S
 *
 */

static VALUE
mgk_image_compress_colormap(VALUE obj)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    CompressColormap(mgk->ptr);

    return obj;
}

static VALUE
mgk_image_get_number_colors(int argc, VALUE *argv, VALUE obj)
{
    MgkImage *mgk;
    VALUE io;
    OpenFile *open_f;
    FILE *fptr;
    int c;

    c = rb_scan_args(argc, argv, "01", &io);
    if ( c == 1 ){
        Check_Type(io, T_FILE);
        GetOpenFile(io, open_f);
        fptr = open_f->f;
    }
    else {
        fptr = (FILE *)NULL;
    }
    Get_MgkImage(obj, mgk);

    return INT2NUM(GetNumberColors(mgk->ptr, fptr));
}

static VALUE
mgk_image_is_grayimage(VALUE obj)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    return IsGrayImage(mgk->ptr) ? Qtrue : Qfalse;
}

static VALUE
mgk_image_is_monochromeimage(VALUE obj)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    return IsMonochromeImage(mgk->ptr) ? Qtrue : Qfalse;
}

static VALUE
mgk_image_is_opaqueimage(VALUE obj)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    return IsOpaqueImage(mgk->ptr) ? Qtrue : Qfalse;
}

static VALUE
mgk_image_is_pseudoclass(VALUE obj)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    return IsPseudoClass(mgk->ptr) ? Qtrue : Qfalse;
}





/*
 *
 *  I M A G E   D E C O R A T A T I O N  M E T H O D S
 *
 */

static void
ary2rectinfo(VALUE rect_ary, RectangleInfo *rect_info)
{
    if (RARRAY(rect_ary)->len !=4)
        rb_raise(rb_eArgError, "wrong # of elements(%d for 4)", RARRAY(rect_ary)->len);

    rect_info->width  = FIX2UINT(*(RARRAY(rect_ary)->ptr));
    rect_info->height = FIX2UINT(*(RARRAY(rect_ary)->ptr+1));
    rect_info->x      = FIX2UINT(*(RARRAY(rect_ary)->ptr+2));
    rect_info->y      = FIX2UINT(*(RARRAY(rect_ary)->ptr+3));

    return;
}

static VALUE
mgk_image_border(VALUE obj, VALUE rect_ary, VALUE color)
{
    MgkImage *mgk;
    Image *new_image;
    ImageInfo *info;
    PixelPacket target_color;
    RectangleInfo rect_info;
    ExceptionInfo e;

    Check_Type(rect_ary, T_ARRAY);
    Check_Type(color, T_STRING);

    GetExceptionInfo(&e);

    Get_MgkImage(obj, mgk);
    GetInfoStruct(mgk, info);

    if (QueryColorDatabase(RSTRING(color)->ptr, &target_color))
        info->border_color = target_color;

    mgk->ptr->border_color = target_color;
    ary2rectinfo(rect_ary, &rect_info);
    new_image = BorderImage(mgk->ptr, &rect_info, &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_frame(VALUE obj, VALUE width, VALUE height,
                VALUE x, VALUE y, VALUE inner_bevel, VALUE outer_bevel)
{
    MgkImage *mgk;
    Image *new_image;
    FrameInfo frame_info;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    frame_info.width = FIX2UINT(width);
    frame_info.height = FIX2UINT(height);
    frame_info.x = FIX2INT(x);
    frame_info.y = FIX2INT(y);
    frame_info.inner_bevel = FIX2INT(inner_bevel);
    frame_info.outer_bevel = FIX2INT(outer_bevel);

    new_image = FrameImage(mgk->ptr, &frame_info, &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_raise(VALUE obj, VALUE rect_ary, VALUE raised)
{
    MgkImage *mgk;
    RectangleInfo rect_info;

    Check_Type(rect_ary, T_ARRAY);
    Get_MgkImage(obj, mgk);

    ary2rectinfo(rect_ary, &rect_info);
    RaiseImage(mgk->ptr, &rect_info, RTEST(raised));

    return obj;
}

/*
 *
 *  I M A G E   E N H A N C E   M E T H O D S
 *
 */

static VALUE
mgk_image_contrast(VALUE obj, VALUE sharpen)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    ContrastImage(mgk->ptr, RTEST(sharpen));

    return obj;
}

static VALUE
mgk_image_equalize(VALUE obj)
{
    MgkImage *mgk;
    int status;

    Get_MgkImage(obj, mgk);
    status = EqualizeImage(mgk->ptr);
    if (!status)
        rb_warn("operation not completed");

    return obj;
}

static VALUE
mgk_image_gamma(VALUE obj, VALUE gamma_str)
{
    MgkImage *mgk;
    int status;

    Check_Type(gamma_str, T_STRING);
    Get_MgkImage(obj, mgk);

    status = GammaImage(mgk->ptr, RSTRING(gamma_str)->ptr);
    if (!status)
        rb_warn("operation not completed");

    return obj;
}

static VALUE
mgk_image_modulate(VALUE obj, VALUE modulation_str)
{
    MgkImage *mgk;
    int status;

    Check_Type(modulation_str, T_STRING);
    Get_MgkImage(obj, mgk);

    status = ModulateImage(mgk->ptr, RSTRING(modulation_str)->ptr);
    if (!status)
        rb_warn("operation not completed");

    return obj;
}

static VALUE
mgk_image_negate(VALUE obj, VALUE grayscale)
{
    MgkImage *mgk;
    int status;

    Get_MgkImage(obj, mgk);

    status = NegateImage(mgk->ptr, RTEST(grayscale));
    if (!status)
        rb_warn("operation not completed: NegateImage()");

    return obj;
}

static VALUE
mgk_image_normalize(VALUE obj)
{
    MgkImage *mgk;
    int status;

    Get_MgkImage(obj, mgk);

    status = NormalizeImage(mgk->ptr);
    if (!status)
        rb_warn("operation not completed: NormalizeImage()");

    return obj;
}

/*
 *
 *  Q U A N T I Z E  I M A G E S
 *
 */

static VALUE
mgk_image_map(VALUE obj, VALUE im, VALUE dither)
{
    MgkImage *mgk, *other;
    int status;

    IsKindOf(im, cImage);
    Get_MgkImage(obj, mgk);
    Get_MgkImage(im, other)

    status = MapImage(mgk->ptr, other->ptr, FIX2UINT(dither));
    if (!status)
        rb_warn("operation not completed: MapImage()");

    return obj;
}

static VALUE
mgk_image_map_images(VALUE obj, VALUE im, VALUE dither)
{
    MgkImage *mgk, *other;
    int status;

    IsKindOf(im, cImage);
    Get_MgkImage(obj, mgk);
    Get_MgkImage(im, other)

    status = MapImages(mgk->ref->image, other->ptr, FIX2UINT(dither));
    if (!status)
        rb_warn("operation not completed: MapImages()");

    return obj;
}

#if MagickLibVersion >= 0x0532
static VALUE
mgk_image_ordered_dither(VALUE obj)
{
    MgkImage *mgk;
    int status;

    Get_MgkImage(obj, mgk);
    status = OrderedDitherImage(mgk->ptr);
    if (!status)
        rb_warn("operation not completed: OrderedDitherImage()");

    return obj;
}
#endif

static VALUE
mgk_image_quantize(VALUE obj, VALUE quantize_info)
{
    MgkImage *mgk;
    QuantizeInfo *qinfo;
    int status;

    Get_MgkImage(obj, mgk);
    Get_QuantizeInfo(quantize_info, qinfo);

    status = QuantizeImage(qinfo, mgk->ptr);
    if (!status)
        rb_warn("operation not completed: QuantizeImage()");

    return obj;
}


static VALUE
mgk_image_quantize_images(VALUE obj, VALUE quantize_info)
{
    MgkImage *mgk;
    QuantizeInfo *qinfo;
    int status;

    Get_MgkImage(obj, mgk);
    Get_QuantizeInfo(quantize_info, qinfo);
    status = QuantizeImages(qinfo, mgk->ptr);
    if (!status)
        rb_warn("operation not completed: QuantizeImages()");

    return obj;
}


/*
 *
 *  S E G M E N T   I M A G E
 *
 */

static VALUE
mgk_image_segment(VALUE obj, VALUE colorspace_type,
                  VALUE verbose, VALUE cluster_threshould, VALUE smooth)
{
    MgkImage *mgk;
    int status;

    Get_MgkImage(obj, mgk);
    status = SegmentImage(mgk->ptr, FIX2INT(colorspace_type),
                          RTEST(verbose), NUM2DBL(cluster_threshould),
                          NUM2DBL(smooth));
    if (!status)
        rb_warn("operation not completed: SegmentImage()");
    return obj;
}


/*
 *
 *  S H E A R   O R   R O T A T E
 *
 */

static VALUE
mgk_image_rotate(VALUE obj, VALUE degree)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = RotateImage(mgk->ptr, NUM2INT(degree), &e);

    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_shear(VALUE obj, VALUE x_shear, VALUE y_shear)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    GetExceptionInfo(&e);
    Get_MgkImage(obj, mgk);

    new_image = ShearImage(mgk->ptr, NUM2DBL(x_shear), NUM2DBL(y_shear), &e);

    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

/*
 *
 *  O T H E R   M I S C E R E N E O U S    M E T H O D S
 *
 */

static VALUE
mgk_image_clone(VALUE obj)
{
    MgkImage *mgk, *mgk_dup;
    Image *im_dup;
    ImageInfo *info, *info_dup;
    ExceptionInfo e;

    Get_MgkImage(obj, mgk);
    GetInfoStruct(mgk, info);

    GetExceptionInfo(&e);
    im_dup = CloneImage(mgk->ptr, 0, 0, 1, &e);
    info_dup = CloneImageInfo(info);

    mgk_dup = init_MgkImage(im_dup, info_dup);
    mgk_dup->ptr = mgk_dup->ref->image;

    return Data_Wrap_Struct(cImage, 0, free_MgkImage, mgk_dup);
}

static VALUE
mgk_image_channel(VALUE obj, VALUE channel_type)
{
    MgkImage *mgk;
    int status;

    Get_MgkImage(obj, mgk);
    status = ChannelImage(mgk->ptr, FIX2INT(channel_type));
    if (!status)
        rb_warn("operation not completed: ChannelImage()");

    return obj;
}

static VALUE
mgk_image_composite(VALUE obj, VALUE other, VALUE comp_op,
                    VALUE offset_x, VALUE offset_y)
{
    MgkImage *mgk, *mgk_other;
    int status;

    IsKindOf(other, cImage);
    Get_MgkImage(obj, mgk);
    Get_MgkImage(other, mgk_other);

    status = CompositeImage(mgk->ptr, FIX2UINT(comp_op), mgk_other->ptr,
                            NUM2INT(offset_x), NUM2INT(offset_y));
    if (!status)
        rb_warn("operation not completed: CompositeImage()");
    return obj;
}


static VALUE
mgk_image_cycle_colormap(VALUE obj, VALUE amount)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    CycleColormapImage(mgk->ptr, FIX2INT(amount));

    return obj;
}

static VALUE
mgk_image_describe(int argc, VALUE *argv, VALUE obj)
{
    MgkImage *mgk;
    VALUE verbose_flag, dest_io;
    OpenFile *f;
    int c;

    c = rb_scan_args(argc, argv, "11", &dest_io, &verbose_flag);

    Check_Type(dest_io, T_FILE);
    GetOpenFile(dest_io, f);

    Get_MgkImage(obj, mgk);
    if (!NIL_P(verbose_flag)){
        DescribeImage(mgk->ptr, f->f, RTEST(verbose_flag));
    }
    else
        DescribeImage(mgk->ptr, f->f, Qfalse);

    return obj;
}

static VALUE
mgk_image_animate(VALUE obj)
{
    MgkImage *mgk;
    ImageInfo *image_info;
    int status;

    Get_MgkImage(obj, mgk);
    GetInfoStruct(mgk, image_info);

    status = AnimateImages(image_info, mgk->ref->image);
    if (!status)
        rb_warn("operation not completed: AnimateImages()");
    return obj;
}

static VALUE
mgk_image_append(int argc, VALUE *argv, VALUE obj)
{
    MgkImage *mgk;
    Image *new_image;
    VALUE direction;
    int c, d;
    ExceptionInfo e;

    Get_MgkImage(obj, mgk);
    c = rb_scan_args(argc, argv, "01", &direction);
    d = NIL_P(direction) ? 0 : 1;
    GetExceptionInfo(&e);
    new_image = AppendImages(mgk->ref->image, d, &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_display(VALUE obj)
{
    MgkImage *mgk;
    Image *p;
    ImageInfo *info;
    int scene = 1;

    Get_MgkImage(obj, mgk);
    GetInfoStruct(mgk, info);
    for (p = mgk->ref->image; p != (Image *)NULL; p = p->next){
        p->scene = scene;
        scene++;
    }
    DisplayImages(info, mgk->ptr);

    return obj;
}

static VALUE
mgk_image_get_image_boundingbox(VALUE obj)
{
    MgkImage *mgk;
    RectangleInfo rect_info;
    char str[MaxTextExtent];

    Get_MgkImage(obj, mgk);
    rect_info = GetImageBoundingBox(mgk->ptr);
    sprintf(str, "%dx%d+%d+%d",
            rect_info.width, rect_info.height,
            rect_info.x, rect_info.y);
    return rb_str_new2(str);
}

static VALUE
mgk_image_get_image_depth(VALUE obj)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    return INT2FIX(GetImageDepth(mgk->ptr));
}

static VALUE
mgk_image_get_image_type(VALUE obj)
{
    MgkImage *mgk;
    int type;
    char *types[] = {"Undefined", "Bilevel", "Grayscale",
                     "Palette", "PaletteMatte", "TrueColor",
                     "TrueColorMatte", "CMYK"};
    Get_MgkImage(obj, mgk);
    type = GetImageType(mgk->ptr);

    return rb_str_new2(types[type]);
}

static VALUE
mgk_image_get_number_scenes(VALUE obj)
{
    MgkImage *mgk;

    Get_MgkImage(obj, mgk);
    return INT2NUM(GetNumberScenes(mgk->ref->image));
}

static VALUE
mgk_image_mogrify(int argc, VALUE *argv, VALUE obj)
{
    MgkImage *mgk;
    ImageInfo *info;
    int i, status;
    char **opts = (char **)NULL;

    opts = (char **)malloc(sizeof(char *)*argc);
    for (i = 0; i < argc; i++){
        Check_Type(argv[i], T_STRING);
        opts[i] = strdup(RSTRING(argv[i])->ptr);
    }
    Get_MgkImage(obj, mgk);
    GetInfoStruct(mgk, info);
    status = MogrifyImages(info, argc, opts, &mgk->ptr);
    if (!status)
        rb_warn("operation not completed: MogrifyImage()");

    for (i = 0; i < argc; i++)
        free(opts[i]);
    free(opts);

    return obj;
}

static VALUE
mgk_image_mosaic(VALUE obj)
{
    MgkImage *mgk;
    Image *new_image;
    ExceptionInfo e;

    Get_MgkImage(obj, mgk);
    GetExceptionInfo(&e);

    new_image = MosaicImages(mgk->ref->image, &e);
    check_exception(new_image, &e);
    replace_image(mgk, new_image);

    return obj;
}

static VALUE
mgk_image_rgb_transform(VALUE obj, VALUE colorspace_type)
{
    MgkImage *mgk;
    int status;

    Get_MgkImage(obj, mgk);
    status = RGBTransformImage(mgk->ptr, FIX2INT(colorspace_type));
    if (!status)
        rb_warn("operation not completed: RGBTransformImage()");
    return obj;
}

static VALUE
mgk_image_texture(VALUE obj, VALUE other)
{
    MgkImage *mgk, *mgk_other;
    Image *clone_image;
    ExceptionInfo e;

    Get_MgkImage(obj, mgk);
    IsKindOf(other, cImage);
    Get_MgkImage(other, mgk_other);
    GetExceptionInfo(&e);

    clone_image = CloneImage(mgk_other->ptr, 0, 0, 1, &e);
    TextureImage(mgk->ptr, clone_image);
    check_exception(mgk->ptr, &e);

    return obj;
}

/*
 *
 *  dump image(s) into String object
 *
 */

static VALUE
mgk_image_to_str(VALUE obj)
{
    MgkImage *mgk;
    Image *p;
    ImageInfo *info;
    void *blob;
    ExceptionInfo e;
    size_t length;
    int scene = 0;
    VALUE ret_ary = rb_ary_new();

    Get_MgkImage(obj, mgk);
    GetInfoStruct(mgk, info);
    for (p = mgk->ref->image; p != (Image *)NULL; p = p->next){
        strcpy(p->filename, info->filename);
        p->scene = scene;
        scene++;
    }
    strcpy(info->magick, mgk->ptr->magick);
#if MagickLibVersion >=0x0534
    SetImageInfo(info, 1, &mgk->ref->image->exception);
#else
    SetImageInfo(info, 1);
#endif /* MagickLibVersion >= 0x0534 */
    GetExceptionInfo(&e);
    for (p = mgk->ref->image; p != (Image *)NULL; p = p->next){
        length = 0;
        blob = ImageToBlob(info, p, &length, &e);
        if (!blob)
            rb_raise(rb_eRuntimeError, "Unable to create BLOB");
        rb_ary_push(ret_ary,rb_str_new((char *)blob, length));
        LiberateMemory((void **)&blob);
    }
    return ret_ary;
}

static VALUE
mgk_s_to_image(VALUE klass, VALUE ary)
{
    MgkImage *mgk;
    Image *next;
    ImageInfo *image_info;
    ExceptionInfo e;
    int scene;
    VALUE str_ary;

    Check_Type(ary, T_ARRAY);
    GetExceptionInfo(&e);
    image_info = CloneImageInfo(NULL);
#if MagickLibVersion >=0x0534
    SetImageInfo(image_info, 1, &e);
#else
    SetImageInfo(image_info, 1);
#endif /* MagickLibVersion >= 0x0534 */

    mgk = init_MgkImage(NULL, image_info);
    str_ary = *(RARRAY(ary)->ptr); /* [["foo",...]] to ["foo",...] */
    for (scene = 0; scene < RARRAY(str_ary)->len; scene++) {
        VALUE str = *(RARRAY(str_ary)->ptr+scene);
        GetExceptionInfo(&e);
        next = BlobToImage(image_info, (char *)RSTRING(str)->ptr,
                           RSTRING(str)->len, &e);
        check_exception(next, &e);
        if (!next)
            continue;
        /* create the head image */
        if (mgk->ref->image == (Image *)NULL) {
            mgk->ref->image = next;
            mgk->ref->image->previous = (Image *)NULL;
        }
        /* create the linked image */
        else {
            register Image *q;
            for (q = mgk->ref->image; q->next != (Image *)NULL; q = q->next)
                ; /* do nothing */
            q->next = next;
            next->previous = q;
            next->next = (Image *)NULL;
        }
    }
    mgk->ptr = mgk->ref->image;

    return Data_Wrap_Struct(klass, 0, free_MgkImage, mgk);
}

/*
 *
 *  Some utility functions that will be used in lib/magick.rb
 *
 */

static VALUE
mgk_m_query_color(VALUE m, VALUE str)
{
    PixelPacket packet;
    int status;
    char *color;
    VALUE ary[3];

    Check_Type(str, T_STRING);
    color = RSTRING(str)->ptr;
    status = QueryColorDatabase(color, &packet);
    if (!status) {
        rb_warn("Unable to find RGB values for %s", color);
        return Qnil;
    }
    ary[0] = INT2FIX(packet.red);
    ary[1] = INT2FIX(packet.green);
    ary[2] = INT2FIX(packet.blue);

    return rb_ary_new4(3, ary);
}

#if MagickLibVersion >= 0x0534
static VALUE
mgk_m_query_name(VALUE m, VALUE r, VALUE g, VALUE b)
{
    int status;
    PixelPacket packet;
    char *ret = (char *)NULL;

    Check_Type(r, T_FIXNUM);
    Check_Type(g, T_FIXNUM);
    Check_Type(b, T_FIXNUM);

    packet.red = FIX2INT(r);
    packet.blue = FIX2INT(g);
    packet.green = FIX2INT(b);

    status = QueryColorname((Image *)NULL, &packet, 1, ret);
    if (!status)
        return Qnil;

    return rb_str_new2(ret);
}
#endif /* MagickLibVersion */

static VALUE
mgk_m_parse_geometry(VALUE m, VALUE str)
{
    int flags;
    RectangleInfo r;
    VALUE elts[5];

    Check_Type(str, T_STRING);
    if (!IsGeometry(RSTRING(str)->ptr))
        rb_raise(rb_eArgError, "Invalid geometry specification");
    flags = ParseGeometry(RSTRING(str)->ptr, &r.x, &r.y, &r.width, &r.height);
    elts[0] = INT2FIX(flags);
    elts[1] = INT2NUM(r.x); elts[2] = INT2NUM(r.y);
    elts[3] = INT2NUM(r.width); elts[4] = INT2NUM(r.height);

    return rb_ary_new4(5, elts);
}

static VALUE
mgk_m_parse_image_geometry(VALUE m, VALUE str)
{
    int flags;
    RectangleInfo r;
    VALUE elts[5];

    Check_Type(str, T_STRING);
    if (!IsGeometry(RSTRING(str)->ptr))
        rb_raise(rb_eArgError, "Invalid geometry specification");
    flags = ParseImageGeometry(RSTRING(str)->ptr, &r.x, &r.y, &r.width, &r.height);
    elts[0] = INT2FIX(flags);
    elts[1] = INT2NUM(r.x); elts[2] = INT2NUM(r.y);
    elts[3] = INT2NUM(r.width); elts[4] = INT2NUM(r.height);

    return rb_ary_new4(5, elts);
}





void Init_c_magick()
{
    /* defines ImageMagick module */
    mMagick = rb_define_module("Magick");

    SetWarningHandler(mgk_warn);
    SetErrorHandler(mgk_raise);

    /* Constants initialization */
    rb_define_const(mMagick, "Copyright", rb_str_new2(MagickCopyright));
    rb_define_const(mMagick, "LibVersion", INT2NUM(MagickLibVersion));
    rb_define_const(mMagick, "Version", rb_str_new2(MagickVersion));
    rb_define_const(mMagick, "MaxRGB", INT2NUM(MaxRGB));
    rb_define_const(mMagick, "TransparentOpacity", INT2NUM(TransparentOpacity));
    Init_enums();
    rb_include_module(mMagick, mClassType);
    rb_include_module(mMagick, mColorspaceType);
    rb_include_module(mMagick, mCompositeOp);
    rb_include_module(mMagick, mCompressionType);
    rb_include_module(mMagick, mDecorationType);
    rb_include_module(mMagick, mFilterTypes);
    rb_include_module(mMagick, mGravityType);
/*    rb_include_module(mMagick, mImageType); */
    rb_include_module(mMagick, mInterlaceType);
    rb_include_module(mMagick, mNoiseType);
    rb_include_module(mMagick, mPaintMethod);
    rb_include_module(mMagick, mPreviewType);
/*    rb_include_module(mMagick, mQuantumType); */
    rb_include_module(mMagick, mRenderingIntent);
    rb_include_module(mMagick, mResolutionType);

    /* Magick::Image class */
    cImage  = rb_define_class_under(mMagick, "Magick_c_Image", rb_cObject);
    rb_define_singleton_method(cImage, "new", mgk_image_s_new, -1);
    rb_define_private_method(cImage, "initialize", mgk_image_initialize, -1);
    rb_define_private_method(cImage, "do_read_from_files", mgk_image_do_read_from_files, 1);
    rb_define_singleton_method(cImage, "ping", mgk_s_image_ping, 1);
    rb_define_singleton_method(cImage, "to_image", mgk_s_to_image, -2);

    rb_define_method(cImage, "[]", mgk_image_aref, 1);
    rb_define_method(cImage, "[]=", mgk_image_aset, 2);
    rb_define_method(cImage, "at", mgk_image_at, 1);
    rb_define_method(cImage, "set", mgk_image_set, 1);
    rb_define_method(cImage, "get", mgk_image_get, -2);
    rb_define_method(cImage, "each", mgk_image_each, 0);
    rb_define_method(cImage, "push", mgk_image_push, 1);

    rb_define_method(cImage, "write", mgk_image_write_image, -1);

    rb_define_method(cImage, "magnify", mgk_image_magnify, 0);
    rb_define_method(cImage, "minify", mgk_image_minify, 0);
    rb_define_private_method(cImage, "do_resize", mgk_image_resize, 3);
    rb_define_private_method(cImage, "do_sample", mgk_image_sample, 1);
    rb_define_private_method(cImage, "do_scale", mgk_image_scale, 1);
    rb_define_private_method(cImage, "do_zoom", mgk_image_zoom, 1);

    rb_define_private_method(cImage, "do_chop", mgk_image_chop, 1);
    rb_define_method(cImage, "coalesce", mgk_image_coalesce, 0);
    rb_define_private_method(cImage, "do_crop", mgk_image_crop, 1);
    rb_define_method(cImage, "deconstruct", mgk_image_deconstruct, 0);
    rb_define_method(cImage, "flip", mgk_image_flip, 0);
    rb_define_method(cImage, "flop", mgk_image_flop, 0);
    rb_define_method(cImage, "profile", mgk_image_profile, 2);
    rb_define_method(cImage, "roll", mgk_image_roll, 2);
    rb_define_method(cImage, "transform", mgk_image_transform, 2);

    /*
     * methods for drawing primitives on an image
     * are implemented in lib/magick.rb
     */
    rb_define_private_method(cImage, "do_draw", mgk_image_do_draw, 1);
    rb_define_private_method(cImage, "do_color_floodfill", mgk_image_colorfloodfill, 4);
    rb_define_private_method(cImage, "do_matte_floodfill", mgk_image_mattefloodfill,4);
    rb_define_method(cImage, "opaque", mgk_image_opaque, 2);
    rb_define_method(cImage, "transparent", mgk_image_transparent, -1);

    /*
     * methods for annotating an image
     * are implementd in lib/magick.rb
     */
    rb_define_private_method(cImage, "do_annotate", mgk_image_do_annotate, 1);

    rb_define_method(cImage, "compress_colormap", mgk_image_compress_colormap, 0);
    rb_define_method(cImage, "get_number_colors", mgk_image_get_number_colors, -1);
    rb_define_method(cImage, "is_gray_image?", mgk_image_is_grayimage, 0);
    rb_define_method(cImage, "is_monochrome_image?", mgk_image_is_monochromeimage, 0);
    rb_define_method(cImage, "is_opaque_image?", mgk_image_is_opaqueimage, 0);
    rb_define_method(cImage, "is_pseudo_class?", mgk_image_is_pseudoclass, 0);

    /*
     * methods for decorating an image
     * are implemented in lib/magick.rb
     */
    rb_define_private_method(cImage, "do_border", mgk_image_border, 2);
    rb_define_private_method(cImage, "do_frame", mgk_image_frame, 6);
    rb_define_private_method(cImage, "do_raise", mgk_image_raise, 2);

    rb_define_method(cImage, "add_noise", mgk_image_add_noise, 1);
    rb_define_method(cImage, "blur", mgk_image_blur, 2);
    rb_define_method(cImage, "colorize", mgk_image_colorize, 2);
    rb_define_method(cImage, "convolve", mgk_image_convolve, 1);
    rb_define_method(cImage, "despeckle", mgk_image_despeckle, 0);
    rb_define_method(cImage, "edge", mgk_image_edge, 1);
    rb_define_method(cImage, "emboss", mgk_image_emboss, 2);
    rb_define_method(cImage, "enhance", mgk_image_enhance, 0);
    rb_define_method(cImage, "gaussian_blur", mgk_image_gaussian_blur, 2);
    rb_define_method(cImage, "implode", mgk_image_implode, 1);
    rb_define_method(cImage, "median_filter", mgk_image_median_filter, 1);
    rb_define_method(cImage, "morph", mgk_image_morph, 1);
    rb_define_method(cImage, "motion_blur", mgk_image_motion_blur, 3);
    rb_define_method(cImage, "oilpaint", mgk_image_oilpaint, 1);
    rb_define_method(cImage, "plasma", mgk_image_plasma, 3);
    rb_define_method(cImage, "reduce_noise", mgk_image_reduce_noise, 1);
    rb_define_method(cImage, "shade", mgk_image_shade, 3);
    rb_define_method(cImage, "sharpen", mgk_image_sharpen, 2);
    rb_define_method(cImage, "solarize", mgk_image_solarize, 1);
    rb_define_method(cImage, "spread", mgk_image_spread, 1);
    rb_define_method(cImage, "stegano", mgk_image_stegano, 1);
    rb_define_method(cImage, "stereo", mgk_image_stereo, 1);
    rb_define_method(cImage, "swirl", mgk_image_swirl, 1);
    rb_define_method(cImage, "threshold", mgk_image_threshold, 1);
    rb_define_method(cImage, "unsharp_mask", mgk_image_unsharp_mask, 4);
    rb_define_method(cImage, "wave", mgk_image_wave, 2);

    rb_define_method(cImage, "contrast", mgk_image_contrast, 1);
    rb_define_method(cImage, "equalize", mgk_image_equalize, 0);
    rb_define_method(cImage, "gamma", mgk_image_gamma, 1);
    rb_define_method(cImage, "modulate", mgk_image_modulate, 1);
    rb_define_method(cImage, "negate", mgk_image_negate, 1);
    rb_define_method(cImage, "normalize", mgk_image_normalize, 0);

    rb_define_method(cImage, "map", mgk_image_map, 2);
    rb_define_method(cImage, "map_images", mgk_image_map_images, 2);
#if MagickLibVersion >= 0x0532
    rb_define_method(cImage, "ordered_dither", mgk_image_ordered_dither, 0);
#endif

    /*
     *  method for quantizing image(s) are
     *  implemented in lib/magick.rb
     */
    rb_define_private_method(cImage, "do_quantize_image", mgk_image_quantize, 1);
    rb_define_private_method(cImage, "do_quantize_images", mgk_image_quantize_images, 1);

    /*
     * method 'montage' is implemented
     * in lib/magick.rb
     */
    rb_define_private_method(cImage, "do_montage", mgk_image_do_montage, 1);

    rb_define_method(cImage, "segment", mgk_image_segment, 4);

    rb_define_method(cImage, "rotate", mgk_image_rotate, 1);
    rb_define_method(cImage, "shear", mgk_image_shear, 2);

    rb_define_protected_method(cImage, "clone", mgk_image_clone, 0);
    rb_define_method(cImage, "channel", mgk_image_channel, 1);
    rb_define_method(cImage, "composite", mgk_image_composite, 4);
    rb_define_method(cImage, "cycle_colormap", mgk_image_cycle_colormap, 1);
    rb_define_method(cImage, "describe", mgk_image_describe, -1);
    rb_define_method(cImage, "animate", mgk_image_animate, 0);
    rb_define_method(cImage, "append", mgk_image_append, -1);
    rb_define_method(cImage, "display", mgk_image_display, 0);
    rb_define_method(cImage, "bounding_box", mgk_image_get_image_boundingbox, 0);
    rb_define_method(cImage, "depth", mgk_image_get_image_depth, 0);
    rb_define_method(cImage, "image_type", mgk_image_get_image_type, 0);
    rb_define_method(cImage, "scenes", mgk_image_get_number_scenes, 0);
    rb_define_method(cImage, "mogrify", mgk_image_mogrify, -1);
    rb_define_method(cImage, "mosaic", mgk_image_mosaic, 0);
    rb_define_method(cImage, "rgb_transform", mgk_image_rgb_transform, 1);
    rb_define_method(cImage, "texture", mgk_image_texture, 1);

#if 0
    rb_define_method(cImage, "to_blob", mgk_image_to_blob, 1);
#endif

    rb_define_method(cImage, "to_str", mgk_image_to_str, 0);

    /* in magick_private.c */
    rb_define_private_method(cImage, "do_get_font_metrics", mgk_do_get_font_metrics,1);
#if 0
    /* Magick::Blob class */
    cBlob = rb_define_class_under(mMagick, "Blob", rb_cObject);
    rb_define_method(cBlob, "length", mgk_blob_get_length, 0);
    rb_define_method(cBlob, "quantum=", mgk_blob_set_quantum, 1);
    rb_define_alias(cBlob, "size", "length");
#endif

    /* module functions */
    rb_define_module_function(mMagick, "query_color", mgk_m_query_color, 1);
#if MagickLibVersion >= 0x0534
    rb_define_module_function(mMagick, "query_name", mgk_m_query_name, 3);
#endif /* MagickLibVersion */
    rb_define_module_function(mMagick, "parse_geometry", mgk_m_parse_geometry, 1);
    rb_define_module_function(mMagick, "parse_image_geometry", mgk_m_parse_image_geometry, 1);
    eImageError = rb_define_class_under(mMagick, "ImageError", rb_eStandardError);
    /* Private classes initialization */
    Init_MagickPrivate();
}
