/* rsrce -- a Macintosh resource fork editor
 * Copyright (C) 2004  Jeremie Koenig
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include "config.h"
#include "resource.h"
#include "translate.h"
#include "command.h"

struct res_fork *cmd_rf;
struct resource *cmd_res;

int cmd_selectresource(const char *spec)
{
	restype_t type;
	int id;
	char *colon;

	if(!spec) {
		fprintf(stderr, "Please specify a resource to act on.\n");
		return 1;
	}

	colon = strrchr(spec, ':');
	if(!colon || colon > spec+4) {
		fprintf(stderr, "Incorrect resource specification\n");
		return 1;
	}
	memset(type, ' ', sizeof(type));
	memcpy(type, spec, colon - spec);
	id = atoi(colon + 1);
	cmd_res = res_lookup(cmd_rf, type, id);
	if(!cmd_res) {
		fprintf(stderr, "No such resource.\n");
		return 1;
	}

	return 0;
}


int cmd_read(char **argv)
{
	struct res_fork *rf;
	FILE *f;
	
	if(!argv[1]) {
		fprintf(stderr, "Read which file ?\n");
		return -1;
	}

	f = fopen(argv[1], "r");
	if(!f) {
		perror(argv[1]);
		return -1;
	}

	rf = res_read(f);
	if(!rf) {
		fprintf(stderr, "Error while reading %s\n", argv[1]);
		return -1;
	}

	res_delfork(cmd_rf);
	cmd_rf = rf;
	return 0;
}

int cmd_write(char **argv)
{
	FILE *f;
	
	if(!argv[1]) {
		fprintf(stderr, "Write to which file ?\n");
		return -1;
	}

	f = fopen(argv[1], "w");
	if(!f) {
		perror(argv[1]);
		return -1;
	}

	if(res_write(cmd_rf, f)) {
		fprintf(stderr, "Write error\n");
		return -1;
	}

	return 0;
}

int cmd_ls(char **argv)
{
	res_ls(stdout, cmd_rf);
	return 0;
}

int cmd_hexdump(char **argv)
{
	int i, j;
	unsigned char *data;
	int len;

	if(cmd_selectresource(argv[1]))
		return 1;

	res_getdata(cmd_res, (void **) &data, &len);
	
	for(i=0 ; i < len ; i += 16) {
		printf("%8x ", i);
		for(j=0 ; j < 16 ; j++) {
			if(i+j < len)
				printf(" %02x", data[i+j]);
			else
				printf("   ");
		}
		printf("  |");
		for(j=0 ; j < 16 && i+j < len ; j++)
			printf("%c", isprint(data[i+j]) ? data[i+j] : '.');
		printf("|\n");
	}

	return 0;
}


int cmd_export(char **argv)
{
	struct translator *tr;
	FILE *f;
	char *c;
	int what, r;
	
	what = (strcmp(argv[0], "export") == 0);
	
	if(!argv[2]) {
		fprintf(stderr, "%s <resource> <file> [<type>]\n", argv[0]);
		return -1;
	}
	if(!argv[3]) {
		c = strrchr(argv[2], '.');
		argv[3] = c ? c+1 : NULL;
	}

	if(cmd_selectresource(argv[1]))
		return -1;
	
	tr = tr_lookup(cmd_res, argv[3]);
	if(!tr)
		return -1;
	
	f = (strcmp(argv[2], "-") == 0) ? (what ? stdout : stdin)
					: fopen(argv[2], what ? "w" : "r");
	if(!f) {
		perror(argv[2]);
		return -1;
	}

	r = (what ? tr_export : tr_import)(tr, cmd_res, f);
	if(strcmp(argv[2], "-") != 0)
		fclose(f);

	return r;
}

/* Export the resource data to a temporary file and return its name */
static char *cmd_edit_export(struct translator *tr)
{
	char filename[] = "/tmp/rsrce.XXXXXX", *nfname;
	const char *ext;
	int fd;
	FILE *f;
	
	fd  = mkstemp(filename);
	ext = tr_ext(tr);
	nfname = malloc(strlen(filename) + strlen(ext) + 2);
	sprintf(nfname, "%s.%s", filename, ext);
	if(rename(filename, nfname) != 0 || (f = fdopen(fd, "w")) == NULL) {
		perror(nfname);
		close(fd);
		free(nfname);
		return NULL;
	}
	if(tr_export(tr, cmd_res, f) != 0) {
		fprintf(stderr,"Couldn't export resource data to %s\n",nfname);
		fclose(f);
		free(nfname);
		return NULL;
	}
	fclose(f);

	return nfname;
}

/* Invoke the external editor */
static int cmd_edit_edit(const char *fname)
{
	char *cmd;
	int ret;

	cmd = malloc(strlen(CFG_EDITOR) + strlen(fname) + 2);
	sprintf(cmd, "%s %s", CFG_EDITOR, fname);
	ret = system(cmd);
	free(cmd);

	return ret;
}

/* Re-import the resource data */
static int cmd_edit_import(struct translator *tr, const char *filename)
{
	FILE *f;
	int ret;

	f = fopen(filename, "r");
	if(!f) {
		perror(filename);
		return -1;
	}

	ret = tr_import(tr, cmd_res, f);
	fclose(f);
	unlink(filename);

	return ret;
}

int cmd_edit(char **argv)
{
	struct translator *tr;
	int ret;
	char *fname;
	
	if(cmd_selectresource(argv[1]) != 0)
		return 1;

	tr = tr_lookup(cmd_res, argv[2]);
	fname = cmd_edit_export(tr);
	if(!fname)
		return 1;
	ret = (cmd_edit_edit(fname) == 0 && cmd_edit_import(tr, fname) == 0);
	free(fname);

	return ret ? 0 : 1;
}

int cmd_help(char **argv);

int cmd_exit(char **argv)
{
	exit(0);
}

struct command {
	const char *cmd;
	int (*f)(char **argv);
	const char *help;
} cmd_table[] = {
	{"read", 	cmd_read,	"Read the resource fork from a file"},
	{"write",	cmd_write,	"Write the resource fork to a file"},
	{"ls",		cmd_ls,		"List resources"},
	{"hexdump",	cmd_hexdump,	"Show an hexdump of the resource data"},
	{"import",	cmd_export,	"Import resource data from a file"},
	{"export",	cmd_export,	"Export resource data to a file"},
	{"edit",	cmd_edit,	"Edit resource data"},
	{"help",	cmd_help,	"This short help listing"},
	{"exit",	cmd_exit,	"Exit (don't forget to write first!)"},
	{NULL,		NULL,		NULL}
};

int cmd_help(char **argv)
{
	int i;

	for(i=0 ; cmd_table[i].cmd ; i++) {
		printf("%10s -- %s\n", cmd_table[i].cmd, cmd_table[i].help);
	}

	return 0;
}


#define MAXARGC 7

int cmd_exec(char *cmd)
{
	char *argv[MAXARGC+1];
	int argc, i;

	memset(argv, 0, sizeof(argv));
	argc = 0;
	while((argv[argc++] = strsep(&cmd, " ")) && argc < MAXARGC+1);
	
	if(argv[MAXARGC]) {
		fprintf(stderr, "Too many arguments\n");
		return -1;
	}

	for(i=0 ; cmd_table[i].cmd ; i++) {
		if(strcmp(cmd_table[i].cmd, argv[0]) == 0)
			return cmd_table[i].f(argv);
	}

	fprintf(stderr, "%s: no such command\n", argv[0]);
	return -1;
}

void cmd_init(FILE *f)
{
	cmd_rf = f ? res_read(f) : res_newfork();
	if(!cmd_rf) {
		exit(1);
	}
}
