
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>


#define MAX_CHUNK_SIZE 128


static void
rarray_append_chunk (RArray *array)
{
    RArrayChunkEntry *entry;

    /* Do a quick check to see if the curentry is indeed the last chunk.
     * If it's not, we may be doing push/pop stuff accross a chunk
     * boundary */

    if (array->curentry != rlist_last (&array->chunks)) {

        /* They are not equal, so we move the curentry
         * to the next one as it's already allocated */
        array->curentry = rlist_next (array->curentry);
        array->curindex = -1;

        return;
    }

    /* Otherwise, we are just growing normally.. */
    entry = rchunk_alloc (sizeof (RArrayChunkEntry));

    if (array->element_size * array->elements_per_chunk <= MAX_CHUNK_SIZE) {
        entry->chunk = rchunk_alloc (array->element_size * array->elements_per_chunk);
    } else {
        entry->chunk = rmem_alloc (array->element_size * array->elements_per_chunk);
    }

    rlist_append (&array->chunks, entry);
    array->curentry = entry;
    array->curindex = -1;
}

RArray *
rarray_new (unsigned int element_size, unsigned int elements_per_chunk)
{
    RArray *array;

    array = rchunk_alloc (sizeof (RArray));
    array->elements_per_chunk = elements_per_chunk;
    array->element_size = element_size;
    array->curentry = NULL;
    array->curindex = -1;

    RLIST_INIT (&array->chunks);

    rarray_append_chunk (array);

    return (array);
}

void *
rarray_append (RArray *array)
{
    unsigned int index;
    void *mem;


    if ((array->curindex + 1) >= (signed int) array->elements_per_chunk) {
        rarray_append_chunk (array);
    }

    array->curindex++;

    index = array->curindex * array->element_size;
    mem = &array->curentry->chunk[index];

    /* Act like the memory is new0'ed memory. */
    memset (mem, 0, array->element_size);

    return (mem);
}

void *
rarray_pop (RArray *array)
{
    void *mem;

    mem = &array->curentry->chunk[array->curindex * array->element_size];

    if (array->curindex <= 0) {
        /* If this is the first chunk, we have a bit of special
         * logic to do */
        if (array->curentry == rlist_first (&array->chunks)) {
            if (array->curindex == 0) {
                array->curindex = -1;
                return (array->curentry->chunk);
            } else {
                return (NULL);
            }
        }

        array->curentry = rlist_prev (array->curentry);
        array->curindex = array->elements_per_chunk - 1;
    } else {
        array->curindex--;
    }

    return (mem);
}

void *
rarray_last (RArray *array)
{
    void *mem;

    mem = &array->curentry->chunk[array->curindex * array->element_size];

    return (mem);
}

void *
rarray_nth (RArray *array, unsigned int nth)
{
    unsigned int chunk_num;
    unsigned int chunk_index;
    RArrayChunkEntry *entry;

    if (rarray_len (array) <= nth)
      return NULL;

    chunk_num = nth / array->elements_per_chunk;

    entry = rlist_nth (&array->chunks, chunk_num);
    if (!entry)
        return (NULL);

    chunk_index = nth % array->elements_per_chunk;

    return (&entry->chunk[chunk_index * array->element_size]);
}

unsigned int
rarray_len (RArray *array)
{
    unsigned int length;
    unsigned int num_chunks = 0;
    RArrayChunkEntry *entry;

    RLIST_FOREACH (&array->chunks, entry) {
        if (entry == array->curentry)
            goto count_done;
        num_chunks++;
    } RFOREACH_CLOSE;

count_done:

    length = num_chunks * array->elements_per_chunk;
    length += array->curindex + 1;

    return (length);
}

void
rarray_set_len (RArray *array, unsigned int length)
{
   unsigned int len;

   len = rarray_len (array);

   if (length < len) {
       /* Shrink. */
       while (length < len) {
          rarray_pop (array);
          len--;
       }
   } else if (len < length) {
       /* Grow. */
       while (len < length) {
          rarray_append (array);
          length--;
       }
   }
   /* Else they are the same so ignore. */
}


void
rarray_free (RArray *array)
{
    RArrayChunkEntry *entry;

    RLIST_FOREACH (&array->chunks, entry) {

        rlist_remove (&array->chunks, entry);

        if (array->element_size * array->elements_per_chunk <= MAX_CHUNK_SIZE) {
            rchunk_free (entry->chunk, array->element_size * array->elements_per_chunk);
        } else {
            rmem_free (entry->chunk);
        }

        rchunk_free (entry, sizeof (RArrayChunkEntry));
    } RFOREACH_CLOSE;

    rchunk_free (array, sizeof (RArray));
}




