/*
 * robmanage.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file robmanage.cpp
  * Implements the sim managers.
  */

#include "robmanage.h"
#include "robbase.h"
#include "robsupervis.h"

#include <rtstring.h>
#include <rtcollect.h>
#include <rtsystem.h>

using namespace lrt;

namespace rt {

//////////////// TourInitInfo ////////////////
TourInitInfo::TourInitInfo(int numParticipants, int numSimulations)
	: numParticipants(numParticipants), numSimulations(numSimulations)
{
}

//////////////// TourResult //////////////////

int TourResult::compareByPoints(const TourResult& t1, const TourResult& t2)
{
	return int(t2.points - t1.points);
}

TourResult::TourResult() : fileName(), wins(0), looses(0), ties(0),
	rank(0), points(0)
{}
TourResult::TourResult(const StringMap<Program::HeaderField>& headers, const String& fileName) :
	headers(headers), fileName(fileName), wins(0), looses(0), ties(0),
	rank(0), points(0)
{}

///////////////// SimOptions /////////////////
SimOptions::SimOptions(const ErrorHandler* frontend) :
	numRepeats(10), files(0), supervisors(0), execSupervisors(0),
	tourDisps(0), errHandler(frontend), glob(frontend)
{}

SimOptions::~SimOptions() 
{
	deleteAll(supervisors);
	deleteAll(execSupervisors);
	deleteAll(tourDisps);
}


////////////// SimManager ////////////////////
void SimManager::setup(SimOptions* const options)
{
	if(options == 0) options->errHandler->handleSystemError(36, 
		"options may not be 0 in SimManager::setup!");
	this->options = options;
	createPrograms();
	// now we can ask about tour information
	TourInitInfo initInfo = getTourInitInfo();
	// ...and send it to the displayers...
	for(int d = 0; d < options->tourDisps.length(); d++)
	  options->tourDisps[d]->init(initInfo);
	
	createSupervisors();
}

SimManager::~SimManager()
{
	deleteAll(managerSupervisors);
	deleteAll(programs);
}

void SimManager::createSupervisors()
{
	managerSupervisors += new StdBotCreator();
	managerSupervisors += new StdVictorySupervisor();
}

void SimManager::createPrograms()
{
	if(Program::exampleLoaders.length() == 0) options->errHandler->handleSystemError(37, 
		"cannot load bots: no example loaders!");
	
	for(int i = 0; i < options->files.length(); i++)
	{
		Program* newProgram = Program::create(options->files[i], options->glob, programs.length());
		if(newProgram == 0)
		{
			String formats;
			for(int pl = 0; pl < Program::exampleLoaders.length(); pl++)
				formats += "\n  " + Program::exampleLoaders[pl]->getFormatName();
			options->errHandler->handleLoadError(
				options->files[i], "unknown type of file. Known file formats:" + formats);
		}
		else {
			programs += newProgram;
			results += TourResult(newProgram->headers, options->files[i]);
		}
	}
}

TourStatusType SimManager::doAllStdSims(const Array<int>& programNums)
{
	TourStatusType tourStatus = tourContinue; 

	for(int i = 0; i < options->numRepeats; i++) {
		tourStatus = doStdSim(programNums);
		if(tourStatus == tourAbort)
			break;
	}
	return tourStatus; 
}

TourStatusType SimManager::doStdSim(const Array<int>& programNums)
{
	/*if curSim != 0*/ /*delete curSim; */
	curSim = new Simulation(options->glob, options->errHandler);
	// add handlers
	for(int s = 0; s < options->supervisors.length(); s++)
		curSim->addSupervisor(options->supervisors[s]);
	for(int m = 0; m < managerSupervisors.length(); m++)
		curSim->addSupervisor(managerSupervisors[m]);
	for(int e = 0; e < options->execSupervisors.length(); e++)
		curSim->addExecSupervisor(options->execSupervisors[e]);
	// add programs
	for(int p = 0; p < programNums.length(); p++)
		curSim->addProgram(programs[programNums[p]]);

	// run sim
	Program* winner = curSim->run();

	// do std scoring
	if(winner == 0) { // handle tie : everyone get +1 ties, +1 points
		for(int q = 0; q < programNums.length(); q++)
		{
			results[programNums[q]].ties += 1;
			results[programNums[q]].points += 1;
		}
	}
	else { // handle win: winner get +3 points and +1 wins, others +1 loose
		for(int q = 0; q < programNums.length(); q++)
		{
			if(programs[programNums[q]] == winner) { // here comes the winner
				results[programNums[q]].wins += 1;
				results[programNums[q]].points += 3;
			} 
			else {
				results[programNums[q]].looses += 1;
			}
		}
	}

	// call TourDisplayers
	TourStatusType tourStatus = tourContinue; 

	for(int d = 0; d < options->tourDisps.length(); d++)
	{
		tourStatus = options->tourDisps[d]->update(results);
		if(tourStatus == tourAbort)
			break;
	}

	// cleanup sim
	delete curSim;
	curSim = 0;
	return tourStatus; 
}

SimManager::SimManager() : options(0), curSim(0), managerSupervisors(0), programs(0), results(0)
{
}

//////////////// ChartsManager ////////////////////
ChartsManager::ChartsManager(bool playAll) : playAll(playAll)
{
}

ChartsManager::~ChartsManager() 
{}

void ChartsManager::run()
{
	// inform the tourDisplayers about the tournament
	for(int di = 0; di < options->tourDisps.length(); di++)
		options->tourDisps[di]->update(results);
	

	Array<int> programNums(2);
	
	int end = (playAll ? programs.length() : 1);

	TourStatusType tourStatus = tourContinue;

	for(int n = 0; n < options->numRepeats; n++)
	{
		for(int p = 0; p < end; p++) {
			programNums[0] = p;
			for(int q = p + 1; q < programs.length(); q++) {
				programNums[1] = q;
				tourStatus = doStdSim(programNums);
				if(tourStatus == tourAbort)
					goto tourEnd;
			}
		}
	}

tourEnd:

	for(int d = 0; d < options->tourDisps.length(); d++)
		options->tourDisps[d]->exit(tourStatus, results);
}

TourInitInfo ChartsManager::getTourInitInfo()
{
	int nPlayers = options->files.length();
	int nSims;
	if(playAll)
		nSims = nPlayers * (nPlayers-1) * options->numRepeats / 2;
	else
		nSims = (nPlayers-1) * options->numRepeats;

	return TourInitInfo(nPlayers, nSims);
}

//////////////// AllInOneManager //////////////////

AllInOneManager::AllInOneManager() 
{}

AllInOneManager::~AllInOneManager()
{}

void AllInOneManager::run()
{
	// inform the tourDisplayers about the tournament
	for(int di = 0; di < options->tourDisps.length(); di++)
		options->tourDisps[di]->update(results);
	
	// create array including all programs
	Array<int> programNums(programs.length());
	for(int i = 0; i < programs.length(); i++)
		programNums[i] = i;
	
	TourStatusType tourStatus = tourContinue; 
	tourStatus = doAllStdSims(programNums);

	for(int d = 0; d < options->tourDisps.length(); d++)
		options->tourDisps[d]->exit(tourStatus, results);
}

TourInitInfo AllInOneManager::getTourInitInfo()
{
	int nPlayers = options->files.length();
	int nSims = options->numRepeats;
	return TourInitInfo(nPlayers, nSims);
}


} // namespace

