/*
 * robbase.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file robbase.cpp
  * Implements the core RoboTour classes. */

#include "robbase.h"
#include "robvars.h"
#include "robstrings.h"
#include "robglob.h"

#include <rtsystem.h>
#include <rtstring.h>
#include <rtiterator.h>
#include <rtlist.h>
#include <rtmap.h>

using namespace lrt;

namespace rt {

////////// Simulation

Simulation::Simulation(const Globals& glob, const ErrorHandler* errHandler) : 
    errHandler(errHandler), glob(glob),
	/*vars(simVarLength),*/ fieldWidth(glob.fieldsX), 
	fieldHeight(glob.fieldsY), field(fieldWidth * fieldHeight), bots(), curCycle(0),
	programs(0), supervisors(), started(false)
	
{
	vars[simFields] = glob.fields;
	vars[simPub] = 0;
	vars[simMaxVars] = glob.maxVars;
	vars[simMaxTasks] = (glob.enableMultitasking ? glob.maxTasks : 1);
	vars[simMaxGeneration] = glob.maxGeneration;
	vars[simMaxMybots] = glob.maxMybots;
	vars[simTimeout] = (rint)(glob.timeout / 1000);
	vars[simElimTrigger] = (glob.enableElim ? (rint)glob.elim : (rint)-1);
	vars[simMaxLifetime] = (glob.enableLifetime ? (rint)(glob.maxLifetime / 1000) : (rint)-1);

	for(int f = 0; f < field.length(); f++)
		field[f] = 0;

    
}

Simulation::~Simulation()
{
	deleteAll(bots.begin());
}

Program* Simulation::run()
{
	// inform the supervisors that the sim is about to start
	for(List<SimSupervisor*>::Node* n = supervisors.first(); n; n = n->getNext())
		n->accessElement()->initSim(this);
	for(List<ExecSupervisor*>::Node* ne = execSupervisors.first(); ne; ne = ne->getNext())
		ne->accessElement()->initSim(this);
	
	started = true;
	SimSupervisor::GameState state;

	// do the sim
	for( curCycle = 0; curCycle < glob.timeout; curCycle++)
	{
		vars[simTime] = (rint)((curCycle + 1) / 1000);

		rint allBots = getAllNumBots();

		// setup $OTHERBOTS ($MYBOTS is handled by Bot() and ~Bot())
		for(int p = 0; p < programs.length(); p++)
		{
			programs[p]->vars[progOtherbots] = allBots - programs[p]->vars[progMybots];
		}

		processBots(); // do the cycle

		for(List<SimSupervisor*>::Node* n2 = supervisors.first(); n2; n2 = n2->getNext())
		{
			SimSupervisor* super = n2->accessElement();
			if(!(curCycle % super->getStepCycles())) {
				state = super->exec(this);
				if(state.event != gameNothing)
					goto breakSim;
			}
		}
	}
	// normally exited for loop => tie
	state.event = gameTie;

breakSim:
	// inform the supervisors that the sim is about to exit
	for(List<SimSupervisor*>::Node* n3 = supervisors.first(); n3; n3 = n3->getNext())
		n3->accessElement()->exitSim(this, state);
	for(List<ExecSupervisor*>::Node* ne3 = execSupervisors.first(); ne3; ne3 = ne3->getNext())
		ne3->accessElement()->exitSim(this);

	if(state.event == gameFinished) // return the winner
		return programs[state.affectedProgram];
	else return 0;
}

rint Simulation::getAllNumBots()
{
	return bots.length();
/*	rint ret = 0;
	for(int p = 0; p < programs.length(); p++)
		ret += programs[p]->vars[progMybots];
	return ret;
*/
}

void Simulation::addProgram(Program* program)
{
	if(started) return; // can't add programs to started simulations

	programs += program;
	program->init(programs.length() - 1, this);
}

int Simulation::getNumPrograms()
{
	return programs.length();
}

Program* Simulation::getProgram(int index)
{
	return programs[index];
}

void Simulation::addSupervisor(SimSupervisor* supervis)
{
	int supPos = supervis->getPreferredPosition();
	for(Iterator<SimSupervisor*> iter = supervisors.begin(); iter.hasElement(); ++iter)
		if((*iter)->getPreferredPosition() > supPos) {
			supervisors.insertBefore(iter, supervis);
			return;
		}
	supervisors.append(supervis);
}

void Simulation::addExecSupervisor(ExecSupervisor* es)
{
	execSupervisors.append(es);
}

//#ifdef __MSVC__ 
#define deleteBotFromList(l,n) deleteNode(l,n)
//#else
//#define deleteBotFromList(l,n) deleteNode<Bot>(l,n)
//#endif

void Simulation::processBots()
{
	// process bots
	for(List<Bot*>::Node* n = bots.first(); n; )
	{
		Bot* current = n->accessElement();
		ExecReturnType ret;
		if((ret = current->exec()) != execOK)
		{
		  if(errHandler != 0) errHandler->handleBotError(current, ret);
		  deleteBotFromList(bots, n);
		}
		else
		  n = n->getNext();
	}
	// process age
	if(glob.enableLifetime) 
	{
		for(List<Bot*>::Node* n2 = bots.first(); n2; )
		{
			Bot* current = n2->accessElement();
			if((curCycle - current->creationCycle) >= glob.maxLifetime)
			{
			  if(errHandler != 0) errHandler->handleBotError(current, failAge);
			  deleteBotFromList(bots, n2);
			}
			else
			  n2 = n2->getNext();
		}
	}
}

void Simulation::getPosBefore(rint& x, rint& y, rint dir)
{
	switch(dir) 
	{
		case dirRight: x += 1; break;
		case dirLeft:  x -= 1; x += fieldWidth;   break;
		case dirUp:    y -= 1; y += fieldHeight;  break;
		case dirDown:  y += 1; break;
	}

	x %= fieldWidth;
	y %= fieldHeight;
}


///////////// Program

Program::HeaderField::HeaderField(const String& name, const String& value, HeaderType type) :
	name(name), value(value), type(type)
{}

Program::HeaderField::HeaderField() :
	name(), value(), type(headerPublished)	
{}

Vector<ProgramLoader*> Program::exampleLoaders(0);

ProgramLoader* Program::getLoader(const String& file)
{
	for(int l = 0; l < exampleLoaders.length(); l++)
		if(exampleLoaders[l]->canLoad(file)) {
			return exampleLoaders[l]; 
		}

	return 0; 
}

Program* Program::create(const String& file, const Globals& glob, int globalNum)
{
	Program* ret = 0; 
	ProgramLoader* loader = getLoader(file);
	if(loader)
		ret = new Program(loader->create(file, glob));
	if(ret)
		ret->globalNum = globalNum;
	return ret;
}


String Program::print()
{
	String ret;
	for(StringMap<HeaderField>::Iterator iter = headers.begin(); iter.hasElement(); ++iter)
	{
		HeaderField field = iter.get().getValue();
		if(field.type == headerPublished) ret += "Published ";
		else if(field.type == headerSecret) ret += "Secret ";
		else ret += "Internal ";
		ret += field.name;
		ret += ' ';
		ret += field.value;
		ret += '\n';
	}
	ret += '\n';
	for(int b = 0; b < banks.length(); b++)
		ret += banks[b]->print();
	return ret;
}

const Vector<Bank*>& Program::getBanks()
{
	if(banks.length() == 0) {
		Vector<Bank*>* bp = loader->getBanks();
		if(bp != 0) banks = *bp;
	}
	return banks;
}

ProgramLoader* Program::getLoader()
{
	return loader;
}

Program::Program(ProgramLoader* loader) : sim(0), /*vars(progVarLength),*/
	headers(false), programNum(-1), globalNum(-1), emptyBank(0), loader(loader), banks(0)
{
	loader->owner = this;
	headers["Name"] = HeaderField("name", "Unnamed Bot", headerPublished);
	headers["Author"] = HeaderField("author", "Unknown Author", headerPublished);
	headers["Country"] = HeaderField("country", "Unknown Country", headerPublished);
	StringMap<HeaderField>* hp = loader->getHeaders();
	if(hp != 0) headers.putAll(*hp);
	emptyBank = new Bank(this, "Empty Bank");
}

void Program::init(rint num, Simulation* sim)
{
	this->sim = sim;
	this->programNum = num;
	this->vars[progMybots] = 0;
	this->vars[progOtherbots] = 0;
	this->vars[progId] = Math::rand(sim->glob.minId, sim->glob.maxId);
}

Program::~Program()
{
    delete emptyBank;
	delete loader;
}

////////////////// Bot //////////

Bot::Bot(rint instrSet, rint numBanks, rint mobile,
		rint posX, rint posY, rint dir, rint generation, 
		Program* owner) :
	owner(owner), /*vars(botVarLength), stdVars(owner->sim->glob.maxVars + 1),*/
	banks(numBanks), tasks(0), curTask(0), creationCycle(owner->sim->curCycle),
	deactCycle(0)
{
	stdVars = new rint[owner->sim->glob.maxVars + 1];
	owner->vars[progMybots]++;

	vars[botInstrSet]   = instrSet;
	vars[botNumBanks]   = numBanks;
	vars[botMobile]     = mobile;
	vars[botPosX]       = posX;
	vars[botPosY]       = posY;
	vars[botGeneration] = generation;

	vars[botActive]     = 0;
	vars[botNumTasks]   = 0;

	for(int v = 0; v <= owner->sim->glob.maxVars; v++)
		stdVars[v] = 0;

	owner->sim->field[posX + posY * owner->sim->fieldWidth] = this;
	owner->sim->bots.append(this);

	for(int b = 0; b < numBanks; b++)
		banks[b] = owner->emptyBank;

	new Task(this, 0, dir);
}

Bot::~Bot()
{
	delete[] stdVars;
	for(int t = tasks.length() - 1; t >= 0; t--) 
		delete tasks[t]; // can't use deleteAll() here as it would remove() tasks twice
	if(owner) {
		owner->sim->field[ vars[botPosX] + vars[botPosY] * owner->sim->fieldWidth ] = 0;
		owner->vars[progMybots]--;
	}
}

ExecReturnType Bot::exec()
{
	if(vars[botActive] <= 0 ) return execOK;

	if(tasks.length() == 0)
		return failUnemploy;
	// the for loop avoids getting us stuck if all tasks sleep
	for(int nt = 0; nt < tasks.length(); nt++) 
	{
		if(!tasks[curTask]->seized) curTask++; // switch tasks if not seized
		curTask %= tasks.length();

		ExecReturnType ret = tasks[curTask]->exec();
		if(ret <= execOK)
		{
			if((ret == failQuit) || // quit when several tasks present
				((ret == failData) && (tasks.length() > 1))) 
			{ // special handling of data hunger in multitasking: just current task dies
				delete tasks[curTask];
				return execOK;
			}
			return ret;
		}
	}
	// all tasks sleep
	return execOK;
}

////////////// Task //////////
Task::Task(Bot* bot, Task* creator, rint dir) : 
	bot(bot), sim(bot->owner->sim), /*vars(taskVarLength),*/ curInstr(0), seized(false)
{
	// enter myself into bot->tasks, using multitasking rules
	bool canAdd = false;

	// we can add another task: no problem!
	if(bot->tasks.length() < sim->glob.maxTasks)
	{ 
		bot->tasks += this;
		canAdd = true;
	}
	else { // bot->tasks is full, so check if we can delete one
	  if(sim->glob.maxTasks > 1) // if only one task is allowed, no task can be deleted
	  {
		for(int t = 0; t < bot->tasks.length(); t++)
		{
			if(bot->tasks[t] != creator) // now we've got the oldest task which is not our creator
			{ // => remove it, append us instead.
				delete bot->tasks[t];
				bot->tasks += this;
				canAdd = true;
				break;
			}
		}
	  }
	}
	if(!canAdd) // couldn't add myself 
	{
		delete this;
		return;
	}

	bot->vars[botNumTasks]++;
	vars[taskCurBank]      = 0;
	vars[taskCurInstr]     = -1;
	vars[taskDir]          = dir;
	vars[taskNumCycles]    = 0;
	vars[taskCyclesDone]   = 0;
	instrStarted		   = sim->curCycle;
	vars[taskTimingMode]   = timingNumExec;
}

Task::~Task()
{
	// find myself in bot->tasks and remove() me
	for(int t = 0; t < bot->tasks.length(); t++) {
		if(bot->tasks[t] == this)
		{
			bot->tasks.remove(t);
			bot->vars[botNumTasks]--;
			if(bot->vars[botNumTasks] >= 1) {
				bot->curTask += bot->vars[botNumTasks] - 1; // go one task backwards
				bot->curTask %= bot->vars[botNumTasks];		// (process the task at my prior place again)
			}
			break;
		}
	}
}

ExecReturnType Task::exec()
{
  vars[taskCyclesDone]++;

  if(((vars[taskTimingMode] == timingNumExec) && (vars[taskCyclesDone] >= vars[taskNumCycles]))
	  || ((vars[taskTimingMode] == timingNumWaited) && (sim->curCycle >= instrStarted + vars[taskNumCycles])))
  {
      rint ret = execOK, ret2 = execOK;

      if(curInstr) // there is a current instr which can be executed
      {
          // re-get amount of cycles
		  rint cycles = curInstr->getNumCycles(this);

          if(cycles < 0) // Bot killed in getNumCycles?
            return ExecReturnType(cycles);

		  if(cycles > sim->glob.maxInstrDur) // Instruction takes too long
			return failInstrDur;

          // instruction takes longer now then when it was begun (FAT!)
		  if((!sim->glob.enableFAT) && (vars[taskNumCycles] < cycles))
          {
              vars[taskNumCycles] = cycles;
			  // wait longer
			  return ExecReturnType(vars[taskTimingMode]);
          }

		  // is the instruction still present in the bot? (NutCracker!) 
		  if((!sim->glob.enableNutcracker) // no NutCracker enabled => always execute now
			  || ((bot->banks[vars[taskCurBank]]->instr.length() > vars[taskCurInstr]) // NutCracker enabled => check bank present
			      && (bot->banks[vars[taskCurBank]]->instr[vars[taskCurInstr]] == curInstr))) // and instruction not changed
		  {
			  // inform the execSupervisors
			  if(sim->execSupervisors.first()) // are there any? 
			  {
				  for(List<ExecSupervisor*>::Node* n = sim->execSupervisors.first(); n; n = n->getNext())
					  n->accessElement()->onExec(curInstr, bot->banks[vars[taskCurBank]], this);
				  // Note: if we someday moved this behind the actual instruction execution,
				  // we would first have to check that the current bank exists.
			  }
			  // really execute the instruction
			  ret = curInstr->exec(this);
			  if(ret < 0) return ExecReturnType(ret); // error during execution
		  }
		  else // instead of executing overwritten instruction, ignore it & continue with next
		  {
			  ret = 0; 
			  vars[taskCurInstr]--; // re-queue the new instruction
			  //System::println("[NutCracker] Missing instruction ignored: " + curInstr->print());
		  }
      }

      // is there a current bank?
      if((vars[taskCurBank] >= bot->vars[botNumBanks]) || (vars[taskCurBank]< 0) )
      { // no, so reboot
          ret2 = reboot();

          if( ret2 < 0 ) // auto-reboot failed
              return ExecReturnType(ret2);
      }

      vars[taskCurInstr]++;

      // is the current instruction present? 
      if( (vars[taskCurInstr] >= bot->banks[vars[taskCurBank]]->instr.length()) ||
          (vars[taskCurInstr] < 0) )
      { // no, so reboot
          ret2 = reboot();

          if( ret2 < 0 ) // reboot failed
              return ExecReturnType(ret2);

          vars[taskCurInstr]++; // to get the same "curInstr" after reboot
      }

      // get new instr
	  curInstr = bot->banks[vars[taskCurBank]]->instr[vars[taskCurInstr]];
      ret2 = curInstr->getNumCycles(this);

      if( ret2 < 0 ) // bot killed in getNumCycles?
        return ExecReturnType(ret2);

	  if( ret2 > sim->glob.maxInstrDur ) // Instruction takes too long
		return failInstrDur;

      vars[taskNumCycles] = ret2;
      vars[taskCyclesDone] = 0;
	  vars[taskTimingMode] = curInstr->getTimingMode();
	  instrStarted = sim->curCycle;

      return ExecReturnType(ret);
  }

  // nothing to be done: instruction still waits.
  // so return execSleeps if sleeping (timingMode = 1) or execOK if not
  return ExecReturnType(vars[taskTimingMode]);

}

ExecReturnType Task::reboot()
{
	vars[taskCurBank] = 0; // go to first bank

	// is there a first bank, and does it have any instructions?
	if((!bot->banks[0]) || (bot->banks[0]->instr.length() <= 0)) 
	{ // no => data hunger
		return failData;
	}

	curInstr = 0;
	vars[taskCyclesDone] = 0;
	vars[taskNumCycles]  = 0;
	vars[taskTimingMode] = timingNumExec;
	instrStarted		   = sim->curCycle;
	vars[taskCurInstr]   =  -1; // is increased before usage in exec()

	return execOK;
}

Bot* Task::getRefBot()
{
	rint x,y;
	getRefBotPos(x, y);
	return sim->field[x + y * sim->fieldWidth];
}

void Task::getRefBotPos(rint& x, rint& y)
{
	x = bot->vars[botPosX];
	y = bot->vars[botPosY];

	sim->getPosBefore(x, y, vars[taskDir]);
}



////////////// Bank ///////////////
Bank::Bank(Program* const owner, const String& name) : 
	owner(owner), name(name), instr(0), isSecret(false)
{}

Bank::~Bank()
{
	deleteAll(instr);
}

String Bank::print()
{
	String ret;
	if(isSecret)
	{
		ret += "Secret Bank\n";
	}
	else {
		ret += name;
		ret += '\n';
		for(int i = 0; i < instr.length(); i++) {
			ret += ' ';
			ret += instr[i]->print();
			ret += '\n';
		}
	}
	ret += '\n';
	return ret;
}


/////////////// base class Instr ////////////////
Instr::Instr(int numParams) : ops(0), numOps(numParams), cycles(0)
{
	if(numOps > 0)
		ops = new Op[numOps];
}

Instr::~Instr()
{
	delete[] ops;
}

/** HACK: Task ops return their value + 1. This will work well as long as 
  * $InstrPos is the only task op. Reason is that curInstr starts from 0 in RT,
  * but is expected to start from 1 in RC3. */
rint Instr::getValue(Task* task, int opNum, bool absolute)
{
	Op& op = ops[opNum];

	switch(op.type){							
		case opTypeNumber:     return op.ptr;
		case opTypeTask:	   return task->vars[op.ptr] + 1;
		case opTypeActive: // same as named var, falls through!
		case opTypeNamedVar:
		{
			if(!op.isRemote)
				return task->bot->vars[op.ptr];
			// op is remote
			Bot* refBot = task->getRefBot();
			if(refBot != 0)
				return refBot->vars[op.ptr];
			else
				return 0;
		}
		case opTypeUnnamedVar: return task->bot->stdVars[op.ptr];
		case opTypeArray:
		{
			int index = task->bot->stdVars[op.ptr];
			if((index >= 1) && (index <= task->sim->glob.maxVars))
			  return task->bot->stdVars[index];
			else return 0;
		}
		case opTypeProgram:    return task->bot->owner->vars[op.ptr]; 
		case opTypeGlobal:     return task->sim->vars[op.ptr];
		case opTypeFields:
		{
			rint dir = task->vars[taskDir];
			if((dir == dirLeft) || (dir == dirRight))
				return task->sim->fieldWidth;
			else return task->sim->fieldHeight;
		}
		case opTypeAge:
		{
			Bot* bot = (op.isRemote ? task->getRefBot() : task->bot);
			if(bot == 0) return 0;
			return (rint)((task->sim->curCycle - bot->creationCycle) / 1000);
		}
		case opTypeLabel:
		{
			if(absolute) return op.ptr;
			else {
				rint pos = task->vars[taskCurInstr];
				return op.ptr - pos - 1;

			}
		}
		case opTypeAbsLabel:   return op.ptr;
		case opTypeRandom:     return Math::rand(1, task->sim->glob.maxBanks);
	}
	return 0;
}

void Instr::setValue(Task* task, int opNum, rint value)
{
    Op& op = ops[opNum];

	switch(op.type)	
	{						
		case opTypeActive:
		{
		  Bot* target;
		  if(op.isRemote) {
			  target = task->getRefBot();
			  if(target == 0) break;
		  }
		  else target = task->bot;

		  // handle silly sleep property: while deactivated it doesn't sleep
		  rint prev = target->vars[botActive];
		  if((prev > 0) && (value <= 0)) // deactivation
			  target->deactCycle = task->sim->curCycle;
		  else if((prev <= 0) && (value > 0) && (target->deactCycle > 0)) // activation
		  {
			  int deactTime = task->sim->curCycle - target->deactCycle;
			  for(int t = 0; t < target->tasks.length(); t++)
			  {
				  Task* ttask = target->tasks[t];
				  // if the current instruction sleeps
				  if(ttask->curInstr && (ttask->curInstr->getTimingMode() == timingNumWaited))
					  ttask->instrStarted += deactTime;
			  }
		  }

		  target->vars[botActive] = value;

		  break;
		}
		case opTypeNamedVar:
		{
		  if(!op.isRemote) {
		    task->bot->vars[op.ptr] = value;			
		    break;
		  }
		  // op is remote
		  Bot* refBot = task->getRefBot();
		  if(refBot != 0)				
		  {						
			refBot->vars[op.ptr] = value;		
		  }						
		  break;
		}
		case opTypeUnnamedVar:				
		  task->bot->stdVars[op.ptr] = value;			
		  break;					
		case opTypeArray:
		{
		  int index = task->bot->stdVars[op.ptr];
		  if((index >= 1) && (index <= task->sim->glob.maxVars))
		    task->bot->stdVars[index] = value;
		  break;
		}
		case opTypeGlobal:				
		  task->sim->vars[op.ptr] = value;
		  break;					
		default:
		  task->sim->errHandler->handleSystemError(-17, 
			  String("Attempt to set var of unsupported type ") + op.type);
		  break;
    }
}

void Instr::initOps(const lrt::Array<Op>& oops)
{
	for(int o = 0; o < numOps; o++)
		ops[o] = oops[o];
}

/* // more exact, but ugly!!!
String Instr::printOp(int opNum)
{
	Op& op = ops[opNum];
	String ret(op.ptr);
	ret += ":";
	ret += op.type;
	return ret;
}
*/

String Instr::print(int instrPos)
{
  return getName() + " " + printOps(instrPos);
}

String Instr::printOp(int opNum, int instrPos)
{
	Op& op = ops[opNum];
	switch(op.type){							
		case opTypeNumber:     return String(op.ptr);
		case opTypeTask:	   return taskVarNames[op.ptr];
		case opTypeActive: // same as named var, falls through!
		case opTypeNamedVar:
		{
			String ret = botVarNames[op.ptr];
			if(op.isRemote)
				ret[0] = '%';
			return ret;
		}
		case opTypeUnnamedVar: return String("#") + op.ptr;
		case opTypeArray:      return String("##") + op.ptr;
		case opTypeProgram:    return progVarNames[op.ptr];
		case opTypeGlobal:     return simVarNames[op.ptr];
		case opTypeFields:     return "$Fields";
		case opTypeAge:        return (op.isRemote ? "%Age" : "$Age");
		case opTypeLabel:	   return String(op.ptr - instrPos - 1);
		case opTypeAbsLabel:   return String(op.ptr);
		case opTypeRandom:     return "$RandNum";
	}
	return "???";
}


String Instr::printOps(int instrPos)
{
	String ret;
	for(int i = 0; i < numOps - 1; i++)
	{
		ret += printOp(i, instrPos);
		ret += ", ";
	}
	if(numOps != 0)
		ret += printOp(numOps - 1, instrPos);
	return ret;
}

rint Instr::getNumCycles(Task* task)
{
	return cycles;
}

TimingMode Instr::getTimingMode()
{
	return timingNumExec;
}

////////// SimSupervisor
SimSuperPosition SimSupervisor::getPreferredPosition()
{
	return supPosDontCare;
}

void SimSupervisor::initSim(Simulation* const curSim)
{
}

void SimSupervisor::exitSim(Simulation* const curSim, const GameState& simResult)
{
}

////////// ExecSupervisor
void ExecSupervisor::initSim(Simulation* const curSim)
{
}
void ExecSupervisor::exitSim(Simulation* const curSim)
{
}


////////// ProgramLoader

ProgramLoader::~ProgramLoader()
{
	delete headers;
	if( banks )
	  deleteAll(*banks);
	delete banks;
}

StringMap<Program::HeaderField>* ProgramLoader::getHeaders()
{
	if(state == lsNotLoaded) 
	{
		LoadReturnType ret = loadHeaders();

		// check language
		if(headers->isSet("language")) {
			if(!isLanguageVersionSupported(headers->get("language").value))
			{
				ret.success = false;
				ret.message = genLoadFailMsg[genFailLanguageVersion];
			}
		}

		if(ret.success) {
			state = lsHeadersLoaded;
		} else {
			state = lsError;
			if(handler) handler->handleLoadError(filename, ret.message);
		}
	}
	if(state == lsError) return 0;
	else return headers;
}

Vector<Bank*>* ProgramLoader::getBanks()
{
	if((state == lsNotLoaded) || (state == lsHeadersLoaded)) 
	{
		LoadReturnType ret = load();
		if(ret.success) {
			state = lsLoaded;
		} else {
			state = lsError;
			if(handler) handler->handleLoadError(filename, ret.message);
		}
	}
	if(state == lsError) 
		return 0;
	else return banks;
}

const ErrorHandler* ProgramLoader::getErrorHandler()
{
	return handler;
}

void ProgramLoader::setErrorHandler(const ErrorHandler* handler)
{
	this->handler = handler;
}

bool ProgramLoader::isLanguageVersionSupported(const String& language)
{
	String lang = language.upperCase();
	if(!lang.startsWith("RC"))
		return false;
	lang.remove(0, 2);
	int lv = lang.intValue(-1);
	if(lv < 0) // incorrect format
		return false;
	if(lv > languageVersion) // too high
		return false;
	return true;
}

ProgramLoader::ProgramLoader(const Globals& glob, const ErrorHandler* handler, const String& filename) : 
	state(lsNotLoaded), headers(0), banks(0), glob(glob), handler(handler), filename(filename)
{}


} // namespace 
