/*
 * rtfile.win32.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

#ifndef __LRT_FILE_WIN32__
#define __LRT_FILE_WIN32__

#include "rtfile.h"

#include <windows.h>
#include "rtcollect.h"
#include "rtstring.h"
#include "rtchar.h"

// how do we write a LONGLONG constant?
// MSVC wants it as-is, GCC wants a LL appended to it
#ifdef __MSVC__
#define LLCONST(a) a
#else
#define LLCONST(a) a ## LL
#endif

namespace lrt {

const int File::minNameCount = 1;
const char File::separatorChar = '\\';
const String File::separator = "\\";
const bool File::caseSensitive = false;

void File::initialize(const String &executableFileName)
{
    ::DWORD curDirLen = 1000; 
    ::LPTSTR curDir; 
    ::TCHAR curDirBuffer[1000]; 
 
    curDir = curDirBuffer; 
    GetCurrentDirectory(curDirLen, curDir); 

	currentFolder = new File(curDir);
	currentFolder->attribs->isFolder = ATTRIB_TRUE;
	executableFile = new File(executableFileName);
	
	// query registry for paths
	bool regSuccess = true; // succeeded querying registry?
	::HKEY key;
	::DWORD valueType, dataSize = 1000;
	::BYTE dataBuffer[1000];

	if(::RegOpenKeyEx(HKEY_CURRENT_USER, 
		"Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders", 0,
		KEY_READ, &key) == ERROR_SUCCESS)
	{
		// application data folder
		if((::RegQueryValueEx(key, "AppData", NULL, &valueType, dataBuffer, &dataSize)
			== ERROR_SUCCESS) && (valueType == REG_SZ))
		{
			String appData((char*)dataBuffer);
			if(!appData.endsWith("\\"))
			  appData += "\\";
			settingsFolder = new File(appData);
		}
		else { // not found
			regSuccess = false;
		}
		// my documents folder
		if((::RegQueryValueEx(key, "Personal", NULL, &valueType, dataBuffer, &dataSize)
			== ERROR_SUCCESS) && (valueType == REG_SZ))
		{
			String myDocs((char*)dataBuffer);
			if(!myDocs.endsWith("\\"))
			  myDocs += "\\";
			homeFolder = new File(myDocs);
		}
		else { // not found
			regSuccess = false;
		}
		::RegCloseKey(key);
	}
	else regSuccess = false;

	if(!regSuccess) { // fallback
		// one of them may have worked
		if(!homeFolder)
		  homeFolder = new File(*currentFolder);
		if(!settingsFolder)
		{
			File execFolder = executableFile->getParentFile();
			settingsFolder = new File(execFolder);
		}
	}
}

bool File::createSingleFolder(const File& folder)
{
	String fname = folder.getName();
	if(::CreateDirectory(fname.cStr(), NULL))
		return true;
	else return false;

}

bool File::resolveExtra(File *file, const String &fname) const
{
	if(fname.startsWith("\\"))
	{
		Array<String> *nNames = new Array<String>(file->names->length() + 1);
		(*nNames)[0] = (*names)[0];
		for(int i = 0; i < file->names->length(); i++)
			(*nNames)[i+1] = (*(file->names))[i];
		file->set(nNames);
		return true;
	}
	return false;
}

void File::fetchAttributes()
{
	String myName = getName();
	if(myName.endsWith(separator))
		myName.remove(myName.length() - 1); // delete \ at the end
	int attr = GetFileAttributes(myName.cStr());
	if(attr == -1) // file not found
	{
		attribs->exists = File::ATTRIB_FALSE;
		attribs->canRead = File::ATTRIB_FALSE; // the file doesn't exist, so what would you want to read from it anyway?
		attribs->canWrite = File::ATTRIB_TRUE; // in Windows, it's quite probable that you'll be able to create the file
		return;
	}
	else {
		attribs->exists = File::ATTRIB_TRUE;
		attribs->isFolder = ((attr & FILE_ATTRIBUTE_DIRECTORY) ? File::ATTRIB_TRUE : File::ATTRIB_FALSE);
		attribs->canRead = File::ATTRIB_TRUE;
		attribs->canWrite = ((attr & FILE_ATTRIBUTE_READONLY) ? File::ATTRIB_FALSE : File::ATTRIB_TRUE);

		::WIN32_FIND_DATA data;
		::HANDLE hFile;
		if((hFile = FindFirstFile(myName.cStr(), &data)) 
			!= INVALID_HANDLE_VALUE) {
			attribs->size = data.nFileSizeLow;
			LONGLONG ll = ((LONGLONG)data.ftLastWriteTime.dwLowDateTime) + 
				(((LONGLONG)data.ftLastWriteTime.dwHighDateTime) << 32);
			ll -= LLCONST(116444736000000000);
			ll /= 10000000;
			attribs->lastModified.sec = (int)ll;
			FindClose(hFile);
		}

//		else { System::println("CreateFile(" + myName + ") failed!"); }
	}
}

bool File::isAbsolute(const String &fname)
{
	if(fname.length() < 2) return false;
	if((fname[1] == ':') && Char::isLetter(fname[0]))
		return true;
	return false;

}

String File::getAbsolutePrefix()
{
  return String();
}

Array<File> File::listImpl(const IFilenameFilter *filter)
{
	bool acceptAll = false;
	if(filter == 0) 
		acceptAll = true;
	
	File parentFile = (isFolder() ? (*this) : getParentFile());
	String search = (isFolder() ? getName() : parentFile.getName());
	search += "*";

	WIN32_FIND_DATA FindFileData;
	HANDLE hFind;

	Vector<File> ret(0);

	hFind = FindFirstFile(search.cStr(), &FindFileData);

	if(hFind == INVALID_HANDLE_VALUE) {
		return ret;
	} else {
		String file = FindFileData.cFileName;
		if((file != ".") && (file != "..")) { // don't include them
		  if(FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) 
			file += separatorChar;
		  if(acceptAll || filter->accept(parentFile, file))
		    ret += File(parentFile, file);
		}
	}

	while(FindNextFile(hFind, &FindFileData)) {
		String file = FindFileData.cFileName;
		if((file == ".") || (file == "..")) // don't include them
			continue;
		if(FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) 
			file += separatorChar;
		if(acceptAll || filter->accept(parentFile, file))
		  ret += File(parentFile, file);
	}

	FindClose(hFind);
	
	return ret;
}

} // namespace


#endif
