

/*  This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License , or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by:
       e-mail:  hlub@knoware.nl
*/



#include "rlwrap.h"

FILE *log_fh;

static void
utils_warn(const char * message, va_list ap)
{

  int saved_errno = errno;
  char buffer[BUFFSIZE];
  

  snprintf1(buffer, sizeof(buffer) - 1, "%s: ", program_name);
  vsnprintf(buffer + strlen(buffer),sizeof(buffer)-strlen(buffer)-1,
	    message, ap);
  if (saved_errno)
    snprintf(buffer + strlen(buffer) ,sizeof(buffer)-strlen(buffer)-1,
	     ": %s", strerror(saved_errno));
  mystrlcat(buffer, "\n", sizeof(buffer));
  
  fflush(stdout);
  fputs(buffer, stderr);
  fflush(stderr);
  DPRINTF1(DEBUG_ALL, "Error/warning: %s", buffer); /* we want this because sometimes error messages (esp. from client) are dropped) */
  
}

void
mywarn(const char * message, ...)
{
  va_list ap;
  char *warning_message;
  if (nowarn)
    return;
  warning_message = add2strings("warning: ", message);
  va_start(ap, message);
  utils_warn(warning_message, ap);
  va_end(ap); 
}

void
myerror(const char * message, ...)
{
  va_list ap;
  char *error_message = add2strings("error: ", message);
  va_start(ap, message);
  utils_warn(error_message, ap);
  va_end(ap);
  if(! i_am_child) 
    cleanup_rlwrap_and_exit(EXIT_FAILURE);
  else
    exit(-1);
}




char *
mybasename (char *filename)
{ /* determine basename of "filename" */
  char *p;
  /* find last '/' in name (if any) */
  for (p=filename + strlen(filename) - 1; p > filename; p--)
    if (*(p-1) == '/')
      break;     	
  return p;
}

char *
mydirname (char *filename)
{ /* determine directory component of "name" */
  char *p;
  /* find last '/' in name (if any) */
  for (p=filename + strlen(filename) - 1; p > filename; p--)
    if (*(p-1) == '/')
      break;            
  return (p==filename ? "." : mystrndup(filename, p - filename));
}



void * 
mymalloc(size_t size) 
{ /* malloc with simplistic error handling: just bail out when out of memory */
  void *ptr;
  ptr = malloc(size);
  if (ptr == NULL) 
    myerror("Tried in vain to allocate %d bytes", size);
  return ptr;
}


/* wrappers around strlcat and strlcpy, if available, otherwise emulations
   of them. Both versions *assure* 0-termination, but don't check for
   truncation: return type is void */

void 
mystrlcpy(char *dst, const char *src, size_t size) {
#ifdef HAVE_STRLCPY
  strlcpy(dst, src, size);
#else
  strncpy(dst, src, size-1);
  dst[size-1] = '\0';
#endif
}

void 
mystrlcat(char *dst, const char *src, size_t size) {
#ifdef HAVE_STRLCAT
  strlcat(dst, src, size);
  dst[size-1] = '\0'; /* we don't check for truncation, just assure '\0'-termination. */
#else
  strncat(dst, src, size - strnlen(dst,size) - 1);
  dst[size-1] = '\0';
#endif
}




char *
mysavestring(const char* string)
{  
  return mystrndup(string, strlen(string));
}

char *
mystrndup(const char* string, int len)
{
  /* allocate copy of string on the heap */ 
  char *buf;
  buf = (char *) mymalloc (len + 1);
  mystrlcpy (buf, string, len + 1);
  return buf;
}



char* 
add3strings(const char *str1, const char *str2, const char *str3) 
{ 
  int size= strlen(str1) + strlen(str2) + strlen(str3) +1; /* total length plus 0 byte */
  char *buf = (char *) mymalloc (size);
  mystrlcpy (buf, str1, size);
  mystrlcat (buf, str2, size);
  mystrlcat (buf, str3, size);
  return buf;
}


void 
open_logfile(const char *filename)
{
  time_t now;
  log_fh = fopen(filename, "a");
  if (!log_fh) 
    myerror ("Cannot write to logfile %s", filename);
  now = time(NULL);
  fprintf(log_fh, "\n\n[rlwrap] %s\n", ctime(&now)); 
}

void
write_logfile(const char* str)
{
  if (log_fh) 
    fputs(str, log_fh);
}

void  
close_logfile()
{
  if (log_fh)
    fclose (log_fh);
}

static char mangle_buf[BUFFSIZE];

char *
mangle_string_for_debug_log(char *string, int len) { 
  char *tmp = mangle_buf;
  char oct[4];
  int i, j;
  
  assert(len + 3 < BUFFSIZE - 1); 

 
  for(i = j = 0; j < len - 3  && string[i] ; ) {
    if (isprint(string[i]) &&  string[i] != '\n')
      tmp[j++] = string[i++];
    else {
      tmp[j++]='\\';
      snprintf1(oct, sizeof(oct), "%03o", string[i++]);
      strncpy(&tmp[j], oct,3);
      j+=3;
    }
  }

  if (j == len - 3)
    for (; j < len; j++)
      tmp[j] = '.';
    
  tmp[j] = '\0';
  
  return tmp;
}


/* this is a complete mess, should find some better way @@@ */
   
void
usage()
{
  fprintf(stderr,
"Usage: %s [options] command ...\n"
"\n"
"Options:\n"	  
#ifdef HAVE_GETOPT_LONG
"   -a, --always-readline[=password:] %0.0s\n" /* %0.0s is ugly kludge to get rid of argument */
"   -b, --break-chars <chars>\n" 
"   -c, --complete-filenames\n"
"   -C, --command-name <name|N>\n"	  
#ifdef DEBUG
"   -d, --debug[=mask] (output sent to to %s)\n"
#endif
"   -f, --file <file>\n"
"   -h, --help\n"
"   -H, --history-filename <file>\n"
"   -l, --logfile <file>\n"
"   -n, --no-warnings\n"
"   -r, --remember\n"
"   -v, --version\n"
"   -s, --histsize <size>  (negative: readonly)\n" 
#else
"   -a%s always readline\n"     
"   -b <chars>    break chars\n" 
"   -c            complete filenames\n"
"   -C <name|N>   use <name> as command name\n"	  
#ifdef DEBUG	  
"   -d[mask]      debug (output sent to to %s)\n"
#endif
"   -f <file>     completion file \n"
"   -h            help\n"
"   -H <file>     history file\n"
"   -l <file>     logfile\n"
"   -n            don't warn\n"
"   -r            remember\n"
"   -v            print version\n"
"   -s <size>     history size  (negative: readonly)\n" 
#endif
	  , program_name	  
#ifdef GETOPT_GROKS_OPTIONAL_ARGS
	 , "[password:]"
#else
	 ,"<password:>"
#endif
#ifdef DEBUG
	  , DEBUG_FILENAME  /* DEBUG_FILENAME defined in rlwrap.h */
#endif
	  );

#ifdef DEBUG
fprintf(stderr,
"             \n"
"The debugging mask is a bitmask obtaining by adding:\n"
"    %3d    if you want to debug termio\n"
"    %3d    if you want to debug signals\n"
"    %3d    if you want to debug readline handling\n"
	"    omitting the mask means: debug everything \n"
	, DEBUG_TERMIO, DEBUG_SIGNALS, DEBUG_READLINE);
#endif
 
fprintf(stderr, 
"\n"
"bug reports, suggestions, updates:\n"
	"http://utopia.knoware.nl/~hlub/uck/rlwrap/\n");

exit(EXIT_SUCCESS);
}




























