%{
/*
 mfcalc.y, mfcalc.cpp, Copyright (c) 2004, 2005 R.Lackner
 parse string and simple math: based on the bison 'mfcalc' example

    This file is part of RLPlot.

    RLPlot is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RLPlot is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RLPlot; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <math.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include "rlplot.h"

struct symrec {
	int type, row, col;
	unsigned int h_name;
	char *name, *text;
	struct {
		double var;
		double (*fnctptr)(...);
		} value;
	int arg_type;
	struct symrec *next;
};

// syntactical information
struct syntax_info {
	int last_tok;			//resolve ambiguous ':'
	double clval;			//current value for where - clause
	int cl1, cl2;			//limits of clause formula in buffer
	struct syntax_info *next;
	};
static syntax_info *syntax_level = 0L;


typedef struct{
	double  val;
	int type;
	symrec  *tptr;
	double *a_data;
	char *text;
	int a_count;

}YYSTYPE;

static symrec *putsym (unsigned int h_name, int sym_type, int arg_type);
static symrec *getsym (unsigned int h_name, char *sym_name = 0L);
static int push(YYSTYPE *res, YYSTYPE *val);
static void store_res(YYSTYPE *res);
static char *PushString(char *text);
static char *add_strings(char *st1, char *st2);
static char *string_value(YYSTYPE *exp);
static int get_range(YYSTYPE *res, char *first, char *last);
static void exec_clause(YYSTYPE *res);
static YYSTYPE *proc_clause(YYSTYPE *res);
static void yyerror(char *s);
static int yylex(void);

static char res_txt[1000];
static anyResult line_res = {ET_UNKNOWN, 0.0, res_txt};
static double line_result;
static DataObj *curr_data;

//the current command buffer
static char *buffer = 0L;
static int buff_pos = 0;
%}

%token <val>  NUM ARR STR PI E CLVAL
%token <tptr> VAR FNCT TXT
%type  <val>  exp str_exp

%right  '='
%left	','			/* list separator */
%left	CLAUSE			/* clause (where) operator */
%right	COLR COLC		/* range sep., conditional sep. */
%right	'?'			/* conditional assignment */
%left	AND OR
%left   EQ NE GT GE LT LE
%left   '-' '+'
%left   '*' '/'
%left   NEG 		/* negation-unary minus */
%right  '^'	 	/* exponentiation       */

/* Grammar follows */
%%
input:    /* empty string */
	| input line
;

line:	 '\n' ';'
	| exp '\n'		{store_res(&yyvsp[-1]); return 0;}
	| exp ';'		{store_res(&yyvsp[-1]); return 0;}
	| str_exp '\n'		{store_res(&yyvsp[-1]); return 0;}
	| str_exp ';'		{store_res(&yyvsp[-1]); return 0;}
	| error '\n'		{yyerrok;}
;

str_exp:
	STR			{yyval.type=STR;}
	|str_exp '+' exp	{yyval.text=add_strings(yyvsp[-2].text, string_value(&yyvsp[0])); yyval.type=STR;}
	|exp '+' str_exp	{yyval.text=add_strings(string_value(&yyvsp[-2]), yyvsp[0].text); yyval.type=STR;}
	|str_exp '+' str_exp	{yyval.text=add_strings(yyvsp[-2].text, yyvsp[0].text); yyval.type=STR;}
;

exp:	NUM				{$$ = $1; yyval.type = NUM;}
        |TXT				{$$ = 0.0;}
	|CLVAL				{$$ = syntax_level ? syntax_level->clval : 0.0; }
	|PI				{$$ = 3.14159265358979; yyval.type = NUM;}
	|E				{$$ = 2.71828182845905; yyval.type = NUM;}
	|VAR				{$$ = $1->value.var; yyval.type=VAR}
	|VAR '=' exp		{$$ = $1->value.var = $3;
					if($1->row >=0 && $1->col >= 0 && curr_data) curr_data->SetValue($1->row, $1->col, $3);}
	|FNCT '(' exp ')'	{$$ = ($1->arg_type == ARR) ? ((*$1->value.fnctptr)(proc_clause(&yyvsp[-1]))) : 
					(*($1->value.fnctptr))($3);}
	|exp AND exp		{$$ = (($1 != 0) && ($3 != 0))? 1 : 0;}
	|exp OR exp		{$$ = (($1 != 0) || ($3 != 0))? 1 : 0;}
	|exp EQ exp		{$$ = ($1 == $3) ? 1 : 0;}
	|exp NE exp		{$$ = ($1 != $3) ? 1 : 0;}
	|exp GT exp		{$$ = ($1 > $3) ? 1 : 0;}
	|exp GE exp		{$$ = ($1 >= $3) ? 1 : 0;}
	|exp LT exp		{$$ = ($1 < $3) ? 1 : 0;}
	|exp LE exp		{$$ = ($1 <= $3) ? 1 : 0;}
	|exp '+' exp		{$$ = $1 + $3;}
	|exp '-' exp		{$$ = $1 - $3;}
	|exp '*' exp		{$$ = $1 * $3;}
	|exp '/' exp		{if($3 != 0.0) $$ = $1 / $3;
					else $$ = (getsym(HashValue((unsigned char*)"zdiv")))->value.var; }
	|exp ',' exp		{push(&yyval, &yyvsp[0]);}
	|VAR COLR VAR		{get_range(&yyval, $1->name, $3->name);}
	|'-' exp  %prec NEG	{$$ = -$2;}
	|exp '^' exp		{$$ = pow($1, $3);}
	|exp CLAUSE exp		{$$ = $1; exec_clause(&yyval);} 
	|'(' exp ')'		{memcpy(&yyval, &yyvsp[-1], sizeof(YYSTYPE))}
	|exp '?' exp COLC exp	{memcpy(&yyval, $1 != 0.0 ? &yyvsp[-2] : &yyvsp[0], sizeof(YYSTYPE))}
	|exp '?' STR COLC STR	{memcpy(&yyval, $1 != 0.0 ? &yyvsp[-2] : &yyvsp[0], sizeof(YYSTYPE))}
	|exp '?' STR COLC exp	{memcpy(&yyval, $1 != 0.0 ? &yyvsp[-2] : &yyvsp[0], sizeof(YYSTYPE))}
	|exp '?' exp COLC STR	{memcpy(&yyval, $1 != 0.0 ? &yyvsp[-2] : &yyvsp[0], sizeof(YYSTYPE))}
;
%%

static char *last_error = 0L;
static void yyerror(char *s)
{  
	//Called by yyparse on error
	if(curr_data) curr_data->Command(CMD_ERROR, last_error = s, 0L);
	else printf("%s\n", s);
}

static void store_res(YYSTYPE *res)
{
	if(res->type == STR) {
		line_result = 0.0;
		line_res.type = ET_TEXT;
		line_res. value = 0.0;
		if(res->text) strcpy(res_txt, res->text);
		}
	else if(res->type == ARR || (res->a_data)) {
		line_result = 0.0;
		line_res.type = ET_TEXT;
		line_res. value = 0.0;
		strcpy(res_txt, "#ARRAY");
		}
	else if(res->tptr && res->tptr->type == TXT) {
		line_result = 0.0;
		line_res.type = ET_TEXT;
		line_res.value = 0.0;
		if(res->tptr->text) strcpy(res_txt, res->tptr->text);
		}
	else {
		line_result = res->val;
		line_res.type = ET_VALUE;
		line_res.value = res->val;
		}
}

static char *add_strings(char *st1, char *st2)
{
	char *newstr, *ret;

	if(st1 && st2) {
		if(newstr = (char*)malloc(strlen(st1) +strlen(st2) +4)) {
			sprintf(newstr, "%s%s", st1, st2);
			ret = PushString(newstr);
			free(newstr);
			return ret;
			}
		else return 0L;
		}
	if(st1) return st1;
	if(st2) return st2;
	return 0L;
}

static char *string_value(YYSTYPE *exp)
{
	char *st1, tmp[50];

	if(exp->type == STR){
		st1 = exp->text;
		}
	else if(exp->tptr && exp->tptr->type == TXT) {
		st1 = exp->tptr->text;
		}
	else {
		sprintf(tmp,"%g", exp->val);
		st1 = tmp;
		}
	return PushString(st1);
}

// store syntactical information
static void push_syntax()
{
	syntax_info *next;

	if(!(next = (syntax_info*)calloc(1, sizeof(syntax_info)))) return;
	if(syntax_level)memcpy(next, syntax_level, sizeof(syntax_info));
	next->next=syntax_level;
	syntax_level = next;
}

static void pop_syntax()
{
	syntax_info *si;

	if(si = syntax_level) {
		syntax_level = si->next;
		free(si);
		}
}

// more functions
static double sign(double v)
{
	if(v > 0.0) return 1.0;
	if(v < 0.0) return -1.0;
	return 0.0;
}

static double factorial(double v)
{
	return factrl((int)v);
}

static void close_arr_func(YYSTYPE *sr)
{
	free(sr->a_data);
	sr->a_data = 0L;		sr->a_count = 0;
	if(sr->type == ARR) sr->type = NUM;
}

#undef min
static double min(YYSTYPE *sr) 
{
	int i;

	if(!sr) return 0.0;
	if(sr->a_data && sr->a_data){
		for(i = 1, sr->val = sr->a_data[0]; i < sr->a_count; i++) 
			if(sr->a_data[i] < sr->val) sr->val = sr->a_data[i];
		close_arr_func(sr);
		}
	return sr->val;
}

#undef max
static double max(YYSTYPE *sr) 
{
	int i;

	if(!sr) return 0.0;
	if(sr->a_data){
		for(i = 1, sr->val = sr->a_data[0]; i < sr->a_count; i++) 
			if(sr->a_data[i] > sr->val) sr->val = sr->a_data[i];
		close_arr_func(sr);
		}
	return sr->val;
}

static double count(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->a_data){
		sr->val = (double)sr->a_count;
		close_arr_func(sr);
		}
	else sr->val = 0.0;
	return sr->val;
}

static double sum(YYSTYPE *sr) 
{
	int i;

	if(!sr) return 0.0;
	if(sr->a_data){
		for(i = 1, sr->val = sr->a_data[0]; i < sr->a_count; i++) sr->val += sr->a_data[i];
		close_arr_func(sr);
		}
	return sr->val;
}

static double calc_variance(double *values, int n)
{
	int i;
	double ss, d, mean = d_amean(n, values);

	for(i=0, ss=0.0; i < n; i++) ss += ((d=values[i]-mean)*d);
	return (ss/(n-1));
}

static double mean(YYSTYPE *sr) 
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		sr->val = d_amean(sr->a_count, sr->a_data );
		close_arr_func(sr);
		}
	return sr->val;
}

static double gmean(YYSTYPE *sr) 
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		sr->val = d_gmean(sr->a_count, sr->a_data );
		close_arr_func(sr);
		}
	return sr->val;
}

static double hmean(YYSTYPE *sr) 
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		sr->val = d_hmean(sr->a_count, sr->a_data );
		close_arr_func(sr);
		}
	return sr->val;
}

static double quartile1(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		d_quartile(sr->a_count, sr->a_data, &sr->val, 0L, 0L);
		close_arr_func(sr);
		}
	return sr->val;
}

static double quartile2(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		d_quartile(sr->a_count, sr->a_data, 0L, &sr->val, 0L);
		close_arr_func(sr);
		}
	return sr->val;
}

static double quartile3(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	if(sr->a_data && sr->a_count){
		d_quartile(sr->a_count, sr->a_data, 0L, 0L, &sr->val);
		close_arr_func(sr);
		}
	return sr->val;
}

static double variance(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count){
		sr->val = calc_variance(sr->a_data, sr->a_count);
		close_arr_func(sr);
		}
	return sr->val;
}

static double stdev(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count){
		sr->val = sqrt(calc_variance(sr->a_data, sr->a_count));
		close_arr_func(sr);
		}
	return sr->val;
}

static double sterr(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count){
		sr->val = sqrt(calc_variance(sr->a_data, sr->a_count))/sqrt(sr->a_count);
		close_arr_func(sr);
		}
	return sr->val;
}

static double tdist(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0.0;
	if(sr->a_data && sr->a_count == 2){
		sr->val = t_dist(sr->a_data[0], sr->a_data[1]);
		close_arr_func(sr);
		}
	return sr->val;
}

static double fdist(YYSTYPE *sr)
{
	if(!sr) return 0.0;
	sr->val = 0;
	if(sr->a_data && sr->a_count == 3){
		sr->val = f_dist(sr->a_data[0], sr->a_data[1], sr->a_data[2]);
		close_arr_func(sr);
		}
	return sr->val;
}

struct init
{
	unsigned int h_name;
	double (*fnct)(double);
	int arg_type;
};

static struct init arith_fncts[] = {
	{HashValue((unsigned char*)"variance"), (double(*)(double))&variance, ARR},
	{HashValue((unsigned char*)"stdev"), (double(*)(double))&stdev, ARR},
	{HashValue((unsigned char*)"sterr"), (double(*)(double))&sterr, ARR},
	{HashValue((unsigned char*)"min"), (double(*)(double))&min, ARR},
	{HashValue((unsigned char*)"max"), (double(*)(double))&max, ARR},
	{HashValue((unsigned char*)"count"), (double(*)(double))&count, ARR},
	{HashValue((unsigned char*)"sum"), (double(*)(double))&sum, ARR},
	{HashValue((unsigned char*)"mean"), (double(*)(double))&mean, ARR},
	{HashValue((unsigned char*)"median"), (double(*)(double))&quartile2, ARR},
	{HashValue((unsigned char*)"quartile1"), (double(*)(double))&quartile1, ARR},
	{HashValue((unsigned char*)"quartile2"), (double(*)(double))&quartile2, ARR},
	{HashValue((unsigned char*)"quartile3"), (double(*)(double))&quartile3, ARR},
	{HashValue((unsigned char*)"gmean"), (double(*)(double))&gmean, ARR},
	{HashValue((unsigned char*)"hmean"), (double(*)(double))&hmean, ARR},
	{HashValue((unsigned char*)"tdist"), (double(*)(double))&tdist, ARR},
	{HashValue((unsigned char*)"fdist"), (double(*)(double))&fdist, ARR},
	{HashValue((unsigned char*)"sign"), sign, VAR},
	{HashValue((unsigned char*)"gammaln"), gammln, VAR},
	{HashValue((unsigned char*)"factorial"), factorial, VAR},
	{HashValue((unsigned char*)"abs"), fabs, VAR},
	{HashValue((unsigned char*)"asin"), asin, VAR},
	{HashValue((unsigned char*)"acos"), acos, VAR},
	{HashValue((unsigned char*)"atan"), atan, VAR},
	{HashValue((unsigned char*)"sinh"), sinh, VAR},
	{HashValue((unsigned char*)"cosh"), cosh, VAR},
	{HashValue((unsigned char*)"tanh"), tanh, VAR},
	{HashValue((unsigned char*)"sin"),  sin, VAR},
	{HashValue((unsigned char*)"cos"),  cos, VAR},
	{HashValue((unsigned char*)"atan"), atan, VAR},
	{HashValue((unsigned char*)"log10"), log10, VAR},
	{HashValue((unsigned char*)"ln"),   log, VAR},
	{HashValue((unsigned char*)"log"),   log, VAR},
	{HashValue((unsigned char*)"exp"),  exp, VAR},
	{HashValue((unsigned char*)"sqrt"), sqrt, VAR},
	{0, 0, 0}};

// Store strings in a list
static char **str_list = 0L;
static int n_str = 0;

static char *PushString(char *text)
{
	if(text && text[0]) {
		if(str_list = (char**)realloc(str_list, sizeof(char*)*(n_str+1)))
			str_list[n_str] = strdup(text);
		return str_list[n_str++];
		}
	return 0L;
}

//The symbol table: a chain of `struct symrec'
static symrec *sym_table = (symrec *) 0;

// Put arithmetic functions and predifened variables in table
static void init_table (void)
{
	int i;
	symrec *ptr;

	for (i = 0; arith_fncts[i].h_name; i++) {
		ptr = putsym (arith_fncts[i].h_name, FNCT, arith_fncts[i].arg_type);
		ptr->value.fnctptr = (double (*)(...))arith_fncts[i].fnct;
		}
	ptr = putsym(HashValue((unsigned char*)"zdiv"), VAR, 0);	ptr->value.var = 1.0;
	str_list = 0L;		n_str = 0;
	push_syntax();
}

static void clear_table()
{
	int i;
	symrec *ptr, *next;

	for (ptr = sym_table; ptr != (symrec *) 0;){
		if(ptr) {
			if(ptr->name) free(ptr->name);
			if(ptr->text) free(ptr->text);
			next = (symrec*)ptr->next;
			free(ptr);
			}
		ptr = next;
		}
	sym_table = (symrec *) 0;
	if(str_list) {
		for(i = 0; i < n_str; i++) if(str_list[i]) free(str_list[i]);
		free(str_list);		str_list = 0L;		n_str = 0;
		}
	pop_syntax();
}

static symrec *
putsym (unsigned int h_name, int sym_type, int arg_type)
{
	symrec *ptr;

	ptr = (symrec *) malloc (sizeof (symrec));
	ptr->h_name = h_name;
	ptr->type = sym_type;
	ptr->name = ptr->text = 0L;
	ptr->value.var = 0; /* set value to 0 even if fctn.  */
	ptr->arg_type = arg_type;
	ptr->col = ptr->row = -1;
	ptr->next = (struct symrec *)sym_table;
	sym_table = ptr;
	return ptr;
}

static symrec *
getsym (unsigned int h_name, char *sym_name)
{
	symrec *ptr;
	int row, col;
	AccRange *ar;
	anyResult ares;

	if(!h_name) return 0;
	for (ptr = sym_table; ptr != (symrec *) 0; ptr = (symrec *)ptr->next)
		if (ptr->h_name == h_name){
			if(sym_name && !ptr->name) ptr->name = ptr->name=strdup(sym_name);
			return ptr;
			}
        if((sym_name && curr_data) && (isalpha(sym_name[0]) || sym_name[0] == '$') && isdigit(sym_name[strlen(sym_name)-1])) {
		if((ar = new AccRange(sym_name)) && ar->GetFirst(&col, &row) && 
			(ptr = putsym(h_name, VAR, 0))) {
			ptr->name = strdup(sym_name);
			if(curr_data->GetResult(&ares, row, col)){
				if(ares.type == ET_VALUE) {
					ptr->type = VAR;	ptr->value.var = ares.value;
					ptr->text = 0L;
					}
				else if(ares.type == ET_TEXT && ares.text) {
					ptr->type = TXT;	ptr->value.var = 0.0;
					ptr->text = strdup(ares.text);
					}
				else {
					ptr->type = VAR;	ptr->value.var = 0.0;
					ptr->text = 0L;
					}
				}
			ptr->row = row;	ptr->col = col;
			delete(ar);
			return ptr;
			}
		}
	return 0;
}

static int
push(YYSTYPE *res, YYSTYPE *val)
{
	if(val->a_data && val->a_count) {
		if(!(res->a_data)) {
			if(!(val->a_data=(double*)realloc(val->a_data, (val->a_count+2)*sizeof(double))))return 0;
			val->a_data[val->a_count++] = res->val;
			res->a_data = val->a_data;		res->a_count = val->a_count;
			val->a_data = 0L;			val->a_count = 0;
			val->val = res->val;			return 1;
			}
		else {
			if(!(res->a_data=(double*)realloc(res->a_data, (val->a_count+res->a_count)*sizeof(double))))return 0;
			memcpy(&res->a_data[res->a_count], val->a_data, val->a_count*sizeof(double));
			res->a_count += val->a_count;		free(val->a_data);
			val->a_data = 0L;			val->a_count = 0;
			return 1;
			}
		}
	if(!(res->a_data )){
		if(!(res->a_data =  (double*)malloc(2*sizeof(double))))return 0;
		res->a_data[0] = res->val;			res->a_data[1] = val->val;
		res->a_count = 2;
		return 1;
		}
	else {
		if(!(res->a_data = (double*)realloc(res->a_data, (res->a_count+2)*sizeof(double))))return 0; 
		res->a_data[res->a_count] = val->val;		res->a_count++;
		return 1;
		}
	return 0;
}

static int
get_range(YYSTYPE *res, char *first, char *last)
{
	char r_txt[40];
	AccRange *r;
	int row, col;
	double value;
	YYSTYPE tmp;

	if(!res || !first || !last || !curr_data) return 0;
	sprintf(r_txt, "%s:%s", first, last);
	if(!(res->a_data ) && (r = new AccRange(r_txt)) && r->GetFirst(&col, &row)) {
		if(!(res->a_data =  (double*)malloc(r->CountItems() * sizeof(double)))) return 0;
		res->a_count = 0;
		for( ; r->GetNext(&col, &row); ) {
			if(curr_data->GetValue(row, col, &value)) res->a_data[res->a_count++] = value;
			}
		delete r;		return 1;
		}
	if((r = new AccRange(r_txt)) && r->GetFirst(&col, &row)) {
		//it is probably a bit slow to push every element
		tmp.type = NUM;
		for( ; r->GetNext(&col, &row); ) {
			if(curr_data->GetValue(row, col, &value)) {
				tmp.val = value;
				push(res, &tmp);
				}
			}
		delete r;		return 1;
		}
	return 0;
}

static YYSTYPE *proc_clause(YYSTYPE *res)
{
	int i, n, o_pos;
	char *o_cmd;
	double *n_data;

	if(!(syntax_level) || !syntax_level->cl1 || syntax_level->cl2 <= syntax_level->cl1) return res;
	if(!res->text) return res;
	if(!res->a_data && (res->a_data = (double*)malloc(sizeof(double)))) {
		res->a_data[0] = res->type == VAR && res->tptr ? res->tptr->value.var : res->val;
		res->a_count = 1;
		}
	else if(!res->a_data) return res;
	if(!(n_data = (double*)malloc(res->a_count * sizeof(double)))) return res;
	o_pos = buff_pos;	o_cmd = buffer;
	for(i = n = 0; i < res->a_count; i++) {
		buffer = res->text;	buff_pos = 0;
		if(!syntax_level) break;
		syntax_level->clval = res->a_data[i];
		yyparse();
		if(line_res.type == ET_VALUE && line_res.value != 0.0) n_data[n++] = res->a_data[i];
		}
	free(res->a_data);	res->a_data = n_data;		res->a_count = n;
	free(res->text);	res->text=0L;
	syntax_level->cl1 = syntax_level->cl2 = 0;
	buffer = o_cmd;	buff_pos = o_pos;
	return res;
}

static void exec_clause(YYSTYPE *res)
{
	int i, j;
	char *cmd;

	if(!(syntax_level) || !syntax_level->cl1 || syntax_level->cl2 <= syntax_level->cl1) return;
	if(!(cmd = (char*)malloc(syntax_level->cl2 - syntax_level->cl1 +2)))return;
	while(buffer[syntax_level->cl1] <= ' ' && syntax_level->cl1 < syntax_level->cl2) syntax_level->cl1++;
	for(j = 0, i = syntax_level->cl1; i< syntax_level->cl2; i++) {
		cmd[j++] = buffer[i];
		}
	cmd[j++] = ';';		cmd[j++] = 0;
	res->text = cmd;
}

struct parse_info  {
	char *buffer;
	int buff_pos;
	double line_result;
	DataObj *curr_data;
	symrec *sym_table;
	YYSTYPE yylval;
	struct parse_info *next;
	char **str_list;
	int n_str;
};
static parse_info *parse_stack = 0L;

static void push_parser()
{
	parse_info *ptr;

	ptr = (parse_info *) malloc(sizeof(parse_info));
	ptr->buffer = buffer;			ptr->buff_pos = buff_pos;
	ptr->line_result = line_result;		ptr->curr_data = curr_data;
	ptr->sym_table = sym_table;		sym_table = 0L;
	memcpy(&ptr->yylval, &yylval, sizeof(YYSTYPE));
	ptr->next = parse_stack;
	ptr->str_list = str_list;		str_list = 0L;
	ptr->n_str = n_str;			n_str = 0;
	parse_stack = ptr;
	push_syntax();
}

static void pop_parser()
{
	parse_info *ptr;

	if(ptr = parse_stack) {
		parse_stack = ptr->next;
		buffer = ptr->buffer;			buff_pos = ptr->buff_pos;
		line_result = ptr->line_result;		curr_data = ptr->curr_data;
		sym_table = ptr->sym_table;
		memcpy(&yylval, &ptr->yylval, sizeof(YYSTYPE));
		str_list = ptr->str_list;		n_str = ptr->n_str;
		free(ptr);
		}
	pop_syntax();
}

static int is_ttoken(int h_nam)
{
	switch(h_nam) {
	case 69:		return E;
	case 393:		return PI;
	case 28381:
		if(syntax_level) syntax_level->cl1 = buff_pos;
		return CLAUSE;
	case 20:		return CLVAL;
		}
	return 0;
}

static symrec *curr_sym;
static int yylex (void)
{
	int i, c, tok;
	unsigned int h_nam;
	char tmp_txt[80];
	symrec *s;

	while((c = buffer[buff_pos++]) == ' ' || c == '\t');	//get first nonwhite char
	if(!c) return 0;
	//test for number
	if(c == '.' || isdigit(c)) {
		for(buff_pos--, i = 0; i < 79 && ((c = buffer[buff_pos]) == '.' || isdigit(c)); buff_pos++) {
			tmp_txt[i++] = (char)c;
			if(buffer[buff_pos+1] == 'e' && (buffer[buff_pos+2] == '-' || buffer[buff_pos+2] == '+')){
				tmp_txt[i++] = buffer[++buff_pos];
				tmp_txt[i++] = buffer[++buff_pos];
				}
			}
		tmp_txt[i] = 0;
		sscanf(tmp_txt, "%lf", &yylval.val);
		yylval.type = NUM;
		return NUM;
		}
	//test for name or stringtoken
	if(isalpha(c) || c=='$') {
 		for(buff_pos--, i = 0; i < 79 && ((c = buffer[buff_pos]) && (isalnum(c) || c == '$')); buff_pos++) {
			tmp_txt[i++] = (char)c; 
			}
		tmp_txt[i] = 0;
		h_nam = HashValue((unsigned char*)tmp_txt);
		if(tok = is_ttoken(h_nam)) 
			return tok;
		if(!(strcmp(tmp_txt, "pi"))) return PI;
		if(!(strcmp(tmp_txt, "e"))) return E;
		if(!(s = getsym(h_nam, tmp_txt))){
			s = putsym(h_nam, VAR, 0);
			s->name = strdup(tmp_txt);
			}
		
		curr_sym = yylval.tptr = s;	return s->type;
		}
	//test for string
	if(c == '"' || c == '\'') {
		for(i= 0; i < 79 && ((tok = buffer[buff_pos]) && (tok != c)); buff_pos++) {
			tmp_txt[i++] = (char)tok;
			}
		if(buffer[buff_pos] == c)buff_pos++;
		tmp_txt[i] = 0;
		yylval.text = PushString(tmp_txt);
		return yylval.type = STR;
		}
	tok = 0;
	switch(c) {
	case '=':
		if(buffer[buff_pos] == '=') tok = EQ;
		break;
	case '!':
		if(buffer[buff_pos] == '=') tok = NE;
		break;
	case '>':
		if(buffer[buff_pos] == '=') tok = GE;
		else return GT;
		break;
	case '<':
		if(buffer[buff_pos] == '=') tok = LE;
		else if(buffer[buff_pos] == '>') tok = NE;
		else return LT;
		break;
	case '&':
		if(buffer[buff_pos] == '&') tok = AND;
		break;
	case '|':
		if(buffer[buff_pos] == '|') tok = OR;
		break;
	case ')':
		if(syntax_level) {
			if(syntax_level->cl1 && syntax_level->next) {
				syntax_level->next->cl1 = syntax_level->cl1;
				syntax_level->next->cl2 = buff_pos-1;
				}
			}
		pop_syntax();
		break;
	case '(':
		push_syntax();
	case '?':
		if(syntax_level) syntax_level->last_tok = c;
		break;
	case ':':
		if(syntax_level) {
			if(syntax_level->last_tok == '(') return COLR;
			else if(syntax_level->last_tok == '?') return COLC;
			}
		break;
		}
	if(tok) {
		buff_pos++;		return tok;
		}
	//Any other character is a token by itself
	return c;
}

bool do_xyfunc(DataObj *d, double x1, double x2, double step, char *expr, lfPOINT **pts, long *npts, char *param)
{
	double x, y;
	symrec *s;
	lfPOINT *new_points;
	long npoints = 0;
	int length;
	unsigned int hn_x = HashValue((unsigned char *)"x");
	unsigned int hn_y = HashValue((unsigned char *)"y");

	if(x1 < x2) step = fabs(step);
	else step = -fabs(step);
	if(!(new_points = (lfPOINT*)calloc((iround(fabs(x2-x1)/fabs(step))+2), sizeof(lfPOINT))))
		return false;
	if(d) curr_data = d;
	init_table();
	if(param) {
		length = strlen(param);
		if(!(buffer = (char*)malloc(length+2))){
			pop_parser();
			return false;
			}
		strcpy(buffer, param);	buffer[length++] = ';';
		buffer[length] = 0;	buff_pos = 0;
		do {
			yyparse();
			}while(buff_pos < length);
		free(buffer);		buffer = 0L;
		}		
	length = strlen(expr);
	buffer = expr;		s = putsym(hn_x, VAR, 0);
	for(x = x1; step > 0.0 ? x <= x2 : x >= x2; x += step) {
		if(s = getsym(hn_x)){
			s->value.var = x;	buff_pos = 0;
			do {
				yyparse();
				}while(buff_pos < length);
			if(s = getsym(hn_y)) y = s->value.var;
			else y = line_result;
			new_points[npoints].fx = (getsym(hn_x))->value.var;
			new_points[npoints++].fy = y;
			}
		}
	*pts = new_points;	*npts = npoints;
	clear_table();
	if(curr_data) {
		curr_data->Command(CMD_CLEAR_ERROR, 0L, 0L);
		curr_data->Command(CMD_REDRAW, 0L, 0L);
		}
	return true;
}

anyResult *do_formula(DataObj *d, char *expr)
{
	int length;
	static anyResult ret;

	if(d) curr_data = d;
	ret.type = ET_ERROR;		ret.text = 0L;
	if(!expr || !expr[0]) return &ret;
	push_parser();		//make code reentrant
	init_table();		length = strlen(expr);
	if(!(buffer = (char*)malloc(length+2))){
		pop_parser();
		return &ret;
		}
	strcpy(buffer, expr);	buffer[length++] = ';';
	buffer[length] = 0;	buff_pos = 0;
	do {
		yyparse();
		}while(buff_pos < length);
	if(curr_data && last_error) {
		curr_data->Command(CMD_ERROR, last_error = 0L, 0L);
		return &ret;
		}
	free(buffer);		buffer = 0L;
	clear_table();
	pop_parser();
	return &line_res;
}

bool MoveFormula(DataObj *d, char *of, char *nf, int dx, int dy)
{
	int length, tok, pos, i;
	char *res, desc1[2], desc2[2];

	if(d) curr_data = d;
	if(!curr_data || !of || !nf) return false;
	push_parser();		//make code reentrant
	init_table();		length = strlen(of);
	if(!(buffer = (char*)malloc(length+2))){
		pop_parser();
		return false;
		}
	strcpy(buffer, of);	buffer[length++] = ';';
	buffer[length] = 0;	buff_pos = pos = 0;
	res = (char *)calloc(length*2+10, sizeof(char));
	do {
		tok = yylex ();
		if(tok && tok < 256) {
			if(res[pos-1] == ' ') pos--;
			res[pos++] = (char)tok;
			}
		else switch(tok) {
			case NUM:
				pos += sprintf(res+pos, "%g ", yylval.val);
				break;
			case FNCT:
				pos += sprintf(res+pos, "%s", curr_sym->name);
				break;
			case COLR:
				pos += sprintf(res+pos, ":");
				break;
			case COLC:
				pos += sprintf(res+pos, ": ");
				break;
			case CLVAL:
				pos += sprintf(res+pos, "$$ ");
				break;
			case CLAUSE:
				pos += sprintf(res+pos, " where ");
				break;
			case VAR:
				if(curr_sym->col >= 0 && curr_sym->row >= 0) {
					desc1[0] = desc1[1] = desc2[0] = desc2[1] = 0;
					for(i=strlen(curr_sym->name)-1; i>0 && isdigit(curr_sym->name[i]); i--);
					if(curr_sym->name[0] == '$') desc1[0] = '$';
					if(curr_sym->name[i] == '$') desc2[0] = '$';
					pos += sprintf(res+pos, "%s%s%s%d", desc1, 
						Int2ColLabel(desc1[0] ? curr_sym->col : curr_sym->col+dx, false),
						desc2, desc2[0]? curr_sym->row+1 : curr_sym->row+1+dy);
					}
				else pos += sprintf(res+pos, "%s ", curr_sym->name);
				break;
			case STR:
				pos += sprintf(res+pos, "\"%s\"", yylval.text && yylval.text[0] ? yylval.text : "");
				break;
			case PI:
				pos += sprintf(res+pos, "pi ");
				break;
			case E:
				pos += sprintf(res+pos, "e ");
				break;
			case AND:
				pos += sprintf(res+pos, " && ");
				break;
			case OR:
				pos += sprintf(res+pos, " || ");
				break;
			case EQ:
				pos += sprintf(res+pos, " == ");
				break;
			case NE:
				pos += sprintf(res+pos, " != ");
				break;
			case GT:
				pos += sprintf(res+pos, " > ");
				break;
			case GE:
				pos += sprintf(res+pos, " >= ");
				break;
			case LT:
				pos += sprintf(res+pos, " < ");
				break;
			case LE:
				pos += sprintf(res+pos, " <= ");
				break;
			}
		}while(buff_pos < length);
	while((res[pos-1] == ';' || res[pos-1] == ' ') && pos > 0) { res[pos-1] = 0; pos--;} 
	strcpy(nf, res);	free(res);
	free(buffer);		buffer = 0L;
	clear_table();
	pop_parser();
	return true;
}

static char *txt_formula;	//function to fit
static double **parval;		//pointers to parameter values
static void fcurve(double x, double z, double **a, double *y, double dyda[], int ma)
{
	int i, length;
	double tmp, y1, y2;
	symrec *symx, *s=0L;
	unsigned int hn_x = HashValue((unsigned char *)"x");
	unsigned int hn_y = HashValue((unsigned char *)"y");

	if(!(symx = getsym(hn_x))) symx = putsym(hn_x, VAR, 0);
	//swap parameters to requested set
	if(a != parval) for(i = 0; i < ma; i++) {
		tmp = *parval[i];	*parval[i]  = *a[i];	*a[i] = tmp;
		}
	//calc result
	symx->value.var = x;	buffer = txt_formula;
	buff_pos = 0;		length = strlen(txt_formula);
	do {	yyparse();	}while(buff_pos < length);
	if(s = getsym(hn_y)) *y = s->value.var;
	else *y = line_result;
	if(*y == HUGE_VAL || *y == -HUGE_VAL) {
		for(i = 0, *y = 0.0; i < ma; dyda[i++] = 0.0);
		return;
		}
	//partial derivatives for each parameter by numerical differentiation
	for(i = 0; i < ma; i++) {
		if(*parval[i] != 0.0) {
			tmp = *parval[i];
			*parval[i] = tmp*.995;
			buff_pos = 0;
			do {	yyparse();	}while(buff_pos < length);
			y1 = s ? s->value.var : line_result;
			*parval[i] = tmp*1.005;
			buff_pos = 0;
			do {	yyparse();	}while(buff_pos < length);
			y2 = s ? s->value.var : line_result;
			*parval[i] = tmp;
			dyda[i] = (y2-y1)*100.0/tmp;
			}
		else dyda[i] = 0.0;
		}
	//swap parameters back to original
	if(a != parval) for(i = 0; i < ma; i++) {
		tmp = *parval[i];	*parval[i]  = *a[i];	*a[i] = tmp;
		}
}

int do_fitfunc(DataObj *d, char *rx, char *ry, char *rz, char **par, char *expr, double conv, int maxiter, double *chi_2)
{
	int length, i, j, k, l, ndata, nparam, r1, r2, r3, c1, c2, c3, *lista, itst, itst1;
	symrec *tab1, *tab2, *csr, **parsym;
	AccRange *arx, *ary, *arz;
	double *x, *y, *z, currx, curry, currz, alamda, chisq, ochisq;
	double **covar, **alpha;
	char tmp_txt[500];

	if(d) curr_data = d;
	if(chi_2) *chi_2 = 0.0;
	txt_formula = expr;
	if(!curr_data || !par || !expr || !rx || !ry) return 0;
	//process ranges and create arrays
	arx = ary = arz = 0L;	x = y = z = 0L;	parval = 0L;	parsym = 0L;
	if(!(arx = new AccRange(rx)))return 0;
	i = arx->CountItems()+1;
	if(!(ary = new AccRange(ry))){
		delete arx;	return 0;
		}
	if(rz && !(arz = new AccRange(rz))){
		delete ary;	delete arx;	return 0;
		}
	if(!(x = (double*)malloc(i * sizeof(double)))){
		if(arz) delete arz;
		delete ary;	delete arx;	return 0;
		}
	if(!(y = (double*)malloc(i * sizeof(double)))){
		if(arz) delete arz;
		free(x);	delete arx;	delete ary;	return 0;
		}
	if(rz && !(y = (double*)malloc(i * sizeof(double)))){
		if(arz) delete arz;
		free(y);	free(x);	delete arx;	delete ary;	return 0;
		}
	arx->GetFirst(&c1, &r1);	ary->GetFirst(&c2, &r2);
	if(rz) arz->GetFirst(&c3, &r3);
	for(ndata = j = 0; j < i; j++) {
		if(rz) {
			if(arx->GetNext(&c1, &r1) && ary->GetNext(&c2, & r2) && arz->GetNext(&c3, &r3) &&
				curr_data->GetValue(r1, c1, &currx) && curr_data->GetValue(r2, c2, &curry) &&
				curr_data->GetValue(r3, c3, &currz)) {
				x[ndata] = currx;	y[ndata] = curry;	z[ndata] = currz;	ndata++;
				}
			}
		else {
			if(arx->GetNext(&c1, &r1) && ary->GetNext(&c2, & r2) &&
				curr_data->GetValue(r1, c1, &currx) && curr_data->GetValue(r2, c2, &curry)) {
				x[ndata] = currx;	y[ndata] = curry;	ndata++;
				}
			}
		}
	//common initialization for parser tasks
	push_parser();		//make code reentrant
	init_table();		length = strlen(*par);
	//process parameters
	if(!(buffer = (char*)malloc(length+2))){
		clear_table();	pop_parser();
		if(arz) delete arz;
		free(y);	free(x);	delete arx;	delete ary;
		return 0;
		}
	strcpy(buffer, *par);	buffer[length++] = ';';
	buffer[length] = 0;	buff_pos = 0;
	tab1 = sym_table;
	do {
		yyparse();
		}while(buff_pos < length);
	tab2 = sym_table;	free(buffer);	buffer =0L;
	for(nparam = 0, csr=tab2; csr != tab1; nparam++, csr = csr->next);
	parsym = (symrec**)malloc((nparam+1)*sizeof(symrec*));
	parval = (double**)malloc((nparam+1)*sizeof(double*));
	for(i = 0, csr=tab2; csr != tab1 && i < nparam; i++, csr = csr->next){
		parsym[i] = csr;	parval[i] = &csr->value.var;
		}
	//do iteratations to optimize fit
	lista = (int*)malloc(sizeof(int)*nparam);
	for(i = 0; i< nparam; i++) lista[i] = i;
	covar = dmatrix(1, nparam, 1, nparam);
	alpha = dmatrix(1, nparam, 1, nparam);
	alamda = -1.0;		itst = 0;
	mrqmin(x, y, z, ndata, parval, nparam, lista, nparam, covar, alpha, &chisq, fcurve, &alamda);
	if(!Check_MRQerror()) {
		for(itst = itst1 = 0, ochisq = chisq; itst < maxiter && chisq > conv && ochisq >= chisq && itst1 < 9; itst++) {
			ochisq = chisq;
			mrqmin(x, y, z, ndata, parval, nparam, lista, nparam, covar, alpha, &chisq, fcurve, &alamda);
			if(ochisq == chisq) itst1++;
			else itst1 = 0;
			}
		alamda = 0.0;
		mrqmin(x, y, z, ndata, parval, nparam, lista, nparam, covar, alpha, &chisq, fcurve, &alamda);
		Check_MRQerror();
		}
	for(i = nparam-1, j = k = l = 0; i >= 0; l = 0, i--) {
		if(k > 20) {
			if(tmp_txt[j-1] == ' ') j--;
			if(tmp_txt[j-1] == ';') j--;
			l = sprintf(tmp_txt+j, "\n");
			j += l;		k = 0;
			}
		l += sprintf(tmp_txt+j, "%s%s=%g;", j && k ? " " : "", parsym[i]->name, parsym[i]->value.var);
		j += l;			k += l;
		}
	free(*par);	*par = strdup(tmp_txt);
	if(chi_2) *chi_2 = chisq;
	//write back spreadsheet data if necessary
	buffer = *par;	length = strlen(buffer);
	do {
		yyparse();
		}while(buff_pos < length);
	buffer = 0L;
	free_dmatrix(alpha, 1, nparam, 1, nparam);
	free_dmatrix(covar, 1, nparam, 1, nparam);
	if(arz) delete arz;		if(z) free(z);
	free(y);	free(x);	delete arx;	delete ary;
	if(parval) free(parval);	if(parsym) free(parsym);
	clear_table();
	pop_parser();
	if(curr_data){
		curr_data->Command(CMD_CLEAR_ERROR, 0L, 0L);
		curr_data->Command(CMD_REDRAW, 0L, 0L);
		}
	return itst < maxiter ? itst+1 : maxiter;
}






























