/*
 *  appointment-window.c -
 *
 *  Created: 20020221
 *
 *  Copyright (c) 2001, 2002 by Tomasz Trojanowski
 *  All rights reserved
 *
 *  $Id: appointment-window.c,v 1.38.2.2 2002/03/26 18:50:13 tomek Exp $
 *
 */

#ifdef HAVE_CONFIG
# include <config.h>
#endif

#include <appointment-window.h>

#include <time.h>
#include <stdlib.h>
#include <stdio.h>
#include <intl.h>
#include <common.h>
#include <about-dialog.h>
#include <confirm-dialog.h>
#include <windows-list.h>
#include <database.h>
#include <fields-list.h>
#include <contact-window.h>

/* Funkcja przepisujca rok rozpoczcia lub zakoczenia z okienka terminu do struktury
   danych terminu. */
static void copy_data_year(appointment_data *data, gboolean start_year)
{
    GtkWidget *entry;
    gchar *text;
    gchar *new_text;
    guint *year;
    guint new_year;

    /* Pobieranie danych z pola edycji roku. */
    entry = gtk_object_get_data(GTK_OBJECT(data->appointment_window), 
                                start_year ? "start_year" : "end_year");
    g_assert(entry != NULL);
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    year = start_year ? &data->start_year : &data->end_year;
    new_year = atoi(text);
    if (*year != new_year)
    {
        *year = new_year;
	data->modified = TRUE;
	
	create_days_list(data, start_year);
    }
    
    /* Poprawianie zawartoci pola edycji roku. */
    new_text = g_strdup_printf("%d", new_year);
    if (strcmp(text, new_text) != 0)
	gtk_entry_set_text(GTK_ENTRY(entry), new_text);
    g_free(new_text);
}

/* Funkcja przepisujca czas rozpoczcia lub zakoczenia z okienka terminu do struktury
   danych terminu. */
static void copy_data_time(appointment_data *data, gboolean start_time)
{
    GtkWidget *entry;
    gchar *text;
    gchar *new_text;
    guint *hour;
    guint *mins;
    guint new_hour;
    guint new_mins;
    
    /* Pobieranie danych z pola edycji pola wyboru czasu. */
    entry = gtk_object_get_data(GTK_OBJECT(data->appointment_window),
                                start_time ? "start_time" : "end_time");
    g_assert(entry != NULL);
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    hour = start_time ? &data->start_hour : &data->end_hour;
    mins = start_time ? &data->start_mins : &data->end_mins;
    sscanf(text, "%d:%d", &new_hour, &new_mins);
    if (new_hour >= 24)
        new_hour = 0;
    if (new_mins >= 60)
        new_mins = 0;
    if (*hour != new_hour || *mins != new_mins)
    {
        *hour = new_hour;
	*mins = new_mins;
	data->modified = TRUE;
    }
    
    /* Poprawiamie zawartoci pola edycji pola wyboru czasu. */
    new_text = g_strdup_printf("%02d:%02d", new_hour, new_mins);
    if (strcmp(text, new_text) != 0)
        gtk_entry_set_text(GTK_ENTRY(entry), new_text);
    g_free(new_text);
}

/* Funkcja przepisujaca dane z okienka terminu do struktury danych terminu. */
static void copy_data(appointment_data *data)
{
    GtkWidget *entry;
    gchar *text;
    guint length;

    g_assert(data != NULL);
    
    /* Przepisywanie danych elementw okna dialogowego do struktury danych terminu
       i sprawdzanie czy dane zostay zmodyfikowane, a jeeli tak - ustawianie flagi
       modified struktury danych terminu. */
    entry = gtk_object_get_data(GTK_OBJECT(data->appointment_window), "summary");
    g_assert(entry != NULL);
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->summary) != 0)
    {
        strcpy(data->summary, text);
	data->modified = TRUE;
    }

    entry = gtk_object_get_data(GTK_OBJECT(data->appointment_window), "location");
    g_assert(entry != NULL);
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->location) != 0)
    {
        strcpy(data->location, text);
	data->modified = TRUE;
    }
    
    copy_data_year(data, TRUE);
    copy_data_year(data, FALSE);
    
    copy_data_time(data, TRUE);
    copy_data_time(data, FALSE);
    
    entry = gtk_object_get_data(GTK_OBJECT(data->appointment_window), "description");
    g_assert(entry != NULL);
    length = gtk_text_get_length(GTK_TEXT(entry));
    if (length > MAX_DESCRIPTION)
        length = MAX_DESCRIPTION;
    text = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, length);
    if (strcmp(text, data->description) != 0)
    {
        strcpy(data->description, text);
	data->modified = TRUE;
    }
    g_free(text);
}

/* Funkcja wstawiajca dane ze zmiennej zawierajacej dane terminu do bazy. */
static void insert_data(appointment_data *data)
{
    g_assert(data != NULL);
    
    /* Jeeli identyfikator terminu jest rwny 0, oznacza to e jest tworzony nowy termin,
       w przeciwnym wypadku - uaktualniany istniejcy */
    if (data->id == 0)
        /* Zapisanie danych terminu do bazy danych. */
        db_appointment_add(data);
    else
        /* Uaktualnienie danych terminu w bazie danych. */
	db_appointment_update(data);

    /* Wywoanie funkcji uaktualniajcej list terminw w gwnym oknie programu. */
    show_fields_list(LT_APPOINTMENT, FALSE);

    /* Zerowanie flagi modified struktury danych terminu. */
    data->modified = FALSE;
}

/* Funkcja obsugujca menu Plik-Nowy-Termin otwierajca nowe okno terminu. */
void new_appointment_menu(GtkWidget *widget, gpointer data)
{
    create_appointment_window(0);
}

/* Funkcja obsugujca menu Plik-Zapisz okienka terminw. */
static void save_menu(GtkWidget *widget, appointment_data *data)
{
    g_assert(data != NULL);
    
    /* Kopiowanie danych terminu do struktury. */
    copy_data(data);
    
    if (data->modified)
        /* Zapisanie danych do bazy. */
	insert_data(data);
}

/* Funkcja obsugujca menu Plik-Usu. */
static void delete_menu(GtkWidget *widget, appointment_data *data)
{
    gboolean may_delete = TRUE;

    g_assert(data != NULL);
    
    /* Sprawdzenie czy jest konieczne wywoanie okienka potwierdzajacego. */
    if (data->modified || data->id != 0)
    {
        /* Wywoanie okienka potwierdzajacego. */
	if (G_CONFIRM_YES != confirm_dialog(data->appointment_window,
	                                    _("Are you sure you want to delete this appointment?"),
					    G_CONFIRM_YES | G_CONFIRM_NO))
	    may_delete = FALSE;
    }
    
    if (may_delete)
    {
        if (data->id != 0)
	{
	    /* Usuwanie kontaktu z bazy. */
	    db_appointment_delete(data->id);
	    
	    /* Uaktualnianie listy kontaktw. */
	    show_fields_list(LT_APPOINTMENT, FALSE);
	}
	
	/* Usunicie okienka z listy terminw. */
	remove_appointment_window(data->appointment_window);
	
	/* Zamknicie okienka terminu. */
	gtk_widget_destroy(data->appointment_window);
	
	/* Zwolnienie pamici zajmowanej przez dane terminu. */
	g_free(data);
    }
}

/* Funkcja obsugujca menu Plik-Zamknij okienka terminu. */
static void close_menu(GtkWidget *widget, appointment_data *data)
{
    close_appointment_window(data, TRUE);
}

/* Funkcja obsugujca sygna "delete_event" okienka terminu. */
static gint wnd_delete_event(GtkWidget *widget, GdkEvent *event, appointment_data *data)
{
    return (!close_appointment_window(data, TRUE));
}

/* Funkcja przepisujca miesic rozpoczcia lub zakoczenia z okienka terminu do struktury
   danych terminu. */
static void months_select_child(GtkList          *list,
                                GtkWidget        *widget,
				appointment_data *data,
				gboolean         start_month)
{
    gint *month;
    gint new_month;

    g_assert(list != NULL);
    g_assert(widget != NULL);
    g_assert(data != NULL);
    
    month = start_month ? &data->start_month : &data->end_month;
    new_month = gtk_list_child_position(list, widget) + 1;

    if (*month != new_month)
    {
        *month = new_month;
	data->modified = TRUE;
	
	/* Uaktualnienie listy pola wyboru dni. */
	create_days_list(data, start_month);
    }
}

/* Funkcja obsugujca sygna "select_child" listy pola wyboru miesica daty pocztkowej. */
static void start_months_select_child(GtkList *list, GtkWidget *widget, appointment_data *data)
{
    months_select_child(list, widget, data, TRUE);
}

/* Funkcja obsugujca sygna "select_child" listy pola wyboru miesica daty kocowej. */
static void end_months_select_child(GtkList *list, GtkWidget *widget, appointment_data *data)
{
    months_select_child(list, widget, data, FALSE);
}

/* Funkcja przepisujca dzie rozpoczcia lub zakoczenia z okienka terminu do struktury
   danych terminu. */
static void days_select_child(GtkList          *list,
                              GtkWidget        *widget,
			      appointment_data *data,
			      gboolean         start_day)
{
    gint *day;
    gint new_day;

    g_assert(list != NULL);
    g_assert(widget != NULL);
    g_assert(data != NULL);
    
    day = start_day ? &data->start_day : &data->end_day;
    new_day = gtk_list_child_position(list, widget) + 1;

    if (*day != new_day)
    {
        *day = new_day;
	data->modified = TRUE;
    }
}

/* Funkcja obsugujca sygna "select_child" listy pola wyboru dnia daty pocztkowej. */
static void start_days_select_child(GtkList *list, GtkWidget *widget, appointment_data *data)
{
    days_select_child(list, widget, data, TRUE);
}

/* Funkcja obsugujca sygna "select_child" listy pola wyboru dnia daty kocowej. */
static void end_days_select_child(GtkList *list, GtkWidget *widget, appointment_data *data)
{
    days_select_child(list, widget, data, FALSE);
}

/* Funkcja obsugujca sygna "focus_out_event" pola edycji roku daty pocztkowej. */
static void start_year_focus_out(GtkEntry *entry, GdkEventFocus *event, appointment_data *data)
{
    g_assert(data != NULL);
    
    copy_data_year(data, TRUE);
}

/* Funkcja obsugujca sygna "focus_out_event" pola edycji roku daty kocowej. */
static void end_year_focus_out(GtkEntry *entry, GdkEventFocus *event, appointment_data *data)
{
    g_assert(data != NULL);
    
    copy_data_year(data, FALSE);
}

/* Funkcja obsugujca sygna "focus_out_event" pola edycji pola wyboru czasu rozpoczcia. */
static void start_time_focus_out(GtkEntry *entry, GdkEventFocus *event, appointment_data *data)
{
    g_assert(data != NULL);
    
    copy_data_time(data, TRUE);
}

/* Funkcja obsugujca sygna "focus_out_event" pola edycji pola wyboru czasu zakoczenia. */
static void end_time_focus_out(GtkEntry *entry, GdkEventFocus *event, appointment_data *data)
{
    g_assert(data != NULL);
    
    copy_data_time(data, FALSE);
}

static char *months_list_items[] = 
{
    N_("January"),
    N_("February"),
    N_("March"),
    N_("April"),
    N_("May"),
    N_("June"),
    N_("July"),
    N_("August"),
    N_("September"),
    N_("October"),
    N_("November"),
    N_("December")
};

/* Funkcja tworzca pola wprowadzania daty rozpoczcia i zakoczenia.*/
static void create_start_end_date_entries(GtkWidget *fixed, appointment_data *data)
{
    gint no = 0;
    
    for (no = 0; no < 2; no++)
    {
        GtkWidget *label;
	GtkWidget *combo;
	GtkWidget *entry;
	GtkWidget *list;
	
	gchar *text;
	
	GList *combo_items = NULL;
	int i;
    
        int y = (no == 0) ? 80 : 112;
    
        /* Etykieta "Data rozpoczcia", "Data zakoczenia". */
        label = gtk_label_new((no == 0) ? _("Start time:") : _("End time:"));
        gtk_widget_show(label);
    
        gtk_fixed_put(GTK_FIXED(fixed), label, 8, y);
        gtk_widget_set_uposition(label, 8, y);
        gtk_widget_set_usize(label, 120, 24);
        gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    
        /* Pole wyboru miesica. */
        combo = gtk_combo_new();
        gtk_widget_show(combo);
	
        gtk_fixed_put(GTK_FIXED(fixed), combo, 136, y);
        gtk_widget_set_uposition(combo, 136, y);
        gtk_widget_set_usize(combo, 96, 24);
	gtk_combo_set_value_in_list(GTK_COMBO(combo), TRUE, FALSE);
    
            /* Tworzenie listy elementw pola wyboru miesica. */
	    for (i = 0; i < 12; i++)
	        combo_items = g_list_append(combo_items, (gpointer)gettext(months_list_items[i]));
	    gtk_combo_set_popdown_strings(GTK_COMBO(combo), combo_items);
	    g_list_free(combo_items);
	
	    /* Pole edycji pola wyboru miesiaca. */
	    entry = GTK_COMBO(combo)->entry;
	    gtk_widget_show(entry);
	    
	    gtk_entry_set_text(GTK_ENTRY(entry),
	        gettext(months_list_items[((no == 0) ? data->start_month : data->end_month) - 1]));
	    gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);
	    
	    /* Lista pola wyboru. */
	    list = GTK_COMBO(combo)->list;
	    gtk_widget_show(list);
 
	    gtk_signal_connect(GTK_OBJECT(list), "select_child",
	        GTK_SIGNAL_FUNC((no == 0) ? start_months_select_child : end_months_select_child),
		data);

        /* Pole wyboru dnia. */
        combo = gtk_combo_new();
        gtk_widget_show(combo);

	gtk_object_set_data(GTK_OBJECT(data->appointment_window),
	                    (no == 0) ? "start_day_combo" : "end_day_combo", combo);
    
        gtk_fixed_put(GTK_FIXED(fixed), combo, 240, y);
        gtk_widget_set_uposition(combo, 240, y);
        gtk_widget_set_usize(combo, 48, 24);
	gtk_combo_set_value_in_list(GTK_COMBO(combo), TRUE, FALSE);
    
            /* Tworzenie listy elementw pola wyboru dnia rozpoczcia. */
	    create_days_list(data, (no == 0));

	    /* Pole edycji pola wyboru dnia rozpoczcia. */
	    entry = GTK_COMBO(combo)->entry;
	    gtk_widget_show(entry);

	    gtk_entry_set_editable(GTK_ENTRY(entry), FALSE);

	    /* Lista pola wyboru. */
	    list = GTK_COMBO(combo)->list;
	    gtk_widget_show(list);
 
	    gtk_signal_connect(GTK_OBJECT(list), "select_child",
	        GTK_SIGNAL_FUNC((no == 0) ? start_days_select_child : end_days_select_child),
		data);
    
        /* Pole edycji roku daty rozpoczcia. */
        entry = gtk_entry_new();
        gtk_widget_show(entry);

	gtk_signal_connect(GTK_OBJECT(entry), "focus_out_event",
	                   GTK_SIGNAL_FUNC((no == 0) ? start_year_focus_out : end_year_focus_out),
			   data);
	    
	gtk_object_set_data(GTK_OBJECT(data->appointment_window),
	                    (no == 0) ? "start_year" : "end_year", entry);

        gtk_fixed_put(GTK_FIXED(fixed), entry, 296, y);
        gtk_widget_set_uposition(entry, 296, y);
        gtk_widget_set_usize(entry, 40, 24);

        gtk_entry_set_text(GTK_ENTRY(entry),
	                   g_strdup_printf("%d", (no == 0) ? data->start_year : data->end_year));
    
        /* Pole wyboru godziny rozpoczcia. */
        combo = gtk_combo_new();
        gtk_widget_show(combo);
    
        gtk_fixed_put(GTK_FIXED(fixed), combo, 344, y);
        gtk_widget_set_uposition(combo, 344, y);
        gtk_widget_set_usize(combo, 64, 24);
    
            /* Tworzenie listy elementw pola wyboru godziny rozpoczcia. */
	    combo_items = NULL;
	    for (i = 0; i <= 23; i++)
	    {
	        combo_items = g_list_append(combo_items, (gpointer)g_strdup_printf("%02d:00", i));
	        combo_items = g_list_append(combo_items, (gpointer)g_strdup_printf("%02d:30", i));
	    }
	    gtk_combo_set_popdown_strings(GTK_COMBO(combo), combo_items);
	    g_list_foreach(combo_items, free_list_item_callback, NULL);
	    g_list_free(combo_items);
	
	    /* Pole edycji pola wyboru godziny rozpoczcia. */
	    entry = GTK_COMBO(combo)->entry;
	    gtk_widget_show(entry);

	    gtk_signal_connect(GTK_OBJECT(entry), "focus_out_event",
	                       GTK_SIGNAL_FUNC((no == 0) ? start_time_focus_out : end_time_focus_out),
			       data);
	    
	    gtk_object_set_data(GTK_OBJECT(data->appointment_window),
	                        (no == 0) ? "start_time" : "end_time", entry);

            text = g_strdup_printf("%02d:%02d", (no == 0) ? data->start_hour : data->end_hour,
				   (no == 0) ? data->start_mins : data->end_mins);
	    gtk_entry_set_text(GTK_ENTRY(entry), text);
	    g_free(text);
    }
}

/* Struktura zawierajca menu okienka terminw. */
static GtkItemFactoryEntry appointment_menu[] =
{
    { NULL, NULL, NULL,                 0, "<Branch>" },      /*  0 */
    { NULL, NULL, NULL,                 0, "<Branch>" },      /*  1 */
    { NULL, NULL, new_appointment_menu, 0, NULL },            /*  2 */
    { NULL, NULL, NULL,                 0, "<Separator>" },   /*  3 */
    { NULL, NULL, new_contact_menu,     0, NULL },            /*  4 */
    { NULL, NULL, NULL,                 0, "<Separator>" },   /*  5 */
    { NULL, NULL, save_menu,            0, NULL },            /*  6 */
    { NULL, NULL, NULL,                 0, "<Separator>" },   /*  7 */
    { NULL, NULL, delete_menu,          0, NULL },            /*  8 */
    { NULL, NULL, NULL,                 0, "<Separator>" },   /*  9 */
    { NULL, NULL, close_menu,           0, NULL },            /* 10 */
    { NULL, NULL, NULL,                 0, "<Branch>" },      /* 11 */
    { NULL, NULL, new_appointment_menu, 0, NULL },            /* 12 */
    { NULL, NULL, NULL,                 0, "<LastBranch>" },  /* 13 */
    { NULL, NULL, about_menu,           0, NULL }             /* 14 */
};

/* Funkcja tworzca okno terminw.

   Struktura okna:
   
     GtkWindow (appointment_window)
      +- GtkVBox (main_vbox)
          +- GtkMenuBar (menubar)
	  +- GtkNotebook (notebook)
	  |   +- GtkFixed                                            "Oglne"
	  |   |   +- GtkLabel                                         "Temat"
	  |   |   +- GtkEntry ("summary")
	  |   |   +- GtkLabel                                   "Lokalizacja"
	  |   |   +- GtkEntry ("location")
	  |   |   +- GtkHSeparator
	  |   |   +- GtkLabel                              "Data rozpoczcia"
	  |   |   +- GtkCombo
	  |   |   |   +- GtkEntry
	  |   |   +- GtkCombo ("start_day_combo")
	  |   |   |   +- GtkEntry
	  |   |   +- GtkEntry ("start_year")
	  |   |   +- GtkCombo
	  |   |   |   +- GtkEntry ("start_time"
	  |   |   +- GtkLabel                              "Data zakoczenia"
	  |   |   +- GtkCombo
	  |   |   |   +- GtkEntry
	  |   |   +- GtkCombo ("end_day_combo")
	  |   |   |   +- GtkEntry
	  |   |   +- GtkEntry ("end_year")
	  |   |   +- GtkCombo
	  |   |       +- GtkEntry ("end_time")
	  |   +- GtkLabel
	  |   +- GtkFixed                                              "Opis"
	  |   |   +- GtkText
	  |   +- GtkLabel
	  +- GtkStatusBar (statusbar)

   Klucze zwizane z okienkiem terminw:
   
     summary         - wskanik do pola edycji tytuu terminu
     location        - wskanik do pola edycji lokalizacji terminu
     start_day_combo - wskanik do pola wyboru dnia daty rozpoczcia
     start_year      - wskanik do pola edycji roku daty rozpoczcia
     start_time      - wskanik do pola edycji pola wybory czasu rozpoczcia
     end_day_combo   - wskanik do pola wyboru dnia daty zakoczenia
     end_year        - wskanik do pola edycji roku daty zakoczenia
     end_time        - wskanik do pola edycji pola wyboru czasu rozpoczcia
     description     - wskanik do pola tekstowego opisu terminu
   
*/
void create_appointment_window(gint id)
{
    GtkWidget *appointment_window;
    GtkWidget *main_vbox;
    GtkWidget *menubar;
    GtkWidget *statusbar;
    GtkWidget *notebook;
    GtkWidget *fixed;
    GtkWidget *label;
    GtkWidget *entry;
    GtkWidget *hseparator;
    GtkWidget *text;
    
    GtkItemFactory *item_factory;
    GtkAccelGroup *accel_group;
    
    appointment_data *data = NULL;
    
    /* Sprawdzanie czy istnieje ju okienko przechowujace termin o zadanym identyfikatorze,
       i jeeli tak - uaktywnienie go. */
    if (id != 0)
    {
        appointment_window = find_appointment_window_id(id);
	if (appointment_window != NULL)
	{
	    gtk_widget_hide(appointment_window);
	    gtk_widget_show(appointment_window);
	    return;
	}
    }
    
    /* Utworzenie okienka terminu. */
    appointment_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    
    /* Dodanie okienka do listy okienek terminw. */
    add_appointment_window(appointment_window);

    /* Utworzenie zmiennej zawierajacej dane okna terminu. */
    data = (appointment_data *)g_malloc0(sizeof(appointment_data));
    
    /* Wypenienie zmiennej zawierajcej dane okna kontaktw. */
    data->appointment_window = appointment_window;
    
    if (id != 0)
        db_appointment_get(data, id);
    else
    {
        time_t t;
        struct tm *current_time;
	
	time(&t);
	current_time = localtime(&t);
	
	data->start_year = data->end_year = current_time->tm_year + 1900;
	data->start_month = data->end_month = current_time->tm_mon + 1;
	data->start_day = data->end_day = current_time->tm_mday;
	data->start_hour = data->end_hour = current_time->tm_hour;
	data->end_mins = 30;
    }
    
    /* Tworzenie klucza "data" przechowujacego adres struktury danych kontaktu. */
    gtk_object_set_data(GTK_OBJECT(appointment_window), "data", data);
    
    /* Przypisanie funkcji obsugi sygnau "delete_event" do okienka kontaktu. */
    gtk_signal_connect(GTK_OBJECT(appointment_window), "delete_event",
                       GTK_SIGNAL_FUNC(wnd_delete_event), data);
    
    /* Ustalanie tytuu i rozmiarw okna. */
    gtk_window_set_title(GTK_WINDOW(appointment_window), _("Appointment"));
    gtk_widget_set_usize(GTK_WIDGET(appointment_window), 422, 224);
    
        /* Obszar zawartoci okna terminu. */
	main_vbox = gtk_vbox_new(FALSE, 1);
	gtk_widget_show(main_vbox);
	
	gtk_container_border_width(GTK_CONTAINER(main_vbox), 1);
	gtk_container_add(GTK_CONTAINER(appointment_window), main_vbox);
	
	    /* Uzupenienie struktury zawierajacej menu okienka terminw. */
	    appointment_menu[6].callback_action = (guint)data;
	    appointment_menu[8].callback_action = (guint)data;
	    appointment_menu[10].callback_action = (guint)data;
	    appointment_menu[14].callback_action = (guint)appointment_window;
	
	    /* Tworzenie menu okna terminu. */
	    accel_group = gtk_accel_group_new();
	    
	    item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", accel_group);
	    gtk_item_factory_create_items(item_factory,
	                                  sizeof(appointment_menu) / sizeof(appointment_menu[0]),
					  appointment_menu, NULL);
	    gtk_window_add_accel_group(GTK_WINDOW(appointment_window), accel_group);
	    menubar = gtk_item_factory_get_widget(item_factory, "<main>");
	    
	    gtk_widget_show(menubar);
	    
	    gtk_box_pack_start(GTK_BOX(main_vbox), menubar, FALSE, FALSE, 0);
	    
	    /* Tworzenie gwnej czci okna. */
	    notebook = gtk_notebook_new();
	    gtk_widget_show(notebook);
	    
	    gtk_box_pack_start(GTK_BOX(main_vbox), notebook, TRUE, TRUE, 0);
	    
	        /* Tworzenie zawartoci pierwszej zakadki. */
		fixed = gtk_fixed_new();
		gtk_widget_show(fixed);
		
		    /* Etykieta "Temat". */
		    label = gtk_label_new(_("Summary:"));
		    gtk_widget_show(label);
		   
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 8);
		    gtk_widget_set_uposition(label, 8, 8);
		    gtk_widget_set_usize(label, 120, 24);
		    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
		   
		    /* Pole edycji "Temat". */
		    entry = gtk_entry_new_with_max_length(MAX_SUMMARY);
		    gtk_widget_show(entry);
		    
		    gtk_object_set_data(GTK_OBJECT(appointment_window), "summary", entry);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), entry, 136, 8);
		    gtk_widget_set_uposition(entry, 136, 8);
		    gtk_widget_set_usize(entry, 272, 24);
		    gtk_entry_set_text(GTK_ENTRY(entry), data->summary);
		    
		    /* Etykieta "Lokalizacja". */
		    label = gtk_label_new(_("Location:"));
		    gtk_widget_show(label);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 40);
		    gtk_widget_set_uposition(label, 8, 40);
		    gtk_widget_set_usize(label, 120, 24);
		    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
		    
		    /* Pole edycji "Lokalizacja". */
		    entry = gtk_entry_new_with_max_length(MAX_LOCATION);
		    gtk_widget_show(entry);
		    
		    gtk_object_set_data(GTK_OBJECT(appointment_window), "location", entry);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), entry, 136, 40);
		    gtk_widget_set_uposition(entry, 136, 40);
		    gtk_widget_set_usize(entry, 272, 24);
		    gtk_entry_set_text(GTK_ENTRY(entry), data->location);
		    
		    /* Linia oddzielajca. */
		    hseparator = gtk_hseparator_new();
		    gtk_widget_show(hseparator);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), hseparator, 8, 64);
		    gtk_widget_set_uposition(hseparator, 8, 64);
		    gtk_widget_set_usize(hseparator, 400, 16);
		    
		    /* Tworzenie pl wprowadzania daty rozpoczcia i zakoczenia. */
		    create_start_end_date_entries(fixed, data);
		    
		/* Tworzenie tytuu pierwszej zakadki. */
		label = gtk_label_new(_("General"));
		gtk_widget_show(label);
		
	    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), fixed, label);
	    
	        /* Tworzenie zawartoci drugiej zakadki. */
		fixed = gtk_fixed_new();
		gtk_widget_show(fixed);
		
		    /* Pole tekstowe "Opis". */
		    text = gtk_text_new(NULL, NULL);
		    gtk_widget_show(text);
		    
		    gtk_object_set_data(GTK_OBJECT(appointment_window), "description", text);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), text, 8, 8);
		    gtk_widget_set_uposition(text, 8, 8);
		    gtk_widget_set_usize(text, 400, 128);
		    gtk_text_set_editable(GTK_TEXT(text), TRUE);
		    gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, data->description,
		                    strlen(data->description));
		
		/* Tworzenie tytuu drugiej zakadki. */
		label = gtk_label_new(_("Description"));
		gtk_widget_show(label);
		
	    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), fixed, label);
	    
	    /* Tworzenie linii statusu. */
	    statusbar = gtk_statusbar_new();
	    gtk_widget_show(statusbar);
	    
	    gtk_box_pack_start(GTK_BOX(main_vbox), statusbar, FALSE, FALSE, 0);
    
    /* Wywietlanie okienka kontaktw. */
    gtk_widget_show(appointment_window);
}

/* Funkcja kopiujca dane terminu do struktury i wstawiajca je, o ile jest to konieczne, do
   bazy danych, a nastpnie zamykajca okienko terminu. Zwraca TRUE jeeli okienko zostao
   zamknite. */
gint close_appointment_window(appointment_data *data, gboolean remove_from_list)
{
    g_assert(data != NULL);
    
    /* Kopiowanie danych terminu do struktury. */
    copy_data(data);
    
    /* Sprawdzanie czy jest konieczne wywoywanie okienka potwierdzajcego. */
    if (data->modified)
    {
        /* Wywoanie okienka potwierdzajcego. */
	switch (confirm_dialog(data->appointment_window, _("Do you want to save changes?"),
	                       G_CONFIRM_YES | G_CONFIRM_NO | G_CONFIRM_CANCEL))
	{
	    case G_CONFIRM_YES:
	        /* Wstawianie danych do bazy. */
		insert_data(data);
	    
	    case G_CONFIRM_NO:
                /* Usunicie okienka z listy okienek terminw. */
                if (remove_from_list)
                    remove_appointment_window(data->appointment_window);
    
                /* Zamknicie okienka terminu. */
                gtk_widget_destroy(data->appointment_window);
    
                /* Zwolnienie pamici zajmowanej przez dane terminu. */
                g_free(data);
		return TRUE;
	    default:
	        g_assert_not_reached();
	}
    }
    else
    {
        /* Usunicie okienka z listy okienek terminw. */
        if (remove_from_list)
            remove_appointment_window(data->appointment_window);
    
        /* Zamknicie okienka terminu. */
        gtk_widget_destroy(data->appointment_window);
    
        /* Zwolnienie pamici zajmowanej przez dane terminu. */
        g_free(data);
        return TRUE;
    }
    
    return FALSE;
}    

/* Funkcja inicjalizujca menu okienka terminw. */
void init_gettext_appointment_window()
{
    appointment_menu[0].path = _("/_File");
    appointment_menu[1].path = _("/File/_New");
    appointment_menu[2].path = _("/File/New/_Appointment");
    appointment_menu[3].path = _("/File/New/sep1");
    appointment_menu[4].path = _("/File/New/_Contact");
    appointment_menu[5].path = _("/File/sep2");
    appointment_menu[6].path = _("/File/_Save");
    appointment_menu[7].path = _("/File/sep3");
    appointment_menu[8].path = _("/File/_Delete");
    appointment_menu[9].path = _("/File/sep4");
    appointment_menu[10].path = _("/File/_Close");
    appointment_menu[11].path = _("/_Appointment");
    appointment_menu[12].path = _("/Appointment/_New appointment");
    appointment_menu[13].path = _("/_Help");
    appointment_menu[14].path = _("/Help/_About");
}
