/***************************************************************************
		   gui_playlist.cpp  -  description
			 -------------------
begin                : Thu Mar 7 2002
copyright            : (C) 2001 by Holger Sattel
email                : hsattel@rumms.uni-mannheim.de

This file contributed by Tim Lee ;)
****************************************************************************/

/***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
**************************************************************************/

#include "gui_playlist.h"

#include <qlayout.h>
#include <qframe.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qptrlist.h>
#include <qlistview.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qtooltip.h>

#include <libintl.h>
#define _(String) gettext (String)

#include "gui.h"

#include "jobmanager.h"
#include "job_modifyplaylists.h"
#include "job_modifyplaylist_tracks.h"
#include "job_queryplaylist_tracks.h"
#include "job_querydatabase.h"
#include "job_generateplaylist.h"
#include "database.h"

#include <iostream>

GUI_Playlist::GUI_Playlist(QWidget *parent, const char *name)
  : QTabWidget(parent, name), firsttime( true )
{

//  setFixedWidth(160);
//    setMinimumWidth(160);
//    setFixedWidth(220);

  vsplit = new QSplitter(QSplitter::Vertical, this);
  vsplit->setOpaqueResize( true );

  pl_selframe = new QFrame( vsplit);
  QVBoxLayout *pl_selblay = new QVBoxLayout(pl_selframe);

//  QGroupBox *pl_box = new QGroupBox(1, Qt::Vertical, "Playlists", pl_selframe);
//  QFrame *pl_box_frame = new QFrame(pl_box);
//  QVBoxLayout *pl_box_blay = new QVBoxLayout(pl_box_frame);
  playlist_list = new ExtendedDropListView(pl_selframe);
  char* temps = _( "Change playlists\nRight click to play.");
  QToolTip::add( playlist_list, temps );
  QToolTip::add( playlist_list->viewport(), temps );
  playlist_list->addColumn(_("Playlists"),220);
  playlist_list->setItemMargin(1);
  playlist_list->setSelectionMode(QListView::Single);
  playlist_list->setSorting(0);
  playlist_list->setHScrollBarMode(QScrollView::AlwaysOff);
  playlist_list->setColumnWidthMode(0,QListView::Manual);
  playlist_list->setShowToolTips(true);
  playlist_list->setAcceptDrops( true );

  connect(playlist_list, SIGNAL(selectionChanged(QListViewItem*)),
          this, SLOT(slot_playlistSelectionChanged(QListViewItem*)));
  connect(playlist_list, SIGNAL(doubleClicked(QListViewItem*)),
          this, SLOT(slot_playlistDoubleClicked(QListViewItem*)));
  connect(playlist_list, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
          this, SLOT(slot_contextMenu(QListViewItem*, const QPoint&, int)));

//  pl_box_blay->addWidget(playlist_list,1);

  pl_selblay->addWidget(playlist_list,1);

  QFrame *pl_butframe = new QFrame(pl_selframe);
  QGridLayout *pl_butgrid = new QGridLayout(pl_butframe, 1, 3);
  buttonNew = new QPushButton(_("New"), pl_butframe);
  buttonNew->setMinimumWidth(20);
  pl_butgrid->addWidget(buttonNew, 0, 0);
  buttonRename = new QPushButton(_("Rename"), pl_butframe);
  buttonRename->setMinimumWidth(20);
  pl_butgrid->addWidget(buttonRename, 0, 1);
  buttonRemove = new QPushButton(_("Remove"), pl_butframe);
  buttonRemove->setMinimumWidth(20);
  pl_butgrid->addWidget(buttonRemove, 0, 2);

  connect(buttonNew, SIGNAL(clicked()), this, SLOT(slot_newButton()));
  connect(buttonRename, SIGNAL(clicked()), this, SLOT(slot_renameButton()));
  connect(buttonRemove, SIGNAL(clicked()), this, SLOT(slot_removeButton()));

  pl_selblay->addWidget(pl_butframe);


  QFrame *pl_selframe_low = new QFrame(vsplit);
  QVBoxLayout *pl_selblay_low = new QVBoxLayout(pl_selframe_low);

  tracklist = new ExtendedDropListView(pl_selframe_low);
  tracklist->addColumn(_("Tracks in Playlist"),220);
  tracklist->setItemMargin(0);
  
//  tracklist->addColumn("Artist",120);
//  tracklist->addColumn("Title",160);

//  tracklist->setAllColumnsShowFocus(true);
  tracklist->setSelectionMode(QListView::Extended);
  tracklist->setSorting(-1,true);
  tracklist->setHScrollBarMode(QScrollView::AlwaysOff);
  tracklist->setColumnWidthMode(0,QListView::Manual);
  tracklist->setShowToolTips(true);
  tracklist->setAcceptDrops( true );

  connect(tracklist, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
          this, SLOT(slot_contextMenu(QListViewItem*, const QPoint&, int)));
  connect(tracklist, SIGNAL(selectionChanged()),
	  this, SLOT(slot_trackSelectionChanged()));
  connect(tracklist, SIGNAL(doubleClicked(QListViewItem*)),
	  this, SLOT(slot_playlistTrackDoubleClicked(QListViewItem*)));

  pl_selblay_low->addWidget(tracklist,1);

  QFrame *pl_titframe = new QFrame(pl_selframe_low);
  QGridLayout *pl_titgrid = new QGridLayout(pl_titframe, 1, 3);
  QPushButton *buttonUp = new QPushButton(_("Up"), pl_titframe);
  buttonUp->setMinimumWidth(20);
  pl_titgrid->addWidget(buttonUp, 0, 0);
  QPushButton *buttonDown = new QPushButton(_("Down"), pl_titframe);
  buttonDown->setMinimumWidth(20);
  pl_titgrid->addWidget(buttonDown, 0, 1);
  QPushButton *buttonSave = new QPushButton(_("Save"), pl_titframe);
  buttonSave->setMinimumWidth(20);
  pl_titgrid->addWidget(buttonSave, 0, 2);

	connect(buttonUp, SIGNAL(clicked()), this, SLOT(slot_movePlaylistTrackUp()));
	connect(buttonDown, SIGNAL(clicked()), this, SLOT(slot_movePlaylistTrackDown()));
	connect(buttonSave, SIGNAL(clicked()), this, SLOT(slot_savePlaylistTracks()));

  pl_selblay_low->addWidget(pl_titframe);

  current_tracklist = new QListView();
  current_tracklist->addColumn("#",-1);
  current_tracklist->addColumn(_("Artist"),120);
  current_tracklist->addColumn(_("Title"),160);
  current_tracklist->setAllColumnsShowFocus(true);
  current_tracklist->setSelectionMode(QListView::Extended);
  current_tracklist->setSorting(-1,true);

  connect(current_tracklist, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
	  this, SLOT(slot_contextMenu(QListViewItem*, const QPoint&, int)));
  connect(current_tracklist, SIGNAL(selectionChanged()),
	  this, SLOT(slot_trackSelectionChanged()));

  tabGenerator = new QFrame(this);
  QToolTip::add( tabGenerator, _( "Generate playlists by randomily picking tracks,\n within user-defined constraints" ) );
  
  QGridLayout *generatorLay = new QGridLayout(tabGenerator, 6, 2);
  int row = 0;
  
  QLabel *labelMinTrackLength = new QLabel(QString(_("Min Track Length")) + ": ", tabGenerator);
  labelMinTrackLength->setTextFormat( Qt::RichText );
  generatorLay->addWidget(labelMinTrackLength, row, 0);
  generatorSpinMinTrackLength = new QSpinBox(0, 600, 10, tabGenerator);
  generatorSpinMinTrackLength->setButtonSymbols(QSpinBox::PlusMinus);
  generatorLay->addWidget(generatorSpinMinTrackLength, row++, 1);

  QLabel *labelMaxTrackLength = new QLabel(QString(_("Max Track Length")) + ": ", tabGenerator);
  labelMaxTrackLength->setTextFormat( Qt::RichText );
  generatorLay->addWidget(labelMaxTrackLength, row, 0);
  generatorSpinMaxTrackLength = new QSpinBox(0, 600, 10, tabGenerator);
  generatorSpinMaxTrackLength->setButtonSymbols(QSpinBox::PlusMinus);
  generatorLay->addWidget(generatorSpinMaxTrackLength, row++, 1);

  /******************************** work in progress....
  QLabel *labelArtists = new QLabel(QString(_("Artists")) + ": ", tabGenerator);
  generatorLay->addWidget(labelArtists, row, 0);
  generatorComboArtists = new QComboBox(false, tabGenerator);
  generatorLay->addWidget(generatorComboArtists, row++, 1);
  *********************************/

  QLabel *labTracks = new QLabel(QString(_("Number of Tracks")) + ": ", tabGenerator);
  labTracks->setTextFormat( Qt::RichText );
  generatorLay->addWidget(labTracks, row, 0);
  generatorSpinTracks = new QSpinBox(10, 100, 10, tabGenerator);
  generatorSpinTracks->setButtonSymbols(QSpinBox::PlusMinus);
  generatorLay->addWidget(generatorSpinTracks, row++, 1);

  QLabel *labFavouriteArtists = new QLabel(QString(_("Only fav. Artists")) + ": ", tabGenerator);
  labFavouriteArtists->setTextFormat( Qt::RichText );
  generatorLay->addWidget(labFavouriteArtists, row, 0);
  genaratorCheckBoxOnlyFavouriteArtists = new QCheckBox(tabGenerator);    
  generatorLay->addWidget(genaratorCheckBoxOnlyFavouriteArtists, row++, 1);

  QPushButton *buttonReset = new QPushButton(_("Reset"), tabGenerator);
  generatorLay->addWidget(buttonReset, row, 0);
  buttonReset->setMinimumWidth(20);

  QPushButton *buttonGenerate = new QPushButton(_("Generate"), tabGenerator);
  generatorLay->addWidget(buttonGenerate, row++, 1);
  buttonGenerate->setMinimumWidth(20);

  QLabel *labMedium = new QLabel(QString(_("Only tracks on hardisk are supported!")) , tabGenerator);
  labMedium->setTextFormat( Qt::RichText );
  generatorLay->addMultiCellWidget(labMedium, row++, row, 0, 1);

  slot_resetButton();
  connect(generatorSpinMinTrackLength, SIGNAL(valueChanged(int)), this, SLOT(slot_generatorMinTrackLength(int)));
  connect(generatorSpinMaxTrackLength, SIGNAL(valueChanged(int)), this, SLOT(slot_generatorMaxTrackLength(int)));
  connect(buttonReset, SIGNAL(clicked()), this, SLOT(slot_resetButton()));
  connect(buttonGenerate, SIGNAL(clicked()), this, SLOT(slot_generateButton()));

  addTab(vsplit, _("Editor"));
  addTab(tabGenerator, _("Generator"));

  connect(this, SIGNAL(currentChanged(QWidget*)), this, SLOT(slot_tabChange(QWidget*)));

  displayed_playlist = NULL;
}

GUI_Playlist::~GUI_Playlist()
{
}

void GUI_Playlist::setConnectionState(bool state)
{
  isConnected = state;
  if(!isConnected) clear();
}

void GUI_Playlist::notifyNewPlaylistBasis(QList<PLAYLIST> *play_list)
{
  if(!isConnected) return;

  playlist_list->clear();

  for(PLAYLIST *curr = play_list->first();curr != 0; curr = play_list->next())
    notifyNewPlaylist(curr);
}

void GUI_Playlist::notifyNewPlaylistTracksBasis(QList<PLAYLIST_TRACK> *tracks_list, int playlistID)
{
  if(!isConnected) return;
  if(playlistID < 0) return;

  if(playlistID == 0) {
    current_tracklist->clear();
    mapCurrent.clear();
  }
  else {
    tracklist->clear();
    mapEditor.clear();
  }
  selectedItems.clear();
 
  for(PLAYLIST_TRACK *curr = tracks_list->first(); curr != 0; curr = tracks_list->next())
    notifyNewPlaylistTrack(curr,playlistID);
}

void GUI_Playlist::notifyNewPlaylist(PLAYLIST *playlist, bool isNewCreated)
{
  if(!isConnected) return;

  LVI_Playlist *item;
  item = new LVI_Playlist(playlist_list, playlist);
  item->setText(0,playlist->name);
	if(isNewCreated) playlist_list->setSelected(item, true);

  jobman->addJob(new Job_QueryPlaylist_Tracks(playlist->id));
}

void GUI_Playlist::notifyNewPlaylistTrack(PLAYLIST_TRACK *pl_track, int playlistID)
{
  if(!isConnected) return;

  LVI_Playlist *checklvi = static_cast<LVI_Playlist*>(playlist_list->selectedItem());
  if((playlistID != 0) && ((checklvi == NULL) || (checklvi->getID() != playlistID)))
    return;

  if((playlistID == 0 &&
      mapCurrent.find(pl_track->track_id) != mapCurrent.end()) ||
     (playlistID > 0 &&
      mapEditor.find(pl_track->track_id) != mapEditor.end()))
    return;

  QListView *temp_listview;
  if(playlistID == 0)
    temp_listview = current_tracklist;
  else
    temp_listview = tracklist;

  LVI_PlaylistTrack *item;
  item = new LVI_PlaylistTrack(temp_listview, pl_track, mapType[pl_track->medium_id]);
  item->setAvailability(mapAvail[pl_track->medium_id]);

  if(playlistID == 0)
    mapCurrent[pl_track->track_id] = item;
  else 
    mapEditor[pl_track->track_id] = item;

  //  temp_listview->setSorting(0,true);  PHF removed 28/12/2002
  //  temp_listview->sort();              PHF removed 28/12/2002
  temp_listview->setSorting(-1);
}

void GUI_Playlist::notifyRemovePlaylist(LVI_Playlist *playlist)
{
  if(!isConnected) return;

  if (displayed_playlist == playlist) {
    if(playlist->itemAbove())
      displayed_playlist = dynamic_cast<LVI_Playlist*>(playlist->itemAbove());
    else
      displayed_playlist = dynamic_cast<LVI_Playlist*>(playlist->itemBelow());

    if(displayed_playlist != NULL) {
      tracklist->setSelected(dynamic_cast<QListViewItem*>(playlist),true);
      tracklist->setCurrentItem(dynamic_cast<QListViewItem*>(playlist));
    }
    tracklist->clear();
    selectedItems.clear();
  }

  delete playlist;
}

void GUI_Playlist::notifyRemovePlaylistTracks(QList<LVI_PlaylistTrack> *pl_track_list)
{
  for(LVI_PlaylistTrack *lvi = pl_track_list->first(); lvi != 0; lvi = pl_track_list->next()) {
    if(lvi->getPlaylistID() == 0) {
      mapCurrent.remove(lvi->getTrackID());
      current_tracklist->takeItem(static_cast<QListViewItem*>(lvi));
    }
    else {
      mapEditor.remove(lvi->getTrackID());
      tracklist->takeItem(static_cast<QListViewItem*>(lvi));
    }
    delete lvi;
  }
}

void GUI_Playlist::notifyRenamePlaylist(LVI_Playlist *playlist, QString newname)
{
  playlist->setText(0, newname);
  playlist->setName(newname);
  playlist_list->sort();
}

void GUI_Playlist::notifyMovePlaylistTrack()
{
/*
  if(current_tracklist->hasFocus())
    current_tracklist->sort();
  else
    tracklist->sort();
*/
}

void GUI_Playlist::notifyNewMediumBasis(QList<MEDIUM> *list)
{
  for(MEDIUM *medium = list->first(); medium != 0; medium = list->next()) notifyNewMedium(medium);
}

void GUI_Playlist::notifyNewMedium(MEDIUM *medium)
{
  mapType[medium->id] = medium->type;
  mapAvail[medium->id] = (medium->type == MEDIUM_HARDDISK ? true : false);
}

void GUI_Playlist::notifyMediumRemoved(int id)
{
  mapType.remove(id);
  mapAvail.remove(id);
}

void GUI_Playlist::notifyMediumAvailabilityChanged(int mediumID, QString path, bool avail)
{
  if(avail == true) mapPath[mediumID] = path;

  mapAvail[mediumID] = avail;
  for(LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(current_tracklist->firstChild());
      lvi != 0; lvi = dynamic_cast<LVI_PlaylistTrack*>(lvi->nextSibling())) {
    if(lvi->getMediumID() == mediumID) lvi->setAvailability(avail);
  }

  for(LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(tracklist->firstChild());
      lvi != 0; lvi = dynamic_cast<LVI_PlaylistTrack*>(lvi->nextSibling())) {
    if(lvi->getMediumID() == mediumID) lvi->setAvailability(avail);
  }
}

void GUI_Playlist::clear()
{
  tracklist->clear();
  current_tracklist->clear();
  playlist_list->clear();
  selectedItems.clear();
  mapCurrent.clear();
  mapEditor.clear();
}

int GUI_Playlist::getDisplayedPlaylistID()
{
  if(displayed_playlist)
    return displayed_playlist->getID();

  return -1;
}

// ********************
// *** PRIVATE PART ***
// ********************

void GUI_Playlist::getSelectedItems(bool current)
{
  selectedItems.clear();
  QListViewItem *curr;
  curr = (current) ? current_tracklist->firstChild() : tracklist->firstChild();
  while(curr) {
    if(curr->isSelected())
      selectedItems.append(curr);

    curr = curr->nextSibling();
  }
}

bool GUI_Playlist::existsPlaylist(const QString &s)
{
  if(!isConnected) return false;

  return (playlist_list->findItem(s,0,ExactMatch) != 0);
}

// #################################################
// # callbacks for new/rename/remove buttons
// #################################################
void GUI_Playlist::slot_newButton()
{
  if(!isConnected) return;

  bool ok = false, done;
  QString name;
  do {
    name = QInputDialog::getText(_("New playlist"), QString(_("Enter name of new playlist")) + ":", QLineEdit::Normal, name, &ok);
    if(ok) {
      done = (!name.isEmpty() && !existsPlaylist(name));
      if(!done) {
	QMessageBox::information(gui, _("Message"),
				 _("Playlist name is empty\nOR\nPlaylist with that name already exists!"),
				 QMessageBox::Ok);
      }
    } else 
      done = true;
  } while(!done);

  if(ok) 
    jobman->addJob(new Job_ModifyPlaylists(APPEND_PLAYLIST, name));
}

void GUI_Playlist::slot_renameButton()
{
  if(!isConnected) return;

  QListViewItem *item;
  item = playlist_list->selectedItem();
  if(item == 0) {
    return;
  }

  bool ok = false, done;
  QString name;
  do {
    name = QInputDialog::getText(_("Rename playlist"), QString(_("Enter new name of playlist")) + ":", QLineEdit::Normal, name, &ok);
    if(ok) {
      done = (!name.isEmpty() && !existsPlaylist(name));
      if(!done) {
	QMessageBox::information(gui, _("Message"),
				 _("Playlist name is empty\nOR\nPlaylist with that name already exists!"),
				 QMessageBox::Ok);
      }
    } else 
      done = true;
  } while(!done);

  if(ok) {
    jobman->addJob(new Job_ModifyPlaylists(RENAME_PLAYLIST, dynamic_cast<LVI_Playlist*>(item), name));
  }
}

void GUI_Playlist::slot_removeButton()
{
  if(!isConnected) return;

  QListViewItem *item;
  item = playlist_list->selectedItem();
  if(item == 0)
    return;

  jobman->addJob(new Job_ModifyPlaylists(DELETE_PLAYLIST, dynamic_cast<LVI_Playlist*>(item)));
}

void GUI_Playlist::slot_resetButton()
{
  generatorSpinMinTrackLength->setValue(120);
  generatorSpinMaxTrackLength->setValue(300);
  generatorSpinTracks->setValue(10);
  genaratorCheckBoxOnlyFavouriteArtists->setChecked(FALSE);
}

void GUI_Playlist::slot_generateButton()
{
  if(!isConnected) return;

  jobman->addJob(new Job_GeneratePlaylist(generatorSpinTracks->value(), generatorSpinMinTrackLength->value(), generatorSpinMaxTrackLength->value(), genaratorCheckBoxOnlyFavouriteArtists->isChecked()));
}

void GUI_Playlist::slot_generatorMinTrackLength(int length)
{
  generatorSpinMaxTrackLength->setMinValue(length);
}

void GUI_Playlist::slot_generatorMaxTrackLength(int length)
{
  generatorSpinMinTrackLength->setMaxValue(length);
}


// #################################################
// # callbacks for move track up/down buttons
// #################################################
void GUI_Playlist::slot_movePlaylistTrackUp()
{
	if(!isConnected) return;
	
	QListViewItem *item;
	item = tracklist->currentItem();
	if(item == 0)
		return;
		
	if(item == tracklist->firstChild()) return;

	if(item->itemAbove() == tracklist->firstChild()) tracklist->firstChild()->moveItem(item);
	else item->moveItem(item->itemAbove()->itemAbove());

	slot_savePlaylistTracks();  //PHF added 28/12/2002

}

void GUI_Playlist::slot_movePlaylistTrackDown()
{
  if(!isConnected) return;
  
  QListViewItem *item;
  item = tracklist->currentItem();
  if(item == 0)
  	return;
  
  if(item->itemBelow() == 0) return;
  
  item->moveItem(item->itemBelow());

  slot_savePlaylistTracks();  // PHF added 28/12/2002

}

void GUI_Playlist::slot_savePlaylistTracks()
{
  if(!displayed_playlist || tracklist->firstChild() == 0) return;

  QList<int> *poslist = new QList<int>;
  for(QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling()) {
	LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
	int *id = new int;
	*id = lvi->getTrackID();
	poslist->append(id);
  }
  jobman->addJob(new Job_ModifyPlaylist_Tracks(SAVE_POSITIONS, displayed_playlist->getID(), poslist));
}


// #################################################
// # callback for context menu
// #################################################
void GUI_Playlist::slot_contextMenu(QListViewItem *lvi, const QPoint &p, int column)
{
  if(!isConnected) return;

  QPopupMenu *menu = new QPopupMenu(this);
  
  if(lvi->listView() == playlist_list) {
//    menu->insertItem("ADD this playlist to current", this, SLOT(slot_menuAddPlaylistToCurrent()),0,1);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_xmms.png"), _("PLAY this playlist"), this, SLOT(slot_menuPlayPlaylist()),0,2);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_xmms.png"), _("ENQUEUE this playlist"), this, SLOT(slot_menuEnqueuePlaylist()),0,3);
	menu->insertSeparator();
	menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_updateonly.png"), _("Playlist to Selection"), this, SLOT(slot_menuPlaylistToSelection()),0,4);
	menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_burncd.png"), _("Burn CD from playlist"), this, SLOT(slot_menuBurnCD()),0,5);
  }
  else if(lvi->listView() == tracklist || lvi->listView() == current_tracklist) {
	if(dynamic_cast<LVI_PlaylistTrack*>(lvi)->getIsAvailable()) {
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_xmms.png"), _("PLAY track(s)"), this, SLOT(slot_menuPlayPlaylistTracks()),0,1);
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_xmms.png"), _("ENQUEUE tracks(s)"), this, SLOT(slot_menuEnqueuePlaylistTracks()),0,2);
	menu->insertSeparator();
	}
    menu->insertItem(QPixmap(PROKYON_HOME + "/images/action_remove.png"), _("remove track(s) from playlist"), this, SLOT(slot_menuRemovePlaylistTracks()),0,3);
  }
  menu->exec(p,0);
}

void GUI_Playlist::slot_menuBurnCD()
{
  QString command;
  QString tmpCmd;
  
  command.sprintf("xterm -e %s", config->getBurnCDCmd().stripWhiteSpace().latin1() );
  
  for (QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling())
    {
      LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
      PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
      QString prepath="";
      if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];

      if(lvi->getIsAvailable()) 
	command.append( tmpCmd.sprintf(" \"%s\"", 
				       (const char*)adjustString(QString(prepath + track->path + "/" 
									 + track->filename).stripWhiteSpace().utf8())) );
    }

  QString listfiles(_("<b>List of files to burn:</b><br><br><font color=blue>"));
  int totaltime=0;
 
  QList<TRACK>* ltrack  = database->queryTracksByPlaylist(displayed_playlist->getID());
  for(TRACK *cur = ltrack->first(); cur != 0; cur = ltrack->next()) {
    listfiles.append(QString(cur->path + "/" + cur->filename +"<br>").stripWhiteSpace().utf8());
    totaltime += cur->length + 2;  // 2 second blank between tracks
    if (cur->samplerate != 44100) {
      QMessageBox::critical(this, _("Message"), QString(_("<font color=red><b>File %1/%2</font><br>"
							  "use a samplerate different from 44100Hz!</b><br><br>"
							  "It cannot be burned to an audio CD without special conversion."
							  "Remove it from the playlist if you want to proceed.<br>")
							).arg(cur->path).arg(cur->filename));
      return;
    }
  }


  int userIsSure = QMessageBox::warning(gui, _("Burning a CD"), listfiles+QString(_("</font><br><br><center><b><u>Total CD length will be:</u> %1 min %2 sec</center></b><br><br>")).arg(totaltime/60).arg(totaltime%60)+
					_("Are you really sure you want to proceed?<br>"), 
					_("&Yes"), _("&No"));
  if(userIsSure == 0) {
    command.append(" &");
    //    qWarning( command.latin1()); 
    system(command.latin1());
  }
}


void GUI_Playlist::slot_menuAddPlaylistToCurrent()
{
  QList<LVI_PlaylistTrack> *temp_list = new QList<LVI_PlaylistTrack>;
  for(QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling()) {
	LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
	temp_list->append(lvi);
  }
  jobman->addJob(new Job_ModifyPlaylist_Tracks(APPEND_TO_CURRENT,temp_list));
}

void GUI_Playlist::slot_menuPlaylistToSelection()
{
    jobman->addJob(new Job_QueryDatabase(QUERY_BY_PLAYLIST, displayed_playlist->getID()));
}

QString GUI_Playlist::adjustString(QString s)
{
  for(unsigned i=0; i<s.length(); i++) {
    if(s[i] == '\"' || s[i] == '\\') s.insert(i++, '\\');
  }
  return s;
}

void GUI_Playlist::slot_menuPlayPlaylist()
{
  QString command;
  QString tmpCmd;
  
  command.sprintf("%s", config->getPlayCmd().stripWhiteSpace().latin1() );
  
  for (QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling())
  {
     LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
     PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
	QString prepath="";
	if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];

     if(lvi->getIsAvailable()) command.append( tmpCmd.sprintf(" \"%s\"", (const char*)adjustString(QString(prepath + track->path + "/" + track->filename).stripWhiteSpace().utf8())) );
  }
  command.append(" &");
  system(command.latin1());
}

void GUI_Playlist::slot_menuEnqueuePlaylist()
{
  QString command;
  QString tmpCmd;
  
  command.sprintf("%s", config->getEnqueueCmd().stripWhiteSpace().latin1() );
  
  for (QListViewItem *curr = tracklist->firstChild(); curr != 0; curr = curr->nextSibling())
  {
     LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
     PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
	QString prepath="";
	if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];

     if(lvi->getIsAvailable()) command.append( tmpCmd.sprintf(" \"%s\"", (const char*)adjustString(QString(prepath + track->path + "/" + track->filename).stripWhiteSpace().utf8())) );
  }
  command.append(" &");
  system(command.latin1());
}

void GUI_Playlist::slot_menuRemovePlaylistTracks()
{
  QList<LVI_PlaylistTrack> *temp_list = new QList<LVI_PlaylistTrack>;
  for(QListViewItem *curr = selectedItems.first(); curr != 0; curr = selectedItems.next()) {
    LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
    temp_list->append(lvi);
  }
  jobman->addJob(new Job_ModifyPlaylist_Tracks(DELETE_PLAYLIST_TRACK,temp_list));
}

void GUI_Playlist::slot_menuPlayPlaylistTracks()
{
  QString command;
  QString tmpCmd;
  
  command.sprintf("%s", config->getPlayCmd().stripWhiteSpace().latin1() );
  
  for (QListViewItem *curr = selectedItems.first(); curr != 0; curr = selectedItems.next())
  {
     LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
     PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
	QString prepath="";
	if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];

     if(lvi->getIsAvailable()) command.append( tmpCmd.sprintf(" \"%s\"", (const char*)adjustString(QString(prepath + track->path + "/" + track->filename).stripWhiteSpace().utf8())) );
  }
  command.append(" &");
  system(command.latin1());
}

void GUI_Playlist::slot_menuEnqueuePlaylistTracks()
{
  QString command;
  QString tmpCmd;
  
  command.sprintf("%s", config->getEnqueueCmd().stripWhiteSpace().latin1() );
  
  for (QListViewItem *curr = selectedItems.first(); curr != 0; curr = selectedItems.next())
  {
     LVI_PlaylistTrack *lvi = dynamic_cast<LVI_PlaylistTrack*>(curr);
     PLAYLIST_TRACK *track = lvi->getPlaylistTrack();
	QString prepath="";
	if(mapPath[lvi->getMediumID()] != 0) prepath = mapPath[lvi->getMediumID()];
     if(lvi->getIsAvailable()) command.append( tmpCmd.sprintf(" \"%s\"", (const char*)adjustString(QString(prepath + track->path + "/" + track->filename).stripWhiteSpace().utf8())) );
  }
  command.append(" &");
  system(command.latin1());
}


// #################################################
// # callback for playlist selection change
// #################################################
void GUI_Playlist::slot_playlistSelectionChanged(QListViewItem *lvi)
{
  if(!isConnected) return;

  if (lvi == NULL) {
  	displayed_playlist = NULL;	
  	return;
  }
  
  LVI_Playlist *item = dynamic_cast<LVI_Playlist*>(lvi);
  if (item == NULL) {
  	displayed_playlist = NULL;
  	return;
  }

  selectedItems.clear();
  jobman->addJob(new Job_QueryPlaylist_Tracks(item->getID()));
  displayed_playlist = item;
}

// #################################################
// # callback for playlist track selection change
// #################################################
void GUI_Playlist::slot_trackSelectionChanged()
{
  if(!isConnected) return;

  if(current_tracklist->hasFocus())
    getSelectedItems(true);
  else
    getSelectedItems(false);
}

// #################################################
// # callback for selected tab changed
// #################################################
void GUI_Playlist::slot_tabChange(QWidget *widget)
{
  if(widget == current_tracklist) {
    selectedItems.clear();
    getSelectedItems(true);
    current_tracklist->setFocus();
    if (isConnected) jobman->addJob(new Job_QueryPlaylist_Tracks(0));
  }
  else {
    selectedItems.clear();
    getSelectedItems(false);
    vsplit->setFocus();
  }
}

// #################################################
// # callback for doubleClicked
// #################################################
void GUI_Playlist::slot_playlistDoubleClicked(QListViewItem *lvi)
{
  slot_menuPlayPlaylist();
}

void GUI_Playlist::slot_playlistTrackDoubleClicked(QListViewItem *lvi)
{
  slot_menuPlayPlaylistTracks();
}

// #################################################
// # drag and drop support
// # 
// #  We assume that internal drag is enabled 
// #      only from QListView's of valid LVI_Tracks.  
// #  All selected Tracks are added to the play_list
// #################################################

ExtendedDropListView::ExtendedDropListView(QWidget *parent, const char *name)
  : QListView(parent, name)
{}

void ExtendedDropListView::dragEnterEvent(QDragEnterEvent* event)
{ 
  event->accept( true );
}

void ExtendedDropListView::dropEvent(QDropEvent* event)
{  

  QList<TRACK> *templist = new QList<TRACK>;
  int ID;

  QListView * listdrop = dynamic_cast<QListView*>( event->source() );
  if ( listdrop ) {   
    for(QListViewItem *item = listdrop->firstChild(); item != 0; item = item->nextSibling()) {
      if(item->isSelected() && dynamic_cast<LVI_Track*>(item)->getOriginalTrack() )  // robustness only
	templist->append(dynamic_cast<LVI_Track*>(item)->getOriginalTrack());
    }

    // First check if user dropped to a playlist name
    QPoint pt = mapToGlobal(event->pos());
    LVI_Playlist *pl = dynamic_cast<LVI_Playlist*>( itemAt( viewport()->mapFromGlobal( pt)  ));
    if ( pl) {
      if ( pl->getID()>0 && templist->count()>0)  // robustness only
	jobman->addJob(new Job_ModifyPlaylist_Tracks(APPEND_PLAYLIST_TRACK, templist, pl->getID()));
    }  else 
      // otherwise add to the currently selected playlist, if any.
      if((ID = gui->getPlaylisting()->getDisplayedPlaylistID()) > 0) 
	jobman->addJob(new Job_ModifyPlaylist_Tracks(APPEND_PLAYLIST_TRACK, templist, ID));
  }
}
