#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>

#include <config.h>
#include <support.h>
#include <sysmsg.h>

#include "log.h"
#include "console.h"
#include "option.h"
#include "frame.h"
#include "filter.h"

extern int IpFilterAdd();
extern void IpFilterShow(), IpFilterDel();
extern f_act_t IpFilterCheck();

static struct filtergroup_s {
    const char *name;
    int	arg;
    int (*add)();
    void (*show)();
    void (*del)();
    f_act_t (*check)();
} filterGroup[]={
    {
	"IP", IPPROTO_IP,
	IpFilterAdd, IpFilterShow, IpFilterDel, IpFilterCheck,
    },
    {
	"ICMP", IPPROTO_ICMP,
	IpFilterAdd, IpFilterShow, IpFilterDel, IpFilterCheck,
    },
    {
	"TCP", IPPROTO_TCP,
	IpFilterAdd, IpFilterShow, IpFilterDel, IpFilterCheck,
    },
    {
	"UDP", IPPROTO_UDP,
	IpFilterAdd, IpFilterShow, IpFilterDel, IpFilterCheck,
    },
};

#define	NUM_GROUPS	(sizeof(filterGroup)/sizeof(filterGroup[0]))

/*
 * filter <group> <operation><filter><type> <options>
 *
 * operation:
 *	+	add
 *	-	delete
 *	none	list
 *
 * filter:
 *	t	Trigger
 *	i	Input
 *	o	Output
 *
 * type:
 *	a	accept
 *	r	reject
 *	x	translate
 *	n	neglect (idle timer)
 *	k	keepalive (idle timer)
 *
 * group:
 * option:
 */

int
CmdFilter(int argc, char **argv)
{
    unsigned int group;
    int nargc, n, ff=F_TYPE_NUM;
    f_act_t act;
    char **nargv, *cmd, op=0;
    struct filtergroup_s *fgp=NULL;
    time_t keeptime = pppOpt.i_to;
    int id;
    priority_t pri = 0;

    if (argc < 2) {
	fgp= &filterGroup[0]; /* filter ip */
    } else if (argc > 1) for (group = 0; group < NUM_GROUPS; group ++) {
	if (!strcasecmp(filterGroup[group].name, argv[1])) {
	    fgp = &filterGroup[group];
	    break;
        }
    }
    if (!fgp) {
	ConsoleMsg(MS_E_PROTO_UNKNOWN, NULL);
	return(-1);
    }

    act = F_ACT_NONE;
    if (argc > 2) {
	cmd = argv[2];

	while (*cmd) {
	    switch (*cmd) {
	    case 'T':
	    case 't':
		ff = F_TYPE_TRIGGER;
		break;
	    case 'I':
	    case 'i':
		ff = F_TYPE_INPUT;
		break;
	    case 'O':
	    case 'o':
		ff = F_TYPE_OUTPUT;
		break;
	    case '+':
	    case '-':
		op = *cmd;
		break;
	    case 'a':
		act = F_ACT_ACCEPT;
		break;
	    case 'r':
		act = F_ACT_REJECT;
		break;
	    case 'm':
		act = F_ACT_MAPPER;
		break;
	    case 'k':
		act = F_ACT_KEEPTIME;
		break;
	    case 'p':
		act = F_ACT_PRIORITY;
		break;
	    case 'n':
		act = F_DACT_NEGLECT;
		break;
	    case 'h':
		act = F_DACT_HPRIORITY;
		break;
	    case 'l':
		act = F_DACT_LPRIORITY;
		break;
	    case 'd':
		act = F_DACT_MPRIORITY;
		break;
	    }
	    cmd ++;
	}
    }
    if (op) {
	if (ff == F_TYPE_NUM) {
	    Logf(LOG_ERROR, "unknown filter\n");
	    ConsoleMsg(MS_E_FILTER_NOTSPECIFIED, NULL);
	    return(-1);
	} else if  (act == F_ACT_NONE) {
	    Logf(LOG_ERROR, "unknown type\n");
	    ConsoleMsg(MS_E_TYPE_NOTSPECIFIED, NULL);
	    return(-1);
	} else {
	    id = (argc > 3)
		? ((!strcmp(argv[3], "all")) ? -1: atoi(argv[3])): 0;
	    if (id) {
		nargc = argc - 4;
		nargv = &argv[4];
	    } else {
		nargc = argc - 3;
		nargv = &argv[3];
	    }
	}
        switch (op) { /* operation */
        case '+':
	    if (id < 0) {
		Logf(LOG_ERROR, "bad filter id\n");
		ConsoleMsg(MS_E_FILTER_BADID, NULL);
		return(-1);
	    }
	    if (act & ~F_ACT_MASK) {
		switch (act) {
		case F_DACT_NEGLECT:
		    keeptime = 0;
		    break;
		case F_DACT_LPRIORITY:
		case F_DACT_HPRIORITY:
		case F_DACT_MPRIORITY:
		    pri = 0;
		    break;
		}
		act &= F_ACT_MASK;
	    }
	    fgp->add(fgp->arg, ff, id,
		     act, keeptime, pri, nargc, nargv);
	    break;
        case '-':
            if (id) fgp->del(fgp->arg, ff, id);
	    else {
		Logf(LOG_ERROR, "bad filter id\n");
		ConsoleMsg(MS_E_FILTER_BADID, NULL);
		return(-1);
	    }
	    break;
	}
    } else {
	if (ff == F_TYPE_NUM) {
	    for (ff = F_TYPE_TRIGGER; ff <= F_TYPE_OUTPUT; ff ++)
		fgp->show(fgp->arg, ff);
	} else {
	    fgp->show(fgp->arg, ff);
        }
    }
    return(0);
}
