/*
 * raw_ia64_smpl.c - IA-64 raw output for sampling buffer
 *
 * Copyright (C) 2002-2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#include "pfmon.h"

#include <perfmon/pfmlib_generic_ia64.h>
#include <perfmon/perfmon_default_smpl.h>

#define SMPL_MOD_NAME	"raw-ia64"
#define CHECK_VERSION(h)	(PFM_VERSION_MAJOR((h)) != PFM_VERSION_MAJOR(PFM_DEFAULT_SMPL_VERSION))
#define RAW_DFL_SMPL_ENTRIES	2048UL

typedef struct {
	unsigned long smpl_entries;
} raw_options_t;

static raw_options_t raw_options;

/*
 * forward declaration
 */
pfmon_smpl_module_t raw_ia64_smpl_module;

static int
raw_process_samples(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr;
	void *first_pos;
	size_t sz;
	ssize_t ret;

	hdr       = csmpl->smpl_hdr;
	first_pos = (void *)(hdr+1);
	sz        = hdr->hdr_cur_offs - sizeof(*hdr);

	ret = write(fileno(csmpl->smpl_fp), first_pos, sz);
	if (ret < sz) goto error;

	if (options.opt_aggr) {
		*csmpl->aggr_count += hdr->hdr_count;
	} else {
		csmpl->entry_count += hdr->hdr_count;
	}
	return 0;
error:
	warning("cannot write to raw sampling file: %s\n", strerror(errno));
	/* not reached */
	return -1;
}

static int
raw_check_new_samples(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr; 

	hdr   = csmpl->smpl_hdr;

	if (hdr->hdr_overflows <= csmpl->last_ovfl && csmpl->last_ovfl != ~0UL) {
		DPRINT(("skipping identical set of samples %lu <= %lu\n", hdr->hdr_overflows, csmpl->last_ovfl)); 
		return -1;
	}
	/*
	 * XXX: module specific field in generic structure (last_ovfl)
	 */
	csmpl->last_ovfl = hdr->hdr_overflows;

	return 0;
}

/*
 * 000-255   reserved for generic options
 * 400-499   reserved for PMU specific options
 * 500-599   reserved for format specific options
 */
static struct option raw_cmd_options[]={
	{ "smpl-entries", 1, 0, 500},
	{ NULL, 0, 0, 0}
};

static void
raw_show_options(void)
{
	printf("\t--smpl-entries=val\t\tset number of entries for sampling buffer (default %lu)\n", RAW_DFL_SMPL_ENTRIES);
}

/*
 * 0  means we understood the option
 * -1 unknown option
 */
static int
raw_parse_options(int code, char *optarg, pfmon_lib_param_t *evt)
{
	char *endptr = NULL;

	switch(code) {
		case  500:
			if (raw_options.smpl_entries != RAW_DFL_SMPL_ENTRIES) 
				fatal_error("smpl-entries already defined\n");

			raw_options.smpl_entries = strtoul(optarg, &endptr, 0);

			if (*endptr != '\0') 
				fatal_error("invalid number of entries: %s\n", optarg);
			break;
		default:
			return -1;
	}
	return 0;

}


static size_t
raw_get_fmt_arg_size(void)
{
	return sizeof(pfm_default_smpl_ctx_arg_t);
}

#define MAX_PMD_COUNT		64
#define FUDGE_FACTOR		(sizeof(pfm_default_smpl_entry_t)+(sizeof(unsigned long)*MAX_PMD_COUNT))
#define ENTRY_SIZE(npmd)	(sizeof(pfm_default_smpl_entry_t)+((npmd)*sizeof(unsigned long)))
static int
raw_initialize_ctx_arg(pfmon_ctx_arg_t *arg, unsigned max_pmds_sample)
{
	pfm_default_smpl_ctx_arg_t *ctx_arg = (pfm_default_smpl_ctx_arg_t *)arg;
	unsigned long entry_size;

	entry_size = ENTRY_SIZE(max_pmds_sample);
	
	/*
	 * the fudge factor allows us to get exactly the number of entries specified by the
	 * user (or the default). The kernel module is such that it will consider the buffer
	 * full if less than PFM_DEFAULT_MAX_ENTRY_SIZE bytes are left in the buffer. Any
	 * entry size is <= PFM_DEFAULT_MAX_ENTRY_SIZE, therefore we will always record
	 * less than the specified number unless we increase the buffer size slightly.
	 */
	ctx_arg->buf_arg.buf_size = sizeof(pfm_default_smpl_hdr_t)
		                  + (FUDGE_FACTOR-entry_size)
				  + raw_options.smpl_entries*entry_size;

	/*
	 * copy the uuid of the format we are using
	 */
	memcpy(ctx_arg->ctx_arg.ctx_smpl_buf_id, raw_ia64_smpl_module.uuid, sizeof(pfm_uuid_t));

	vbprintf("min buffer entries=%lu buffer_size=%lu max PMD/entry=%u\n", 
		raw_options.smpl_entries, 
		ctx_arg->buf_arg.buf_size, 
		max_pmds_sample);

	DPRINT(("max_pmds_sample=%u buf_size=%lu fudge=%lu buffer_header=%lu entry_header=%lu (max)entry_size=%lu\n", 
		max_pmds_sample, 
		ctx_arg->buf_arg.buf_size, 
		FUDGE_FACTOR, 
		sizeof(pfm_default_smpl_hdr_t), 
		sizeof(pfm_default_smpl_entry_t), 
		entry_size));
		

	return 0;
}

/*
 * module initialization
 */
static int
raw_initialize_module(void)
{
	raw_options.smpl_entries = RAW_DFL_SMPL_ENTRIES;

	return pfmon_register_smpl_mod_options(raw_cmd_options, sizeof(raw_cmd_options));
}


static int
raw_check_version(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr;

	hdr   = csmpl->smpl_hdr;

	if (CHECK_VERSION(hdr->hdr_version)) {
		warning("format %s expect format v%u.x not v%u.%u\n", 
				SMPL_MOD_NAME,
				PFM_VERSION_MAJOR(PFM_DEFAULT_SMPL_VERSION),
				PFM_VERSION_MAJOR(hdr->hdr_version),
				PFM_VERSION_MINOR(hdr->hdr_version));
		return -1;
	}
	return 0;
}

pfmon_smpl_module_t raw_ia64_smpl_module ={
	.name		    = SMPL_MOD_NAME,
	.pmu_mask	    = PFMON_PMU_MASK(PFMLIB_GENERIC_IA64_PMU),
	.description	    = "IA-64 raw values in binary format",
	.check_version	    = raw_check_version,
	.process_samples    = raw_process_samples,
	.get_fmt_arg_size   = raw_get_fmt_arg_size,
	.initialize_ctx_arg = raw_initialize_ctx_arg,
	.check_new_samples  = raw_check_new_samples,
	.show_options       = raw_show_options,
	.parse_options      = raw_parse_options,
	.initialize_module  = raw_initialize_module,
	.uuid		    = PFM_DEFAULT_SMPL_UUID,
};
