/*
 * pfmon_results.c 
 *
 * Copyright (C) 2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#include "pfmon.h"
#include <sys/utsname.h>

static void
print_sdesc_header(FILE *fp, pfmon_sdesc_t *sdesc)
{
	fprintf(fp, "# command line     : %s\n#\n"
		    "# process id       : %d\n"
		    "# thread  id       : %d\n"
		    "# parent process id: %d\n"
		    "# parent thread  id: %d\n"
		    "# exec count       : %u\n#\n",
		    sdesc->cmdline,
		    sdesc->pid,
		    sdesc->tid,
		    sdesc->ppid,
		    sdesc->ptid,
		    sdesc->exec_count);
}

void
print_standard_header(FILE *fp, unsigned int cpu, pfmon_sdesc_t *sdesc)
{
	char **argv = options.argv;
	struct utsname uts;
	unsigned long m1, m2, pgsz;
	time_t t;
	unsigned int i;
	unsigned int num_cpus;
	char name[PFMON_MAX_EVTNAME_LEN];

	uname(&uts);
	time(&t);

	fprintf(fp, "#\n# date: %s", asctime(localtime(&t)));
	fprintf(fp, "#\n# hostname: %s\n", uts.nodename);
	fprintf(fp, "#\n# kernel version: %s %s %s\n", 
			uts.sysname, 
			uts.release, 
			uts.version);

	fprintf(fp, "#\n# pfmon version: "PFMON_VERSION"\n");
	fprintf(fp, "# kernel perfmon version: %u.%u\n#\n", 
			PFM_VERSION_MAJOR(options.pfm_version),
			PFM_VERSION_MINOR(options.pfm_version));

	fprintf(fp, "#\n# page size: %u bytes\n#\n", options.page_size);
	fprintf(fp, "# CLK_TCK: %u ticks/second\n#\n", options.clock_tick);
	fprintf(fp, "# CPU configured: %lu\n# CPU online    : %lu\n#\n", 
			options.config_cpus,
			options.online_cpus);

	pgsz = options.page_size;

	m1 = sysconf(_SC_PHYS_PAGES)*pgsz;
	m2 = sysconf(_SC_AVPHYS_PAGES)*pgsz;

	fprintf(fp, "# physical memory          : %lu bytes (%.1f MB)\n"
		    "# physical memory available: %lu bytes (%.1f MB)\n#\n", 
		    m1, (1.0*m1) / (1024*1024),
		    m2, (1.0*m2) / (1024*1024));

	pfmon_print_cpuinfo(fp);

	fprintf(fp, "#\n#\n# captured events:\n");

	for(i=0; i < PFMON_MAX_PMDS; i++) {
		if (options.rev_pc[i] == -1) continue;

		pfm_get_event_name(options.events[options.rev_pc[i]].event, name, PFMON_MAX_EVTNAME_LEN);

		fprintf(fp, "#\tPMD%u: %-s\n", i, name);
	} 

	fprintf(fp, "#\n# privilege levels:\n");

	for(i=0; i < PFMON_MAX_PMDS; i++) {
		if (options.rev_pc[i] == -1) continue;

		pfm_get_event_name(options.events[options.rev_pc[i]].event, name, PFMON_MAX_EVTNAME_LEN);

		fprintf(fp, "#\tPMD%u: %-*s = %s\n", 
			i,
			(int)options.max_event_name_len,
			name,
			priv_level_str(options.events[options.rev_pc[i]].plm));
	} 
	fprintf(fp, "#\n");

	if (options.opt_syst_wide) {
		fprintf(fp, "# monitoring mode: system wide\n");
	} else {
		fprintf(fp, "# monitoring mode: per-process\n");
	}

	/*
	 * invoke CPU model specific routine to print any additional information
	 */
	if (pfmon_current->pfmon_print_header) {
		pfmon_current->pfmon_print_header(fp);
	}

	fprintf(fp, "#\n# command:");
	while (*argv) fprintf(fp, " %s", *argv++);

	fprintf(fp, "\n#\n");
	
	if (options.opt_syst_wide) {
		unsigned int i;

		fprintf(fp, "# results captured on: ");
		if (options.opt_aggr == 0) {
				fprintf(fp, "CPU%u", cpu);
		} else {
			num_cpus = options.online_cpus;
			for(i=0; i < num_cpus; i++) {
				if (PFMON_CPUMASK_ISSET(options.cpu_mask, i)) fprintf(fp, "CPU%u ", i);
			}
		}
		fprintf(fp, "\n");
	} else if (options.opt_aggr == 0) {
		if (sdesc) print_sdesc_header(fp, sdesc);
	}

	fprintf(fp, "#\n#\n");
}

/*
 * Does the pretty printing of results
 *
 * In the case where results are aggregated, the routine expect the results
 * in pd[] and will not generate CPU (or pid) specific filenames
 */
int
print_results(pfarg_reg_t *pd, pfmon_sdesc_t *sdesc, unsigned int cpu)
{
	FILE *fp = NULL;
	char *suffix, *p = NULL;
	char *format_str;
	pid_t pid = 0, tid = 0, ppid = 0;
	int ret = 0, is_syswide;
	unsigned int i;
	unsigned long max_len, the_len;
	char name[PFMON_MAX_EVTNAME_LEN];
	char format_base[32];
	char filename[PFMON_MAX_FILENAME_LEN];
	char counter_str[32];
	char use_max_len = 0;

	static pthread_mutex_t results_lock = PTHREAD_MUTEX_INITIALIZER;

	is_syswide = options.opt_syst_wide;

	if (options.opt_use_smpl && options.opt_smpl_print_counts == 0) return 0;

	if (options.outfile) {
		if (options.opt_aggr == 0 && is_regular_file(options.outfile)) {
			if (is_syswide) {
				sprintf(filename, "%s.cpu%u", options.outfile, cpu);
			} else {
				if (options.opt_follows) {
					if (options.opt_split_exec) {
						sprintf(filename, "%s.%d.%d.%u", 
							options.outfile, 
							sdesc->pid, 
							sdesc->tid, 
							sdesc->exec_count);
					} else {
						sprintf(filename, "%s.%d.%d", 
							options.outfile, 
							sdesc->pid, 
							sdesc->tid);
					}
				} else {
					sprintf(filename, "%s", options.outfile);
				}
			}
		} else {
			strcpy(filename, options.outfile);
		}

		fp = fopen(filename, options.opt_append ? "a" : "w");
		if (fp == NULL) {
			/*
			 * XXX: may be to revisit this one
			 */
			warning("cannot open %s for writing, defaulting to stdout\n", filename);
		}
	}

	if (fp == NULL)	{
		fp = stdout;
		/*
		 * need to serialize printf to avoid confusion on the terminal
		 */
		pthread_mutex_lock(&results_lock);
		if (options.opt_verbose) vbprintf_block();
	} else {
		if (is_syswide) 
			vbprintf("CPU%-3u results are in file \"%s\"\n", cpu, filename);
		else
			vbprintf("[%d] results are in file \"%s\"\n", sdesc->tid, filename);
	}	

	if (options.opt_with_header) {
		print_standard_header(fp, cpu, sdesc);
	}

	suffix     = "";
	format_str = "%26s %-*s\n";
	if (is_syswide) {
		if (fp == stdout && options.opt_aggr == 0) {
			sprintf(format_base, "CPU%-2u %%26s %%-*s\n", cpu);
			format_str = format_base;
		} 
	} else if (options.opt_aggr == 0 && options.opt_follows && options.outfile == NULL) {
			/* isolate command */
			p = strchr(sdesc->cmdline, ' ');
			if (p) *p = '\0';

			/*
			 * XXX: could hard code suffix, tid, pid
			 */
			format_str = "%26s %-*s %s (%d,%d,%d)\n";

			suffix = sdesc->cmdline;
			pid = sdesc->pid; tid = sdesc->tid; ppid = sdesc->ppid;
			use_max_len = 1;
	} 
	max_len = options.max_event_name_len;

	for (i=0; i < options.monitor_count; i++) {

		pfm_get_event_name(options.events[i].event, name, PFMON_MAX_EVTNAME_LEN);

		counter2str(pd[i].reg_value, counter_str);
		/*
		 * ensures we do not add spaces when we don't need them
		 */
		the_len = use_max_len ? max_len : strlen(name);

		fprintf(fp, format_str, 
			counter_str,
			the_len, name,
			suffix,
			pid, 
			tid, ppid);
	}
	/* restore space character, if needed */
	if (p) *p = ' ';

	if (fp == stdout) {
		pthread_mutex_unlock(&results_lock);
		if (options.opt_verbose) vbprintf_unblock();
	} else {
		fclose(fp);
	}
	return ret;
}
