/*
 * transmitter.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/srmv2/net/transmitter.cc,v 1.16 2002/02/03 03:03:19 lim Exp $ (LBL)";

#include <stdio.h>
#include <stdlib.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <errno.h>
#include <string.h>
#ifdef WIN32
#include <winsock.h>
#include <io.h>
#include <sys/stat.h>
#else
#include <sys/param.h>
#include <sys/uio.h>
#include <netinet/in.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#endif
#include "transmitter.h"
#include "net.h"
#include "hfsc_fun.h"
#include "hfsc_queue.h"

#if defined(sun) && !defined(__svr4__) || (defined(_AIX) && !defined(_AIX41))
extern "C" writev(int, iovec*, int);
#endif

SRMv2_Transmitter::SRMv2_Transmitter(int bps) :
	mtu_(1024),
	nb_(0),
	nf_(0),
	np_(0),
	bps_(bps),
	nextpkttime_(0.),
	busy_(0),
	head_(0),
	tail_(0),
	loopback_(1),
	loop_layer_(1000)
{
}


/*
 * Each srm_session_t object within the application instance has a
 * separate hierarchy of traffic classes. srm_create_session should
 * also create and initialize the traffic class hierarchy for the
 * session.
 * Create the root class and one child per message class, and set
 * root bandwidth to zero. Default class hierarchy is as follows: 

 -- Root 
  -- Application Data 
  -- Session Announcements 
  -- NACKs
  
 * Note: init_tc creates the root of the traffic class hierarchy
 * and also initializes the default classes that
 * are controlled by libsrm. create_tc is used to
 * create and initialize the parameters of an individual
 * traffic class. init_tc uses create_tc to create the initial
 * traffic class structure. 
 */

void  SRMv2_Transmitter::init_hfsc(unsigned int total_bps)
{
	ClassItem *hfsc_root;
	
	/* Create the root class. */
	hfsc_root = newClassItem(0, 0, newSC((int) total_bps, 0, (int) total_bps));
	hfsc_queue_ = new HFSCQueue(hfsc_root);
	
	/*
	 * Create a class per protocol message type.
	 * How to set bandwidths? Maybe using some heuristics:
	 * e.g., data 90% and control 10%
	 */
	for (int i=1; i<=3; i++) {
		insertClassItem(hfsc_root, ROOT_CLASS, newClassItem(i, 1, newSC(0, 0, 0)));
	}

	/* Used in allocating TC ids. */
	next_tcid_ = 4;
}

ClassItem *SRMv2_Transmitter::hfscRoot()
{
	return (hfsc_queue_->hfscRoot());
}


void SRMv2_Transmitter::idle()
{
}

inline double SRMv2_Transmitter::gettimeofday() const
{
	timeval tv;
	::gettimeofday(&tv, 0);
	return (tv.tv_sec + 1e-6 * tv.tv_usec);
}

void SRMv2_Transmitter::loopback(SRMv2_pktbuf* pb)
{
	pb->release();
}

void SRMv2_Transmitter::update_stats(SRMv2_pktbuf* pb)
{
	int cc = pb->len;
	nb_ += cc;
	++np_;
}

int SRMv2_Transmitter::dumpfd_ = -1;
void SRMv2_Transmitter::dump(int fd)
{
	dumpfd_ = fd;
#define MAGIC "RTPCLIP 1.0"
	(void)write(fd, MAGIC, sizeof(MAGIC));
}

/*FIXME*/
#ifdef WIN32
int writev(int fd, iovec* iov, int iovlen)
{
	int len = 0, n;
	for (int i = 0; i < iovlen; i++) {
		if ((n = write(fd, iov[i].iov_base, iov[i].iov_len)) == -1) {
			perror("writev");
			exit(1);
		}
		len += n;
	}
	return(len);
}
#endif

void SRMv2_Transmitter::dump(int fd, const SRMv2_pktbuf* pb) const
{
	char cliphdr[4];
	*(u_int16_t*)cliphdr = htons(pb->len);
	cliphdr[2] = 0; /* data packet (i.e., not an rtcp packet) */
	cliphdr[3] = 0; /* ? */

	/*FIXME use stdio? */
	(void)write(fd, cliphdr, 4);
	(void)write(fd, pb->data, pb->len);
}

/*
 * Time it takes in seconds to send this
 * packet at the configured bandwidth.
 */
double SRMv2_Transmitter::txtime(SRMv2_pktbuf* pb)
{
	return (8 * pb->len / double(bps_));
}


#ifdef USE_HFSC_SCHEDULER
void SRMv2_Transmitter::recv(SRMv2_pktbuf* pb)
{
	/* Initialize HFSC timestamps. */
	pb->pkt_measure.pm_arrive = pb->pkt_srcTimestamp = gettimeofday();
	
	if (!busy_) {
		double delay = txtime(pb);
		nextpkttime_ = gettimeofday() + delay;
		output(pb);
		/*
		 * emulate a transmit interrupt --
		 * assume we will have more to send.
		 */
		printf("Scheduling for %f\n", delay);
		msched(int(delay * 1e3));
		busy_ = 1;
	} else {
		printf("INSERTING into queue...\n");
		hfsc_queue_->insert(pb);
	}
}

void SRMv2_Transmitter::timeout()
{
	double now = gettimeofday();
	for (;;) {
		printf(" *** Timeout: Time to transmit packet.\n");
		SRMv2_pktbuf* p = hfsc_queue_->remove();
		printf("Packet address is 0x%x\n", p);
		if (p != 0) {
			nextpkttime_ += txtime(p);
			output(p);
			int ms = int(1e3 * (nextpkttime_ - now));
			/* make sure we will wait more than 10ms */
			if (ms > 0) {
				msched(ms);
				return;
			} 
		} else { /* No more backlogged packets. */
			busy_ = 0;
			break;
		}
	}
}

/* Output all backlogged packets. */
void SRMv2_Transmitter::flush()
{
	if (busy_) {
		busy_ = 0;
		cancel();
	}

	SRMv2_pktbuf* p = hfsc_queue_->remove();
	while (p != 0) {
	  //		printf("Writing packet...\n");
		output(p);
		p = hfsc_queue_->remove();
	}
}
#else
void SRMv2_Transmitter::recv(SRMv2_pktbuf* pb)
{

	if (!busy_) {
		double delay = txtime(pb);
		nextpkttime_ = gettimeofday() + delay;
		output(pb);
		/*
		 * emulate a transmit interrupt --
		 * assume we will have more to send.
		 */
		msched(int(delay * 1e3));
		busy_ = 1;
	} else {
		if (head_ != 0) {
                        tail_->next = pb;
			tail_ = pb;
		} else
			tail_ = head_ = pb;
		pb->next = 0;
	}
}

void SRMv2_Transmitter::timeout()
{
	double now = gettimeofday();
	for (;;) {
		SRMv2_pktbuf* p = head_;
		if (p != 0) {
			head_ = p->next;
			nextpkttime_ += txtime(p);
			output(p);
			int ms = int(1e3 * (nextpkttime_ - now));
			/* make sure we will wait more than 10ms */
			if (ms > 0) {
				msched(ms);
				return;
			} 
		} else {
			busy_ = 0;
			break;
		}
	}
}

void SRMv2_Transmitter::flush()
{
	if (busy_) {
		busy_ = 0;
		cancel();
	}

	SRMv2_pktbuf* p = head_;
	while (p != 0) {
		SRMv2_pktbuf* n = p->next;
		output(p);
		p = n;
	}
	head_ = 0;
}

#endif /* USE_HFSC_SCHEDULER */

void SRMv2_Transmitter::output(SRMv2_pktbuf* pb)
{
  //	printf("Writing packet...\n");
	if (dumpfd_ >= 0)
		dump(dumpfd_, pb);
	transmit(pb);
	update_stats(pb);
	/*
	 * Always loop back the packet from here.
	 * The subclass is responsible for releasing
	 * the packet buffer (and checking the loopback_ flag).
	 */
	loopback(pb);
}

