/*
 * transducer-gauge.cc --
 *
 *      Receives audio levels and filters them with some moving sums.
 *
 * Copyright (c) 2000-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "transducer-gauge.h"
#include "tclcl.h"

#include <stdlib.h> // for atoi

static class GaugeTransducerClass : public TclClass {
public:
	GaugeTransducerClass() : TclClass("Transducer/Gauge") {}
	TclObject* create(int argc, const char*const* argv) {
		return(new GaugeTransducer(argc, argv));
	}
} gauge_transducer_class;

GaugeTransducer::GaugeTransducer(int argc, const char* const* argv)
{
  // args passed in tcl start at argv[4]
  /*
  int x;
  for(x = 0; x < argc; ++x)
  {
    printf("argc %d = %s\n", x, argv[x]);
  }
  */
  slaveTransducer_ = NULL;
  useSpoof = 0;
  spoofLevel = 0.0;
  threshold = 50;

  // get the moving sum history size from the 5th argument
  if(argc > 4)
  {
    historySize = atoi(argv[4]);
  }
  else
  {
    // default of 20
    historySize = 55;
  }

  levelSumPtr = new MovingSum(historySize);
  activeSumPtr = new MovingSum(historySize);

  // get the gauge name from argument 6
  if(argc > 5)
  {
    if(strlen(argv[5]) < 100)
    {
      strcpy(gaugeName, argv[5]);
    }
    else
    {
      fprintf(stderr, "GaugeTransducer::GaugeTransducer: gauge name must be < 100 chars\n");
      strcpy(gaugeName, "DefaultGaugeName");
    }
  }
  else
  {
    strcpy(gaugeName, "DefaultGaugeName");
  }
}

GaugeTransducer::~GaugeTransducer()
{
//  printf("destroying gauge transducer\n");
  delete levelSumPtr;
  delete activeSumPtr;
}

// by experimentation, this gets called every ~60 ms => ~16.6667 samples/sec
//    => 50 samples every 3 seconds
//
// by experimentation, it also appears that this "double" is always of the
//    form 60.00000, 32.0000, etc, i.e., converting it to an int loses no
//    precision
//
// for logging, 50 samples/3 seconds => 1000 samples/minute
//    1000 samples/min * 5 bytes/sample (one for \n) = 5000 bytes/min
//    5000 bytes/min = 300,000 bytes/hour
//
void GaugeTransducer::set(double d)
{
  int sum;

  // spoofing, ignore the result from the audio card if we want and use the
  //   spoofLevel, which can be set from tcl
  if(useSpoof)
  {
    d = spoofLevel;
  }

  if(slaveTransducer_ != NULL)
  {
    slaveTransducer_->set(d);
  }

  levelSumPtr->nextVal((int)d);

  if(d > threshold)
  {
    //printf("GaugeTransducer::set: setting to level %lf, name = %s\n", d, gaugeName);
    sum = activeSumPtr->nextVal(1);
  }
  else
  {
    sum = activeSumPtr->nextVal(0);
  }
  //printf("GaugeTransducer::set: moving active sum is %d, name = %s\n", sum, gaugeName);
}

void GaugeTransducer::setThreshold(double newThreshold)
{
//  printf("GaugeTransducer::setThreshold: setting threshold to %lf\n",	 newThreshold);
  threshold = newThreshold;
}

void GaugeTransducer::attachTransducer(const char* trans)
{
  slaveTransducer_ = (Transducer*)TclObject::lookup(trans);
}

int GaugeTransducer::command(int argc, const char* const* argv)
{
  const char* cmd = argv[1];
  Tcl& tcl = Tcl::instance();

  if(argc == 2)
  {
    if(!strcmp(cmd, "enableSpoofing"))
    {
      useSpoof = 1;
      return(TCL_OK);
    }
    if(!strcmp(cmd, "disableSpoofing"))
    {
      useSpoof = 0;
      return(TCL_OK);
    }
    if(!strcmp(cmd, "getActiveSum"))
    {
      tcl.resultf("%d", activeSumPtr->getCurrentSum());
      return(TCL_OK);
    }
    if(!strcmp(cmd, "getLevelSum"))
    {
      tcl.resultf("%d", levelSumPtr->getCurrentSum());
      return(TCL_OK);
    }
  }
  if(argc == 3)
  {
    // use this to manually call set() from Tcl
    if(!strcmp(cmd, "setCall"))
    {
      int val = atoi(argv[2]);
      set((double)val);
      return(TCL_OK);
    }
    if(!strcmp(cmd, "setSpoofLevel"))
    {
      int val = atoi(argv[2]);
      spoofLevel = (double)val;
      return(TCL_OK);
    }
    if(!strcmp(cmd, "setThreshold"))
    {
      int val = atoi(argv[2]);
      setThreshold((double)val);
      return(TCL_OK);
    }
    if(!strcmp(cmd, "attachTransducer"))
    {
      attachTransducer(argv[2]);
      return(TCL_OK);
    }
  }

  // default to TclObject
  return(TclObject::command(argc, argv));
}

