# qm.tcl --
#
#       QuestionMonitor.  This listens to the mic audio inputs and controls
#       which camera view is shown.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Import enable

import 405Client RTPSimpleAgent QMStateMachine

# use the Tcl namespace facility for global variables to prevent conflicts
namespace eval QuestionMonitor {
    variable show
    variable spoof
    variable enabled
    variable mark
    variable debugOn
}

# FIXME - this is silly to have to do this in apps that use these objects
Audio/OSSStereo set duplex_ 1

Class QuestionMonitor

# the spec argument is a list of all the other arguments
#
# 0 withGui
# 1 mode [record, spoof]
# 2 audio address
# 3 audio port
# 4 logfile out directory
# 5 logfile in directory (for Spoof Transducer)
# 6 speaker threshold
# 7 audienceLeft threshold
# 8 audienceCenter threshold
# 9 audienceRight threshold
#
QuestionMonitor instproc init {base spec} {
    $self instvar numDevices_         # how many sound cards are used
    $self instvar audio_              # arrays of audio devices and audio
    $self instvar controller_         #    controllers

    # transducers
    $self instvar gaugeTransducer_    # array of gauge transducers
    $self instvar masterTransducer_   # array of master transducers
    $self instvar logTransducer_      # array of log transducers to log raw
                                      #    set inputs to a file
    $self instvar spoofTransducer_    # array of spoof transducers to log raw
                                      #    set inputs to a file
    $self instvar meter_              # array of VUMeters associated with
                                      #    each input

    # miscellaneous
    $self instvar devList_            # mapping of unix devices to qm devs
    $self instvar timeSlice_          # interval between checks of mic inputs
    $self instvar 405_                # 405Client handle
    $self instvar output_             # encoding RVC we are working with
    $self instvar inputs_             # array of microphone input names
    $self instvar widgets_            # array of all the widgets used
    $self instvar inputsInfo_         # array containing info about each of
                                      #    the microphone inputs, such as name
    $self instvar withGui_            # 0 or 1, determines if gui is there
    $self instvar thresholds_

    # state machine stuff
    $self instvar stateMachine_       # actual state machine

    # for modeFile stuff
    $self instvar modeLogFile_

    # for switch log stuff
    $self instvar switchLogFile_

    # for camera log stuff
    $self instvar cameraLogFile_

    # for development/testing
    $self instvar markFile_           # file for marking special events such
                                      #    as question begin/end events
    $self instvar spoofMarkFile_      # file for marking special events such
                                      #    as question begin/end events
    $self instvar mode_               # record mode or spoof mode
    $self instvar audioAddress_
    $self instvar audioPort_
    $self instvar logOutDir_
    $self instvar logOutSuffix_
    $self instvar logInDir_

    # debug on/off
    #
    # this controls whether or not all the log.* files are generated for
    #   the development and test environment
    set QuestionMonitor::debugOn 1

    # get the args from the spec arguments
    set withGui_ [lindex $spec 0]
    set mode_ [lindex $spec 1]
    set audioAddress_ [lindex $spec 2]
    set audioPort_ [lindex $spec 3]
    set logOutDir_ [lindex $spec 4]
    set logInDir_ [lindex $spec 5]
    set thresholds_(speaker) [lindex $spec 6]
    set thresholds_(audienceLeft) [lindex $spec 7]
    set thresholds_(audienceCenter) [lindex $spec 8]
    set thresholds_(audienceRight) [lindex $spec 9]

    set QuestionMonitor::enabled 0

    # for question marking stuff
    set QuestionMonitor::mark(left) 0
    set QuestionMonitor::mark(center) 0
    set QuestionMonitor::mark(right) 0
    set QuestionMonitor::mark(generic) 0

    set numDevices_ 2
    for {set x 0} {$x < $numDevices_} {incr x 1} {
	set audio_($x) ""
	set devList_($x) -1
    }
    # you must modify devList(x) before calling open_devices to open specific
    #    sound devices
    # FIXME - this is a total hack for development ease - there should be
    #    some command line options or a configuration file that can be read in
    set hostname [info hostname]
    switch -exact -- $hostname {
	garfield.cs.berkeley.edu {
	    # garfield
	    set devList_(0) 2
	    set devList_(1) 3
	}
	htsr2.bmrc.berkeley.edu {
	    # htsr2
	    set devList_(0) 0
	    set devList_(1) 1
	}
	default {
	    puts stderr "unrecognized system, assuming audio ports 0 and 1"
	    set devList_(0) 0
	    set devList_(1) 1
	}
    }

    # assume:
    #    0,left = speaker
    #    0,right = audience left
    #    1,left = audience center
    #    1,right = audience right
    #
    # audience left/right is defined as facing the front of the room
    #
    if {$numDevices_ > 1} {
    set inputs_ [list "speaker" "audienceLeft" "audienceCenter" "audienceRight"]
    } else {
	set inputs_ [list "speaker" "audienceLeft"]
    }

    set inputsInfo_(speaker,device) 0
    set inputsInfo_(speaker,channel) left
    set inputsInfo_(speaker,shortName) "S"
    set inputsInfo_(audienceLeft,device) 0
    set inputsInfo_(audienceLeft,channel) right
    set inputsInfo_(audienceLeft,shortName) "AL"
    if {$numDevices_ > 1} {
	set inputsInfo_(audienceCenter,device) 1
	set inputsInfo_(audienceCenter,channel) left
	set inputsInfo_(audienceCenter,shortName) "AC"
	set inputsInfo_(audienceRight,device) 1
	set inputsInfo_(audienceRight,channel) right
	set inputsInfo_(audienceRight,shortName) "AR"
    }

    set output_ htsr

    # number of ms between checking of mic levels
    #
    # I did some experiments with the simulator and 250 is a good value,
    #   it follows the curve pretty closely; sim.samples can be used to
    #   check if the timeSlice_ value is small enough to accurately sample
    #   the microphone volume levels
    set timeSlice_ 250

    if {$withGui_} {
	$self initUI $base
    }

#    puts stdout "***********************made UI"

    $self open_devices

#    puts stdout "***********************opened devices"

    # FIXME - should check to make sure successfully got the sound cards
    for {set x 0} {$x < $numDevices_} {incr x 1} {
	set result [$audio_($x) obtain]
    }

    if {$withGui_} {
	$self attachMeters $base
    }


    # figure out the suffix
    set logOutSuffix_ [$self getSuffix $logOutDir_]

    foreach input $inputs_ {
	# set up the transducers
	#
	# with debugOff:
	# master --> gauge
	#        |-> VuMeter
	#
	# with debugOn:
	# spoof --> master --> gauge
	#                  |-> log
	#                  |-> VuMeter
	#
	# spoof reads from a file and passes the value to master
	# master passes to gauge and log
	# note that Spoofing from the GUI affects the master transducer
	#

	# master transducer
	set masterTransducer_($input) [new Transducer/Master]

	# gauge transducer
	# the 55 is the history size
	set gaugeTransducer_($input) [new Transducer/Gauge 55 "$input"]
	$masterTransducer_($input) attachTransducer $gaugeTransducer_($input)

	if {!$QuestionMonitor::debugOn} {
	    # attach the lead transducer to the controller
	    set cmd "$inputsInfo_($input,channel)-meter"
	    set dev "$inputsInfo_($input,device)"
	    $controller_($dev) $cmd $masterTransducer_($input)
	} else {
	    # logging transducer
	    set filename "$logOutDir_/log.$input${logOutSuffix_}"
	    set logTransducer_($input) [new Transducer/Log $filename]
	    $masterTransducer_($input) attachTransducer $logTransducer_($input)
	    $logTransducer_($input) enableLogging

	    # set up the spoof transducer
	    set filename "$logInDir_/log.$input"
	    set spoofTransducer_($input) [new Transducer/Spoof $filename]
	    if {$mode_ == "spoof"} {
		$spoofTransducer_($input) enableSpoofing
	    }
	    # attach the master transducer to the spoofTransducer
	    $spoofTransducer_($input) attachTransducer $masterTransducer_($input)
	    # attach the lead transducer to the controller
	    set cmd "$inputsInfo_($input,channel)-meter"
	    set dev "$inputsInfo_($input,device)"
	    $controller_($dev) $cmd $spoofTransducer_($input)
	}

	# this sets the threshold value for each gauge transducer
	$self setThreshold $input $thresholds_($input)

	# if we have a gui, attach the VuMeters to the master transducer
	if {$withGui_} {
	    set meter_($input) [new Meter/Linear $base.displays.${input}.meter]
	    pack $base.displays.${input}.meter -side left -padx 1 -pady 1 -fill y
	    $masterTransducer_($input) attachTransducer $meter_($input)
	}
    }

#    puts stdout "***********************created transducers"

    # FIXME - 0 is for the mike, 1 for line in, as determined by running vat,
    #   but we should do like audio agent and do some mapping
    for {set x 0} {$x < $numDevices_} {incr x 1} {
	$audio_($x) set_input_port 1
	$audio_($x) set_input_mute 0
    }

    $self adjustParams

#    puts stdout "***********************adjusted params"

    set 405_ [new 405Client]

    # set up state machine
    set gaugeList [array get gaugeTransducer_]
    set stateFileName "$logOutDir_/log.states${logOutSuffix_}"
    if {$QuestionMonitor::debugOn} {
	set getIndexCallback "$self getCurrentIndex"
    } else {
	set getIndexCallback ""
    }
    set stateMachine_ [new QMStateMachine $inputs_ $gaugeList 250 "$self switchCamera" $stateFileName $getIndexCallback]

#    puts stdout "***********************set up state machine"

    if {$QuestionMonitor::debugOn} {
	# for question marking stuff
	set filename "$logOutDir_/log.mark${logOutSuffix_}"
	set markFile_ [open $filename "w"]
	if {$mode_ == "spoof"} {
	    set filename "$logOutDir_/log.spoofMark${logOutSuffix_}"
	    set spoofMarkFile_ [open $filename "w"]
	}

	# for mode log stuff
	$self initModeLogFile

	# for switch log stuff
	$self initSwitchLogFile

	# for switch log stuff
	$self initCameraLogFile

	# this is for recording/spoofing, not needed for live control
	$self createRtpHandler $audioAddress_ $audioPort_
    }
}

QuestionMonitor instproc destroy {} {
    $self instvar numDevices_ audio_ gaugeTransducer_ inputs_ 405_ controller_
    $self instvar meter_ masterTransducer_ logTransducer_
    $self instvar markFile_ agent_ dHandler_ cHandler_ switchLogFile_ mode_
    $self instvar spoofMarkFile_ modeLogFile_

    # need to destroy before the transducers or disable the call to
    #    logT->getIndex before destroying the transducers
    if {[info exists agent_]} {
	$agent_ destroy
    }
    if {[info exists dHandler_]} {
	if {$dHandler_ != ""} {
	    $dHandler_ destroy
	}
    }
    if {[info exists cHandler_]} {
	if {$cHandler_ != ""} {
	    $cHandler_ destroy
	}
    }

    for {set x 0} {$x < $numDevices_} {incr x 1} {
	$audio_($x) release
    }

    foreach input $inputs_ {
	$masterTransducer_($input) destroy
	$gaugeTransducer_($input) destroy
	if {[info exists logTransducer_($input)]} {
	    $logTransducer_($input) destroy
	}
	if {[info exists meter_($input)]} {
#	    puts stdout "destroying meter $input"
	    $meter_($input) destroy
	}
    }
    $405_ destroy
    # need to destroy controllers and vumeters
    for {set x 0} {$x < $numDevices_} {incr x 1} {
	$controller_($x) destroy
    }

    catch {close $markFile_}
    if {$mode_ == "spoof"} {
	close $spoofMarkFile_
    }
    if {[info exists switchLogFile_]} {
	close $switchLogFile_
    }
    if {[info exists modeLogFile_]} {
	close $modeLogFile_
    }
}


QuestionMonitor public setOutput {output} {
    $self instvar output_ 405_

    switch -exact -- $output {
	projector -
	htsr -
	htsr2 {
#	    puts stdout "setting output to $output"
	    set output_ $output

	    # we have to reregister the callback with the updated filter to
	    #   only get the stuff we want
	    set filter [$self getFilter $output]
	    set callback "$self processAmxMsg"
	    $405_ callback_register $callback $filter
	    $405_ callback_enable $callback
	}
	default {
	    return -code error "QuestionMonitor::setOutput: output must be \[htsr, htsr2\]"
	}
    }
}

QuestionMonitor public enable {} {
    set QuestionMonitor::enabled 1
    $self processEnabledDisabled
}

QuestionMonitor public disable {} {
    set QuestionMonitor::enabled 0
    $self processEnabledDisabled
}

QuestionMonitor instproc processEnabledDisabled {} {
    $self instvar stateMachine_


    if {$QuestionMonitor::enabled} {
#	puts stdout "enabling question monitor"
	$stateMachine_ enable
    } else {
#	puts stdout "disabling question monitor"
	$stateMachine_ disable
    }
}

####################### UI Stuff #######################

QuestionMonitor instproc initUI {base} {
    $self instvar inputs_ numDevices_ frames_ checks_ inputsInfo_ mode_
    $self instvar indexWidget_

    frame $base.controls -borderwidth 3
    pack $base.controls -side top -fill x

    checkbutton $base.controls.onOff -text "Enable Question Monitor" -variable QuestionMonitor::enabled -command "$self processEnabledDisabled"
    pack $base.controls.onOff -side left

    # initially disabled
    set QuestionMonitor::enabled 0

    # display/hide other parts of the GUI
    set controlList [list]
    lappend controlList [list "displays" "Displays"]
    lappend controlList [list "spoof" "Spoof"]

    set curFrame $base.choiceFrame
    frame $curFrame
    pack $curFrame -side left

    foreach info $controlList {
	set control [lindex $info 0]
	set controlText [lindex $info 1]
	set curFrame $base.choiceFrame
	frame $curFrame.$control
	pack $curFrame.$control -side top -fill x
	set checks($control) [checkbutton $curFrame.$control.button -text $controlText -variable QuestionMonitor::show($control) -command "$self show $control"]
	pack $curFrame.$control.button -side left
    }

    # display frame and stuff
    set curFrame $base.displays
    set frames_(displays) [frame $curFrame]
#    pack $curFrame -side top

    set curFrame $base.displays.checks
    frame $curFrame -borderwidth 2 -relief sunken
    pack $curFrame -side bottom
    set checks_(all) [checkbutton $base.displays.checks.all -text all -variable QuestionMonitor::show(all) -command "$self showDisplays all"]
    pack $base.displays.checks.all -side left

    foreach input $inputs_ {
	set curFrame $base.displays.$input
	set frames_($input) [frame $curFrame -borderwidth 1 -relief solid]
	label $curFrame.label -text "$input"
	pack $curFrame.label -side top
	$self makeBar $curFrame "$input,threshold" "T" 0 100 left 200
	set checks_($input) [checkbutton $base.displays.checks.$input -text "$inputsInfo_($input,shortName)" -variable QuestionMonitor::show($input) -command "$self showDisplays $input"]
	pack $base.displays.checks.$input -side left
    }

    set curFrame $base.displays.device
    set frames_(device) [frame $curFrame -borderwidth 1 -relief solid]
    label $curFrame.label -text "Device"
    pack $curFrame.label -side top
    set checks_(device) [checkbutton $base.displays.checks.device -text device -variable QuestionMonitor::show(device) -command "$self showDisplays device"]
    pack $base.displays.checks.device -side left

    $self makeBar $curFrame "dummy,gain" "IG" 1 255 left 200
    $self makeBar $curFrame "dummy,silence" "ST" 1 100 left 200

    # spoofing stuff
    set curFrame $base.spoof
    set frames_(spoof) [frame $curFrame]
    #pack $curFrame -side top
    label $curFrame.label -text "Spoof"
    pack $curFrame.label

    foreach input $inputs_ {
	set curFrame $base.spoof.$input
	frame $curFrame -borderwidth 1 -relief solid
	pack $curFrame -side left
	label $curFrame.label -text "$inputsInfo_($input,shortName)"
	pack $curFrame.label -side top
	$self makeBar $curFrame "$input,spoof" "" 1 100 top 100
	checkbutton $curFrame.check -text "On" -variable QuestionMonitor::spoof($input) -command "$self handleSpoofCheck $input"
	pack $curFrame.check -side bottom
    }

    # to make debugging easier
#    $checks(displays) invoke
    $checks_(speaker) invoke
    $checks_(audienceLeft) invoke
    if {$numDevices_ > 1} {
	$checks_(audienceCenter) invoke
	$checks_(audienceRight) invoke
    }

    if {$QuestionMonitor::debugOn} {
	$self makeMarkUI $base
    }

    if {$mode_ == "spoof"} {
	set curFrame $base.indexNum
	frame $curFrame
	pack $curFrame -side top
	label $curFrame.label -text "Index:"
	pack $curFrame.label -side left
	set indexWidget_ [label $curFrame.value -text "0"]
	pack $curFrame.value -side right
    }
}


# this overrides the input from the spoofTransducer, since we cut in
#   at the master transducer
QuestionMonitor instproc handleSpoofCheck {name} {
    $self instvar masterTransducer_

    if {$QuestionMonitor::spoof($name)} {
	if {[info exists masterTransducer_($name)]} {
	    $masterTransducer_($name) enableSpoofing
	}
    } else {
	if {[info exists masterTransducer_($name)]} {
	    $masterTransducer_($name) disableSpoofing
	}
    }
}

QuestionMonitor instproc show {control} {
    $self instvar frames_

    if {$QuestionMonitor::show($control)} {
	pack $frames_($control) -side top -padx 2 -pady 2 -fill x
    } else {
	pack unpack $frames_($control)
    }
}

QuestionMonitor instproc showDisplays {name} {
    $self instvar frames_ inputs_ checks_

    if {$name == "all"} {
	foreach index [array names checks_] {
	    if {$index != "all"} {
		#puts stdout "index is $index"
		if {$QuestionMonitor::show(all)} {
		    $checks_($index) select
		} else {
		    $checks_($index) deselect
		}
	    }
	}
    }

    if {!$QuestionMonitor::show($name)} {
	$checks_(all) deselect
    }

    # do it this way so the interface always looks the same
    foreach index $inputs_ {
	pack unpack $frames_($index)
    }
    pack unpack $frames_(device)
    foreach index $inputs_ {
	if {$QuestionMonitor::show($index)} {
	    pack $frames_($index) -side left -padx 1 -pady 1
	}
    }
    if {$QuestionMonitor::show(device)} {
	pack $frames_(device) -side left -padx 1 -pady 1
    }
}

QuestionMonitor instproc makeBar {base name textLabel min max side size} {
    $self instvar widgets_

    set curFrame "$base.$name"
    frame $curFrame
    pack $curFrame -side $side

    set widgets_($name) [scale $curFrame.bar -from $max -to $min -length $size -variable x.$name -orient vertical -label $textLabel -showvalue true -command "$self barHandler $name"]
    pack $curFrame.bar -side $side
#    if {$makeButton == "true"} {
#	button $curFrame.button -text "Set" -command "$self setBarValue $name"
#	pack $curFrame.button -side left
#    }
}

QuestionMonitor instproc barHandler {name value} {
    $self instvar inputs_ gaugeTransducer_ controller_ numDevices_ audio_
    $self instvar masterTransducer_

    #puts stdout "$name bar has new value $value"

    set temp [split $name ","]
#    puts stdout "temp is $temp, gaugeTransducer_ is [array get gaugeTransducer_]"
    set input [lindex $temp 0]
    set action [lindex $temp 1]

    # need to check if the transducer has been created since GUI is made first

    #puts stdout "doing $action to $input"
    switch -exact -- $action {
	threshold {
	    if {[info exists gaugeTransducer_($input)]} {
		$gaugeTransducer_($input) setThreshold $value
	    }
	}
	gain {
	    for {set x 0} {$x < $numDevices_} {incr x 1} {
		$audio_($x) set_input_gain $value
	    }
	}
	silence {
	    for {set x 0} {$x < $numDevices_} {incr x 1} {
	    	$controller_($x) silence-thresh $value
	    }
	}
	spoof {
	    if {[info exists masterTransducer_($input)]} {
		$masterTransducer_($input) setSpoofLevel $value
	    }
	}
	default {
	    return -code error "QuestionMonitor::barHandler: unrecognized action $action"
	}
    }
}

########################## device stuff ############################

QuestionMonitor instproc setThreshold {input value} {
    $self instvar gaugeTransducer_ withGui_ widgets_

    if {[info exists gaugeTransducer_($input)]} {
	if {$withGui_} {
	    # this will cause barHandler to get called which sets the threshold
	    $widgets_($input,threshold) set $value
	} else {
	    $gaugeTransducer_($input) setThreshold $value
	}
    }
}

QuestionMonitor instproc open_devices {} {
    $self instvar numDevices_ audio_ controller_ devList_

    # FIXME - according to agent-audio.tcl, we have to do this, but it doesn't
    #   make much sense
    AudioControllerErik set max_playout_ 500

    set dev "Audio/OSSStereo"
    set duplex "FullDuplex"

    for {set x 0} {$x < $numDevices_} {incr x 1} {
	set audio_($x) [new $dev]
#	puts stdout "setting device $x"
	if {$devList_($x) != -1} {
	    $audio_($x) set_device $devList_($x)
	}
#	puts stdout "set device $x with $devList_($x)"
	set controller_($x) [new AudioControllerErik/$duplex]
	$controller_($x) audio $audio_($x)
    }
}

# this just sets some variables needed for the VuMeters
QuestionMonitor instproc attachMeters {base} {
    option add *VatVU.foreground black startupFile
    option add *VatVU.peak gray50 startupFile
    option add *VatVU.hot firebrick1 startupFile
    option add *VatVU.hotLevel 90 startupFile
}

QuestionMonitor instproc adjustParams {} {
    $self instvar numDevices_ audio_ controller_ widgets_ withGui_

    if {$withGui_} {
	$widgets_(dummy,gain) set 150
	$widgets_(dummy,silence) set 20
    }
    # if the frame is not displayed, barHandler won't be called and the
    #    values won't be set, so call explicitly
    $self barHandler "dummy,gain" 150

    # this is the default for Vat; I don't think it affects things
    $self barHandler "dummy,silence" 20

    # need to set the output gain or it will be set to 0 and programs
    #   opening the device may not set the output level
    for {set x 0} {$x < $numDevices_} {incr x 1} {
	$audio_($x) set_output_gain 180
	$controller_($x) talk-thresh 0
    }
}

########################## switching stuff ############################

# FIXME - switching b/w audienceLeft and audienceRight looks bad because
#   the camera pans very quickly.  We would like to be able to detect
#   that we are switching from one audience view to another.  When we do
#   we should show the wide camera view while the audience camera is moving.
#
# Currently, we don't know what we are currently displaying.  We need to
#   add AMX event monitoring to QM so it can know what camera view is
#   being shown.  This would also help out with situations such as long
#   questions where the VD times out to the speaker mode before the audience
#   member is done asking his question.
#
# Unfortunately, then you have to deal with avoiding "rapid switching",
#   where you switch camera views too rapidly.  Perhaps the state machine
#   itself should be aware of the current camera view and disable itself
#   for RECENT_DURATION whenever it sees a camera switch.  This would simplify
#   the state machine and make it more general.

QuestionMonitor instproc switchCamera {newMode} {
    $self instvar 405_ output_ modeLogFile_

    if {$QuestionMonitor::debugOn} {
	set index [$self getCurrentIndex]
	set sysTime [gettimeofday]
	puts $modeLogFile_ "$index $newMode $sysTime"
    }
    # FIXME - for debugging only
    puts stdout "\t\tswitching to $newMode"
    switch -exact -- $newMode {
	speaker {
	    $405_ matrix_switchVideoStream "speakerCamera" $output_
	}
	audienceLeft {
	    set audienceVcc3_ [$405_ getVcc3Client "audience"]
	    $audienceVcc3_ do vcc3_fastMove -15 -190 270
	    $405_ matrix_switchVideoStream "audienceCamera" $output_
	}
	audienceCenter {
	    set audienceVcc3_ [$405_ getVcc3Client "audience"]
	    $audienceVcc3_ do vcc3_fastMove -229 -154 317
	    $405_ matrix_switchVideoStream "audienceCamera" $output_
	}
	audienceRight {
	    set audienceVcc3_ [$405_ getVcc3Client "audience"]
	    $audienceVcc3_ do vcc3_fastMove -420 -144 440
	    $405_ matrix_switchVideoStream "audienceCamera" $output_
	}
	default {
	}
    }
}

QuestionMonitor instproc initModeLogFile {} {
    $self instvar logOutDir_ logOutSuffix_ modeLogFile_

    set filename "$logOutDir_/log.modes${logOutSuffix_}"
    if {[catch {set modeLogFile_ [open $filename "w"]}]} {
	unset modeLogFile_
    }
}

################################ question marker #######################

QuestionMonitor instproc makeMarkUI {base} {
    # question marker
    set curFrame $base.mark
    frame $curFrame -borderwidth 1 -relief solid
    pack $curFrame -side top

    label $curFrame.label -text "Mark Question"
    pack $curFrame.label -side top

    set curFrame $base.mark.buttons
    frame $curFrame
    pack $curFrame -side top

    checkbutton $curFrame.left -text "L" -variable QuestionMonitor::mark(left) -command "$self markQuestion left"
    pack $curFrame.left -side left
    checkbutton $curFrame.center -text "C" -variable QuestionMonitor::mark(center) -command "$self markQuestion center"
    pack $curFrame.center -side left
    checkbutton $curFrame.right -text "R" -variable QuestionMonitor::mark(right) -command "$self markQuestion right"
    pack $curFrame.right -side left
    # in case you can't tell where it's coming from
    checkbutton $curFrame.generic -text "G" -variable QuestionMonitor::mark(generic) -command "$self markQuestion generic"
    pack $curFrame.generic -side left
}

QuestionMonitor instproc markQuestion {region} {
    if {$QuestionMonitor::mark($region)} {
	$self markQuestionStart $region
    } else {
	$self markQuestionEnd $region
    }
}

QuestionMonitor instproc markQuestionStart {region} {
    $self instvar markFile_ mode_ spoofMarkFile_

    set index [$self getCurrentIndex]
    set sysTime [gettimeofday]
    puts $markFile_ "start $region $index $sysTime"
    if {$mode_ == "spoof"} {
	set index [$self getSpoofIndex]
	puts $spoofMarkFile_ "start $region $index $sysTime"
    }
}

QuestionMonitor instproc markQuestionEnd {region} {
    $self instvar markFile_ mode_ spoofMarkFile_

    set index [$self getCurrentIndex]
    set sysTime [gettimeofday]
    puts $markFile_ "end $region $index $sysTime"
    if {$mode_ == "spoof"} {
	set index [$self getSpoofIndex]
	puts $spoofMarkFile_ "end $region $index $sysTime"
    }
}

################################ record/spoof stuff #######################

QuestionMonitor instproc createRtpHandler {addr port} {
    $self instvar logTransducer_ agent_ dHandler_ cHandler_ dSlave_
    $self instvar mode_ logOutDir_ logOutSuffix_ logInDir_

    set dSlave_ ""
    if {$mode_ == "spoof"} {
	set callback "$self setIndex"
	set tsInFile "$logInDir_/log.timestamps.processed"
	set dSlave_ [new Module/RTPQMIndexer $tsInFile $callback]
    }
    set tsOutFile "$logOutDir_/log.timestamps${logOutSuffix_}"

    set dHandler_ [new Module/RTPQMRcvr $tsOutFile $logTransducer_(speaker) $dSlave_]
    # don't need a control handler since MARS doesn't record RTCP packets
    set cHandler_ ""
    set agent_ [new RTPSimpleAgent "audio" $addr/$port $dHandler_ $cHandler_]
}

# this is used in spoofing mode
QuestionMonitor instproc setIndex {index} {
    $self instvar inputs_ spoofTransducer_ indexWidget_ mode_

    foreach input $inputs_ {
	if {[info exists spoofTransducer_($input)]} {
	    $spoofTransducer_($input) setIndex $index
	}
    }

    if {$mode_ == "spoof"} {
	$indexWidget_ configure -text "$index"
    }
}

QuestionMonitor instproc getCurrentIndex {} {
    $self instvar logTransducer_ spoofTransducer_

    # assume speaker is always there
    set index [$logTransducer_(speaker) getIndex]
    return $index
}

QuestionMonitor instproc getSpoofIndex {} {
    $self instvar logTransducer_ spoofTransducer_

    # assume speaker is always there
    set index [$spoofTransducer_(speaker) getIndex]
    return $index
}

########################### switch log stuff ############################

QuestionMonitor instproc initSwitchLogFile {} {
    $self instvar logOutDir_ logOutSuffix_ mode_ 405_ output_ switchLogFile_

    if {$mode_ == "spoof"} {
	# don't do this part
	return ""
    }

    set filename "$logOutDir_/log.switches${logOutSuffix_}"
    if {[catch {set switchLogFile_ [open $filename "w"]}]} {
	unset switchLogFile_
    }

    set callback "$self processAmxMsg"
    set filter [$self getFilter $output_]
    $405_ callback_register $callback $filter
    $405_ callback_enable $callback

    set output $output_
    set input [$405_ matrix_getInputSource $output]
    set sysTime [gettimeofday]
    set index [$self getCurrentIndex]
    puts $switchLogFile_ "$output $input $index $sysTime"
}

QuestionMonitor instproc getFilter {watchOutput} {
    # This filters out all events except for matrixSwitch events that have an
    #   output of $watchOutput
    set filter ""
    set filter "$filter set doCallback 0\n"
    set filter "$filter set info \[lindex \$arg 0\]\n"
    set filter "$filter set type \[lindex \$info 0\]\n"
    set filter "$filter set data \[lindex \$arg 1\]\n"
    set filter "$filter if \{\$type == \"matrixSwitch\"\} \{\n"
    set filter "$filter set output \[lindex \$data 1\]\n"
    set filter "$filter if \{(\$output == \"$watchOutput\")\} \{\n"
    set filter "$filter set doCallback 1\n"
    set filter "$filter \}\n"
    set filter "$filter \}\n"
    return $filter
}

QuestionMonitor public processAmxMsg { amxMsg } {
    $self instvar switchLogFile_ output_

    if {[llength $amxMsg] != 0} {
	set eventInfo [lindex $amxMsg 0]
	set type [lindex $eventInfo 0]
	set eventData [lindex $amxMsg 1]
    }

    if {$type == "matrixSwitch"} {
	set input [lindex $eventData 0]
	set output [lindex $eventData 1]
	if {$output == $output_} {
	    set sysTime [gettimeofday]
	    set index [$self getCurrentIndex]
#	    puts stdout "got matrix switch from $input to $output"
	    puts $switchLogFile_ "$output $input $index $sysTime"

	    if {$input == "audienceCamera"} {
	        after 500 "$self recordAudiencePosition 10"
	    }
	}
    }

    return ""
}

########################### camera log stuff ############################

QuestionMonitor instproc initCameraLogFile {} {
    $self instvar logOutDir_ logOutSuffix_ cameraLogFile_ audienceVcc3_ 405_

    set filename "$logOutDir_/log.camera${logOutSuffix_}"
    if {[catch {set cameraLogFile_ [open $filename "w"]}]} {
	unset cameraLogFile_
    }
    catch {set audienceVcc3_ [$405_ getVcc3Client "audience"]}
}

QuestionMonitor instproc recordAudiencePosition {count} {
    $self instvar cameraLogFile_ audienceVcc3_

    if {$count > 0} {
	set sysTime [gettimeofday]
	set index [$self getCurrentIndex]
	set info [$audienceVcc3_ do vcc3_getPanTiltInfo]
	if {[llength $info] > 0} {
	    set panPos [lindex $info 2]
	    set tiltPos [lindex $info 3]
	    puts $cameraLogFile_ "$index $panPos $tiltPos $sysTime"
	}
	set count [expr $count - 1]
	after 500 "$self recordAudiencePosition $count"
    }
}

######################## log out suffix stuff #####################

QuestionMonitor instproc getSuffix {dir} {
    # FIXME - I just use the log.speaker file to check for the suffix; this
    #   is sort of a hack
    if {!$QuestionMonitor::debugOn} {
	return ""
    }
    set filename "$dir/log.speaker"
    if {[catch {open $filename "r"}]} {
	# no log files exist, so it's safe to use no suffix
	return ""
    }
    # if we got here, then a log.speaker already exists, find a suitable suffix
    for {set x 1} {$x < 1000} {incr x 1} {
	set filename "$dir/log.speaker.$x"
	if {[catch {open $filename "r"}]} {
	    return ".$x"
	}
    }
    # if we got here, then there are over 1000 log files with suffixes!
    return ".toomany"
}