# panel.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/vat/panel.tcl,v 1.21 2002/02/03 04:30:33 lim Exp $


import AudioArbiter CheckButton


#
# This UI element interfaces to the AudioAgent, providing a speaker meter,
# mike meter, buttons for muting and switching input/output ports, and a
# button for demanding control over the host's audio device.
#
Class AudioPanel

#
# Given the AudioAgent, <i>agent</i>, create an AudioArbiter, and a
# frame within the supplied widgetpath, <i>top</i>, that contains a
# speaker meter, a mike meter, and a button for "Keep Audio".
#
AudioPanel instproc init { top agent } {
	$self add_default audioFont [$self get_option helv10]
	$self add_default ctrlFont [$self get_option helv10b]

	# used by Meter/Linear c++ code
	global tcl_platform
	# use the default colors if we are in windows
	if {$tcl_platform(platform) != "windows"} {
		option add *VatVU.background gray85 startupFile
	}
	option add *VatVU.foreground black startupFile
	option add *VatVU.peak gray50 startupFile
	option add *VatVU.hot firebrick1 startupFile
	option add *VatVU.hotLevel 90 startupFile
	if { [winfo depth .] == 1 } {
		# make mono look better
		option add *VatVU.background white startupFile
		option add *VatVU.hot gray50 startupFile
	}

	frame $top.panel
	pack $top.panel -side right -fill y

	set w $top.panel.audio
	frame $w
	pack $w -expand 1 -fill y

	option add *AudioPanelPane*borderWidth 2 widgetDefault
	frame $w.spkr -borderwidth 0 -class AudioPanelPane
	$self instvar spkr_meter_ mike_meter_ agent_ arbiter_
	set agent_ $agent
	set arbiter_ [new AudioArbiter $agent]
	set spkr_meter_ [$self mk.pane $w.spkr output speaker listen]
	frame $w.mike -borderwidth 0 -class AudioPanelPane
	set mike_meter_ [$self mk.pane $w.mike input mike talk]
	pack $w.spkr $w.mike -side left -expand 1 -fill y

	set f [$self get_option ctrlFont]
	checkbutton $top.panel.button -text "Keep Audio" -font $f \
		-command "$self invoke_keep_audio" \
		-variable [$self tkvarname audioHeld] -anchor c \
		-relief ridge -borderwidth 2 -highlightthickness 0
	global keepAudioButton
	set keepAudioButton $top.panel.button
	# sync the arbiter with the default state of audioHeld
	$self invoke_keep_audio
	pack $top.panel.button -fill x -side bottom -anchor c -before $w
}

#
# If the "input" or "output", as specified by <i>which</i>, is unmuted,
# demand the audio device.
#
AudioPanel instproc mute_invoke { which } {
	$self instvar agent_ arbiter_ $which\_mute_
	if { "[[$self set $which\_mute_] get-val]" == "unmuted" } {
		$agent_ set_$which\_mute 0
		if ![$agent_ have_audio] {
			$arbiter_ demand
		}
	} else {
		$agent_ set_$which\_mute 1
	}
}

#
# Demand the audio device.
#
AudioPanel instproc invoke_keep_audio {} {
	$self instvar arbiter_
	$self tkvar audioHeld
	$arbiter_ hold $audioHeld
}

#
# Set the gain of the "input" or "output", as specified by <i>which</i> to <i>level</i>.
#
AudioPanel instproc setgain { which level } {
	$self instvar agent_
	$agent_ set_$which\_gain $level
}

#
# If <i>yesno</i> is true, bind the transducer input/output to the speaker/mike meters.
#
AudioPanel instproc enable_meters yesno {
	$self instvar spkr_meter_ mike_meter_ agent_
	if $yesno {
		$agent_ bind_transducer output $spkr_meter_
		$agent_ bind_transducer input $mike_meter_
	} else {
		$agent_ bind_transducer output ""
		$agent_ bind_transducer input ""
		$spkr_meter_ set_level 0.
		$mike_meter_ set_level 0.
	}
}

#
# Fetch the bitmap of the image referenced by <i>name</i>.
#
AudioPanel instproc lookup_bitmap { name } {
	switch -glob $name {
		mike { return mike }
		mic* { return mike }
		speaker { return speaker }
		jack { return headphone }
		headphone { return headphone }
		lineout2 { return lineout2 }
		lineout3 { return lineout3 }
		lineout* { return lineout }
		line*in2 { return linein2 }
		cd*       { return linein2 }
		linein3 { return linein3 }
		line*in { return linein }
		mix*    { return linein3 }
		synth*  { return linein3 }
		default { return linein3 }
	}
}

#
# Set the port for the AudioAgent's "input"/"output" (as specified by
# <i>which</i>) to the provided <i>port</i>, and configure the provided
# <i>button</i> with the appropriate bitmap.  Also, set the gain for the
# provided <i>scale</i>.
#
AudioPanel instproc setPort { which button scale port } {
	$self instvar agent_
	$agent_ set_$which\_port $port
	$button configure -bitmap [$self lookup_bitmap $port]
	$scale set [$agent_ get_$which\_gain]
}

#
# Change the port for the AudioAgent's "input"/"output" (as specified by
# <i>which</i>).  Reconfigure the <i>button</i> and <i>scale</i> appropriately.
#
AudioPanel instproc changePort { which button scale } {
	$self instvar agent_
	set ports [$agent_ get_$which\_ports]
	set n [$agent_ get_$which\_portno]
	if { $n < 0 } {
		return
	}
	incr n
	if { $n >= [llength $ports] } {
		set n 0
	}
	$self setPort $which $button $scale [lindex $ports $n]
}

#
# Create and return, but do not pack, a meter using widgetpath <i>w</i>.
# Use <i>which</i> to specify "input" or "output" port, <i>bitmap</i> to
# identify the image to appear on the change-port button above the meter
# (e.g. "speaker", "mike"), plus a <i>label</i> to appear next to a
# mute-checkbutton above the <i>bitmap</i> (e.g. "listen", "talk").
#
AudioPanel instproc mk.pane { w which bitmap label } {
	set f [$self get_option audioFont]
	frame $w.mute -relief raised -borderwidth 1
	#bd
	set cb [new CheckButton $w.mute.b -text $label -font $f -relief ridge \
		-anchor c \
		-command "$self mute_invoke $which" \
		-onvalue unmuted \
		-offvalue muted \
		-highlightthickness 0]
	$cb set-val unmuted
	$self instvar $which\_mute_
	set $which\_mute_ $cb

	pack $w.mute.b -expand 1 -fill x

	#bd
	frame $w.select -relief raised
	#bd
	button $w.select.b -bitmap $bitmap -relief flat \
		-command "$self changePort $which $w.select.b $w.frame.scale" \
		-height 24 -highlightthickness 1
	pack $w.select.b -expand 1 -fill x
	#
	# Disable button if there aren't any choices
	#
	$self instvar agent_
	if { [llength [$agent_ get_$which\_ports]] <= 1 } {
		$w.select.b configure -state disabled
	}

	#bd
	frame $w.frame -relief raised
	set meter [new Meter/Linear $w.frame.meter]
	#bd
	scale $w.frame.scale -orient vertical \
			-showvalue 0 \
			-from 256 -to 0 \
			-command "$self setgain $which" \
			-relief groove -length 200 \
			-highlightthickness 0
	global $which\Scale $which\PortButton
	set $which\Scale $w.frame.scale
	set $which\PortButton $w.select.b

	pack $w.frame.meter $w.frame.scale -side left -expand 1 -fill y

	pack $w.mute $w.select -fill x
	pack $w.frame -expand 1 -fill both

	return $meter
}

#
AudioPanel instproc ptt-press {} {
	$self instvar input_mute_
	if { "[$input_mute_ get-val]" == "muted" } {
		$input_mute_ invoke
	}
}

#
AudioPanel instproc ptt-release {} {
	$self instvar input_mute_
	if { "[$input_mute_ get-val]" == "unmuted" } {
		$input_mute_ invoke
	}
}

#
# If <i>v</i> is true, mute user's input.
#
AudioPanel instproc set_recv_only v {
	$self instvar input_mute_
	if $v {
		$self ptt-release
		$input_mute_ configure -state disabled
	} else {
		$input_mute_ configure -state normal
	}
}


AudioPanel instproc is_recv_only {} {
	$self instvar input_mute_
	if { [$input_mute_ cget -state] == "disabled" } { return 1 } \
			else { return 0 }
}


#
# If the AudioAgent does not have_audio and the user has his/her output unmuted,
# ask for control of the audio device.
#
AudioPanel instproc action {} {
	$self instvar agent_ arbiter_ output_mute_
	if { ![$agent_ have_audio] && "[$output_mute_ get-val]" == "unmuted" } {
		$arbiter_ request
	}
}

#
# Cancels the AudioArbiter's timer.
#
AudioPanel instproc cancel_timer {} {
	$self instvar arbiter_

	# calls AudioArbiter's destroy method
	delete $arbiter_
}
