# catalog.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/archive/catalog.tcl,v 1.13 2002/02/03 04:25:07 lim Exp $


# This class implements functionality dealing with the catalog file,
# which is used in the archive system.  Catalog files list all the
# stream files which make up a presentation.  Optionally, catalog file
# may contain the sdp announcement from when the presentation was
# recorded.
# Status: Beta
# Author: Yatin Chawathe
Class SessionCatalog


SessionCatalog public init { } {
    $self instvar sdp_

    $self next
    $self set file_ ""
    $self set filename_ ""
    $self set sdp_ ""
    $self set info_ ""
}


SessionCatalog public destroy { } {
    $self close
    $self next
}

# Open a catalog file
SessionCatalog public open { filename { mode "r" } { permissions 0644 } } {
    $self instvar file_ filename_ line_no_
    set file_ [open $filename $mode $permissions]

    $self clear
    set filename_ $filename
}



SessionCatalog private clear { } {
    $self set filename_ ""
    $self set line_no_ 0

    $self instvar streams_
    catch { unset streams_ }
    set streams_(all) ""
}


SessionCatalog instproc close { } {
    $self instvar file_
    if { $file_!="" } {
	close $file_
	set file_ ""
	set filename_ ""
    }
}


SessionCatalog instproc is_opened { } {
    $self instvar file_
    if { $file_=="" } { return 0 } else { return 1 }
}


SessionCatalog instproc filename { } {
    return [$self set filename_]
}


SessionCatalog instproc write_sdp { sdp } {
    $self instvar file_
    if { $file_=="" } { error "file not opened" }
    puts $file_ "START_SDP"
    puts $file_ $sdp
    puts $file_ "END_SDP"
    flush $file_
}


SessionCatalog instproc write_info { info } {
    $self instvar file_
    if { $file_ == "" } { error "file not opened" }
    puts $file_ "START_INFO"
    puts $file_ $info
    puts $file_ "END_INFO"
    flush $file_
}


SessionCatalog instproc write_stream { id session datafile indexfile } {
    $self instvar file_
    if { $file_=="" } { error "file not opened" }
    puts $file_ "START_STREAM"
    puts $file_ "\tid=$id"
    puts $file_ "\tsession=$session"
    puts $file_ "\tdatafile=$datafile"
    puts $file_ "\tindexfile=$indexfile"
    puts $file_ "END_STREAM"
    flush $file_
}

# Parse a catalog file, where the file has already been open ed
SessionCatalog public read { } {
    $self instvar file_ line_no_
    if { $file_=="" } { error "file not opened" }

    while { [$self read_line_ line] } {
	if { ![regexp "START_(.*)" $line dummy block_type] } {
	    # we ought to have got a START_something line
	    error "parse error at line $line_no_ in header file"
	}

	$self read_block_ [string tolower $block_type]
    }
}

# Parse a catalog file, where the content is handed through the msg parameter.
# msg should be a list of lines
SessionCatalog public parse {msg} {
	$self instvar msg_ cur_line_

	$self clear



	set msg_ [split [string trim $msg] "\n"]
	for {set cur_line_ 0} {$cur_line_ < [llength $msg_]} {incr cur_line_} {
		set line [lindex $msg_ $cur_line_]
		if { ![regexp "START_(.*)" $line dummy block_type] } {
			error "parse error ($line)"
		}
		incr cur_line_
		$self parse_block_ [string tolower $block_type]
    }

}



SessionCatalog private read_line_ { lineVar } {
    upvar $lineVar line
    $self instvar file_ line_no_
    while { ![eof $file_] } {
	incr line_no_
	gets $file_ line
	set line [string trim $line]
	if { [string length $line]!=0 && [string index $line 0]!="#"} {
	    # this is neither an empty line nor a comment
	    return 1
	}
    }

    return 0
}

SessionCatalog private parse_block_ { block_type  } {
	$self instvar msg_ cur_line_

	set msg {}
	for {} {$cur_line_ < [llength $msg_]} {incr cur_line_} {
		set line [lindex $msg_ $cur_line_]
		if { [regexp "END_(.*)" $line dummy end_type] } {
			# we got an END; check if it is the correct one,
			# and return
			set end_type [string tolower $end_type]
			if { $block_type != $end_type } {
				# expected an END_"block_type"
				error "expected END_$block_type;\
						got END_$end_type at\
						line $line_no_ in header file"
			}

			$self handle_read_${block_type}_ $msg
			return
		}
		append msg "$line\n"
	}
	error "unexpected EOF at line $cur_line_; expected END_$block_type"
}



SessionCatalog private read_block_ { block_type } {
    set msg {}
    while { [$self read_line_ line] } {
	if { [regexp "END_(.*)" $line dummy end_type] } {
	    # we got an END; check if it is the correct one,
	    # and return
	    set end_type [string tolower $end_type]
	    if { $block_type != $end_type } {
		# expected an END_"block_type"
		error "expected END_$block_type;\
			got END_$end_type at\
			line $line_no_ in header file"
	    }

	    $self handle_read_${block_type}_ $msg
	    return
	}

	append msg "$line\n"
    }

    error "unexpected EOF at line $line_no_; expected END_$block_type"
}


SessionCatalog instproc handle_read_info_ { msg } {
    $self instvar info_
    append info_ $msg
    return
}


# The DESC block is used in the mars/rover system.  It's used to pass
# server address and presentation file name to the rover client.
SessionCatalog private handle_read_descr_ {msg } {
	$self instvar desc_

	set desc_ $msg
	return

}


SessionCatalog private handle_read_sdp_ { msg } {
    $self instvar sdp_

    set sdp_ $msg

# Printing all SDP Messages slows down the mash_server playback_agent.
#    puts "SDP----"
#    puts $msg
#    puts "SDP----"

    return
}


SessionCatalog public get_sdp {} {
    $self instvar sdp_
    return $sdp_
}


SessionCatalog public get_html_time { type } {
	set time [$self get_info record_$type]
	if { $time == {} } { return {} }

	set day   [lindex $time 0]
	set month [lindex $time 1]
	set date  [lindex $time 2]
	set year  [lindex $time 3]
	set time  [lindex $time 5]

	set m [string tolower $month]
	set month [lsearch "january february march april may june july august\
			september october november december" $m]
	if { $month == -1 } {
		set month [lsearch "jan feb mar apr may jun jul aug sep oct\
				nov dec" $m]
		if { $month == -1 } { set month $m } else { incr month }
	} else { incr month }
	set date [string trim $date ,]
	set year [string range $year 2 end]

	return "$day $month/$date/$year $time"
}


SessionCatalog public get_info { type } {

    $self instvar info_

    set return_info ""
    set info_list [split $info_ "=\n"]
    set index [lsearch -exact $info_list $type]
    if { $index != -1 } {
	set return_info [lindex $info_list [expr $index + 1]]
    }
    return $return_info
}


SessionCatalog public get_desc {} {

    $self instvar desc_
    return $desc_
}


SessionCatalog private handle_read_stream_ { msg } {

    $self instvar streams_ filename_ line_no_

    foreach line [split $msg "\n"] {
	if { $line=={} } continue
	set line [split $line "="]
	set attribute [string trim [lindex $line 0]]
	set value     [string trim [lindex $line 1]]
	set header($attribute) $value
    }

    # check if all the fields exist:
    if { ![info exists header(id)] } {
	error "could not find the \"id\" field in STREAM block at\
		line $line_no_"
    }
    set id $header(id)

    if { ![info exists header(session)] } {
	error "could not find the \"session\" field in STREAM block at\
		line $line_no_"
    }
    set streams_($id,session) $header(session)

    if { ![info exists header(datafile)] } {
	error "could not find the \"datafile\" field in STREAM block\
		at line $line_no_"
    } else {
	# if this is a relative path, prepend the directory path of the
	# header file to this path

	set streams_($id,datafile) [file join \
		[file dirname $filename_] $header(datafile)]
    }

    if { [info exists header(indexfile)] } {
	# if this is a relative path, prepend the directory path of the
	# header file to this path

	if { $header(indexfile)=="" } {
	    set streams_($id,indexfile) ""
	} else {
	    set streams_($id,indexfile) [file join [file dirname \
		    $filename_] $header(indexfile)]
	}
    } else {
	# a non-existent index field
	# automatically use [datafile minus extension].idx

	set streams_($id,indexfile) "[file rootname \
		$streams_($id,datafile)].idx"
    }

    lappend streams_(all) $id
}


SessionCatalog instproc info { method args } {
    eval [list $self] [list info.$method] $args
}


SessionCatalog instproc info.streams { } {
    $self instvar streams_
    return $streams_(all)
}


SessionCatalog instproc info.session { id } {
    $self instvar streams_
    return $streams_($id,session)
}


SessionCatalog instproc info.datafile { id } {
    $self instvar streams_
    return $streams_($id,datafile)
}


SessionCatalog instproc info.indexfile { id } {
    $self instvar streams_
    return $streams_($id,indexfile)
}

