/*
 * vw.cc --
 *
 *      Implements the video wiget that's used to display video 
 *      in tools, e.g., vic.  Multiple implementations are provided in 
 *      order to take advantage of various OS-specific performance features.
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/render/vw.cc,v 1.33 2002/02/23 22:55:47 lim Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* stuff needed to include declaration of TkPutImage */
#include <math.h>
extern "C" {
#define HAVE_LIMITS_H 1
#ifndef WIN32
#   define HAVE_UNISTD_H 1
#endif
#include <tkPort.h>
#undef strcasecmp
#define TIMEZONE_DEFINED_
}

#ifdef WIN32
extern "C" unsigned long *win32Colors;
extern "C" int win32NColors;
#endif

extern "C" {
#include <tk.h>
#ifdef USE_SHM
#include <sys/ipc.h>
#include <sys/shm.h>
#if defined(sun) && !defined(__svr4__)
int shmget(key_t, int, int);
char *shmat(int, char*, int);
int shmdt(char*);
int shmctl(int, int, struct shmid_ds*);
#endif
#ifdef __osf__
int XShmGetEventBase(struct _XDisplay *);
#else
int XShmGetEventBase(Display *);
#endif
#ifdef sgi
#define XShmAttach __XShmAttach__
#define XShmDetach __XShmDetach__
#define XShmPutImage __XShmPutImage__
#endif
#include <X11/extensions/XShm.h>
#ifdef sgi
#undef XShmAttach
#undef XShmDetach
#undef XShmPutImage
int XShmAttach(Display*, XShmSegmentInfo*);
int XShmDetach(Display*, XShmSegmentInfo*);
#ifndef sgi
int XShmPutImage(Display*, Drawable, GC, XImage*, int, int, int, int,
		 int, int, int);
#endif

#endif
#endif
#include <X11/Xutil.h>
}
#include "color.h"
#include "tclcl.h"
#include "config.h"
#include "rgb-converter.h"

#include "vw.h"

#ifdef USE_DDRAW

#if WINVER < 0x0500 // if Win98 and Win2k API is not enabled
#define COMPILE_MULTIMON_STUBS // then compile stubs that substitute for missing functions
#include <multimon.h>

// #define SHOW_MSGS // define to have run-time status messages displayed

#endif

#include <tkWin.h> // needed for Tk_GetHWND

#endif

static class VideoWindowClass : public TclClass {
public:
	VideoWindowClass() : TclClass("VideoWindow") {}
	TclObject* create(int argc, const char*const* argv) {
		if (argc != 7)
			/*FIXME*/
			abort();
		const char* path = argv[4];
		int width = atoi(argv[5]);
		int height = atoi(argv[6]);
		VideoWindow* p = new VideoWindow(path);
		p->setsize(width, height);
		return (p);
	}
} class_vw;

class SlowVideoImage : public StandardVideoImage {
public:
	SlowVideoImage(Tk_Window, int width, int height);
	~SlowVideoImage();
	void putimage(Display* dpy, Window window, GC gc,
		      int sx, int sy, int x, int y,
		      int w, int h) const;
};

VideoImage::VideoImage(Tk_Window tk, int w, int h)
	: width_(w), height_(h)
{
	dpy_ = Tk_Display(tk);
	int depth = Tk_Depth(tk);
	/*FIXME*/
	bpp_ =  (depth == 24) ? 32 : depth;
}

VideoImage::~VideoImage()
{
}

int StandardVideoImage::use_shm_ = -1;	/* -1 means we haven't yet figured
					 * out whether to use shm or not. */

int StandardVideoImage::noXShm(ClientData, XErrorEvent*)
{
        /*
         * can get called twice, because on some systems the
         * XShmDetach after a failed XShmAttach is an error
         * (i.e., a second error), while on others (bsdi), the
         * detach is necessary so we cannot omit it
         */
        use_shm_ = 0;
        return (0);
}

StandardVideoImage* StandardVideoImage::allocate(Tk_Window tk, int w, int h)
{
#ifdef USE_SHM
    if (use_shm_ == -1) {
    /*
    * FIXME this is a hack to see if we can used shared memory.
    * if the X server says we can, and we're on the same
    * host as the X server, then we are golden.
	*/
	Display* dpy = Tk_Display(tk);
	
	if (XShmQueryExtension(dpy) == 0) {
	    use_shm_ = 0;
	    goto noshm;
	}
	XShmSegmentInfo si;
	
	si.shmid = shmget(IPC_PRIVATE, 512, IPC_CREAT|0777);
	if (si.shmid < 0) {
	    use_shm_ = 0;
	    goto noshm;
	}
	use_shm_ = 1;
	si.readOnly = 1;
	XSync(dpy, 0);
	Tk_ErrorHandler handler = Tk_CreateErrorHandler(dpy,
	    -1, -1, -1,
	    noXShm, 0);
	XShmAttach(dpy, &si);
	XSync(dpy, 0);
	XShmDetach(dpy, &si);
	Tk_DeleteErrorHandler(handler);
	shmctl(si.shmid, IPC_RMID, 0);
    }
    if (use_shm_) {
	SharedVideoImage* p = new SharedVideoImage(tk, w, h);
	if (p->valid())
	    return (p);
	delete p;
    }
noshm:
#endif
// DirectDraw rendering is disabled for now since it's not stable enough yet
//#ifdef USE_DDRAW
#if 0
    Tcl& t = Tcl::instance();	
    if( w>1 && h>1 && // mash loves to unnecessarily create/destroy widgets that consist of 1 pixel
	((w&0x3)==0) && // pitch (width * bytes/pixel) should be a multiple of 8 for DirectDraw
	Tk_Depth(tk)>=16 && // at least 16 bits/pixel is assumed for DirectDraw
	DDrawVideoImage::DDrawOkay(w,h, Tk_GetHWND(Tk_WindowId(t.tkmain())) ) ) {
#ifdef SHOW_MSGS    
	printf("DirectDraw available; creating DDrawVideoImage\n");
#endif
	return (new DDrawVideoImage(tk, w, h));
    }
    else {
#ifdef SHOW_MSGS    
	printf("DirectDraw not available; creating SlowVideoImage\n");
#endif
	return (new SlowVideoImage(tk, w, h));
    }
#endif
    return (new SlowVideoImage(tk, w, h));
}

StandardVideoImage::StandardVideoImage(Tk_Window tk, int w, int h)
	: VideoImage(tk, w, h)
{
	image_ = XCreateImage(dpy_, Tk_Visual(tk), Tk_Depth(tk),
			      ZPixmap, 0, (char*)0, w, h, 8, 0);
}

StandardVideoImage::~StandardVideoImage()
{
	/*FIXME*/
#ifndef WIN32
	XSync(dpy_, 0);
#endif
	image_->data = 0;
	image_->obdata = 0;
	XDestroyImage(image_);
}

SlowVideoImage::SlowVideoImage(Tk_Window tk, int w, int h)
	: StandardVideoImage(tk, w, h)
{
	int size = w * h;
	if (bpp_ > 8)
		size *= bpp_ / 8;
	image_->data = new char[size];
}

SlowVideoImage::~SlowVideoImage()
{
	delete[] image_->data;
}

void SlowVideoImage::putimage(Display* dpy, Window window, GC gc,
			      int sx, int sy, int x, int y,
			      int w, int h) const
{
    unsigned long *colors;
    int ncolors;
#ifdef WIN32
    colors = win32Colors;
    ncolors = win32NColors;
#else
    colors = NULL;
    ncolors = 0;
#endif
    TkPutImage(colors, ncolors, dpy, window, gc, image_, sx, sy, x, y, w, h);
}

#ifdef USE_DDRAW

bool DDrawVideoImage::ddrawInitAttempted=false; // tracks if attempt has been made to alloc/use ddraw
bool DDrawVideoImage::ddrawInitSuccessful=false; // marks if ddraw attempt was successful
int DDrawVideoImage::deviceListLength=0; // number of video devices in system
deviceListType DDrawVideoImage::deviceList[MAX_SYSTEM_MONITORS]; // list of video devices

// this function gets calls by the OS to record the video monitors in the system
BOOL WINAPI DDrawDeviceEnumerateCallback(GUID FAR *lpGUID, LPSTR lpDriverDescription, 
							 LPSTR lpDriverName, LPVOID hw,HMONITOR hm)
{
    return DDrawVideoImage::addMonitor(lpGUID, lpDriverDescription, 
	lpDriverName, (HWND)hw, hm, true);
}

// same callback as above, but for older directX versions
BOOL WINAPI DDrawDeviceEnumerateCallbackOld(GUID FAR *lpGUID,LPSTR  lpDriverDescription, 
						   LPSTR  lpDriverName, LPVOID hw)
{
    return DDrawVideoImage::addMonitor(lpGUID, lpDriverDescription, 
	lpDriverName, (HWND)hw, NULL, false);
}

// function is called by callback functions so that the existence of each monitor in the
// system can be recorded
bool DDrawVideoImage::addMonitor(GUID FAR *lpGUID, LPSTR lpDriverDescription, 
				LPSTR lpDriverName, HWND hw, HMONITOR hm, bool multimon) 
{
    if( deviceListLength>=MAX_SYSTEM_MONITORS ) {
        printf( "WARNING: Too many monitors in the system (increase MAX_SYSTEM_MONITORS)\n");
        return false; // stop enumeration
    }
#ifdef SHOW_MSGS    
    printf("addMonitor: adding card #%d\n", deviceListLength+1);
#endif
    if(lpGUID) {
#ifdef SHOW_MSGS    
        printf( "addMonitor: got monitor GUID\n");
#endif
        deviceList[deviceListLength].deviceGUID = *lpGUID;
        deviceList[deviceListLength].haveGUID = true;
    }
    else {
#ifdef SHOW_MSGS    
        printf( "addMonitor: monitor GUID is NULL\n" );
#endif
        if(multimon) {
            // skipping NULL since monitor will be reported twice, once as NULL and once with GUID (only want GUID)
#ifdef SHOW_MSGS    
            printf( "addMonitor: skipping card since multimon & NULL\n");
#endif
            return true; // return true to allow more callbacks to occur
        }
        deviceList[deviceListLength].haveGUID = false;
    }
    
#ifdef SHOW_MSGS    
    printf( "Driver Name: %s\n", lpDriverName);
    printf( "Driver Desc.: %s\n", lpDriverDescription);
#endif

    deviceList[deviceListLength].driverMonitor = hm;
    
    // init monitor info object
    deviceList[deviceListLength].driverMonitorInfo.cbSize = sizeof(MONITORINFO);
    deviceList[deviceListLength].haveMonitorInfo = false;
    
    // if have a monitor, then try to get information about the monitor
    if( NULL!=hm ) {
        if(!GetMonitorInfo(hm, &(deviceList[deviceListLength].driverMonitorInfo) ) ) {
            printf( "getmonitorinfo failed\n" );
            return false; // stop enumeration since unknown error
        }
        else {
#ifdef SHOW_MSGS    
            printf( "getmonitorinfo success\n" );
#endif
            deviceList[deviceListLength].haveMonitorInfo=true;
#ifdef SHOW_MSGS    
            printf("display monitor rectangle (%d,%d)x(%d,%d)\n", 
                ((DDrawVideoImage::deviceList)[DDrawVideoImage::deviceListLength]).driverMonitorInfo.rcMonitor.left,
                ((DDrawVideoImage::deviceList)[DDrawVideoImage::deviceListLength]).driverMonitorInfo.rcMonitor.top,
                ((DDrawVideoImage::deviceList)[DDrawVideoImage::deviceListLength]).driverMonitorInfo.rcMonitor.right,
                ((DDrawVideoImage::deviceList)[DDrawVideoImage::deviceListLength]).driverMonitorInfo.rcMonitor.bottom);
            printf("work area rectangle (%d,%d)x(%d,%d)\n", 
                ((DDrawVideoImage::deviceList)[DDrawVideoImage::deviceListLength]).driverMonitorInfo.rcWork.left,
                ((DDrawVideoImage::deviceList)[DDrawVideoImage::deviceListLength]).driverMonitorInfo.rcWork.top,
                ((DDrawVideoImage::deviceList)[DDrawVideoImage::deviceListLength]).driverMonitorInfo.rcWork.right,
                ((DDrawVideoImage::deviceList)[DDrawVideoImage::deviceListLength]).driverMonitorInfo.rcWork.bottom);
#endif
        }
    }
#ifdef SHOW_MSGS    
    else {
        printf("hm is NULL\n");
    }
#endif
    
    ++deviceListLength; // item was successfully stored, so increment length
    
    // return true, rather ((deviceListLength<MAX_SYSTEM_MONITORS)?true:false) so that 
    // the warning message located at the beginning of this function will be displayed
    return true; 
}

// get all monitor coordinates in a space delimited list of
// the form { left top right bottom }. The first element is
// the virtual screen coordinates, so the return value should
// always contain at least two elements.
void DDrawVideoImage::getTclMonitorList(char* monitor_info) {
    int i = 0;

	sprintf(monitor_info, "{ %d %d %d %d } ",
		GetSystemMetrics(SM_XVIRTUALSCREEN),
		GetSystemMetrics(SM_YVIRTUALSCREEN),
		GetSystemMetrics(SM_CXVIRTUALSCREEN),
		GetSystemMetrics(SM_CYVIRTUALSCREEN));
	for(i = 0; i < deviceListLength; i++) {
        char moninfo[60];
        RECT *r = &(deviceList[i].driverMonitorInfo.rcWork);
        sprintf(moninfo, "{ %d %d %d %d } ",
            r->left, r->top, r->right, r->bottom);
		strcat(monitor_info, moninfo);
	}
}

// prints debug message about the return value from a function call
void debugMsg(LPSTR prefix, HRESULT hRet)
{
    char message[100];
    sprintf(message, "%s returned unknown code %d\n", prefix, hRet);
    if( hRet==DDERR_WASSTILLDRAWING) sprintf(message, "%s returned DDERR_WASSTILLDRAWING\n", prefix);
    if( hRet==DDERR_SURFACEBUSY) sprintf(message, "%s returned DDERR_SURFACEBUSY\n", prefix);
    if( hRet==DDERR_NOBLTHW) sprintf(message, "%s returned DDERR_NOBLTHW\n", prefix);
    if( hRet==DDERR_GENERIC) sprintf(message, "%s returned DDERR_GENERIC\n", prefix);
    if( hRet==DDERR_INVALIDOBJECT) sprintf(message, "%s returned DDERR_INVALIDOBJECT\n", prefix);
    if( hRet==DDERR_INVALIDRECT) sprintf(message, "%s returned DDERR_INVALIDRECT\n", prefix);
    if( hRet==DDERR_NOCLIPLIST) sprintf(message, "%s returned DDERR_NOCLIPLIST\n", prefix);
    if( hRet==DDERR_SURFACELOST) sprintf(message, "%s returned DDERR_SURFACELOST\n", prefix);
    if( hRet==DDERR_UNSUPPORTED) sprintf(message, "%s returned DDERR_UNSUPPORTED\n", prefix);
    if( hRet==DDERR_INVALIDCLIPLIST) sprintf(message, "%s returned DDERR_INVALIDCLIPLIST\n", prefix);
    if( hRet==DDERR_INCOMPATIBLEPRIMARY  ) sprintf(message, "%s returned DDERR_INCOMPATIBLEPRIMARY\n", prefix);
    if( hRet==DDERR_INVALIDCAPS  ) sprintf(message, "%s returned DDERR_INVALIDCAPS\n", prefix);
    if( hRet==DDERR_INVALIDOBJECT  ) sprintf(message, "%s returned DDERR_INVALIDOBJECT\n", prefix);
    if( hRet==DDERR_INVALIDPARAMS  ) sprintf(message, "%s returned DDERR_INVALIDPARAMS\n", prefix);
    if( hRet==DDERR_INVALIDPIXELFORMAT  ) sprintf(message, "%s returned DDERR_INVALIDPIXELFORMAT\n", prefix);
    if( hRet==DDERR_NOALPHAHW  ) sprintf(message, "%s returned DDERR_NOALPHAHW\n", prefix);
    if( hRet==DDERR_NOCOOPERATIVELEVELSET  ) sprintf(message, "%s returned DDERR_NOCOOPERATIVELEVELSET\n", prefix);
    if( hRet==DDERR_NODIRECTDRAWHW  ) sprintf(message, "%s returned DDERR_NODIRECTDRAWHW\n", prefix);
    if( hRet==DDERR_NOEMULATION  ) sprintf(message, "%s returned DDERR_NOEMULATION\n", prefix);
    if( hRet==DDERR_NOEXCLUSIVEMODE  ) sprintf(message, "%s returned DDERR_NOEXCLUSIVEMODE\n", prefix);
    if( hRet==DDERR_NOFLIPHW  ) sprintf(message, "%s returned DDERR_NOFLIPHW\n", prefix);
    if( hRet==DDERR_NOMIPMAPHW  ) sprintf(message, "%s returned DDERR_NOMIPMAPHW\n", prefix);
    if( hRet==DDERR_NOOVERLAYHW  ) sprintf(message, "%s returned DDERR_NOOVERLAYHW\n", prefix);
    if( hRet==DDERR_NOZBUFFERHW  ) sprintf(message, "%s returned DDERR_NOZBUFFERHW\n", prefix);
    if( hRet==DDERR_NODDROPSHW ) sprintf(message, "%s returned DDERR_NODDROPSHW\n", prefix);
    if( hRet==DDERR_NOMIRRORHW ) sprintf(message, "%s returned DDERR_NOMIRRORHW\n", prefix);
    if( hRet==DDERR_NORASTEROPHW ) sprintf(message, "%s returned DDERR_NORASTEROPHW\n", prefix);
    if( hRet==DDERR_NOROTATIONHW ) sprintf(message, "%s returned DDERR_NOROTATIONHW\n", prefix);
    if( hRet==DDERR_NOSTRETCHHW ) sprintf(message, "%s returned DDERR_NOSTRETCHHW\n", prefix);
    if( hRet==DDERR_OUTOFMEMORY  ) sprintf(message, "%s returned DDERR_OUTOFMEMORY\n", prefix);
    if( hRet==DDERR_OUTOFVIDEOMEMORY  ) sprintf(message, "%s returned DDERR_OUTOFVIDEOMEMORY\n", prefix);
    if( hRet==DDERR_PRIMARYSURFACEALREADYEXISTS  ) sprintf(message, "%s returned DDERR_PRIMARYSURFACEALREADYEXISTS\n", prefix);
    if( hRet==DDERR_UNSUPPORTEDMODE  ) sprintf(message, "%s returned DDERR_UNSUPPORTEDMODE\n", prefix);
    if( hRet==DDERR_DEVICEDOESNTOWNSURFACE ) sprintf(message, "%s returned DDERR_DEVICEDOESNTOWNSURFACE\n", prefix);
    printf("%s", message);
}

// used in unimonitor systems to get information about the primary monitor
BOOL CALLBACK monitorEnumProc(HMONITOR hMonitor /*non-null*/,HDC /*monitor DC*/,
			      LPRECT /* monitor intersection rectangle */, LPARAM dwData)
{
    if(NULL==dwData) {
        return false;
    }
    
    // save the HMONITOR data of the primary monitor
    DDrawVideoImage * ddvi=(DDrawVideoImage *)dwData;
    ddvi->setPrimaryMonitor(hMonitor);
    
    return true; // only want the first device, but need to return true to prevent failure
}

// used in unimonitor systems to store information learned about the primary monitor
void DDrawVideoImage::setPrimaryMonitor(HMONITOR hMonitor)
{
    // don't store info if we already have information about the monitor
    if( deviceList[0].haveMonitorInfo ) {
        return;
    }
    
    deviceList[0].driverMonitor=hMonitor;
    deviceList[0].driverMonitorInfo.cbSize=sizeof(MONITORINFO);
    
    if(!GetMonitorInfo(hMonitor, 
        &(deviceList[0].driverMonitorInfo) ) ) {
        printf( "sPM getmonitorinfo failed\n" );
    }
    else {
        deviceList[0].haveMonitorInfo=true;

	// if zero devices in list, then need to increment
	// device list to report the existence of primary monitor
	if( 0==deviceListLength ) {
	    ++deviceListLength;
	}
#ifdef SHOW_MSGS    
        printf( "spm getmonitorinfo success\n" );
        printf( "spm display monitor rectangle (%d,%d)x(%d,%d)\n", 
            (deviceList[0]).driverMonitorInfo.rcMonitor.left,
            (deviceList[0]).driverMonitorInfo.rcMonitor.top,
            (deviceList[0]).driverMonitorInfo.rcMonitor.right,
            (deviceList[0]).driverMonitorInfo.rcMonitor.bottom);
        printf( "spm work area rectangle (%d,%d)x(%d,%d)\n", 
            (deviceList[0]).driverMonitorInfo.rcWork.left,
            (deviceList[0]).driverMonitorInfo.rcWork.top,
            (deviceList[0]).driverMonitorInfo.rcWork.right,
            (deviceList[0]).driverMonitorInfo.rcWork.bottom);
#endif
    }
}

// this function is called to see if ddraw is supported on this system.  As part of this 
// process (on the first time that it is executed), the static directdraw objects 
// are allocated and initialized
bool DDrawVideoImage::DDrawOkay(int w, int h, HWND hWnd)
{
    HRESULT hRet; // stores return result of function calls
    if( ddrawInitAttempted ) {
        return ddrawInitSuccessful;	  
    }
    else { // see if DDraw is supported
#ifdef SHOW_MSGS    
        printf("DDraw support is being tested\n");
#endif
        ddrawInitAttempted = true; // record that an attempt has been made to use DirectDraw
        
        if( GetSystemMetrics(SM_CMONITORS)>1 ) { // returns 0 on error, which is the case for older OSes
            if( 0==GetSystemMetrics(SM_SAMEDISPLAYFORMAT) ) {
                printf("Multiple monitors detected, but they all need to use the same number of colors for DirectX to be used.\n");
                return false;
            }
        }
        
        // get a description of all the monitors in the system
        
        bool multiCapableSystem = false; // tracks if runtime system is capable of enumerating multiple screens
        
        // DirectDrawEnumerateEx requires
        // Windows 98: DirectX 5.0 and later
        // all others: DirectX 6.0 and later
        // check: use GetProcAddress to get address of "DirectDrawEnumerateEx" from "Ddraw.dll"
        // -- use "name DirectDrawEnumerateExA" (ANSI) or "DirectDrawEnumerateExW" (UNICODE)
        // -- if fails, system does not support multiple monitors
        
        HINSTANCE h = LoadLibrary("ddraw.dll"); // alloc a reference to the DLL
        
        if( NULL!=h ) {
            
            // attempt to load the address of DirectDrawEnumerateEx
            LPDIRECTDRAWENUMERATEEX lpDDEnumEx;
#ifdef _UNICODE
            lpDDEnumEx = (LPDIRECTDRAWENUMERATEEX) GetProcAddress(h,"DirectDrawEnumerateExW");
#else
            lpDDEnumEx = (LPDIRECTDRAWENUMERATEEX) GetProcAddress(h,"DirectDrawEnumerateExA");
#endif
            
            if( NULL!=lpDDEnumEx )  { // if function exists, then use it
                multiCapableSystem = true;
#ifdef SHOW_MSGS    
                printf("DDraw supports multiple monitor detection\n");
#endif
                LPDDENUMCALLBACKEX dddec = &DDrawDeviceEnumerateCallback; // windows wants callback addr in a variable
                
                // call DirectDrawEnumerateEx
                hRet = lpDDEnumEx(dddec /* callback function */, hWnd /* app data */, 
                    DDENUM_ATTACHEDSECONDARYDEVICES /* primary plus those attached to desktop */);
            }
            FreeLibrary(h); // free the reference to the ddraw DLL
        }
        
        if( !multiCapableSystem ) { // if system not capable of enumerating multiple monitors, then call old function
#ifdef SHOW_MSGS    
            printf("DDraw supports only single monitor detection\n");
#endif
            LPDDENUMCALLBACK dddeco = &DDrawDeviceEnumerateCallbackOld; // windows wants callback addr in a variable
            
            // DirectDrawEnumerate enumerates only the primary device (and any devices associated with the primary)
            hRet = DirectDrawEnumerate(dddeco, hWnd); 	    
        }
        
        if( DD_OK!=hRet ) // if call to DDrawEnumerate(Ex) failed
        {
            printf("DDrawEnumerate(Ex) call failed\n");
            
            // if no monitors were enumerated, then assume there is exactly one monitor in the system
            if( 0==deviceListLength ) {
                // enumeration failed or returned no devices, so assume that only one monitor exists
                deviceList[deviceListLength].haveGUID = false;
                deviceListLength = 1;
            }
            
            return false; // temporarily returning false here, at least until this type of error is better understood
        }
        
        // if don't have information about the primary monitor, then get it through old API
        if( false==deviceList[0].haveMonitorInfo ) {
            RECT rcClip={5,5,10,10}; // just made somthing up for coordinates in the main screen
            MONITORENUMPROC fnEnum = monitorEnumProc;
            if( !EnumDisplayMonitors(NULL, &rcClip, &monitorEnumProc, (LPARAM)hWnd) ) {
                printf("EnumDisplayMonitors failed\n");
                return false;
            }
        }
        
        // CoInitialize: Initializes the COM library on the current thread and identifies 
        // the concurrency model as single-thread apartment (STA). 
        hRet = CoInitialize( NULL /* MBZ */);
        if( S_OK!=hRet )
        {
            printf("CoInitialize failed\n");
            return false;
        }
        
        // Create a DirectDrawFactory object and get  
        // an IDirectDrawFactory interface pointer.
        IDirectDrawFactory *pDDF = NULL;
        hRet=CoCreateInstance(CLSID_DirectDrawFactory, NULL, CLSCTX_INPROC_SERVER, 
            IID_IDirectDrawFactory, (void **)&pDDF);
        if( S_OK!=hRet || NULL==pDDF )
        {
            printf("CoCreateInstance DirectDrawFactory failed\n");
            return false;
        }
        
        // track if required ddraw features are supported on all devices
        bool ddrawUsable = true;
        
        // for each video window, attempt to allocate a directdraw object
        int i=0;
        for(i=0; i<deviceListLength; ++i) { // for each video monitor
            
            // allocate direct draw object for issuing commands
            
            // Call the IDirectDrawFactory::CreateDirectDraw method to 
            // get the address of an IDirectDraw interface pointer.
            IDirectDraw *pDD = NULL;
            hRet= (pDDF->CreateDirectDraw( (deviceList[i].haveGUID?&(deviceList[i].deviceGUID):NULL),
                GetDesktopWindow() /* HWND of app */, DDSCL_NORMAL /* interop */ , NULL, NULL, &pDD));
            if ( hRet!=DD_OK || pDD==NULL ) {  // error checking (DD_OK is correctly used instead of S_OK)
                printf("CreateDirectDraw Failed\n");
                return false;
            }
#ifdef SHOW_MSGS    
            else {
                printf("CreateDirectDraw Okay\n");
            }
#endif
            
            // Now query for the new IDirectDraw3 interface, and release the old IDirectDraw one.
            hRet = (pDD->QueryInterface(IID_IDirectDraw3, (LPVOID*)&(deviceList[i].g_pDD) ));
            if ( hRet!=S_OK || deviceList[i].g_pDD==NULL) {  // error checking
                printf("QueryInterface IID_IDirectDraw3 Failed\n");
                return false;
            }
#ifdef SHOW_MSGS    
            else {
                printf("QueryInterface IID_IDirectDraw3 success\n");
            }
#endif
            
            // Release IDirectDraw reference
            pDD->Release();
            pDD = NULL;
            
            // we don't need exclusive use of the board, so just request normal
	    // --recently removed since already specified to CreateDirectDraw
            // hRet = (deviceList[i].g_pDD)->SetCooperativeLevel(hWnd, DDSCL_NORMAL); 
            
            // get the hardware capabilities
            memset(&(deviceList[i].hardwareCaps), 0, sizeof(DDCAPS));
            deviceList[i].hardwareCaps.dwSize = sizeof(DDCAPS);
            memset(&(deviceList[i].emulatedCaps), 0, sizeof(DDCAPS));
            deviceList[i].emulatedCaps.dwSize = sizeof(DDCAPS);
            hRet = (deviceList[i].g_pDD)->GetCaps(&(deviceList[i].hardwareCaps), &(deviceList[i].emulatedCaps) ); 
            if (hRet != DD_OK)
            { 
                printf("GetCaps failed\n");
                return false;
            } 
#ifdef SHOW_MSGS
            else {
                printf("Raw hardware capabilities for device #%d will follow\n", i);
            }
#endif
            
            
#ifdef SHOW_MSGS // showing hardware capabilities; feature tests performed against emulated capabilities
            if( 0==(deviceList[i].hardwareCaps.dwCaps&DDCAPS_BLT) ) {
                printf("blt is NOT supported -- Ddraw is unusable.\n");
            }
            else {
                printf("blt is supported\n");
            }
            
            if( 0==(deviceList[i].hardwareCaps.dwCaps&DDCAPS_CANCLIP) ) {
                printf("clipping is NOT supported -- Ddraw is unusable, or will overwrite other windows.\n");
            }
            else {
                printf("clipping is supported\n");
            }
            
            if(deviceList[i].hardwareCaps.dwCaps&DDCAPS_CANBLTSYSMEM) {
                printf("blt using system memory is supported\n");
                if( deviceList[i].hardwareCaps.dwSVBCaps&DDCAPS_BLT ) {
                    printf("blt from system memory to video is supported\n");
                }
                else {
                    printf("blt from system memory to video is NOT supported -- Ddraw is unusable using multiple siml windows.\n");
                }
                
                if( deviceList[i].hardwareCaps.dwSVBCaps&DDCAPS_CANBLTSYSMEM ) {
                    printf("blt from system memory to video is supported (sysmem flag)\n");
                }
                else {
                    printf("blt from system memory to video is NOT supported (sysmem flag) -- Ddraw is unusable using multiple siml windows.\n");
                }
                
                if( deviceList[i].hardwareCaps.dwSVBCaps&DDCAPS_CANCLIP ) {
                    printf("clipping while using system memory is supported\n");
                }
                else {
                    printf("clipping while using system memory is NOT supported -- Ddraw is unusable, or will overwrite other windows.\n");
                }
            }
            else {
                printf("blt using system memory is NOT supported -- Ddraw is unusable using multiple siml windows.\n");
            }
#endif
            
            if( 0==(deviceList[i].emulatedCaps.dwCaps&DDCAPS_BLT) ) {
                printf("EMULATED blt is NOT supported on device #%d.\nDdraw is unusable.\n", i);
                ddrawUsable = false;
            }
#ifdef SHOW_MSGS
            else {
                printf("blt is supported\n");
            }
#endif
            
            if( 0==(deviceList[i].emulatedCaps.dwCaps&DDCAPS_CANCLIP) ) {
                printf("EMULATED clipping is NOT supported on device #%d.\nDdraw is unusable, or will overwrite other windows.\n", i);
                ddrawUsable=false;
            }
#ifdef SHOW_MSGS
            else {
                printf("clipping is supported\n");
            }
#endif
            
            if(deviceList[i].emulatedCaps.dwCaps&DDCAPS_CANBLTSYSMEM) {
#ifdef SHOW_MSGS
		printf("blt using system memory is supported\n");
#endif
                if( 0==(deviceList[i].emulatedCaps.dwSVBCaps&DDCAPS_BLT) ) {
                    printf("EMULATED blt from system memory to video is NOT supported on device #%d.\nDdraw is unusable using multiple siml windows.\n", i);
                    ddrawUsable=false;
                }
#ifdef SHOW_MSGS
                else {
                    printf("blt from system memory to video is supported\n");
                }
#endif
                
                if( 0==(deviceList[i].emulatedCaps.dwSVBCaps&DDCAPS_CANBLTSYSMEM) ) {
                    printf("EMULATED blt from system memory to video is NOT supported (sysmem flag) on device #%d.\nDdraw is unusable using multiple siml windows.\n", i);
                    ddrawUsable=false;
                }
#ifdef SHOW_MSGS
                else {
                    printf("blt from system memory to video is supported (sysmem flag)\n");
                }
#endif
                
                if( 0==(deviceList[i].emulatedCaps.dwSVBCaps&DDCAPS_CANCLIP) ) {
                    printf("EMULATED clipping while using system memory is NOT supported on device #%d.\nDdraw is unusable, or will overwrite other windows.\n", i);
                    ddrawUsable=false;
                }
#ifdef SHOW_MSGS
                else {
                    printf("clipping while using system memory is supported\n");
                }
#endif
            }
            else {
                printf("EMULATED blt using system memory is NOT supported on device #%d.\nDdraw is unusable using multiple siml windows.\n", i);
                ddrawUsable=false;
            }
            
            // allocate the primary surface
            DDSURFACEDESC ddsd;
            ZeroMemory(&ddsd, sizeof(ddsd));
            ddsd.dwSize = sizeof(ddsd);
            
            ddsd.dwFlags = DDSD_CAPS;
            ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
            
            // allocate a primary surface of type IDirectDrawSurface
	    
            LPDIRECTDRAWSURFACE lpDDSTemp=NULL;
            hRet = (deviceList[i].g_pDD)->CreateSurface(&ddsd, 
                &lpDDSTemp, NULL /* MBZ */ );
            if( DD_OK!=hRet || NULL==lpDDSTemp) 
            {
                printf("Create Primary Surface failed\n");
                return false;
            }
            
            // get a IDirectDrawSurface3 interface from the IDirectDrawSurface object
            hRet = lpDDSTemp->QueryInterface( /*__uuidof ( IDirectDrawSurface3 )*/ IID_IDirectDrawSurface3,
                (void**)&deviceList[i].lpDDSPrimarySurface);
            if( S_OK!=hRet ) 
            {
                printf("Create Primary Surface query interface IID_IDirectDrawSurface3 failed\n");
                return false;
            }
            lpDDSTemp->Release(); // release the old IDirectDrawSurface interface
            
#ifndef NOCLIPPER
            // create a clipper that ensures we won't draw outside of window
            hRet = (deviceList[i].g_pDD)->CreateClipper(0 /* MBZ */ , 
                &(deviceList[i].lpDDCPrimarySurfaceClipper), NULL /* MBZ */ );
            if(hRet != DD_OK) 
            {
                printf("Create Primary Surface clipper failed\n");
                return false;
            }
            // set the clipper to the primary surface
            (deviceList[i].lpDDSPrimarySurface)->SetClipper(deviceList[i].lpDDCPrimarySurfaceClipper);
#endif
	}
#ifdef SHOW_MSGS
	printf("Init over, so releasing the DirectDrawFactory\n");
#endif
	
	// Release IDirectDrawFactory
	pDDF->Release();
	
	if( !ddrawUsable ) {
	    printf("DirectDraw %x.%x will NOT be used for rendering.\n",
               (DIRECTDRAW_VERSION >> 8), (DIRECTDRAW_VERSION & 0xFF));
	    
	    // deallocate the objects
	    for(i=0; i<deviceListLength; ++i) { // for each video monitor
		if( NULL!=deviceList[i].lpDDCPrimarySurfaceClipper ) {
		    (deviceList[i].lpDDSPrimarySurface)->SetClipper(NULL); // remove the clipper from surface
		    (deviceList[i].lpDDCPrimarySurfaceClipper)->Release(); // dealloc reference to clipper
		    deviceList[i].lpDDCPrimarySurfaceClipper=NULL;
		}
		(deviceList[i].lpDDSPrimarySurface)->Release(); // dealloc reference to surface
		deviceList[i].lpDDSPrimarySurface=NULL;
		(deviceList[i].g_pDD)->Release(); // dealloc reference to ddraw object
		deviceList[i].g_pDD=NULL;
		
		// Close the COM library
		CoUninitialize(); // safe to use call here since DDraw won't be used any further
	    }
	    return false;	    
	}
	else {
	    printf("DirectDraw %x.%x will be used for rendering.\n",
               (DIRECTDRAW_VERSION >> 8), (DIRECTDRAW_VERSION & 0xFF));
	}
	
	// Close the COM library
	// CoUninitialize(); // this function should be called, but where to put the call?
	// CoUninitialize should be called on application shutdown, as the last call made to the COM 
	// library after the application hides its main windows and falls through its main message loop
	
	ddrawInitSuccessful=true;
	return true;
    }
}


// To get the best performance in a windowed MultiMon application, you need to create a 
// DirectDraw object for each device, maintain off-screen surfaces in parallel on each device,
// keep track of which part of the window resides on each device, and perform separate blits 
// to each device.

// callback function that's called when the widget moves
void windowMoveEvent(ClientData data, XEvent *eventPtr)
{
    DDrawVideoImage *dvi = (DDrawVideoImage *)data;
    if( ConfigureNotify==eventPtr->type ) {
        dvi->signalWindowMoved();
    }
}

// constructor for allocating state for this widget
DDrawVideoImage::DDrawVideoImage(Tk_Window tk, int w, int h)
	: StandardVideoImage(tk, w, h)
{
    // buffer size for video
    int size = w * h;
    if (bpp_ > 8) { // if bit than one byte/pixel, then adjust size accordingly
        size *= bpp_ / 8;
    }
    image_->data = NULL; // normally where buffer would be allocated (now done below)
    updateMonitorCoverage = true; // need to update monitor location info
    
    int i=0;
    for(i=0; i<MAX_WIDGET_MONITORS; ++i) {
        currentMonitors[i]=-1; // just in case: make no monitors exist until location checked
        currentMonitorsSurface[i]=NULL; // no surface has yet been allocated
    }
    
    tk_ = tk;
    Tcl& t = Tcl::instance();
    
    // keep walking up the widget hierarchy until the top window is reached
    for (toplevelTkWindow_ = tk; !Tk_IsTopLevel(toplevelTkWindow_); )
        toplevelTkWindow_ = Tk_Parent(toplevelTkWindow_); // needed for event handling
    
    // add event handler that fire when the window is moved
    Tk_CreateEventHandler(toplevelTkWindow_, StructureNotifyMask, 
        windowMoveEvent, this);
    Tk_CreateEventHandler(tk, StructureNotifyMask, 
        windowMoveEvent, this);
    
    // allocate memory in this way so that memory is 8-byte aligned
    image_->data = (char*)(new __int64[size/sizeof(__int64)]);
    if( NULL==image_->data ) {
        printf("widget buffer local alloc failed\n");
    }
    else {
        ZeroMemory(image_->data, (DWORD)size);
    }
}

// To get the best performance in a windowed MultiMon application, you need to create a 
// DirectDraw object for each device, maintain off-screen surfaces in parallel on each device,
// keep track of which part of the window resides on each device, and perform separate blits 
// to each device.

// mash isn't structured to have multiple, parallel off-screen surfaces that are allocated 
// within directdraw.  Instead, when multiple surfaces are needed (e.g., a window that's 
// present in two windows), the memory will get reassigned to the memory used within mash,
// and thus the various surfaces will all share the same buffer.  Despite the doc's claims, 
// this functionality was probably introduced in version 3 of DirectDraw since that's when
// the GetSurfaceDesc function was introduced.  Although you're supposed to be able to 
// allocate/assign the buffer when the surface is created, this functionality basically
// requires DirectDraw 7 since a newer version of a surface object can be specified at 
// that time.  Other older API's you have to allocate a version 1, request a newer API for
// the object, and then the buffer reassignment can be performed.

LPDIRECTDRAWSURFACE3 DDrawVideoImage::allocateSurface(int deviceNum)
{
    HRESULT hRet; // stores result to DirectX calls
    
    // Initialize the surface description.
    DDSURFACEDESC ddsd;
    ZeroMemory(&ddsd, sizeof(ddsd));
    ZeroMemory(&ddsd.ddpfPixelFormat, sizeof(DDPIXELFORMAT));
    ddsd.dwSize = sizeof(ddsd);
    
    // Set up the pixel format for RGB
    ddsd.ddpfPixelFormat.dwSize = sizeof(DDPIXELFORMAT);
    ddsd.ddpfPixelFormat.dwFlags = DDPF_RGB;
    ddsd.ddpfPixelFormat.dwRGBBitCount = (DWORD) image_->depth;
    
    if( 16==image_->depth )
    {
        ddsd.ddpfPixelFormat.dwRBitMask = 0x7c00; // 5
        ddsd.ddpfPixelFormat.dwGBitMask = 0x03e0; // 5
        ddsd.ddpfPixelFormat.dwBBitMask = 0x001f; // 5
    }
    else // depth==24 or depth==32
    {
        ddsd.ddpfPixelFormat.dwRBitMask = 0x00FF0000; // 8
        ddsd.ddpfPixelFormat.dwGBitMask = 0x0000FF00; // 8
        ddsd.ddpfPixelFormat.dwBBitMask = 0x000000FF; // 8
    }
    
    ddsd.dwFlags = DDSD_WIDTH | DDSD_HEIGHT | DDSD_PITCH | DDSD_PIXELFORMAT | DDSD_CAPS;
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
    ddsd.dwWidth = image_->width;
    ddsd.dwHeight= image_->height;
    ddsd.lPitch  = image_->bytes_per_line; // surface pitch, which must be a QWORD (8 byte) multiple
    
    // allocate a durface of type IDirectDrawSurface
    LPDIRECTDRAWSURFACE lpDDSTemp = NULL;
    hRet = (deviceList[deviceNum].g_pDD)->CreateSurface(&ddsd, &lpDDSTemp, NULL);
    if( DD_OK==hRet && NULL!=lpDDSTemp ) 
    {
#ifdef SHOW_MSGS
        printf("widget buffer hardware create success\n");
#endif
        
        // allocate an instance of the IDirectDrawSurface3 interface from IDirectDrawSurface
        LPDIRECTDRAWSURFACE3 lpDDSImageSurface = NULL;
        hRet = lpDDSTemp->QueryInterface(IID_IDirectDrawSurface3,(void**)&lpDDSImageSurface);
        if( S_OK!=hRet && NULL!=lpDDSImageSurface )
        {
            printf("QueryInterface IDirectDrawSurface3 failed, now what?\n");
            return NULL;
        }
#ifdef SHOW_MSGS
        else {
            printf("QueryInterface IDirectDrawSurface3 success\n");
        }
#endif
        lpDDSTemp->Release();
        
        // change the surface to use our local buffer
        
        // get the current surface description
        DDSURFACEDESC ddsdAddr;
        ZeroMemory(&ddsdAddr, sizeof(ddsdAddr));
        ddsdAddr.dwSize = sizeof(ddsdAddr);
        ddsdAddr.dwFlags = DDSD_WIDTH | DDSD_HEIGHT | DDSD_PITCH;
        hRet = lpDDSImageSurface->GetSurfaceDesc(&ddsdAddr);
        if( hRet != DD_OK )
        {
            printf("get surface desc failed, now what?\n");
            return NULL;
        }
        
#ifdef SHOW_MSGS
        printf("get surface desc success: width=%d, height=%d, pitch=%d\n", ddsdAddr.dwWidth, 
            ddsdAddr.dwHeight, ddsdAddr.lPitch);
#endif
        
        // add the local buffer to the surface description
        ddsdAddr.dwFlags = DDSD_LPSURFACE | DDSD_WIDTH | DDSD_HEIGHT | DDSD_PITCH;
        ddsdAddr.lpSurface = image_->data;
        // SetSurfaceDesc added in the V3 interface: IDirectDrawSurface3
        hRet = lpDDSImageSurface->SetSurfaceDesc(&ddsdAddr, 0);
        if( DD_OK==hRet ) 
        {
#ifdef SHOW_MSGS
            printf("set surface desc success\n");
#endif
            return lpDDSImageSurface;
        }
        else {
            debugMsg("set surface desc failed, now what??", hRet);
            return NULL;
        }
    }
    else {
        printf("widget create surface failed, now what?\n");
        return NULL;
    }
}

void showMonInfo(HMONITOR hm)
{
    if( NULL!=hm) {
        MONITORINFO mi;
	mi.cbSize = sizeof(mi);
	GetMonitorInfo(hm, &mi);
	printf("device info: %s %dx%d-%dx%d\n", (mi.dwFlags&MONITORINFOF_PRIMARY)?"primary device":"NOT primary device",
	    mi.rcMonitor.left, mi.rcMonitor.top, mi.rcMonitor.right, mi.rcMonitor.bottom);
    }
    else {
	printf("monitor reference is null; no device info available\n");
    }
}

// examine window coordinates to see where a window has moved to
void DDrawVideoImage::examineMonitorCoverage(HWND localhWnd) //, int x, int y, int w, int h)
{
   // if window hasn't moved, then no need to check new positon
    if( !updateMonitorCoverage ) { 
        return;
    }

    int i = 0; // used for loop iterations

    // temporarily store the monitors used for the widget's old location
    int tempMonitors[MAX_WIDGET_MONITORS]; // list of monitors that the widget was located on
    memcpy(tempMonitors, currentMonitors, sizeof(int)*MAX_WIDGET_MONITORS);

    // reset all elements of the monitor list to -1, which basically means not used
    for(i=0; i<MAX_WIDGET_MONITORS; ++i) {
	currentMonitors[i] = -1;
    }

    // temporarily store the monitor surfaces used for the widget's old location
    LPDIRECTDRAWSURFACE3 tempMonitorsSurface[MAX_WIDGET_MONITORS]; // surfaces associated with those monitors
    memcpy(tempMonitorsSurface, currentMonitorsSurface, sizeof(LPDIRECTDRAWSURFACE3)*MAX_WIDGET_MONITORS);
    
    // init all the surfaces to NULL
    memset(currentMonitorsSurface, 0, sizeof(LPDIRECTDRAWSURFACE3)*MAX_WIDGET_MONITORS ); // set all to null

    // get the location of the monitor

    RECT dstRect;
    GetClientRect(localhWnd, &dstRect); // get location of window
    ClientToScreen(localhWnd, (POINT*)&dstRect ); // adjust top and left to screen coords
    ClientToScreen(localhWnd, ((POINT*)&dstRect)+1 ); // adjust bottom and right to screen coords
    
    // -- x,y,w,h might change even if the widget hasn't moved, so best to ignore x,y,w,h and make
    // decisions based on the widgets largest size (which is specified by GetClientRect)

    POINT ptPoint[4];
    ptPoint[0].x = dstRect.left;
    ptPoint[0].y = dstRect.top;
    ptPoint[1].x = dstRect.right;
    ptPoint[1].y = dstRect.top;
    ptPoint[2].x = dstRect.left;
    ptPoint[2].y = dstRect.bottom;
    ptPoint[3].x = dstRect.right;
    ptPoint[3].y = dstRect.bottom;
    
    // get the HMONITOR reference that corresponds to each corner of the widget, and stores
    // those references in an array (duplicates not stored)
    // --assumes that NULL is used only for a point that isn't located on a screen

    // this list stores the monitors that correspond to each point, 
    // duplicates are excluded, and NULL represents empty/unused
    HMONITOR hmMonitors[MAX_WIDGET_MONITORS];
    memset(hmMonitors, 0, sizeof(HMONITOR) ); // init all entries to NULL

    int monitorCount = 0; // tracks # of monitors in hmMonitors
    
    for(i=0; i<MAX_WIDGET_MONITORS; ++i) {
        
        // convert a point location to the monitor that it's present on
        HMONITOR hm = MonitorFromPoint(ptPoint[i], MONITOR_DEFAULTTONULL);
        if( hm==NULL ) { // point isn't on a screen
            //printf("null in mfp\n");
            continue;
        }
        else {
            // point is in/on a monitor
            
            // verify that monitor isn't already shared by another point
	    bool foundMatch = false;

            for(int j=0; j<monitorCount; ++j) {
                if( hmMonitors[j]==hm ) {
                    // printf("duplicate in mfp\n");
		    foundMatch = true;
                    break;
                }
            }
            if( !foundMatch ) { // new monitor, so store information
                // printf("new monitor in mfp\n");
                hmMonitors[monitorCount++] = hm;
            }
        }
    }
    
    // convert window identifiers to array index # that corresponds to window GUID
    int currMonitorIndex=0; // index at which to store next entry in currentMonitors[]

    for(i=0; i<monitorCount /*MAX_WIDGET_MONITORS*/; ++i) {
        if( NULL!=hmMonitors[i] ) {
	    int j=0;
            for(j=0; j<deviceListLength; ++j) {
                if( deviceList[j].driverMonitor==hmMonitors[i] ) {
                    //printf("set a current monitor\n");
                    currentMonitors[currMonitorIndex++] = j;
                    break;
                }
            }
            if( j>=deviceListLength ) {
		static bool showedMsg = false;
		if( !showedMsg ) {
		    printf("1-time msg: no matching device for point %dx%d\n", ptPoint[i].x, ptPoint[i].y);
		    printf("--Windows reported %u as the reference for this point\n", hmMonitors[i]);
		    showMonInfo(hmMonitors[i]);
		    for(int k=0; k<deviceListLength; ++k) {
			printf("--device %d has a reference of %u\n", k, deviceList[k].driverMonitor);
			showMonInfo(deviceList[k].driverMonitor);
		    }
		    showedMsg = true;
		}
            }
        }
    }
    //printf("done setting current monitor\n");
    
    // need to assign/allocate surfaces to the active monitors
    for(i=0; i<currMonitorIndex/*i<MAX_WIDGET_MONITORS && 0<=currentMonitors[i]*/; ++i) {
        
        // search to see if surface has already been allocated

        int monitorToFind=currentMonitors[i]; // surface to find a monitor for
        
	// look through the old list of monitors to see if there's one we can reuse
        int j=0;
        for(j=0; j<MAX_WIDGET_MONITORS; ++j) {
            if(tempMonitors[j]==monitorToFind) { // found a surface
                currentMonitorsSurface[i]=tempMonitorsSurface[j]; // copy surface
#ifdef SHOW_MSGS
                if(i!=j) {
                    printf("moving old position %d to %d\n", j, i);
                }
#endif
                // clear temp entry to prevent reuse (and helps to prevent dealloc)
                tempMonitors[j]=-1;
                tempMonitorsSurface[j]=NULL;
                break; //  break since a match was found
            }
        }
        
        // if surface still NULL, then one wasn't found and we need to alloc one
        if( NULL==currentMonitorsSurface[i] ) {
#ifdef SHOW_MSGS
            printf("allocating a surface at position %d for monitor %d\n", i, monitorToFind);
#endif
            currentMonitorsSurface[i] = allocateSurface(monitorToFind);
            if( NULL==currentMonitorsSurface[i] ) {
                printf("surface alloc failed\n");
            }
        }
    } // end of loop to assign/allocate surfaces
    
    // need to deallocate any unused surfaces (i.e., old surfaces that were not reused)
    for(i=0; i<MAX_WIDGET_MONITORS; ++i) {
        if( NULL!=tempMonitorsSurface[i] ) {
#ifdef SHOW_MSGS
            printf("deallocating a surface at old position %d which had monitor %d\n", i, tempMonitors[i]);
#endif
            tempMonitorsSurface[i]->Release();
            tempMonitorsSurface[i] = NULL;
        }
    }
    updateMonitorCoverage = false;
}

// function that places image
void DDrawVideoImage::putimage(Display* dpy, Window window, GC gc,
                               int sx, int sy, int x, int y,
                               int w, int h) const
{
    HWND localhWnd = Tk_GetHWND(window);
    
    // if necessary, update which monitors should be used (if any)
    ((DDrawVideoImage *) this)->examineMonitorCoverage(localhWnd); //, x, y, w, h);
    
    int currmonitor=-1;
    // loop through monitor list, and draw the widget on each one
    for(int i=0; i<MAX_WIDGET_MONITORS && (currmonitor=currentMonitors[i])>=0; ++i) {
        
        HRESULT hRet;
        
#ifndef NOCLIPPER // if using a clipper
        // set clipper to correspond to this widget
        hRet=(deviceList[currmonitor].lpDDCPrimarySurfaceClipper)->
            SetHWnd(0 /* MBZ */, localhWnd);
        if( DD_OK!=hRet ) {
            debugMsg("surface clippers sethwnd error", hRet);
        }
#endif
        RECT srcRect, dstRect;
        
        GetClientRect(localhWnd, &dstRect); // sets right and bottom to window width
        ClientToScreen(localhWnd, (POINT*)&dstRect.left); // convert left and top
        
        // index left and top by the offset
        dstRect.left+=x;
        dstRect.top+=y;
        
	// change origin to be relative to this particular screen
        dstRect.left-=deviceList[currmonitor].driverMonitorInfo.rcMonitor.left;
        dstRect.top-=deviceList[currmonitor].driverMonitorInfo.rcMonitor.top;
        
        // update right and bottom by the size of the image
        dstRect.right = dstRect.left + w;
        dstRect.bottom = dstRect.top + h;
        
        SetRect(&srcRect, sx, sy, sx + w, sy + h);
        
        // To be safe, the code could clip the source and destination rectangles to the actual real estate of each 
	// screen.  However, I'm assuming that the clipper handles this situation and thus this code can ignore the issue

        // copy visual data from client buffer to video buffer
        hRet = (deviceList[currmonitor].lpDDSPrimarySurface)->Blt(&dstRect, 
            currentMonitorsSurface[i], &srcRect, DDBLT_WAIT, 0);
        if ( DD_OK!=hRet )
        {
            printf("local buffer blt error on blt %d, monitor %d\n", i, currmonitor);
            debugMsg("--details: blt error", hRet);
        }
        // break; // can use this to only render on one screen, as a test
    }
}

DDrawVideoImage::~DDrawVideoImage()
{
    // disable the window movement event handlers
    Tk_DeleteEventHandler(toplevelTkWindow_, StructureNotifyMask, windowMoveEvent, this);
    Tk_DeleteEventHandler(tk_, StructureNotifyMask, windowMoveEvent, this);
    
    // need to deallocate the surfaces
    int i=0;
    for(i=0; i<MAX_WIDGET_MONITORS; ++i) {
        if( NULL!=currentMonitorsSurface[i] ) {
#ifdef SHOW_MSGS
            printf("destructor deallocating a surface\n");
#endif
            currentMonitorsSurface[i]->Release();
            currentMonitorsSurface[i] = NULL;
        }
    }
    
    // dealloc image region if it has been allocated by this application
    if( image_ && image_->data) {
        delete [] image_->data; // release the image data
        image_->data=NULL; // release the image data
    }
}
#endif // ddraw

#ifdef USE_SHM
SharedVideoImage::SharedVideoImage(Tk_Window tk, int w, int h)
	: StandardVideoImage(tk, w, h)
{
	int size = w * h;
	if (bpp_ > 8)
		size *= bpp_ / 8;

	shminfo_.shmid = shmget(IPC_PRIVATE, size, IPC_CREAT|0777);
	if (shminfo_.shmid < 0) {
		perror("vic: shmget");
		fprintf(stderr, "\
vic: reverting to non-shared memory; you should reconfigure your system\n\
vic: with more a higher limit on shared memory segments.\n\
vic: refer to the README that accompanies the vic distribution\n");
		use_shm_ = 0;
		return;
	}
	shminfo_.shmaddr = (char*)shmat(shminfo_.shmid, 0, 0);
	if (shminfo_.shmaddr == (char*)-1) {
		perror("shmat");
		exit(1);
	}
	init(tk);
}

/*
 * side affect - shmid is detached from local addr space
 */
SharedVideoImage::SharedVideoImage(Tk_Window tk, int w, int h,
				   u_char* shmaddr, int shmid)
	: StandardVideoImage(tk, w, h)
{
	shminfo_.shmid = shmid;
	shminfo_.shmaddr = (char*)shmaddr;
	init(tk);
}

SharedVideoImage::~SharedVideoImage()
{
	if (valid()) {
		XShmDetach(dpy_, &shminfo_);
		if (shmdt(shminfo_.shmaddr) < 0)
			perror("vic: shmdt");
	}
}

void SharedVideoImage::init(Tk_Window /* tk */)
{
/*FIXME capture-windows need to be writeable */
#ifdef notdef
	shminfo_.readOnly = 1;
#else
	shminfo_.readOnly = 0;
#endif
	XShmAttach(dpy_, &shminfo_);
	/*
	 * Once the X server has attached the shm segments,
	 * we rmid them so they will go away when we exit.
	 * The sync is to make the X server do the attach
	 * before we do the rmid.
	 */
	XSync(dpy_, 0);
	(void)shmctl(shminfo_.shmid, IPC_RMID, 0);

	/*
	 * Wrap segment in an ximage
	 */
	image_->obdata = (char*)&shminfo_;
	image_->data = shminfo_.shmaddr;
}

void SharedVideoImage::putimage(Display* dpy, Window window, GC gc,
				int sx, int sy, int x, int y,
				int w, int h) const
{
	XShmPutImage(dpy, window, gc, image_, sx, sy, x, y, w, h, 0);
}
#endif

BareWindow::BareWindow(const char* path, XVisualInfo* vinfo)
	: width_(0), height_(0), destroyed_(0)
{
	Tcl& tcl = Tcl::instance();
	tk_ = Tk_CreateWindowFromPath(tcl.interp(), tcl.tkmain(),
				      (char*)path, 0);
	if (tk_ == 0)
		abort();
	Tk_SetClass(tk_, "Vic");
	Tk_CreateEventHandler(tk_, ExposureMask|StructureNotifyMask,
			      handle, (ClientData)this);
	dpy_ = Tk_Display(tk_);
	if (vinfo != 0) {
		/*FIXME*/
		Colormap cm = XCreateColormap(dpy_, Tk_WindowId(tcl.tkmain()),
					      vinfo->visual, AllocNone);
		Tk_SetWindowVisual(tk_, vinfo->visual, vinfo->depth, cm);
	}
}

BareWindow::~BareWindow()
{
	Tk_DeleteEventHandler(tk_, ExposureMask|StructureNotifyMask,
			      handle, (ClientData)this);
}

int BareWindow::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
		if (strcmp(argv[1], "width") == 0) {
			sprintf(tcl.buffer(), "%d", width_);
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
		if (strcmp(argv[1], "height") == 0) {
			sprintf(tcl.buffer(), "%d", height_);
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
	} else if (argc == 4) {
		if (strcmp(argv[1], "resize") == 0) {
			setsize(atoi(argv[2]), atoi(argv[3]));
			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}

void BareWindow::handle(ClientData cd, XEvent* ep)
{
	BareWindow* w = (BareWindow*)cd;

	switch (ep->type) {
	case Expose:
		if (ep->xexpose.count == 0)
			w->redraw();
		break;

	case DestroyNotify:
		w->destroy();
		break;

#ifdef notyet
	case ConfigureNotify:
		if (w->width_ != ep->xconfigure.width ||
		    w->height_ != ep->xconfigure.height)
			;
		break;
#endif
	}
}

void BareWindow::destroy()
{
	destroyed_ = 1;
}

void BareWindow::setsize(int w, int h)
{
	width_ = w;
	height_ = h;
	Tk_GeometryRequest(tk_, w, h);
}

GC VideoWindow::gc_;

VideoWindow::VideoWindow(const char* path, XVisualInfo* vinfo)
	: BareWindow(path, vinfo),
	  vi_(0),
	  callback_pending_(0),
	  damage_(0),
	  voff_(0),
	  hoff_(0)
{
	if (gc_ == 0) {
		/*FIXME should use Vic.background */
		XColor* c = Tk_GetColor(Tcl::instance().interp(), tk_,
					Tk_GetUid("gray50"));
		if (c == 0)
			abort();
		XGCValues v;
		v.background = c->pixel;
		v.foreground = c->pixel;
		const u_long mask = GCForeground|GCBackground;
		gc_ = Tk_GetGC(tk_, mask, &v);
	}
}

VideoWindow::~VideoWindow()
{
	if (callback_pending_)
		Tk_CancelIdleCall(display, (ClientData)this);
}

int VideoWindow::command(int argc, const char*const* argv)
{
	if (argc == 2) {
		if (strcmp(argv[1], "redraw") == 0) {
			redraw();
			return (TCL_OK);
		}
		if (strcmp(argv[1], "clear") == 0) {
			clear();
			return (TCL_OK);
		}
		if (strcmp(argv[1], "dim") == 0) {
			dim();
			return (TCL_OK);
		}
	} else if (argc == 3) {
		if (strcmp(argv[1], "voff") == 0) {
			voff_ = atoi(argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "hoff") == 0) {
			hoff_ = atoi(argv[2]);
			return (TCL_OK);
		}
	}
	return (BareWindow::command(argc, argv));
}

void VideoWindow::display(ClientData cd)
{
	VideoWindow* vw = (VideoWindow*)cd;
	vw->callback_pending_ = 0;
	int h = (vw->vi_ != 0) ? vw->vi_->height() : vw->height_;
	vw->draw(0, h, 0, 0);
}

void VideoWindow::draw(int y0, int y1, int x0, int x1)
{
	if (callback_pending_) {
		callback_pending_ = 0;
		Tk_CancelIdleCall(display, (ClientData)this);
	}
	if (destroyed() || !Tk_IsMapped(tk_))
		return;

	Window window = Tk_WindowId(tk_);

	if (vi_ == 0) {
		XFillRectangle(dpy_, window, gc_, 0, 0, width_, height_);
		return;
	}
	int hoff = (width_ - vi_->width()) >> 1;
	int voff = (height_ - vi_->height()) >> 1;
	hoff += hoff_;
	voff += voff_;
	/*FIXME*/
	if (damage_) {
		damage_ = 0;
		y0 = y1 = x0 = x1 = 0;
		if (hoff > 0) {
			XFillRectangle(dpy_, window, gc_,
				       0, 0, hoff, height_ + 1);
			XFillRectangle(dpy_, window, gc_,
				       width_ - hoff, 0, hoff, height_ + 1);
		}
		if (voff > 0) {
			XFillRectangle(dpy_, window, gc_,
				       0, 0, width_, voff + 1);
			XFillRectangle(dpy_, window, gc_,
				       0, height_ - voff, width_, voff + 1);
		}
	}
	int h = y1 - y0;
	if (h == 0)
		h = vi_->height();
	else if (h > vi_->height())
		h = vi_->height();
	else if (h < 0)
		return;
	int w = x1 - x0;
	if (w == 0)
		w = vi_->width();
	else if (w > vi_->width())
		w = vi_->width();
	else if (w < 0)
		return;

	vi_->putimage(dpy_, window, gc_, x0, y0, x0 + hoff, y0 + voff, w, h);
}

/*FIXME*/
void VideoWindow::redraw()
{
	damage_ = 1;
	if (!callback_pending_) {
		callback_pending_ = 1;
		Tk_DoWhenIdle(display, (ClientData)this);
	}
}

void VideoWindow::clear()
{
	if (!callback_pending_) {
		callback_pending_ = 1;
		Tk_DoWhenIdle(doclear, (ClientData)this);
	}
}

int VideoWindow::bpp()
{
    // FIXME: This is better than creating a new StandardVideoImage, but
    // is there a better way?
    XImage *image = XCreateImage(Tk_Display(tk_), Tk_Visual(tk_), Tk_Depth(tk_),
			      ZPixmap, 0, (char*)0, 1, 1, 8, 0);
    int bpp = image->bits_per_pixel;
    XDestroyImage(image);
    return bpp;
}

void VideoWindow::dim()
{
	if (!callback_pending_) {
		callback_pending_ = 1;
		Tk_DoWhenIdle(dodim, (ClientData)this);
	}
}

void VideoWindow::doclear(ClientData cd)
{
	VideoWindow* vw = (VideoWindow*)cd;
	vw->callback_pending_ = 0;
	if (!vw->destroyed() && Tk_IsMapped(vw->tk_)) {
		Window window = Tk_WindowId(vw->tk_);
		XFillRectangle(vw->dpy_, window, vw->gc_,
			       0, 0, vw->width_, vw->height_);
	}
}

void VideoWindow::dodim(ClientData cd)
{
	static Pixmap graypm; /*FIXME*/
	static GC graygc; /*FIXME*/

	VideoWindow* vw = (VideoWindow*)cd;
	vw->callback_pending_ = 0;
	if (!vw->destroyed() && Tk_IsMapped(vw->tk_)) {
		Window window = Tk_WindowId(vw->tk_);
		if (graypm == 0) {
			u_int32_t bm[32];
			for (int i = 0; i < 32; i += 2) {
				bm[i]   = 0x55555555;
				bm[i+1] = 0xaaaaaaaa;
			}
			graypm = XCreateBitmapFromData(vw->dpy_, window,
						       (const char*)bm,
						       32, 32);
			XColor* c = Tk_GetColor(Tcl::instance().interp(),
						vw->tk_, Tk_GetUid("gray50"));
			if (c == 0)
				abort();
			XGCValues v;
			v.background = c->pixel;
			v.foreground = c->pixel;
			v.fill_style = FillStippled;
			v.stipple = graypm;
			graygc = Tk_GetGC(vw->tk_,
			       GCForeground|GCBackground|GCFillStyle|GCStipple,
			       &v);
		}
		XFillRectangle(vw->dpy_, window, graygc,
			       0, 0, vw->width_, vw->height_);
	}
}

void VideoWindow::render(const VideoImage* v, int miny, int maxy,
			 int minx, int maxx)
{
	vi_ = v;
	draw(miny, maxy, minx, maxx);
}

CaptureWindow::CaptureWindow(const char* path, XVisualInfo* vinfo)
	: BareWindow(path, vinfo),
	  base_width_(0),
	  base_height_(0),
	  image_(0)
{
	gc_ = Tk_GetGC(tk_, 0, 0);
}

CaptureWindow::~CaptureWindow()
{
	Tk_FreeGC(dpy_, gc_);
	delete image_;
}

void CaptureWindow::setsize(int w, int h)
{
	BareWindow::setsize(w, h);
	delete image_;
	image_ = StandardVideoImage::allocate(tk_, width_, height_);
}

void CaptureWindow::grab_image()
{
	XImage* image = image_->ximage();
#ifdef USE_SHM
	if (image->obdata != 0)
		XShmGetImage(dpy_, Tk_WindowId(tk_), image,
			     0, 0, AllPlanes);
	else
#endif
#ifndef WIN32
		XGetSubImage(Tk_Display(tk_), Tk_WindowId(tk_),
			     0, 0, image->width, image->height,
			     AllPlanes, ZPixmap, image, 0, 0);
#endif
}

void CaptureWindow::capture(u_int8_t* frm)
{
	/*
	 * Xv requires that the window be unobscured in order
	 * to capture the video.  So we grab the server and
	 * raise the window.  This won't work if the window
	 * isn't mapped.  Also, we block signals while
	 * the server is grabbed.  Otherwise, the process
	 * could be stopped while the display is locked out.
	 */
	if (!destroyed() && Tk_IsMapped(tk_)) {
		raise();
		grab_image();
		converter_->convert((u_int8_t*)image_->pixbuf(),
				    width_, height_, frm);
	}
}
