/*
 * chroma-key.cc --
 *
 *      A Chroma Key effect
 *
 * Copyright (c) 1993-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/fx/effects/chroma-key.cc,v 1.1 2002/02/07 04:18:19 chema Exp $ */


#include <stdio.h>
#include <math.h>
#include <time.h>
#include <dvmbasic.h>
#include <dvmbytegeom.h>


#include "chroma-key.h"



ChromaKeyEffect::ChromaKeyEffect(int decimation) 
	: EffectModule(2, decimation)
{
	// blue chroma key
	cki.key_color.r = 0;
	cki.key_color.g = 0;
	cki.key_color.b = 255;
	angle = 0.9;
	noise_level = 1.0;
	num_loops = 2;

	return;
}



int ChromaKeyEffect::command(int argc, const char*const* argv)
{
	return (EffectModule::command(argc, argv));
}



void ChromaKeyEffect::trigger_effect (Uncompressed* input)
{
	// step 1: check that sizes are the same
	if (input->samesize (buffer_[0])) {
		output_->byte_copy(input);

	} else {
		// step 2: output = chroma_key (foreground=input, background=buffer)
		get_info();
		for (int lc = 0; lc < num_loops; lc++) {
			(void)chroma_key (input, buffer_[0], output_);
		}
	}

	// deliver the frame to the next system
	deliver_frame();

	return;
}



// This code is Michael Ashihkmin's integer-only implementation of 
//	High Quality Chroma Key for a CS294-3 Fall97 project. Check the 
//	web page here: 
//
//	http://bmrc.berkeley.edu/courseware/cs294-3/fall97/projects/ashikhmin/



// This is integer-only chroma key implementation. All warnings about
// reliability of MMX version (see ck.c) apply here as well. This file
// also contains several never used definitions (leftovers from earlier
// versions) and I'm too lazy to clean this up. Look at ck.c for more
// comments.
#define A(x,y) (y * width + x)




//
// ChromaKeyEffect::chroma_key
//
// Main chroma key function.
// Assumes global variables width and height are set correctly.
// I was a bit paranoic about limits checking and explicit type
// conversions. Some of them are not necessary - they are here to
// get operation of this function as close as possible to MMX version.
// This function was written first and MMX implementation from
// ck.c is a translation of this function.
int ChromaKeyEffect::chroma_key (Uncompressed *fg, Uncompressed *bg, 
		Uncompressed *out)
{
	int work_count = 0;
	int kbg;
	int x1, y1;
	short tmp, tmp1;
	char x, z;

	ByteImage *fg_l = fg->lum_;
	ByteImage *fg_cr = fg->cr_;
	ByteImage *fg_cb = fg->cb_;
	ByteImage *bg_l = bg->lum_;
	ByteImage *bg_cr = bg->cr_;
	ByteImage *bg_cb = bg->cb_;
	ByteImage *out_l = out->lum_;
	ByteImage *out_cr = out->cr_;
	ByteImage *out_cb = out->cb_;

//			(void)chroma_key (in_l->width, in_l->height,
//					in_l->firstByte, in_cb->firstByte, in_cr->firstByte,
//					buf_l->firstByte, buf_cb->firstByte, buf_cr->firstByte,
//					out_l->firstByte, out_cb->firstByte, out_cr->firstByte);

 
 
	for (int i = 0; i < height; i++){
		for (int j = 0; j < width; j++){
			// Convert foreground to XZ coords where X direction is defined by
			// the key color
			tmp = ( (short)(fg_cb[A(j,i)] - 128) * cki.key_colorY.cb +
					(short)(fg_cr[A(j,i)] - 128) * cki.key_colorY.cr) >> 7; // / 128
			if (tmp > 127) tmp = 127;
			if (tmp < -128) tmp = -128;
			x = tmp;

			tmp = ( (short)(fg_cr[A(j,i)] - 128) * cki.key_colorY.cb-
					(short)(fg_cb[A(j,i)] - 128) * cki.key_colorY.cr) >> 7; // / 128
			if (tmp > 127) tmp = 127;
			if (tmp < -128) tmp = -128;
			z = tmp;

			// WARNING: accept angle should never be set greater than "somewhat 
			//	less than 90 degrees" to avoid dealing with negative/infinite 
			//	tangents. In reality, 80 degrees should be enough if foreground 
			//	is reasonable. If this seems to be a problem, go to alternative 
			//	ways of checking point position (scalar product or line equations). 
			//	This angle should not be too small either to avoid infinite ctg 
			//	(used to suppress foreground without use of division)
 			tmp = ((short)(x) * cki.accept_angle_tg) >> 4; // /0x10
			if (tmp > 127) tmp = 127; 
			if (abs(z) > tmp){
				// keep foreground Kfg = 0
				out_y[A(j,i)] = fg_y[A(j,i)];
				out_cb[A(j,i)] = fg_cb[A(j,i)];
				out_cr[A(j,i)] = fg_cr[A(j,i)];
				// out_y[A(j,i)] = out_cb[A(j,i)] = out_cr[A(j,i)] = 0;
			} else {
				work_count++;

				// compute Kfg (implicitly) and Kbg, suppress foreground in XZ coord
				//	according to Kfg
				tmp = ((short)(z) * cki.accept_angle_ctg) >> 4; // /0x10
				if (tmp > 127) tmp = 127;
				if (tmp < -128) tmp = -128;
				x1 = abs(tmp);
				y1 = z;

				tmp1 = x - x1;
				if (tmp1 < 0) tmp1 = 0;
				kbg = ((unsigned char)(tmp1) * (unsigned short)(cki.one_over_kc)) / 2;
				if (kbg < 0) kbg = 0;
				if (kbg > 255) kbg = 255;

				tmp = ((unsigned short)(tmp1) * cki.kout_y_scale) >> 4; // /0x10
				if (tmp > 0xff) tmp = 0xff;
				tmp = fg_y[A(j,i)]  - 128 - tmp;
				if (tmp < 0) tmp = 0;
				out_y[A(j,i)] = tmp + 128;
     
				// Convert suppressed foreground back to CbCr
				tmp = ( (char)(x1) * (short)(cki.key_colorY.cb) -
						(char)(y1) * (short)(cki.key_colorY.cr)) >> 7; // /128
				if (tmp < -128) {
					out_cb[A(j,i)] = -128 + 128;
				} else if (tmp > 127) {
					out_cb[A(j,i)] = 127 + 128;
				} else {
					out_cb[A(j,i)] = tmp + 128;
				}
     
				tmp = ( (char)(x1) * (short)(cki.key_colorY.cr) +
						(char)(y1) * (short)(cki.key_colorY.cb)) >> 7; // /128
				if (tmp < -128) {
					out_cr[A(j,i)] = -128 + 128;
				} else if (tmp > 127) {
					out_cr[A(j,i)] = 127 + 128;
				} else {
					out_cr[A(j,i)] = tmp + 128;
				}

				// Deal with noise. For now, a circle around the key color with
				// radius of noise_level treated as exact key color. Introduces
				// sharp transitions.
				tmp = z * (short)(z) + (x - cki.kg) * (short)(x - cki.kg);
				//if (tmp > 0xffff) tmp = 0xffff;
				if (tmp < (noise_level * noise_level)) {
					// Uncomment this if you want total suppression within the noise 
					//	circle
					// out_y[A(j,i)] = out_cb[A(j,i)] = out_cr[A(j,i)] = 0 + 128;
					kbg = 255;
				}

				// Add Kbg*background
				tmp = out_y[A(j,i)] - 128 + ((unsigned short)(kbg) * 
						(bg_y[A(j,i)] - 128)) / 256;
				out_y[A(j,i)] = ((tmp < 255) ? tmp : 255) + 128;

				tmp = out_cb[A(j,i)] - 128 + ((unsigned short)(kbg) * 
						(bg_cb[A(j,i)] - 128)) / 256;
				if (tmp < -128) {
					out_cb[A(j,i)] = -128 + 128;
				} else if (tmp > 127) {
					out_cb[A(j,i)] = 127 + 128;
				} else {
					out_cb[A(j,i)] = tmp + 128;
				}

				tmp = out_cr[A(j,i)]  - 128 + ((unsigned short)(kbg) * 
						(bg_cr[A(j,i)] - 128)) / 256;
				if (tmp < -128) {
					out_cr[A(j,i)] = -128 + 128;
				} else if (tmp > 127) {
					out_cr[A(j,i)] = 127 + 128;
				} else {
					out_cr[A(j,i)] = tmp + 128;
				}
			} 
		}
	}
	return work_count;
}



//
// ChromaKeyEffect::get_info
//
void ChromaKeyEffect::get_info ()
{
	float kgl;
	float tmp;
	// printf("Key color (R,G,B from 0 to 1 each):");
	// scanf("%f,%f,%f",&r,&g,&b);
	kgl = rgb2y (&cki.key_color, &cki.key_colorY);
	cki.accept_angle_cos = cos (M_PI * angle / 180);
	cki.accept_angle_sin = sin (M_PI * angle / 180);

	tmp = 0xf * tan (M_PI * angle / 180);
	if (tmp > 0xff) tmp = 0xff;
	cki.accept_angle_tg = (unsigned char)tmp;

	tmp = 0xf / tan (M_PI * angle / 180);
	if (tmp > 0Xff) tmp = 0Xff;
	cki.accept_angle_ctg = (unsigned char)tmp;

	tmp  = 1 / (kgl);
	cki.one_over_kc = 0xff * 2 * (unsigned char)tmp - 0xff; 

	tmp = 0xf * (float)(cki.key_colorY.y) / kgl;
	if (tmp > 0Xff) tmp = 0xff;
	cki.kout_y_scale = (unsigned char)tmp;

	if (kgl > 127) kgl = 127;
	cki.kg = (char)kgl;  
}



//
// ChromaKeyEffect::rgb2y
//
float ChromaKeyEffect::rgb2y (color *c, colorY *y)
{
	float tmp, tmp1, tmp2;
	y->y =  0.257 * c->r + 0.504 * c->g + 0.098 * c->b;
	tmp1 = -0.148 * c->r - 0.291 * c->g + 0.439 * c->b;
	tmp2 =  0.439 * c->r - 0.368 * c->g - 0.071 * c->b;
	tmp = sqrt (tmp1 * tmp1 + tmp2 * tmp2);
	y->cb = (char)(127 * (tmp1 / tmp));
	y->cr = (char)(127 * (tmp2 / tmp));
	return tmp;
}

