/*
 * djv.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <osfcn.h>
#include <errno.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <fcntl.h>

#include "jpeg.h"
#include "packet.h"

#ifdef JVS
#include "jvs.h"
extern "C" {
#include <sys/param.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#ifdef mips
#include <mips/cachectl.h>
int cacheflush(char* addr, int len, int cache);
#endif
}
#endif

class Grabber {
public:
	virtual ~Grabber();
	int width() { return (width_); }
	int height() { return (height_); }
	int qfactor() { return (qfactor_); }
	virtual u_char* capture(int& len) = 0;
protected:
	int width_;
	int height_;
	int qfactor_;
};

Grabber::~Grabber()
{
}

class FileGrabber : public Grabber {
public:
	~FileGrabber();
	FileGrabber(const char* filename);
	u_char* capture(int& len);
private:
	struct hdr {
		int cc;
		struct vicfrm_jpg ph;
	};
	int fd;
	u_char buffer[64 * 1024];
};

FileGrabber::FileGrabber(const char* filename)
{
	fd = open(filename, O_RDONLY);
	if (fd < 0) {
		/*FIXME*/
		perror(filename);
		exit(1);
	}
	hdr* p = (hdr*)buffer;
	int cc = read(fd, buffer, sizeof(*p));
	if (cc < sizeof(*p)) {
		fprintf(stderr, "%s: not a vic dump file\n");
		exit(1);
	}
	width_ = ntohs(p->ph.width);
	height_ = ntohs(p->ph.height);
	qfactor_ = ntohs(p->ph.quantization);
	(void)lseek(fd, (off_t)0, SEEK_SET);
}

FileGrabber::~FileGrabber()
{
}

u_char* FileGrabber::capture(int& len)
{
	hdr* p = (hdr*)buffer;
	int cc = read(fd, (char*)buffer, sizeof(*p));
	if (cc < 0) {
		perror("read");
		exit(1);
	}
	if (cc < sizeof(*p))
		return (0);
	cc = ntohl(p->cc);
	if (cc < 0 || cc > 100000) {
		fprintf(stderr, "impossible length\n");
		exit(1);
	}
	cc -= sizeof(struct vicfrm_jpg);
	if (read(fd, (char*)buffer, cc) == 0)
		return (0);

	/* Remove the trailing zeros inserted by the DMA process */
	u_char* ip = buffer + cc;
	if (ip[-1] == 0) {
		while(*--ip == 0)
			;
		ip += 2;
	}
	*ip++ = 0xff;
	*ip++ = 0xd9;/* EOI */
	len = ip - buffer;

	return (buffer);
}

class JvGrabber : public Grabber {
public:
	struct dmabuf {
		int shmid;
		u_char* bp;
	};
	JvGrabber(int q);
	~JvGrabber();
	u_char* capture(int& len);
private:
	void kick();
	void sync();
	void SetComp();
	dmabuf* allocdma(int type);
	void freedma(dmabuf*);
	dmabuf* cb0_;
	dmabuf* cb1_;
	dmabuf* cb_;
	int server_;
};

JvGrabber::JvGrabber(int q)
{
	qfactor_ = q;

	/*FIXME probably should do the open in start() -- when it is
	  moved into the destructor (e.g., cannot use server test) */
	server_ = JvsOpen("", JVS_SOCKET);
	if (server_ < 0) {
		printf("djv: can't open jvdriver connection\n");
		exit(1);
	}
	cb0_ = allocdma(JVS_JPEG);
	cb1_ = allocdma(JVS_JPEG);
	cb_ = cb0_;
	SetComp();

	kick();
}

JvGrabber::~JvGrabber()
{
	close(server_);
}

void JvGrabber::SetComp()
{
	/*
	 * Don't bother doing anything fancy with frame subsampling.
	 * On the j300 this doesn't work anyway, and on the jvideo
	 * we set it to one so we get every other field.
	 */
	int skip = 1;
	int w, h;
	int xdec = 2;
	int ydec = 2;
	if (JvsSetComp(server_, qfactor_, xdec, ydec, skip, &w, &h) < 0) {
		fprintf(stderr, "vic: JvsSetComp failed\n");
		exit(1);
	}
	width_ = w;
	height_ = h;
}

JvGrabber::dmabuf* JvGrabber::allocdma(int type)
{
	dmabuf* p = new dmabuf;
	int w = 640;
	int h = 480;
	if (JvsAllocateBuf(server_, JVS_INPUT, type, w, h, &p->shmid) < 0) {
		fprintf(stderr, "vic: JvsAllocateBuf failed\n");
		exit(1);
	}
	p->bp = (u_char *)shmat(p->shmid, 0, 0);
	if (p->bp == (u_char*)-1) {
		perror("vic: shmat");
		exit(1);
	}
	/*
	 * remove the shared memory segment now -- right after
	 * it has been allocated.  the idea is that we'd like
	 * the shared memory segments to go away if we exit
	 * abonormally, and since they will still be attached to
	 * our address space the kernel won't deallocate it
	 * until we exit.  the ultrix man page doesn't seems
	 * to indicate that this is necessarily the case, but
	 * a quick glance at the kernel code for the smctl()
	 * system call verifies it.
	 */
	(void)shmctl(p->shmid, IPC_RMID, 0);

	return (p);
}

void JvGrabber::freedma(JvGrabber::dmabuf* p)
{
	(void)JvsDeallocBuf(server_, p->shmid);
	if (shmdt((char*)p->bp) < 0) {
		perror("vic: shmdt");
		exit(1);
	}
	delete p;
}

void JvGrabber::sync()
{
	/*FIXME wait for last completion so nothing is left
	  in the socket buffer */
	int length, shmid;
	(void)JvsWaitComp(server_, &shmid, &length);
}

void JvGrabber::kick()
{
	cb_ = (cb_ == cb0_) ? cb1_ : cb0_;
	if (JvsStartComp(server_, cb_->shmid) < 0) {
		fprintf(stderr, "vic: JvsStartComp failed\n");
		return;
	}
}

/*
 * Called by IOHandler when rpc reply arrives
 * on server socket.
 */
u_char* JvGrabber::capture(int& len)
{
	int length;
	int shmid;
	/*
	 * Since we have input waiting on this socket, this
	 * rpc should return immediately.
	 */
	while (JvsWaitComp(server_, &shmid, &length) < 0) {
		fprintf(stderr, "vic: JvsWaitComp failed\n");
		/*
		 * (maybe) the compression timed out.
		 * The hardware can be flaky.
		 * Try again.
		 */
		kick();
	}
	dmabuf* p = cb_;
	kick();
	len = length;
	return (p->bp);
}

extern "C" {
void makewin(int, int);
void putrawimage();
void putcolorimage(u_char*, u_char*, u_char*, int);
void init_color_cube();
}
void pstats();

void
adios(int)
{
	printf("\n");
#ifdef notdef
	delete grabber;
#endif
	pstats();
	exit(0);
}

#include <signal.h>

static void
usage()
{
	fprintf(stderr, "usage\n");
	exit(1);
}

extern void JVideoConfig(JpegDecoder::config& c, int w, int h, int q);

int noshm;
int hysteresis;
int hlevel = 1;
int hblock = -1;

int
main(int argc, char **argv)
{
	extern char *optarg;

	extern int optind, opterr;
	int jflag = 0;
	int cflag = 0;
	int qfactor = 100;
	int nflag = 0;
	int scalebits = 0;

	opterr = 0;
	int op;
	while ((op = getopt(argc, argv, "b:ch:jl:nq:rs:")) != -1) {
		switch (op) {

		case 'b':
			hblock = atoi(optarg);
			break;

		case 'c':
			++cflag;
			break;

		case 'h':
			hysteresis = atoi(optarg);
			break;

		case 'j':
			++jflag;
			break;

		case 'l':
			hlevel = atoi(optarg);
			break;

		case 'n':
			++nflag;
			break;

		case 'q':
			qfactor = atoi(optarg);
			break;

		case 'r':
			noshm = 1;
			break;

		case 's':
			scalebits = atoi(optarg);
			break;

		default:
			usage();
			break;
		}
	}
	signal(SIGINT, adios);

	Grabber* grabber;
#ifdef JVS
	if (jflag)
		grabber = new JvGrabber(qfactor);
	else
#endif
	if (argv[optind] != 0)
		grabber = new FileGrabber(argv[optind]);
	else
		usage();

	int w = grabber->width();
	int h = grabber->height();
	int q = grabber->qfactor();

	int xw;
	int xh;
	if (scalebits >= 0) {
		xw = w >> scalebits;
		xh = h >> scalebits;
	} else {
		xw = w << 1;
		xh = h << 1;
	}
	if (!nflag) {
		makewin(xw, xh);
		/*FIXME*/
		init_color_cube();
	}
	JpegDecoder* decoder;
	JpegDecoder::config config;
	JVideoConfig(config, w, h, q);

	/*FIXME*/
	decoder = JpegDecoder::create(config);
#ifdef notdef
	} else {
		int clut[256];
		extern int grey_lut[];
		extern u_char *ximage_data;
		u_char *cp;

		if (scalebits > 0) {
			fprintf(stderr, "only use -s with -c (for now)\n");
			exit(1);
		}
		if (nflag)
			cp = new u_char[xw * xh];
		else
			cp = ximage_data;

		for (int i = 0; i < 32; ++i)
			for (int k = 0; k < 8; ++k)
/*FIXME*/
#ifdef sun
				clut[i << 3 | k] = grey_lut[i];
#else
				clut[i << 3 | k] = grey_lut[i] >> 24;
#endif
		decoder = new GrayJpegDecoder(config, cp, clut, scalebits);
	}
#endif

	timeval start;
	gettimeofday(&start, 0);
	u_char* frame;
	int len;
	int nframe = 0;
	while ((frame = grabber->capture(len)) != 0) {
		decoder->decode(frame, len);
		if (nflag)
			;
		else
			putcolorimage((u_char*)decoder->yplane(),
				      (u_char*)decoder->uplane(),
				      (u_char*)decoder->vplane(),
				      scalebits);

		if (++nframe >= 10) {
			nframe = 0;
			timeval end;
			gettimeofday(&end, 0);
			double v = end.tv_sec - start.tv_sec;
			v += 1e-6 * (end.tv_usec - start.tv_usec);
			printf("%lg\n", 10. / v);
			start = end;
		}
	}
	pstats();

	delete grabber;

	return (0);
}

