/*
 * decoder.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1995-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/audio/decoder.cc,v 1.11 2002/02/03 03:10:46 lim Exp $";

#include <stdlib.h>
#include "decoder.h"
#include "controller.h"
#include "pktbuf.h"

#define	INITIAL_OFFSET (4*AUDIO_FRAMESIZE)


static class PCM_DecoderClass : public TclClass {
    public:
	PCM_DecoderClass() : TclClass("Module/AudioDecoder/PCM") {}
	TclObject* create(int /* argc */, const char*const* /* argv */) {
		return (new PCM_Decoder);
	};
} pcm_decoder_class;

AudioDecoder::AudioDecoder() :
	PacketHandler(0),
	nstat_(0),
	controller_(0),
	var_(INITIAL_OFFSET << (VAR_FILTER - VAR_MULT)),
	hostoffset_(0),
	predicted_drop_(~0),
	playout_(INITIAL_OFFSET << PLAYO_FILTER),
	lastrecv_(0),
	lecture_mode_(0),
	framesize_(AUDIO_FRAMESIZE), /*FIXME*/
	block_size_(AUDIO_FRAMESIZE),
	maxdel_(0)
{
	for (int i = 0; i < MAXSTAT; ++i) {
		stat_[i].name = 0;
		stat_[i].cnt = 0;
	}
#define STAT_LATE	0
	stat_[STAT_LATE].name = "Late-Pkts";
	nstat_ = 1;
}

AudioDecoder::~AudioDecoder()
{
}

void AudioDecoder::info(char* wrk) const
{
	*wrk = 0;
}

void AudioDecoder::stats(char* bp)
{
	if (nstat_ == 0) {
		*bp = 0;
		return;
	}
	for (int i = 0; i < nstat_; ++i) {
		sprintf(bp, "%s %d ", stat_[i].name, stat_[i].cnt);
		bp += strlen(bp);
	}
	bp[-1] = 0;
}

int AudioDecoder::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
		char* bp = tcl.buffer();
		if (strcmp(argv[1], "info") == 0) {
			info(bp);
			tcl.result(bp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "stats") == 0) {
			stats(bp);
			tcl.result(bp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "block-size") == 0) {
			sprintf(bp, "%u", block_size_);
			tcl.result(bp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "playout") == 0) {
			sprintf(bp, "%u", playout());
			tcl.result(bp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "reset-offset") == 0) {
			hostoffset_ = 0;
			var_ = 0;
			playout_ = 0;
			return (TCL_OK);
		}
	} else if (argc == 3) {
		if (strcmp(argv[1], "controller") == 0) {
			controller_ = (Controller*)TclObject::lookup(argv[2]);
			maxdel_ = controller_->maxdel();
			return (TCL_OK);
		}
		if (strcmp(argv[1], "lecture-mode") == 0) {
			lecture_mode_ = atoi(argv[2]);
			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}

void PCM_Decoder::recv(pktbuf* pb)
{
	const struct rtphdr* rh = (rtphdr*)pb->dp;
	const u_int8_t* data = (u_int8_t*)(rh + 1);
	int len = pb->len - sizeof(*rh);
	if (active())
		consume_samples(rh, data, len);
	pb->release();
}

inline int absdiff(int x, int y)
{
	register int r = y - x;
	return (r < 0? -r : r);
}

static inline int newoffset(
	int nvar,
	int playout,
	int maxdel, int mindel, int lecture)
{
	register int offset = nvar;
	if (offset > maxdel)
		offset = maxdel;
	register int diff = playout - offset;

	if (diff > 0) {
		// offset going down: in LectureMode, drop at most
		// one frametime per talkspurt.  In ConferenceMode,
		// drop at most 1/2 of difference.
		if (lecture) {
			if (diff > AUDIO_FRAMESIZE) {
				if (playout > (maxdel * 3) / 4 &&
				    diff > 10 * AUDIO_FRAMESIZE)
					diff = 5 * AUDIO_FRAMESIZE;
				else
					diff = AUDIO_FRAMESIZE;
			}
		} else
			diff >>= 1;
		offset = playout - diff;
	} else if (-diff > maxdel) {
		// offset going way up: only allow 3/4 of max.
		offset = (maxdel * 3) / 4;
	}
	if (offset > (maxdel * 7) / 8)
		offset = (maxdel * 7) / 8;
	else if (offset < mindel)
		offset = mindel;
	return (offset);
}

int AudioDecoder::adapt(const rtphdr* rh, u_int32_t local_clock)
{
	register u_int32_t tstamp = ntohl(rh->rh_ts);
	register int hoff = (int)hostoffset_;
	register int offset = (tstamp + hoff - local_clock) &~ 3;
	int new_ts = rh->rh_flags & htons(RTP_M);
	if (hoff == 0 || new_ts) {
		/*
		 * start of new talk spurt --
		 * use accumulated variance to compute new offset if
		 * this would make a significant change.  We change if
		 *  - the variance is currently 'small', or
		 *  - the change would be a least a packet time
		 */
		register int nvar = var_ >> (VAR_FILTER - VAR_MULT);
		offset = playout_ >> PLAYO_FILTER;
		if (nvar < 3*AUDIO_FRAMESIZE || absdiff(nvar, offset) >= AUDIO_FRAMESIZE) {
			offset = newoffset(nvar, offset, maxdel_,
					   block_size_, lecture_mode_);
			/*
			 * assume that a talk spurt starts with TALK_LEAD
			 * samples of history & subtract them off if possible.
			 */
			if (new_ts) {
				offset -= 4 * AUDIO_FRAMESIZE;
				if (offset < block_size_)
					offset = block_size_;
			}
		}
		hostoffset_ = local_clock - tstamp + offset;
	} else if (offset < 0 || offset > maxdel_) {
		/*
		 * packet out of range -- if last packet also out of
		 * range or if the delay would increase, resync.
		 */
		if (offset < 0 || predicted_drop_ == tstamp) {
			offset = newoffset(var_ >> (VAR_FILTER - VAR_MULT),
					   playout_ >> PLAYO_FILTER,
					   maxdel_, block_size_,
					   lecture_mode_);
			hostoffset_ = local_clock - tstamp + offset;
		} else {
			predicted_drop_ = tstamp + block_size_;
			lastrecv_ = tstamp - local_clock;
			count(STAT_LATE);
			return (-1);
		}
	} else {
		// packet in range, update interarrival var. est.
		register int nvar = var_;
		register int off = tstamp - local_clock - lastrecv_;
		if (off < 0)
			off = -off;
		off -= (nvar >> VAR_FILTER);
		var_ = nvar + off;
	}
	lastrecv_ = tstamp - local_clock;
	register u_int avgplay = playout_;
	playout_ = avgplay + (offset - (avgplay >> PLAYO_FILTER));
	offset &= ~3;
	delvar_ = var_ >> VAR_FILTER;
	return (offset);
}

/*
 * Dummy class for streams that we recognize but cannot decode (for lack
 * of software support).  We still want to put all the stats etc. --
 * especially the format -- but can't decode it.  FIXME We need to somehow
 * mark the stream in the user-interface so that the user knows that
 * vic cannot decode the stream (to avoid assuming something else is
 * wrong with the tranmission)
 */
class AudioNullDecoder : public AudioDecoder {
    public:
	virtual void recv(pktbuf* pb) { pb->release(); }
};

static class NullDecoderClass : public TclClass {
    public:
	NullDecoderClass() : TclClass("Module/AudioDecoder/Null") {}
	TclObject* create(int /* argc */, const char*const* /* argv */) {
		return (new AudioNullDecoder);
	};
} audio_dm_null;

