/* $Id: nws_sensor.c,v 1.185 2004/12/02 04:25:49 graziano Exp $ */

#include "config_nws.h"
#include <unistd.h>           /* getopt() */

#include "nws_sensor.h"
#include "dnsutil.h"
#include "clique_protocol.h"
#include "protocol.h"
#include "host_protocol.h"
#include "diagnostic.h"
#include "messages.h"
#include "osutil.h"
#include "periodic.h"
#include "skills.h"
#include "strutil.h"
#include "nws_sensor.h"
#include "register.h"

/**
 * Search the nameserver for the registration of the memory with host
 * hostName
 */
#define LOOKUP_MEMORY_FILTER "(&(name=%s*)(hostType=memory))"
#define LOOKUP_MEMORY_FILTER_LEN 30
static int
LookupMemory(struct host_cookie *nameServer,
             const char *hostName,
             struct host_cookie *hostCookie) {
	char *filter;
	ObjectSet objSet;
	int ret;

	/* sanity check */
	if (nameServer == NULL || hostName == NULL || hostCookie == NULL) {
		ERROR("LookupMemory: NULL parameters\n");
		return 0;
	}

	/* get all objectclass=nwsHost with this name */
	objSet = NewObjectSet();
	filter = MALLOC(LOOKUP_MEMORY_FILTER_LEN + strlen(hostName) + 1);
	if (filter == NULL || objSet == NULL) {
		ABORT("LookupMemory: out of memory\n");
	}
	sprintf(filter, LOOKUP_MEMORY_FILTER, hostName);
	ret = !RetrieveObjects(nameServer, filter, &objSet, -1);
	free(filter);

	/* let's check if we got something */
	if (ret || (!ret && strlen(objSet) == 0)) {
		FreeObjectSet(&objSet);
		return 0;
	}

	/* set defaults */
	SAFESTRCPY(hostCookie->name, "");
	hostCookie->port = 0;
	hostCookie->sd = NO_SOCKET;

	/* let's try to connect to whatever we got back */
	if (ConnectToObject(NextObject(objSet, NO_OBJECT), hostCookie)) {
		/* found a good memory */
		ret = 1;
	} 

	/* free allocated memory */
	FreeObjectSet(&objSet);

	return ret;
}


/*
 * try to find the leader for clique named #cliqueName#. Put the host in
 * #cookie#. Returns 1 on success 0 otherwise.
 */
static int
FindLeader(	const char *cliqueName,
		struct host_cookie *leader) {
	char *filter;
	ObjectSet retrieved = NULL;
	Object clique = NO_OBJECT;
	int ret = 1;

	/* sanity check */
	if (leader == NULL || cliqueName == NULL) {
		ERROR("FindLeader: NULL parameters\n");
		return 0;
	}

	/* get the registrations */
	filter = (char *) MALLOC(strlen(cliqueName) + 8);
	if (filter == NULL) {
		ERROR("FindLeader: out of memory\n");
		return 0;
	}
	sprintf(filter, "(name=%s)", cliqueName);
	if (!RetrieveFromMyNameserver(filter, &retrieved)) {
		FREE(filter);
		ERROR("FindLeader: error talking with nameserver\n");
		return 0;
	}
	FREE(filter);

	/* did we get the clique? */
	for (clique = NextObject(retrieved, NULL); clique != NULL; clique = NextObject(retrieved, clique)) {
		filter = NwsAttributeValue_r(FindNwsAttribute(clique, "controlName"));
		if (filter != NULL && strcmp(filter, "clique") == 0) {
			/* found the clique */
			break;
		}
	}
	FREE(filter);
	if (clique == NULL) {
		ERROR1("FindLeader: couldn't find clique %s\n", cliqueName);
		return 0;
	}

	/* we got the clique: let's get the leader */
	filter = NwsAttributeValue_r(FindNwsAttribute(clique, "host"));
	if (filter == NULL) {
		ERROR1("FindLeader: clique registration bogus %s\n", clique);
		ret = 0;
	} else {
		/* fill the cookie */
		Host2Cookie(filter, DefaultHostPort(SENSOR_HOST), leader);
	}
	FreeObject(&retrieved);
	FREE(filter);

	return 1;

}

#define NEVER 0.0
#define SENSOR_ARGS "j:i:n:a:c:e:l:M:N:Pp:v:VS:A:fm:x:"

void usage() {
	printf("\nUsage: nws_sensor [OPTIONS]\n");
	printf("Sensor for the Network Weather Service\n");
	printf("\nOPTIONS can be:\n");
	printf("\t-e filename         write error messages to filename\n");
	printf("\t-l filename         write info/debug messages to filename\n");
	printf("\t-i filename         write pid to filename\n");
	printf("\t-M memory           use memory to store measurements\n");
	printf("\t-N nameserver       register with this nameserver\n");
	printf("\t-p port             bind to port instead of the default\n");
	printf("\t-a address          use this address as mine (ie multi-homed hosts)\n");
	printf("\t-n name             use name as my hostname\n");
	printf("\t-c [yes|no]         start CpuMonitor at startup (default yes)\n");
	printf("\t-m [yes|no]         start MemoryMonitor at startup (default no)\n");
	printf("\t-A [yes|no]         start Availability Monitor at startup (default no)\n");
	printf("\t-S [yes|no]         start process start Monitor at startup (default no)\n");
	printf("\t-P password         use password to control sensor\n");
	printf("\t-f                  don't fork (experimental)\n");
	printf("\t-x port             use port when doing network experiment (experimental)\n");
	printf("\t-j clique           join this clique (experimental)\n");
	printf("\t-v level            verbose level (up to 5)\n");
	printf("\t-V                  print version\n");
	printf("Report bugs to <nws@nws.cs.ucsb.edu>.\n\n");
}


int
main(	int argc,
	char *argv[]) {

	char opts[255+1];
	IPAddress addresses[MAX_ADDRESSES], address;
	unsigned int addressesCount;
	int opt,
		verbose = 2,			/* what to print */
		tmp;
	char *name,
		memoryName[127 + 1],		/* hostMemory name */
		password[127 + 1],
		cliqueName[MAX_CLIQUE_NAME_SIZE],
		tmpIP[127+1],
		*pidFile;
	IPAddress tmpAddress;
	struct host_cookie nsCookie,
		sensorCookie,
		memoryCookie;
	double nextBeatTime,
		nextWorkTime,
		wakeup,
		now;
	extern char *optarg;
	const char *c;
	FILE *logFD, *errFD;
	int skillToStart[SKILL_COUNT];

	/* Set up default values */
	for (tmp = 0; tmp < SKILL_COUNT; tmp++) {
		skillToStart[tmp] = 0;
	}
	skillToStart[cpuMonitor] = 1;
	logFD = errFD = stdout;
	cliqueName[0] = password[0] = tmpIP[0] = opts[0] = '\0';
	pidFile = NULL;

	/* zeroes the cookies */
	nsCookie.name[0] = sensorCookie.name[0] = memoryCookie.name[0] = '\0';
	nsCookie.port = sensorCookie.port = memoryCookie.port= 0;

	/* if we don't have a default memory, we try to look for one on
	 * this host */
	name = GetEnvironmentValue("SENSOR_MEMORY", "~", ".nwsrc", MyMachineName());
	if (name == NULL) {
		SAFESTRCPY(memoryName, "");
	} else {
		SAFESTRCPY(memoryName, name);
		FREE(name);
	}

	/* if we don't have a default nameserver, we try to look for one on
	 * this host */
	name = GetEnvironmentValue("NAME_SERVER", "~", ".nwsrc", MyMachineName());
	if (name == NULL) {
		SAFESTRCPY(nsCookie.name, "");
	} else {
		SAFESTRCPY(nsCookie.name, name);
		FREE(name);
	}
	Host2Cookie(nsCookie.name, DefaultHostPort(NAME_SERVER_HOST), &nsCookie);

	/* who am I? I think, thus I exist! */
	Host2Cookie(MyMachineName(), DefaultHostPort(SENSOR_HOST), &sensorCookie);
	addressesCount = IPAddressValues(MyMachineName(),
			&addresses[0],
			MAX_ADDRESSES);
	if (addressesCount == 0) {
		ERROR1("Couldn't resolve my name (%s)\n", MyMachineName());
	}


	/* no error messages from getopt() */
	opterr = 0;

	while((opt = getopt(argc, argv, SENSOR_ARGS)) != EOF) {
		switch(opt) {

		case 'n':
			/* let's check the user knows what is doing */
			if (IPAddressValue(optarg, &tmpAddress)) {
				/* save the inet addresses */
				addressesCount += IPAddressValues(optarg, &addresses[addressesCount], MAX_ADDRESSES - addressesCount);
			} else {
				ERROR1("Unable to resolve '%s': I'll do what you said but expect problems!\n", optarg);
			}

			/* overrride the name of the machine: we hope the
			 * user knows what is doing! */
			SAFESTRCPY(sensorCookie.name, optarg);

			break;

		case 'a':
			/* let's add this IPs to the list of my addresses */
			for (c = optarg; GETTOK(tmpIP, c, ",", &c); ) {
				tmp = IPAddressValues(tmpIP, 
						&addresses[addressesCount],
						MAX_ADDRESSES - addressesCount);
				if (tmp == 0) {
					ERROR1("Unable to convert '%s' into an IP address\n", tmpIP);
				} else {
					addressesCount += tmp;
					/* I want to have these addresses
					 * as first */
					for (; tmp > 0; tmp--) {
						address = addresses[tmp - 1];
						addresses[tmp-1] = addresses[addressesCount-tmp];
						addresses[addressesCount-tmp] = address;
					}
				}
			}
			break;

		 case 'A':
			skillToStart[availabilityMonitor] = (*optarg == 'y') || (*optarg == 'Y');
			break;
		 case 'S':
			skillToStart[startMonitor] = (*optarg == 'y') || (*optarg == 'Y');
			break;

		case 'm':
			skillToStart[memoryMonitor] = (*optarg == 'y') || (*optarg == 'Y');
			break;

		case 'c':
			skillToStart[cpuMonitor] = (*optarg == 'y') || (*optarg == 'Y');
			break;

		case 'f':
			tmp = strlen(opts);
			if (tmp == 0) {
				snprintf(opts + tmp, 255 - tmp, "fork:no");
			} else {
				snprintf(opts + tmp, 255 - tmp, "\tfork:no");
			}
			break;

		case 'e':
			/* open the error file */
			errFD = fopen(optarg, "w");
			if (errFD == NULL) {
				printf("Couldn't open %s!\n", optarg);
				exit(1);
			}
			break;

		case 'l':
			/* open the log file */
			logFD = fopen(optarg, "w");
			if (logFD == NULL) {
				printf("Couldn't open %s!\n", optarg);
				exit(1);
			}
			break;

		case 'i':
			pidFile = strdup(optarg);
			if (pidFile == NULL) {
				ABORT("out of memory!\n");
			}
			break;

		case 'j':
			SAFESTRCPY(cliqueName, optarg);
			break;

		case 'M':
			SAFESTRCPY(memoryName, optarg);
			break;

		case 'N':
			Host2Cookie(optarg, DefaultHostPort(NAME_SERVER_HOST), &nsCookie);
			break;

		case 'P':
			printf("password? ");
			scanf("%s", password);
			break;

		case 'p':
			sensorCookie.port = strtol(optarg, NULL, 10);
			break;

		case 'x':
			tmp = strlen(opts);
			if (tmp == 0) {
				snprintf(opts + tmp, 255 - tmp, "forceport:%d", (unsigned short)atol(optarg));
			} else {
				snprintf(opts + tmp, 255 - tmp, "\tforceport:%d", (unsigned short)atol(optarg));
			}
			break;

		case 'v':
			verbose = (unsigned short)atol(optarg);
			break;

		case 'V':
			printf("nws_sensor for NWS version %s", VERSION);
#ifdef HAVE_PTHREAD_H
			printf(", with thread support");
#endif
#ifdef WITH_DEBUG
			printf(", with debug support");
#endif
			printf("\n\n");
			exit(0);
			break;

		case '?':
			if (optopt == 'v') {
				/* using the first level */
				verbose = 1;
				break;
			}
			/* non-recognized options: printing help */

		default:
			usage();
			exit(1);
			break;

		}
	}

	/* let's set the verbose level  and open up files (if needed) */
	/* fatal errors are always on */
	SetDiagnosticLevel(verbose, errFD, logFD);
	
	/* let's set the memory as we are given */
	Host2Cookie(memoryName, DefaultHostPort(MEMORY_HOST), &memoryCookie);
	SAFESTRCPY(memoryName, HostCImage(&memoryCookie));

	/* let's look for the nameserver */
	if(!ConnectToHost(&nsCookie, &nsCookie.sd)) {
		WARN("Unable to contact name server\n");
	} else {
		/* we are here only if we managed to talk with the NS */
		if (!LookupMemory(&nsCookie, memoryName, &memoryCookie)) {
			WARN("Unable to find memory registration\n");
		}
	}

	/* let's get ready to listen */
	if(!EstablishHost(HostCImage(&sensorCookie),
                    SENSOR_HOST,
                    addresses,
                    addressesCount,
                    sensorCookie.port,
                    password,
                    &nsCookie,
                    &memoryCookie,
                    &NwsSensorExit)) {
		FAIL("Unable to establish host: port already in use?\n");
	}

	/* now that we've got the port, we can print the pid into the
	 * pid file. We thus avoid to override pid files of running
	 * nameservers */
	if (pidFile != NULL) {
		FILE *pidfile = fopen(pidFile, "w");
		if (!pidfile) {
			ABORT1("Can't write pidfile %s\n", pidFile);
		}
		fprintf(pidfile, "%d", (int)getpid());
		fclose(pidfile);
		free(pidFile);
	}

	/* initialize the sensor */
	if (!NwsSensorInit(opts)) {
		WARN("Sensor: failed to init the sensors\n");
	}

	for (tmp = 0; tmp < SKILL_COUNT; tmp++) {
		if (!skillToStart[tmp]) {
			continue;
		}
		/* start the activity with the default name */
		if(!StartPeriodicActivity(NULL, SkillName(tmp), "")) {
			WARN1("StartSensor: auto-start of %s failed\n", SkillName(tmp));
		}
		FREE(name);
	}

	/* have we been asked to join a clique? */
	if (cliqueName[0] != '\0') {
		struct host_cookie k;

		/* yep let's join the clique: first of all find the
		 * leader of the clique */
		if (FindLeader(cliqueName, &k)) {
			/* then ask it to join the club */
			CliqueJoin(&k, cliqueName, HostCImage(&sensorCookie), -1);
		}
	}

	nextBeatTime = 0;

	while(1) {
		/* time to register again with the name server */
		now = CurrentTime();
		if(now >= nextBeatTime) {
			if (RegisterHost(DEFAULT_HOST_BEAT * 2) == 0) {
				INFO("nws_sensor: couldn't register\n");
			}
			nextBeatTime = now + (HostHealthy() ? DEFAULT_HOST_BEAT : SHORT_HOST_BEAT);
		}

		/* let's see when we need to wake up next */
		nextWorkTime = NwsSensorNextWork();
		if((nextWorkTime != NEVER) && (nextWorkTime < nextBeatTime)) {
			/* if now > wakeup ListenForMessage will just
			 * poll for messages. */
			wakeup = nextWorkTime - now;
		} else {
			wakeup = nextBeatTime - now;
		}

		ListenForMessages((wakeup > 0) ? wakeup : -1);

		/* time to do some work? NwsSensorWork will check if it's
		 * the right time to do work */
		NwsSensorWork();
	}
	return 0; /* never reached */
}
