#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>      /* struct timeval */
#include <string.h>
#include <dirent.h>
#include <time.h>
#include "diagnostic.h"  /* FAIL() LOG() */

#include "fstests.h"
#include "config_nws.h"
#include "strutil.h"



#define FILENAMESIZE 16

int FileSystemMonitorAvailable(const char *options) {
  /* need to check if we can run the test */
  return(1);
}


void FileSystemUseSkill( const char *options,
			 int *length,
			 SkillResult **results) {

	int rc, fssize, fsblock, fsbufmode, fsphysmem;
	long fssize_long;
	float readspeed, writespeed;
	char path[63 + 1], *opts, *fstype, *fstmpdir, *tmp; 
	const char *c;

	srand(time(NULL));
	fstmpdir = NULL;
  
	fstype = GetOptionValue(options, "fstype", "block");
	fstmpdir = GetOptionValue(options, "fstmpdir", "/tmp");
  
	tmp = GetOptionValue(options, "fssize", "8");
	fssize = strtol(tmp, NULL, 10);
	FREE(tmp);
	tmp = GetOptionValue(options, "fsblock", "0");
	fsblock = strtol(tmp, NULL, 10);
	FREE(tmp);
	tmp = GetOptionValue(options, "fsbufmode", "0");
	fsbufmode = strtol(tmp, NULL, 10);
	FREE(tmp);
	tmp = GetOptionValue(options, "fsphysmem", "512");
	fsphysmem = strtol(tmp, NULL, 10);
	FREE(tmp);
  
	fssize_long = (long)((long)fssize * (long)1024 * (long)1024);
	fssize = (int)fssize_long;
  
	if (!strcmp(fstype, "char")) {
		fsblock = 1;
	} else if (!fsblock) { 
		/* nspecified, test will try to 'guess' */
		fsblock = 0;
	}
  
	DDEBUG1("fstests: starting test %lf\n", (double)time(NULL));
  
	tmp = GetOptionValue(options, "path", "");
	for(c = tmp; GETTOK(path, c, ",", &c);) {
		opts = malloc(sizeof(char)*strlen(path)+6);
		sprintf(opts, "path:%s", path);
    
		rc = fstest_runreadtest(path, fsbufmode, fssize, fsblock, fsphysmem, fstmpdir, &readspeed);
		if (rc < 0) {
			AppendResult(fsreadSpeed, opts, 0, 0.0, length, results);
		} else {
			AppendResult(fsreadSpeed, opts, 1, readspeed, length, results);
		}
    
		rc = fstest_runwritetest(path, fsbufmode, fssize, fsblock, fsphysmem, fstmpdir, &writespeed);
		if (rc < 0) {
			AppendResult(fswriteSpeed, opts, 0,0.0, length, results);
		} else {
			AppendResult(fswriteSpeed, opts, 1, writespeed, length, results);
		}
		free(opts);
	}
	FREE(tmp);
	FREE(fstype);
	FREE(fstmpdir);

	DDEBUG1("fstests: finishing test %lf\n", (double)time(NULL));
}

int fstest_runreadtest(char *path, int flushmode, long size, int block, int memsize, const char *tmpdir, float *result) {

  char *filename;

  int rc,
    chunks,
    openmode;
  
  struct stat mystat;
  double myclock;

  openmode = 0;
  myclock =0.0;
  filename = NULL;

  rc = stat(path, &mystat);
  if (rc < 0) {
    FAIL1("ERROR: cannot stat path '%s'\n", path);
    return(-1);
  }

  if (!block) {
    block = mystat.st_blksize;
  }

  chunks = size / block;

  /* these two modes are similar */
  if (flushmode == NOFLUSH || flushmode == MEANFLUSH) {
    
    /* for both modes, create a tempfile, on the FS being tested, of
       the requested size */

    filename = malloc(strlen(path) * sizeof(char) + FILENAMESIZE + 2);
    strncpy(filename, path, strlen(path)+1);
    strcat(filename, "/");
    
    rc = createafile(filename, size);
    if (rc < 0) {
      FAIL1("ERROR: could not create a tmpfile to read: %s\n", filename);
      free(filename);
      return(1);
    }

    /* if we're in mean flush mode, flush the buffer cache using any
       means possible */
    if (flushmode == MEANFLUSH) {
      LOG("attempting to flush buffer cache by eating up memory\n");
      rc = flushbuffer(tmpdir, memsize);
      if (rc < 0) {
	unlink(filename);
	free(filename);
	FAIL("error flushing buffer with flushbuffer()\n");
	return(-1);
      }
    }

    /* then read 'size' from file (size == block * chunks) and record
       number of seconds in &myclock */
    
    LOG5("Starting readtest: file=%s size=%d block=%d chunks=%d time=%lf\n", filename, size, block, chunks, (double)time(NULL));
    
    rc = timeread(filename, 0, block, chunks, openmode, &myclock);
    if (rc < 0) {
      unlink(filename);
      free(filename);
      return(-1);
    }
    
    LOG5("Complete readtest: file=%s size=%d block=%d chunks=%d time=%lf\n", filename, size, block, chunks, (double)time(NULL));
    

    unlink(filename);
    free(filename);  
  } else if (flushmode == NICEFLUSH) {    

    /* kind of weird heuristic to try to miss the buffer cache without
       interfering with normal system operation */

    DIR *dirfh;
    struct dirent *mydirent;
    char *fullpath;
    testfile allfiles[1024];
    int fcount, chunks_to_read;
    int readbytes, i;
    double tmpclock;

    int chunksleft, readchunks, randchunks, seekoffset;

    /* open the directory in question and read the sizes of a bunch of files in the dir */

    dirfh = opendir(path);
    if (dirfh == NULL) {
      return(-1);
    }

    fcount = 0;
    mydirent = readdir(dirfh);
    while(mydirent != NULL && (fcount < 1024)) {
      fullpath = malloc(strlen(path) + strlen(mydirent->d_name) + 5);
      sprintf(fullpath, "%s/%s", path, mydirent->d_name);

      rc = stat(fullpath, &mystat);
      if (((rc == 0) && (mystat.st_size > block)) && S_ISREG(mystat.st_mode) && (strstr(fullpath, "nwstst") == NULL)) {
	allfiles[fcount].filename = fullpath;
	allfiles[fcount].size = mystat.st_size;
	fcount++;
      }
      mydirent = readdir(dirfh);
      free(fullpath);
    }
    closedir(dirfh);

    /* 
       now, while the total number of bytes read is less than the
       target number to read, select a random file in the directory,
       choose to read a random number of 'chunks' from the file, seek
       to a random place in the file, and time only the reads, adding
       to a total amount of time when reads finish 
    */

    readbytes = 0;
    readchunks = 0;

    while(readbytes < size) {
      /* choose random file index */
      i = rand() % fcount;
      
      rc = 0;

      /* calculate random seekoffset into file, how many chunks we have left to read total, and a random number of chunks from the file */
      seekoffset = rand() % (allfiles[i].size - block);
      chunksleft = chunks - readchunks;
      randchunks = rand() % (allfiles[i].size - seekoffset)/block;

      /* choose to actually read either chunksleft or randchunks, whicever is smallest */
      if (chunksleft < randchunks) {
	chunks_to_read = chunksleft;
      } else {
	chunks_to_read = randchunks;
      }

      /* read chunks from the file (tmpclock will contain the time it took for only the reads */
      tmpclock = 0.0;
      rc = timeread(allfiles[i].filename, seekoffset, block, chunks_to_read, openmode, &tmpclock);
      if (rc < 0) {
	unlink(filename);
	free(filename);
	return(-1);
      }
      readbytes += rc;

      /* add to total myclock the time taken actually reading in previous step */
      myclock += tmpclock;
      readchunks += chunks_to_read;

      /*      LOG5("file %s size %ld chunks %d read %d myclock %lf\n", allfiles[i].filename, allfiles[i].size, chunks_to_read, readbytes, myclock);*/
    }

  }

  *result = ((float)size/1000000.0) / (float)myclock;

  /* printf("final result: %s %f\n", path, *result);*/

  return(0);
}


int flushbuffer(const char *tmpdir, int memsize) {
  int fd, rc, block;
  char *filename;
  long tmpsize;
  char buf[4096];

  block = 4096;
  /*  buf = malloc(sizeof(char) * block);*/

  filename = malloc(strlen(tmpdir) * sizeof(char) + FILENAMESIZE + 2);
  strncpy(filename, tmpdir, strlen(tmpdir)+1);
  strcat(filename, "/");
  
  tmpsize = memsize * 2 * 1024 * 1024;
  rc = createafile(filename, tmpsize);
  if (rc < 0) {
    FAIL1("ERROR: could not create a tmpfile to read: %s\n", filename);
    free(filename);
    return(1);
  }

  fd = open(filename, O_RDONLY);
  if (fd < 0) {
    FAIL1("ERROR: could not open tmpfile: %s\n", filename);
    free(filename);
    return(-1);
  }

  while(read(fd, buf, block)){}
  close(fd);

  unlink(filename);
  free(filename);
  /*  free(buf);*/
  return(0);
}


long timeread(char *filename, int offset, int block, int chunks, int openmode, double *myclock) {
  int fd, i, rc;
  struct timeval testclock;
  char *buf;

  buf = malloc(block * sizeof(char));
  if (buf == NULL) {
    FAIL("ERROR: out of memory cannot malloc\n");
    return(-1);
  }

  if (openmode == 1) {
    clockstart(&testclock);
  }

  fd = open(filename, O_RDONLY);
  if (fd < 1) {
    FAIL1("ERROR: cannot open %s for read\n", filename);
    free(buf);
    return(-1);
  }

  if (offset) {
    rc = lseek(fd, offset, SEEK_SET);
  }

  if (openmode == 0) {
    clockstart(&testclock);
  }

  for (i=0; i<chunks; i++) {
    rc = read(fd, buf, block);
  }
  
  if (openmode == 0) {
    clockstop(&testclock, myclock);
  }
  if (fd < 0) {
    FAIL1("ERROR: could not read from file %s\n", filename);
    *myclock = 0.0;
    free(buf);
    return(-1);
  }
  

  close(fd);
  if (openmode == 1) {
    clockstop(&testclock, myclock);
  }

  free(buf);
  return(block*chunks);
}


int fstest_runwritetest(char *path, int flushmode, long size, int block, int memsize, const char *tmpdir, float *result) {
  char *filename,
    *buf;
  int rc,
    fd,
    chunks,
    i;
  
  struct timeval testclock;
  struct stat mystat;
  double myclock;

  rc = stat(path, &mystat);
  if (rc < 0) {
    FAIL1("ERROR: cannot stat path '%s'\n", path);
    return(-1);
  }

  if (!block) {
    block = mystat.st_blksize;
  }

  buf = malloc(block * sizeof(char));
  if (buf == NULL) {
    FAIL("ERROR: out of memory, cannot malloc\n");
    return(-1);
  }
  
  /* malloc size: passed in path + random filename + \0 and "/" */
  filename = malloc(strlen(path) * sizeof(char) + FILENAMESIZE + 2);
  strncpy(filename, path, strlen(path)+1);
  strcat(filename, "/");

  gen_filename(filename);
    
  fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0644);
  if (fd < 1) {
    FAIL1("ERROR: cannot open %s for write\n", filename);
    unlink(filename);
    free(filename);
    free(buf);
    return(-1);
  }

  chunks = size / block;

  LOG5("Starting writetest: file=%s size=%d block=%d chunks=%d time=%lf\n", filename, size, block, chunks, (double)time(NULL));
  
  clockstart(&testclock);
  for (i=0; i<chunks; i++) {
    rc = write(fd, buf, block);
  }
  clockstop(&testclock, &myclock);
  
  LOG5("Complete writetest: file=%s size=%d block=%d chunks=%d time=%lf\n", filename, size, block, chunks, (double)time(NULL));

  if (fd < 0) {
    FAIL1("ERROR: could not write to file %s\n", filename);
    unlink(filename);
    free(filename);
    free(buf);
    return(-1);
  }

  close(fd);    
  unlink(filename);

  *result = ((float)size/1000000.0) / (float)myclock;
  /*
    printf("size: %f myclock %f result: %f\n", ((float)size / 1000000.0), (float)clock, *result);
  */

  free(filename);
  free(buf);
  return(0);
}

void gen_filename(char *filename) {
  int i;
  int len;

  len = strlen(filename);

  for (i=len; i<len+FILENAMESIZE-8; i++) {
    filename[i] = (rand() % 26) + 97;
  }
  filename[i] = '\0';
  strcat(filename, ".nwstst");
  filename[i+7] = '\0';
}

int createafile(char *filename, long fsize) {
  int fd;
  char *buf;
  long writebytes = 0;
  int blocksize = 4096;

  buf = malloc(blocksize * sizeof(char));
  
  gen_filename(filename);

  while((fsize % blocksize) != 0) {
    blocksize -= 2;
  }
  
  fd = open(filename, O_CREAT | O_TRUNC | O_RDWR, 0644);
  if (fd < 0) {
    FAIL1("ERROR: cannot open file %s for write\n", filename);
    free(buf);
    return(-1);
  }
  
  while(writebytes < fsize) {
    writebytes += write(fd, buf, blocksize);
  }
  
  fsync(fd);
  close(fd);
  
  free(buf);
  return(0);
}

void clockstart(struct timeval *testclock) {
  
  gettimeofday(testclock, NULL);
  
}

void clockstop(struct timeval *testclock, double *myclock) {
  struct timeval currtime;

  gettimeofday(&currtime, NULL);

  *myclock = (double)(currtime.tv_sec + (currtime.tv_usec/1000000.0)) - (double)(testclock->tv_sec + (testclock->tv_usec/1000000.0));

}
