/* $Id: start_activity.c,v 1.23 2004/10/21 05:49:26 graziano Exp $ */

#include "config_nws.h"
#include <stdio.h>          /* printf() sprintf() */
#include <stdlib.h>         /* atoi() free() REALLOC() */
#include <string.h>         /* strcat() strcpy() strchr() */
#include <unistd.h>         /* getopt() */
#define NWSAPI_SHORTNAMES

#include "nws_sensor.h"
#include "diagnostic.h"     /* NWS diagnostic messages. */
#include "cliques.h"
#include "osutil.h"
#include "register.h"
#include "nws_api.h"        /* Interface functions */


/*
** This program allows the user to launch and terminate NWS activities from the
** command line.  See the USERSGUIDE for a description of the command-line
** options.
*/


/*
** Parses #option# as a "name:value" pair and appends it to the #optionsCount#-
** long string array #optionsList#, updating #optionsCount# as appropriate.
** Allows for the pair to be split between sequential calls (i.e. "name:" one
** call, "value" the next).  Returns 1 if successful, else 0.
*/
static int
AddOption(const char *option,
          char ***optionList,
          size_t *optionsCount) {

  static int isValue = 0;
  char **lastOption;
  const char *valueStart;

  if(isValue) {
    valueStart = option;
  }
  else {
    valueStart = strchr(option, ':');
    if(valueStart == NULL) {
      fprintf(stderr, "AddOption: colon missing from option %s\n", option);
      return 0;
    }
    valueStart++;
  }

  if(isValue) {
    lastOption = (*optionList) + (*optionsCount - 1);
    *lastOption =
      REALLOC(*lastOption, strlen(*lastOption) + strlen(option) + 1);
    strcat(*lastOption, option);
  }
  else {
    (*optionsCount)++;
    *optionList = (char **)
       REALLOC(*optionList, *optionsCount * sizeof(char *));
    (*optionList)[*optionsCount - 1] = strdup(option);
  }
    
  isValue = *valueStart == '\0';
  return 1;

}


/*
** Frees all elements of the #length#-long allocated element array #array#.
*/
static void
FreeAllElements(void **array,
                size_t length) {
  int i;
  for(i = 0; i < length; i++)
    free(array[i]);
}


/*
** Returns the value of the #attributeName# attribute from #object#, or an
** empty string if no such attribute exists. The returned memory needs to be freed
*/
static char *
GetNwsAttributeValue(const Object object,
                  const char *attributeName) {
	NwsAttribute attribute;
	char *tmp;
	int ret;

	ForEachNwsAttribute(object, attribute) {
		tmp = NwsAttributeName_r(attribute);
		if (tmp == NULL) {
			continue;
		}
		ret = strcmp(tmp, attributeName); 
		free(tmp);

		if (ret  == 0) {
      			return NwsAttributeValue_r(attribute);
		}
	}

	return strdup("");
}


/*
** Retrieves control and skill names matching the partial names in #control#
** and #skill# that have been registered by #sensor#, then attempts to narrow
** the choices to one control/skill pair using the options listed in the
** #optionsCount#-long list #options#.  If successful, returns 1 and sets
** #control# and #skill# to the retrieved names; otherwise, returns 0.
*/
static int
GetControlAndSkill(const HostSpec *sensor,
                   char *control,
                   char *skill,
                   const char **options,
                   int optionsCount) {

  Object controlObject;
  ObjectSet controlObjects;
  char filter[255 + 1];
  int i;
  char optionName[255 + 1];
  char possiblePairs[255 + 1];
  char recognizedOptions[255 + 1];
  Object skillObject;
  ObjectSet skillObjects;
  char supportedSkills[255 + 1];
  char *tmp;

  /* Grab lists of matching controls and objects from the name server. */
  sprintf(filter, "(&(%s=%s:%d)%s(%s=*%s*))",
          HOST_ATTR, sensor->machineName, sensor->machinePort,
          CONTROLS, CONTROL_ATTR, control);
  if(!GetObjects(filter, &controlObjects)) {
    fprintf(stderr, "GetControlAndSkill: unable to contact name server\n");
    return 0;
  }
  sprintf(filter, "(&(%s=%s:%d)%s(%s=*%s*))",
          HOST_ATTR, sensor->machineName, sensor->machinePort,
          SKILLS, SKILL_ATTR, skill);
  if(!GetObjects(filter, &skillObjects)) {
    FreeObjectSet(&controlObjects);
    fprintf(stderr, "GetControlAndSkill: unable to contact name server\n");
    return 0;
  }

  /*
  ** For each control/supported skill pair, match the option list against the
  ** control and skill options to see if the pair recognizes all of them.
  ** Throw out pairs where this isn't true and hope that we have exactly one
  ** pair at the end.
  */
  possiblePairs[0] = '\0';
  ForEachObject(controlObjects, controlObject) {

    tmp = GetNwsAttributeValue(controlObject, SKILL_ATTR);
    strcpy(supportedSkills, tmp);
    free(tmp);

    ForEachObject(skillObjects, skillObject) {

      tmp = GetNwsAttributeValue(skillObject, SKILL_ATTR);
      if(strstr(supportedSkills, tmp) == NULL) {
	      free(tmp);
		continue;
      }
      free(tmp);

      strcpy(recognizedOptions, ",");
      tmp = GetNwsAttributeValue(controlObject, OPTION_ATTR);
      strcat(recognizedOptions, tmp);
      free(tmp);
      strcat(recognizedOptions, ",");
      tmp = GetNwsAttributeValue(skillObject, OPTION_ATTR);
      strcat(recognizedOptions, tmp);
      free(tmp);
      strcat(recognizedOptions, ",");

      for(i = 0; i < optionsCount; i++) {
        strcpy(optionName, ",");
        strcat(optionName, options[i]);
        *strchr(optionName, ':') = '\0';
        strcat(optionName, ",");
        if(strstr(recognizedOptions, optionName) == NULL)
          break; /* Unknown option. */
      }
      if(i == optionsCount) {
        tmp = GetNwsAttributeValue(controlObject, CONTROL_ATTR);
        strcpy(control, tmp);
	free(tmp);
        tmp = GetNwsAttributeValue(skillObject, SKILL_ATTR);
        strcpy(skill, tmp);
	free(tmp);
        if(possiblePairs[0] != '\0')
          strcat(possiblePairs, "\n");
        strcat(possiblePairs, CONTROL_ATTR);
        strcat(possiblePairs, ":");
        strcat(possiblePairs, control);
        strcat(possiblePairs, " ");
        strcat(possiblePairs, SKILL_ATTR);
        strcat(possiblePairs, ":");
        strcat(possiblePairs, skill);
      }
    }

  }

  FreeObjectSet(&controlObjects);
  FreeObjectSet(&skillObjects);

  if(possiblePairs[0] == '\0') {
    fprintf(stderr, "%s\n%s\n%s\n",
            "No registered control and skill support all listed options",
            "Please check your spelling, make sure the requested skill is",
           " available on this host and try again");
    return 0;
  }
  else if(strchr(possiblePairs, '\n') != NULL) {
    fprintf(stderr, "Ambiguous control/skill; please specify one of:\n");
    fprintf(stderr, "%s\n", possiblePairs);
    return 0;
  }

  return 1;

}


/*
** Returns 1 or 0 depending or not an object named #name# is registered with
** the name server.
*/
static int
LookupName(const char *name) {

  char filter[127 + 1];
  ObjectSet objects;
  int returnValue;

  sprintf(filter, "(%s=%s)", NAME_ATTR, name);
  if(!GetObjects(filter, &objects)) {
    return 0;
  }
  returnValue = FirstObject(objects) != NULL;
  FreeObjectSet(&objects);
  return returnValue;

}


/*
** Reads #fileName# and appends to #optionsList# any "name:value" pairs
** contained within.  Returns 1 if successful, else 0.
*/
static int
ParseOptionsFile(const char *fileName,
                 char ***optionsList,
                 size_t *optionsCount) {

  char line[255 + 1];
  FILE *optionsFile;
  char *word;

  optionsFile = fopen(fileName, "r");
  if(optionsFile == NULL) {
    fprintf(stderr, "ParseOptionsFile: unable to open file %s\n", fileName);
    return 0;
  }

  while(fgets(line, sizeof(line), optionsFile) != NULL) {
    for(word = strtok(line, " \n\t");
        (word != NULL) && (*word != '#');
        word = strtok(NULL, " \n\t")) {
      if(!AddOption(word, optionsList, optionsCount)) {
        fclose(optionsFile);
        return 0; /* Message will come from AddOption(). */
      }
    }
  }

  (void)fclose(optionsFile);
  return 1;

}


static void
RemoveOption(const char *optionName,
             char ***options,
             size_t *optionsCount,
             char *value) {

  int i;
  int j;
  int nameLen = strlen(optionName);

  *value = '\0';
  for(i = 0; i < *optionsCount; ) {
    if((strncmp((*options)[i], optionName, nameLen) == 0) &&
       ((*options)[i][nameLen] == ':')) {
      strcpy(value, &(*options)[i][nameLen + 1]);
      free((*options)[i]);
      (*optionsCount)--;
      for(j = i; j < *optionsCount; j++)
        (*options)[j] = (*options)[j + 1];
    }
    else {
      i++;
    }
  }

}


#define START_COMMAND "start"
#define SWITCHES "f:Fa"

int
main(int argc,
     char *argv[]) {

  extern char *optarg;
  extern int optind;
  char activityName[127+1];
  char controlName[127 + 1];
  char *tmp;
  HostSpec nsSpec;
  int opt;
  char **options;
  size_t optionsCount;
  char optionsFile[127 + 1];
  HostSpec sensorSpec;
  char skillName[127 + 1];
  const char *USAGE = "start_activity [-f file] [-F] [-a] host [option ...]";

  int force = 0;
  int all = 0;

  optionsFile[0] = '\0';
  DirectDiagnostics(DIAGERROR, stderr);
  DirectDiagnostics(DIAGFATAL, stderr);

  while((opt = getopt(argc, argv, SWITCHES)) != EOF) {

    switch(opt) {

    case 'f':
      strcpy(optionsFile, optarg);
      break;

	case 'F':
	  force = 1;
	  break;

    case 'a':
      all=1;
      break;

    default:
      fprintf(stderr, "Unrecognized switch\n%s\n", USAGE);
      exit(1);
      break;

    }

  }

  if(optind >= argc) {
    fprintf(stderr, "%s\n", USAGE);
    exit(1);
  }

	tmp = EnvironmentValue_r("SENSOR_PORT", NULL);
	if (tmp == NULL) {
		opt = GetDefaultPort(NWSAPI_SENSOR_HOST);
	} else {
		opt = atoi(tmp);
		free(tmp);
	}
	sensorSpec = *MakeHostSpec (argv[optind++], opt);

  if(!GetNameServer(&sensorSpec, &nsSpec)) {
    fprintf(stderr, "Unable to contact sensor %s:%d\n",
            sensorSpec.machineName, sensorSpec.machinePort);
    exit(1);
  }
  else if(!UseNameServer(&nsSpec)) {
    fprintf(stderr, "Unable to contact name server %s:%d\n",
            nsSpec.machineName, nsSpec.machinePort);
    exit(1);
  }

  if (all) {
    printf("Restart all activities on %s:%d.\n",
          sensorSpec.machineName, sensorSpec.machinePort);
    exit (!RestartActivities(&sensorSpec));
  }

  options = NULL;
  optionsCount = 0;

  if((optionsFile[0] != '\0') &&
     !ParseOptionsFile(optionsFile, &options, &optionsCount)) {
    FreeAllElements((void **)options, optionsCount);
    free(options);
    exit(1);  /* Message will come from ParseOptionsFile() */
  }

  for(; optind < argc; optind++) {
    if(!AddOption(argv[optind], &options, &optionsCount)) {
      FreeAllElements((void **)options, optionsCount);
      free(options);
      exit(1);  /* Message will come from AddOption() */
    }
  }

  activityName[0] = '\0';
  controlName[0] = '\0';
  skillName[0] = '\0';
  RemoveOption("name", &options, &optionsCount, activityName);
  RemoveOption(CONTROL_ATTR, &options, &optionsCount, controlName);
  RemoveOption(SKILL_ATTR, &options, &optionsCount, skillName);

  if(!GetControlAndSkill(&sensorSpec,
                         controlName,
                         skillName,
                         (const char **)options,
                         optionsCount)) {
    FreeAllElements((void **)options, optionsCount);
    free(options);
    exit(1);  /* Message will come from GetControlAndSkill() */
  }

	if(activityName[0] == '\0') {
		char ciccio[63 + 1];

		/* there is no name: let's get the default */
		snprintf(ciccio, 64, "%s:%d", sensorSpec.machineName, sensorSpec.machinePort);
		tmp = NameOfActivity_r(ciccio, skillName, "");
		if (tmp != NULL) {
			strncpy(activityName, tmp, sizeof(activityName));
			free(tmp);
		} else {
			ERROR("couldn't generate activity name\n");
			exit(1);
		}
	}

	if (strcmp(controlName, "clique") == 0) {
		if (strlen(activityName) >= MAX_CLIQUE_NAME_SIZE) {
			WARN("Clique name too long: chopping it\n");
			activityName[MAX_CLIQUE_NAME_SIZE-1] = '\0';
		}
	}
  if(LookupName(activityName)) {
	if (force == 0) {
	  fprintf(stderr, "Activity '%s' already registered, you can \n\thalt the activity (with halt_activity)\n\tuse -F to force restart\n\tuse a different name (with the 'name:' option)\n", activityName);
	  exit(1);
	}
	else {
	  fprintf(stderr, "Activity already registered; restarting anyway.\n");
	}
  }

  if(StartActivity(&sensorSpec,
                        activityName,
                        controlName,
                        skillName,
                        (const char **)options,
                        optionsCount)) {
    printf("Started activity %s on %s:%d\n",
           activityName, sensorSpec.machineName, sensorSpec.machinePort);
  }

  FreeAllElements((void **)options, optionsCount);
  free(options);
  return 0;

}
