/* Magic definition maker.
 */

#include "ip.h"

/* Struct we carry during a magic walk.
 */
typedef struct {
	Symbol *sym;		/* Symbol we are magicing */
	int count;			/* Number of parameters */
	List *params;			/* Symbols to be parameters */
	List *locals;			/* Symbols to be locals */
	List *text;			/* Strings for body of defn. */
	char *defn;			/* Definition we make */
} Magic;

/* Make a new magic struct.
 */
static Magic *
make_magic( Symbol *sym )
{
	Magic *magic = IM_NEW( NULL, Magic );

	magic->sym = sym;
	magic->count = 1;
	magic->params = NULL;
	magic->locals = NULL;
	magic->text = NULL;
	magic->defn = NULL;

	return( magic );
}

/* Free an old magic struct.
 */
static void
free_magic( Magic *magic )
{
	im_list_free( &magic->params, NULL, NULL, NULL );
	im_list_free( &magic->locals, NULL, NULL, NULL );
	im_list_free( &magic->text, (im_list_free_fn) im_free, NULL, NULL );
	FREE( magic->defn );
	FREE( magic );
}

/* Sub-fn. of below.
 */
static void *
append_string( char *line, char *buf )
{
	strcat( buf, line );

	return( NULL );
}

/* Sub-fn. of below.
 */
static int
add_length( char *str, int n )
{
	return( n + strlen( str ) );
}

/* Turn a list of strings into a single string.
 */
static char *
make_string( List *text )
{
	int n;
	char *buf;

	/* Count up total characters.
	 */
	n = (int) im_list_fold( text, 
		0, (im_list_fold_fn) add_length, NULL, NULL );
	
	/* Allocate space.
	 */
	buf = im_malloc( NULL, n + 1 );

	/* Copy strings in.
	 */
	*buf = '\0';
	im_list_map( text, 
		(im_list_map_fn) append_string, buf, NULL );

	return( buf );
}

/* Make an ip predicate from a value.
 */
static char *
toip_type( PElement *base )
{
	if( PEISREAL( base ) )
		return( "is_number" );
	else if( heap_isstring( base ) )
		return( "is_string" );
	else if( PEISBOOL( base ) )
		return( "is_bool" );
	else if( PEISCOMPLEX( base ) )
		return( "is_complex" );
	else if( PEISREGION( base ) )
		return( "is_region" );
	else if( PEISIMAGE( base ) )
		return( "is_image" );
	else if( PEISCHAR( base ) )
		return( "is_char" );
	else if( PEISLIST( base ) )
		return( "is_list" );
	else
		return( "<type unknown by magic>" );
}

/* Make a descriptive name from a value.
 */
static char *
decode_type( PElement *base )
{
	if( PEISREAL( base ) )
		return( "real number" );
	else if( heap_isstring( base ) )
		return( "string" );
	else if( PEISBOOL( base ) )
		return( "boolean" );
	else if( PEISCOMPLEX( base ) )
		return( "complex" );
	else if( PEISREGION( base ) )
		return( "region" );
	else if( PEISIMAGE( base ) )
		return( "vips_image" );
	else if( PEISCHAR( base ) )
		return( "char" );
	else if( PEISLIST( base ) )
		return( "list" );
	else
		return( "<type unknown by magic>" );
}

/* Add a type-check line for this parameter.
 */
static void *
add_type_check( Symbol *sym, Magic *magic )
{
	char buf[4096];
	PElement *root = &sym->expr->root;

	im_snprintf( buf, 4096, 
		"      =\terror \"arg %d is not %s\", not %s %s;\n",
		magic->count,
		decode_type( root ), toip_type( root ),
		MODEL( sym )->name );
	im_list_append( &magic->text, im_strdup( NULL, buf ) );
	magic->count += 1;

	return( NULL );
}

/* Add a symbol name, with a space, to a string. Used for formatting
 * parameter line.
 */
static void *
add_name( Symbol *sym, char *buf )
{
	strcat( buf, MODEL( sym )->name );
	strcat( buf, " " );

	return( NULL );
}

/* Add a local definition.
 */
static void *
add_local( Symbol *sym, Magic *magic )
{
	char buf[4096];

	im_snprintf( buf, 4096, "\t%s\n", sym->expr->text );
	im_list_append( &magic->text, im_strdup( NULL, buf ) );

	return( NULL );
}

/* Given a symbol upon which our output depends. Is it suitable for turning
 * into a local or a parameter?
 */
static void *
add_referant( Symbol *sym, Magic *magic )
{
	if( sym->type == SYM_VALUE )
		if( sym->expr->nparam == 0 && sym->expr->row )
			/* Add as a local definition.
			 */
			if( !im_list_member( magic->locals, sym ) ) {
				im_list_add( &magic->locals, sym );

				/* Recurse for anything which this 
				 * local refers to.
				 */
				if( slist_map( sym->expr->children,
					(SListMapFn) add_referant, 
					magic ) )
					return( sym );
			}

	return( NULL );
}

/* Make a magic definition for this symbol.
 */
gboolean
magic_sym( Symbol *sym )
{
	Magic *magic;
	Symbol *osym = workspace_add_symbol( main_workspacegroup->current );
	char buf[ 4096 ];

	if( !osym )
		return( FALSE );

	magic = make_magic( sym );

	/* Build locals to our new definition. Definitions which we refer to
	 * which have values (ie. no parameters) become locals, values with no
	 * definitions which we refer to become parameters.
	 */
	if( slist_map( sym->expr->children, 
		(SListMapFn) add_referant, magic ) ) {
		free_magic( magic );
		return( FALSE );
	}

	/* Did we find anything? If not, do nothing.
	 */
	if( !magic->locals && !magic->params ) {
		free_magic( magic );
		errors( "Trivial magic expression generated\n"
			"Try something more complicated!" );
		return( FALSE );
	}

	/* Build the top of the definition.
	 */
	im_list_append( &magic->text, im_strdup( NULL, 
		"/* Definition created automatically by ip.\n" ) );
	im_list_append( &magic->text, im_strdup( NULL, 
		" * This will need some editing to be useful!\n" ) );
	im_list_append( &magic->text, im_strdup( NULL, 
		" */\n" ) );
	im_list_append( &magic->text, im_strdup( NULL, 
		"\n" ) );

	/* Make the first line.
	 */
	*buf = '\0';
	strcat( buf, MODEL( osym )->name );
	strcat( buf, " " );
	im_list_map( magic->params, 
		(im_list_map_fn) add_name, buf, NULL );
	strcat( buf, "\n" );
	im_list_append( &magic->text, im_strdup( NULL, buf ) );

	/* Make a type-check line for each parameter.
	 */
	im_list_map( magic->params,
		(im_list_map_fn) add_type_check, magic, NULL );
	
	/* Make the main line.
	 */
	im_snprintf( buf, 4096, "      =\t%s", sym->expr->rhstext );
	im_list_append( &magic->text, im_strdup( NULL, buf ) );

	/* Add the locals, if any.
	 */
	if( magic->locals ) {
		im_list_append( &magic->text, im_strdup( NULL, "\n{\n" ) );
		im_list_map( magic->locals,
			(im_list_map_fn) add_local, magic, NULL );
		im_list_append( &magic->text, im_strdup( NULL, "};\n" ) );
	}
	else
		im_list_append( &magic->text, im_strdup( NULL, ";\n" ) );

	/* Make a single string from the list of strings we have made.
	 */
	magic->defn = make_string( magic->text );

	/* Open an editor with this text.
	 */
	pop_def_edit( magic->defn );

	/* Free the magic struct.
	 */
	free_magic( magic );

	return( TRUE );
}

