

# chatola.py
# a simple chat engine

import random


from twisted.cred import portal, checkers
from nevow import inevow
from nevow import loaders
from nevow import rend
from nevow import tags
from nevow import liveevil



REALM, MIND = range(2)

WORDS = open('/usr/share/dict/words').readlines()


class ChatolaRealm:
    """A simple implementor of cred's IRealm.
    The only interface we support is web. We create and return a unique
    instance of Chatola for each new user, and keep track of them in self.clients.
    """
    __implements__ = portal.IRealm

    def __init__(self):
        self.clients = []
        self.topic = "Welcome to Chatola"
        self.messagePattern = inevow.IQ(Chatola.docFactory).patternGenerator('message')
        self.userPattern = inevow.IQ(Chatola.docFactory).patternGenerator('user')

    def requestAvatar(self, avatarId, mind, *interfaces):
        """The mind when using nevow is a LiveEvil instance
        which represents the client side browser. We can
        use it to broadcast messages to viewing browsers.

        When constructing the page the user will see, we pass the realm
        as the data to the page. When the user enters some text to broadcast,
        this page can then notify the realm which broadcasts the message to
        all other clients.

        We also pass the mind. This list becomes available as the main page
        data. The Chatola methods can use the REALM and MIND constants
        defined above to access the appropriate object.
        
        requestAvatar should return a tuple of
            (interface, pageObject, logoutCallable)

        In our case, we are returning an object implementing IResource
        and we use the logout callable to remove a user from the chat
        in case their session expires before the body.onunload javascript
        event fires.
        """
        if inevow.IResource in interfaces:
            if avatarId is checkers.ANONYMOUS:
                mind.userId = random.choice(WORDS).strip()

                def onSessionExpire():
                    self.userLeft(mind)

                return inevow.IResource, Chatola([self, mind]), onSessionExpire

        raise NotImplementedError("Can't support that interface.")

    def usernames(self):
        """Return a list of all the users participating in the chat.
        """
        return [user.userId for user in self.clients]

    def broadcast(self, user, message, focusInput=True):
        """Broadcast a message to every client which is connected to this realm.
        """
        for mind in self.clients:
            mind.append('content',
                self.messagePattern.fillSlots('userid', user.userId).fillSlots('message', message))
            if mind is not user:
                mind.sendScript('scrollDown();')

        if focusInput:
            user.sendScript("focusInput();")
        else:
            user.sendScript('scrollDown();')

    def changeTopic(self, client, newTopic):
        self.topic = newTopic
        self.broadcast(client, "changed the topic to %s." % (newTopic, ), False)
        for mind in self.clients:
            mind.set('topic', newTopic)
            if mind is not client:
                mind.sendScript("document.getElementById('change-topic').value = '%s';" % mind.flt(newTopic))

    def _appendUserDom(self, client):
        user = client.userId
        for c in self.clients:
            c.append('userlist', self.userPattern(id=['user-list.', user]).fillSlots('user-list-entry', user))

    def userJoined(self, client):
        if client in self.clients:
            self.userLeft(client)
        user = client.userId
        self.broadcast(client, "%s has joined." % user)
        self._appendUserDom(client)
        self.clients.append(client)

    def userLeft(self, client):
        # Either onunload or session expiry (due to timeout or logout) will call this
        self.clients.remove(client)
        self.broadcast(client, "%s left." % (client.userId, ))
        for c in self.clients:
            c.call('removeNode', 'user-list.%s' % client.userId)

    def userChangedNick(self, client, newval):
        self.broadcast(client, "is now known as %s." % (newval, ), False)
        user = client.userId
        for c in self.clients:
            c.call('removeNode', 'user-list.%s' % user)
        client.userId = newval
        self._appendUserDom(client)


class Chatola(rend.Page):
    docFactory = loaders.xmlfile('Chatola.html')

    def render_body(self, ctx, data):
        """Do things we need to do upon initiating the rendering of this page.
        Log in to the realm, set up an onunload handler which will expire our
        session, and fill some global slots.
        """
        realm, mind = data
        realm.userJoined(mind)

        ctx.fillSlots('topic', realm.topic)
        ctx.fillSlots('username', mind.userId)

        session = inevow.ISession(ctx)
        def unload(client):
            session.expire()
        return ctx.tag(onunload=liveevil.handler(unload))

    def render_userlist(self, context, data):
        """Render the userlist by cloning the "user" pattern repeatedly
        and filling it with a user's username.
        """
        realm = data[REALM]
        userPattern = context.tag.patternGenerator('user')
        def generate():
            for user in realm.usernames():
                yield userPattern(id=["user-list.", user]).fillSlots('user-list-entry', user)
        return context.tag[ generate() ]

    def render_input(self, context, data):
        """Render an input form; a text box and a submit button. On the form, we add an onsubmit
        event handler; we pass a python callable wrapped by the liveevil.handler function which will
        be called on the server when the client side onsubmit handler is called. liveevil.handler
        automatically returns false in  the browser handler so the normal page submit does not
        occur. Instead, the second argument to handler, a string, is evaluated as javascript in the
        browser context and the result is sent to the server-side onSubmit method.
        """
        realm = data[REALM]
        def onSubmit(client, text):
            """When the onSubmit method is called on the server in response to the client-side
            onsubmit javascript handler, pass the text which was in the user's input box to the
            realm's broadcast method. Broadcast will format the message with the user's nick
            and send javascript to every connected client (including this user) which will append
            this message to the "content" div so all users can see the new chat text.
            """
            realm.broadcast(client, text)

        return context.tag(onsubmit=liveevil.handler(onSubmit, "getValue('inputline')"))

    def render_nick(self, context, data):
        return context.tag(onsubmit=liveevil.handler(data[REALM].userChangedNick, "getValue('nick')"))

    def render_changeTopic(self, context, data):
        return context.tag(onsubmit=liveevil.handler(data[REALM].changeTopic, "getValue('change-topic')"))

    def render_glue(self, context, data):
        """Insert the liveevil glue necessary for the input and output conduits to be present,
        and thus serverToClient and clientToServer events to be passed.
        """
        return liveevil.glue

