/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_trace_h
#define __nd_trace_h

#include <netdude/nd.h>
#include <pcap.h>

#include <gtk/gtk.h>

#include <netdude/nd_dialog.h>
#include <netdude/nd_protocol_inst.h>


struct nd_trace
{
  char        *filename;
  char        *unnamed;
  pcap_t      *pcap;

  GHashTable  *registry;            /* Generic hashtable to associate things */

  ND_Packet   *pl;                  /* Packet list */
  ND_Packet   *pl_end;              /* Last in packet list */

  struct {
    ND_Packet   *sel;               /* Selected packets */
    int          size;

    ND_Packet   *last;
    int          last_index;
    gboolean     last_valid;

    int          handler;           /* tag of Gtk idle handler for setting GUI to last packet */
  } sel;

  GtkWidget   *list;                /* This trace's tcpdump list */

  /* This trace's notebook tab.

     It contains the notebook with the protocols that display a
     packet's content under the key "notebook", the clist widget
     is available via "trace_list".

     The notebook contains each protocol's tab via the
     protocol's name as key. The trace is available from the
     notebook via "trace".
  */
  GtkWidget    *tab;
  GtkWidget    *tab_label;          /* This trace's notebook tab label */

  ND_Packet    *cur_packet;         /* Current packet */
  guchar       *cur_header;

  ND_ProtoInst *cur_pi_sel;         /* The protocol the user has selected recently. */
  ND_ProtoInst *cur_pi;

  int           num_packets;

  /* Stuff needed for the tcpdump connection: */
  struct {
    struct pcap_file_header pfh; /* The file's header */
    int                     fd;  /* Socket to pump data into and read line from */
    int                     pid; /* Pid of the tcpdump process for this trace */
  } tcpdump;

  int          dirty          : 1;
  int          apply_to_all   : 1;
  int          incomplete     : 1;
};


/**
 * nd_trace_new - creates a new trace.
 * @filename: canonical name of the trace file in the filesystem
 *
 * This is the constructor function for new traces. The @filename
 * is the full name of a trace file. You can pass %NULL if you don't
 * want/can provide packet data yet (as in live capture, for example).
 * If a filename is given, the file is checked for existence
 * and readability, if these checks fail, %NULL is returned. Otherwise
 * the trace is read into memory and initialized. The trace is NOT
 * automatically added to the trace registry, use nd_trace_registry_add()
 * for that purpose.
 *
 * Returns: new trace.
 */
ND_Trace      *nd_trace_new(const char *filename);

/**
 * nd_trace_close - closes a trace.
 * @trace: trace to close.
 *
 * Safe version of nd_trace_free() - if the trace was modified,
 * a dialog is opened that asks for confirmation. Only then the
 * trace is actually removed from the registry and freed.
 */
void           nd_trace_close(ND_Trace *trace);

/**
 * nd_trace_free - cleans up trace.
 * @trace: trace to free.
 *
 * This function closes the tcpdump connection for this trace
 * and releases the memory it occupies, together with all its
 * packets.
 */
void           nd_trace_free(ND_Trace *trace);

/**
 * nd_trace_init_header - initializes the pcap_file_header structure.
 * @trace: trace to initialize.
 * @snap_len: maximum capture length for the packets in the trace.
 * @datalink_type: the DLT of this trace.
 *
 * You can initialize a trace that doesn't yet contain any packets
 * using this function. The @datalink_type is a value as returned
 * by pcap_datalink(). Also see .../include/net/bpf.h. You can
 * call this function only once -- subsequent calls will do
 * nothing.
 */
void           nd_trace_init_header(ND_Trace *trace, guint32 snap_len, guint32 datalink_type);

/**
 * nd_trace_initialized - predicate for trace initialization state
 * @trace: trace to check
 *
 * Returns: %TRUE when the trace has been initialized,
 * %FALSE otherwise.
 */
gboolean       nd_trace_initialized(const ND_Trace *trace);

/**
 * nd_trace_init_packets - initializes all packets in the trace.
 * @trace: trace with packets to initialize
 *
 * The function iterates over packets in the trace and initializes
 * them using nd_packet_init(). GUI feedback is provided through
 * the progressbar. The function is called from nd_trace_new(),
 * so no extra call is necessary.
 */
void           nd_trace_init_packets(ND_Trace *trace);

/**
 * nd_trace_add_proto_tab - adds a protocol tab to a trace
 * @trace: trace to add tab to.
 * @proto: protocol for which to add a trace.
 * @nesting: nesting level of protocol.
 *
 * The function creates the necessary widgets to build a new tab
 * that gets added to the trace's protocols notebook. If there already
 * is a tab for the given protocol/nesting combo, the function does
 * nothing and returns.
 */
void           nd_trace_add_proto_tab(ND_Trace *trace,
				      ND_Protocol *proto,
				      guint nesting);

/**
 * nd_trace_get_name - returns tracefile name if available, or dummy name.
 * @trace: trace to query
 *
 * If the trace contains packets from a tracefile, its filename is
 * returned. Otherwise, a dummy name is generated and returned.
 * In both cases, pointers to local data are returned and should
 * be strdupped if you want to keep them around.
 *
 * Returns: trace name.
 */
const char    *nd_trace_get_name(const ND_Trace *trace);

/**
 * nd_trace_set_current_proto_selection - sets the preferred protocol of this trace.
 * @trace: trace to set selection in.
 * @pi: new protocol instance.
 *
 * When browsing through packets in a trace, there is the problem of
 * choosing a protocol to be selected initially (just like you want
 * your cursor to stay in the same column as much as possible in 
 * a text editor). Use this function to set that protocol. The protocol
 * instance data is copied internally, so you don't need to allocate
 * @pi.
 */
void           nd_trace_set_current_proto_selection(ND_Trace *trace,
						    const ND_ProtoInst *pi);


/**
 * nd_trace_set_current_proto - sets current protocol of a trace.
 * @trace: trace to set selection in.
 * @pi: new protocol instance selection.
 * @gui_update: whether to update the GUI.
 *
 * This function sets the current protocol in a trace, updating
 * the GUI. It does not influenced the preferred protocol selction
 * which is set using nd_trace_current_protocol_selection().
 * Sometimes it makes sense to not update the GUI immediately,
 * e.g. when you know that another call to this function will follow
 * soon. In this case, set @gui_update to %FALSE. The protocol
 * instance data is copied internally, so you don't need to allocate
 * @pi.
 */
void           nd_trace_set_current_proto(ND_Trace *trace,
					  const ND_ProtoInst *pi,
					  gboolean gui_update);

/**
 * nd_trace_get_proto_info - returns protocol info structure
 * @trace: trace to query.
 * @proto: protocol type to look up.
 * @nesting: which protocol instance to look up.
 *
 * In a trace's protocol notebook, each tab has an associated
 * #ND_ProtoInfo structure. This function is used to look up these
 * structures.
 *
 * Returns: protocol info if found, %NULL otherwise.
 */
ND_ProtoInfo  *nd_trace_get_proto_info(const ND_Trace *trace,
				       const ND_Protocol *proto,
				       guint nesting);


/**
 * nd_trace_get_current_packet - returns currently selected packet.
 * @trace: trace to look up packet in.
 *
 * Each trace has a currently selected packet, which may be %NULL.
 * This function returns it.
 *
 * Returns: current packet or NULL if n/a.
 */
ND_Packet     *nd_trace_get_current_packet(const ND_Trace *trace);


/**
 * nd_trace_get_current_proto_inst - returns currently selected protocol instance.
 * @trace: trace to return current protocol from.
 *
 * This function returns the currently selected protocol of
 * the given trace.
 *
 * Returns: current protocol instance or %NULL if n/a.
 */
ND_ProtoInst  *nd_trace_get_current_proto_inst(const ND_Trace *trace);


/**
 * nd_trace_get_current_proto_header - returns header of the current protocol in current packet.
 * @trace: trace to return header from.
 *
 * This is a convenience function that returns a pointer to the
 * header of the currently selected protocol in the current packet
 * of the given trace. You can then cast this pointer to the
 * appropriate structure. If this function is not applicable,
 * %NULL is returned.
 *
 * Returns: pointer to current protocol in current packet.
 */
guchar        *nd_trace_get_current_proto_header(const ND_Trace *trace);

/**
 * nd_trace_set_current_packet - sets currently selected packet in trace.
 * @trace: trace to set packet in.
 * @packet: packet to set as current one.
 *
 * This function sets the given packet as the current one in
 * the trace. Normally this could be done with the packet alone
 * (using packet->trace), but this function also allows you to
 * pass %NULL as @packet to unset the current packet. In both
 * cases the GUI is updated accordingly.
 */
void           nd_trace_set_current_packet(ND_Trace *trace,
					   ND_Packet *packet);

/**
 * nd_trace_load_dialog - opens dialog for loading a trace.
 */
void           nd_trace_load_dialog(void);

/**
 * nd_trace_save_as_dialog - opens a dialog for saving a trace.
 * @trace: trace to save.
 * @callback_finished: callback to call after file is saved.
 * @user_data: arbitrary data to pass to @callback_finished.
 *
 * The function opens a dialog to save @trace, if the trace got
 * saved after the user closes the dialog, the callback is
 * called with the given @user_data. This function gets called
 * by nd_trace_save() when the trace doesn't yet have a name.
 */
void           nd_trace_save_as_dialog(ND_Trace *trace,
				       ND_DialogCallback callback_finished,
				       void *user_data);

/* nd_trace_save - saves given trace, using dialog if necessary.
 * @trace: trace to save.
 *
 * The trace is saved to disk. If it doesn't have a name yet,
 * nd_trace_save_as_dialog is called. If saving was successful,
 * %TRUE is returned, %FALSE otherwise.
 *
 * Returns: success status.
 */
gboolean       nd_trace_save(ND_Trace *trace);


/* nd_trace_save_as - saves trace under new name and updates GUI accordingly.
 * @trace: trace to save.
 *
 * The name of the trace is changed and the trace saved subsequentially.
 * If saving was successful, %TRUE is returned, %FALSE otherwise.
 *
 * Returns: success status.
 */
gboolean       nd_trace_save_as(ND_Trace *trace, const char *filename);

/**
 * nd_trace_move_packet - moves packet around in a trace.
 * @trace: trace to operate on.
 * @from_packet: index of packet to move.
 * @to_packet: index to move packet to.
 *
 * This function moves a single packet around in the trace.
 * The GUI is NOT updated, this function is intended as a callback
 * after the GUI has changed. If you want both, use gtk_clist_row_move()
 * on the desired packets, which will cause nd_trace_move_packet()
 * to be called.
 */
void           nd_trace_move_packet(ND_Trace *trace, int from_packet, int to_packet);

/**
 * nd_trace_packte_insert_at_index - inserts a packet into the trace.
 * @trace: trace to insert into.
 * @packet: packet to insert.
 * @index: where to insert.
 *
 * This function inserts a packet into the trace, at the given index.
 * If the packet is already part of the trace, nothing happens.
 */
void           nd_trace_packet_insert_at_index(ND_Trace *trace, ND_Packet *packet, int index);

/**
 * nd_trace_packet_get_nth - returns packet in trace at given index.
 * @trace: trace to return packet from.
 * @n: index of packet to return.
 *
 * This function returns the nth packet of the trace, if possible,
 * %NULL otherwise.
 *
 * Returns: packet.
 */
ND_Packet     *nd_trace_packet_get_nth(const ND_Trace *trace, int n);

/**
 * nd_trace_sel_get - returns first selected packet.
 * @trace: trace to return from.
 * @selected_only: if %TRUE, only look at actually selected packets.
 *
 * This function returns the first selected packet in this trace,
 * for subsequent use of nd_trace_sel_next(). If @selected_only
 * is %FALSE and apply-to-all
 * mode is active, the first packet of the trace is returned. If
 * apply-to-all is inactive or @selected_only is %TRUE, only
 * the actually selected packets are looked at.
 * If there is no acitve selection, %NULL is returned.
 *
 * Returns: start of selection.
 */ 
ND_Packet     *nd_trace_sel_get(ND_Trace *trace, gboolean selected_only);

/**
 * nd_trace_sel_next - returns next selected packet after the given one.
 * @trace: trace to operate on.
 * @packet: packet to search onward from.
 *
 * This function returns the next packet that is selected. If apply-to-all
 * mode is active, this is simply the next packet in the list,
 * otherwise the next packet in the selection list.
 *
 * Returns: next selected packet.
 */
ND_Packet     *nd_trace_sel_next(ND_Trace *trace, ND_Packet *packet);


/**
 * nd_trace_sel_delete - deletes currently selected packets.
 * @trace: trace to modify.
 *
 * The function deletes the selected packets from the trace,
 * including GUI updates.
 */
void           nd_trace_sel_delete(ND_Trace *trace);

/**
 * nd_trace_sel_hide - hides currently selected packets.
 * @trace: trace in which to hide packets.
 *
 * The currently selected packets are hidden, which basically means
 * temporarily removed from the trace.
 */
void           nd_trace_sel_hide(ND_Trace *trace);

/**
 * nd_trace_sel_show - show currently selected packets.
 * @trace: trace in which to show hidden packets.
 *
 * The function shows hidden packets in the current selection.
 */
void           nd_trace_sel_show(ND_Trace *trace);

/**
 * nd_trace_sel_size - returns number of selected packets.
 * @trace: trace to query.
 *
 * The function returns the number of packets currently
 * selected in @trace.
 *
 * Returns: number of packets.
 */
guint          nd_trace_sel_size(ND_Trace *trace);


/**
 * nd_trace_size - returns number of packets.
 * @trace: trace to query.
 *
 * The function returns the number of packets in @trace.
 *
 * Returns: number of packets.
 */
guint          nd_trace_size(ND_Trace *trace);

/**
 * nd_trace_select_packet - selects a packet in the trace
 * @trace: trace to select packet in.
 * @index: index of packet to select.
 *
 * The function selects the indicated packet and marks it as the
 * currently selected one. This function is a callback that gets
 * called when a user selects a packet in the packet list, so when
 * you want full service, use gtk_clist_select_row() instead, which
 * will cause this function to be called.
 */
void           nd_trace_select_packet(ND_Trace *trace, int index);

/**
 * nd_trace_unselect_packet - unselects a packet in the trace.
 * @trace: trace to unselect packet in.
 * @index: index of packet to unselect.
 *
 * The function unselects the indicated packet in the trace.
 * This function is a callback that gets
 * called when a user selects a packet in the packet list, so when
 * you want full service, use gtk_clist_unselect_row() instead, which
 * will cause this function to be called.
 */
void           nd_trace_unselect_packet(ND_Trace *trace, int index);

/**
 * nd_trace_clear_selection - unselects all packets in the trace.
 * @trace: trace in which to clear selection.
 *
 * This function clears the selection in the trace. It only changes
 * the internal state of the trace, and doesn't update the GUI
 * in any way. This function is a callback for the unselect-all
 * signal in the trace's GUI packet list.
 */
void           nd_trace_clear_selection(ND_Trace *trace);

/**
 * nd_trace_full_selection - selects all packets in the trace.
 * @trace: trace in which to select everything.
 *
 * This function selects everything in the trace. It only changes
 * the internal state of the trace, and doesn't update the GUI
 * in any way. This function is a callback for the select-all
 * signal in the trace's GUI packet list.
 */
void           nd_trace_full_selection(ND_Trace *trace);

/**
 * nd_trace_set_dirty - updates dirty state of trace.
 * @trace: trace to update.
 * @dirty: whether we're dirty or not.
 *
 * This function sets the dirty state of @trace and updates the
 * GUI accordingly. If the trace is marked dirty, you cannot close
 * it without confirmation.
 */
void           nd_trace_set_dirty(ND_Trace *trace, gboolean dirty);

/**
 * nd_trace_get_notebook - returns protocol notebook.
 * @trace: trace to query.
 *
 * This function provides access to the GTK notebook that contains
 * the protocols for this trace.
 *
 * Returns: the protocols notebook.
 */
GtkNotebook   *nd_trace_get_notebook(const ND_Trace *trace);

/**
 * nd_trace_goto_top - scroll to top of trace.
 * @trace: trace to scroll.
 *
 * This function scrolls @trace to the top.
 */
void           nd_trace_goto_top(const ND_Trace *trace);

/**
 * nd_trace_goto_bottom - scroll to bottom of trace.
 * @trace: trace to scroll.
 *
 * This function scrolls @trace to the bottom.
 */
void           nd_trace_goto_bottom(const ND_Trace *trace);

/**
 * nd_trace_set_data - registers arbitrary data in the trace.
 * @trace: trace to store data in.
 * @key: key to use for looking up the data.
 * @data: data to register
 *
 * #ND_Trace structures provide a data storage facility that allows
 * arbitrary data to be hooked into traces. Use this function to
 * add data. The @key string is copied internally, so you don't
 * need to allocate it.
 */
void         nd_trace_set_data(ND_Trace *trace, const char *key, void *data);

/**
 * nd_trace_get_data - query data from the trace.
 * @trace: trace to look up data from.
 * @key: key to use for looking up the data.
 *
 * This function looks up registered data in the trace.
 *
 * Returns: queried data, or %NULL if no data was registered with @key.
 */
void       * nd_trace_get_data(const ND_Trace *trace, const char *key);

/**
 * nd_trace_remove_data - removes data from the trace.
 * @trace: trace to remove data from.
 * @key: key for data.
 *
 * This function removes data stored under the given @key from the trace
 * and returns it.
 *
 * Returns: the removed data.
 */
void       * nd_trace_remove_data(ND_Trace *trace, const char *key);

#endif
