
/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <dirent.h>
#include <glib.h>

#include <netdude/nd.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_prefs.h>
#include <netdude/nd_protocol.h>
#include <netdude/nd_protocol_registry.h>
#include <netdude/nd_protocol_plugin.h>
#include <callbacks.h>
#include <interface.h>
#include <support.h>

static GList *proto_plugins = NULL;  /* list<ND_ProtoPlugin*> */

static ND_ProtoPlugin *proto_plugin_new(char *filename);
static void       proto_plugin_scan(const char *dirname);
static void       proto_plugin_hook_up(ND_ProtoPlugin *p);
static void       proto_plugin_load(void);
static void       proto_plugin_setup_gui(void);
static int        proto_plugin_cmp(ND_ProtoPlugin *p1, char *filename);

static const char      *
proto_plugin_dummy_name(void)
{
  return _("Unnamed Plugin.");
}


static const char      *
proto_plugin_dummy_description(void)
{
  return _("No description available.");
}


static const char      *
proto_plugin_dummy_author(void)
{
  return _("No author(s) specified.");
}

static const char      *
proto_plugin_dummy_version(void)
{
  return _("No version specified.");
}


static int
proto_plugin_dummy_op(void)
{
  return 0;
}

static ND_Protocol *
proto_plugin_dummy_init(void)
{
  return NULL;
}


static ND_ProtoPlugin *
proto_plugin_new(char *filename)
{
  ND_ProtoPlugin *plugin;

  plugin = (ND_ProtoPlugin*) g_new0(ND_ProtoPlugin, 1);
  D_ASSERT_PTR(plugin);
  if (!plugin)
    return NULL;


  if (filename)
    plugin->filename = g_strdup(filename);

  plugin->name        = proto_plugin_dummy_name;
  plugin->description = proto_plugin_dummy_description;
  plugin->author      = proto_plugin_dummy_author;
  plugin->version     = proto_plugin_dummy_version;
  plugin->init        = proto_plugin_dummy_init;
  plugin->cleanup     = proto_plugin_dummy_op;

  return (plugin);
}


static int    
proto_plugin_cmp(ND_ProtoPlugin *p1, char *filename)
{
  if (!p1 || !filename)
    return (-1);

  return (strcmp(p1->filename, filename));
}


static void
proto_plugin_scan(const char *dir)
{
  DIR             *dirp;
  struct dirent   *de;
  char            *dot;

  D(("Scanning '%s' for protocols\n", dir));

  if ((dirp = opendir(dir)) != NULL)
    {
      for (de = readdir(dirp); de; de = readdir(dirp))
	{
	  if (de->d_name[0] != '.')
	    {
	      dot = strrchr(de->d_name, '.');

	      if (dot)
		*dot = '\0';

	      if (!g_list_find_custom(proto_plugins, de->d_name,
				      (GCompareFunc)proto_plugin_cmp))
		{
		  proto_plugins = g_list_append(proto_plugins,
						proto_plugin_new(de->d_name));
		}
	    }
	}      
      closedir(dirp);
    }
}


static void       
proto_plugin_hook_up(ND_ProtoPlugin *plugin)
{
  ND_Protocol *proto;
  lt_ptr_t     sym;
  int          i;

  if ((sym = lt_dlsym(plugin->lt, "name")) != NULL)
    plugin->name = sym;
  if ((sym = lt_dlsym(plugin->lt, "description")) != NULL)
    plugin->description = sym;
  if ((sym = lt_dlsym(plugin->lt, "author")) != NULL)
    plugin->author = sym;
  if ((sym = lt_dlsym(plugin->lt, "version")) != NULL)
    plugin->version = sym;

  if ((sym = lt_dlsym(plugin->lt, "init")) != NULL)
    plugin->init = sym;
  if ((sym = lt_dlsym(plugin->lt, "cleanup")) != NULL)
    plugin->cleanup = sym;

  /* Call the init routine the plugin provides: */
  proto = plugin->init();
  D_ASSERT_PTR(proto);

  /* If the plugin provides a header fields table,
     calculate the header size */

  for (i = 0, proto->header_size = 0; proto->fields[i].label; i++)
    proto->header_size += proto->fields[i].bits;
  
  /* register the plugin: */
  proto->plugin = plugin;
  nd_proto_registry_register(proto);
}


static gint
proto_plugin_strcmp(gconstpointer a, gconstpointer b)
{
  ND_ProtoPlugin *p1 = (ND_ProtoPlugin *) a;
  ND_ProtoPlugin *p2 = (ND_ProtoPlugin *) b;

  return strcmp(p1->name(), p2->name());
}

static void       
proto_plugin_load(void)
{
  GList            *l;
  ND_ProtoPlugin   *plugin;
  char              s[MAXPATHLEN];

  for (l = g_list_first(proto_plugins); l; l = g_list_next(l))
    {
      plugin = (ND_ProtoPlugin *) l->data;

      g_snprintf(s, MAXPATHLEN, "%s/%s", nd_prefs_get_proto_dir_user(), plugin->filename);

      if ( (plugin->lt = lt_dlopenext(s)))
	{
	  proto_plugin_hook_up(plugin);
	  continue;
	}

      g_snprintf(s, MAXPATHLEN, "%s/%s", nd_prefs_get_proto_dir_global(), plugin->filename);

      if ( (plugin->lt = lt_dlopenext(s)))
	{
	  proto_plugin_hook_up(plugin);
	  continue;
	}
      
      g_snprintf(s, MAXPATHLEN, 
		 _("Unable to load protocol plugin %s.\n"
		   "Please contact the module author."),
		 plugin->filename);
      nd_dialog_message(_("Plugin error"), s, FALSE);
    }

  /* Now sort the list of plugins alphanumerically,
     so that we automatically get sorted menus etc.
  */
  proto_plugins = g_list_sort(proto_plugins, proto_plugin_strcmp);
}


static void       
proto_plugin_setup_gui(void)
{
  GtkWidget        *about_protocols, *menu, *item;
  GList            *l;
  ND_ProtoPlugin   *plugin;

  /* Hook in protocol menus: */

  about_protocols = gtk_object_get_data(GTK_OBJECT(toplevel), "about_protocols");
  D_ASSERT_PTR(about_protocols);

  menu = gtk_menu_new();
  gtk_widget_ref (menu);
  gtk_object_set_data_full (GTK_OBJECT (toplevel), "about_protocols_menu", menu,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (about_protocols), menu);

  for (l = g_list_first(proto_plugins); l; l = g_list_next(l))
    {
      plugin = (ND_ProtoPlugin*)l->data;

      item = gtk_menu_item_new_with_label(plugin->name());
      gtk_widget_ref(item);
      gtk_object_set_data_full (GTK_OBJECT (menu), plugin->name(), item,
				(GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show(item);
      gtk_object_set_data(GTK_OBJECT(item), "plugin", plugin);
      gtk_container_add(GTK_CONTAINER(menu), item);
      gtk_signal_connect (GTK_OBJECT (item), "activate",
			  GTK_SIGNAL_FUNC (on_about_proto_activate),
			  NULL);
    }
}


void 
nd_proto_plugin_init(void)
{
  if (lt_dlinit() != 0)
    {
      fprintf(stderr, "Could not initialize libltdl -- exiting.\n");
      exit(-1);
    }

  proto_plugin_scan(nd_prefs_get_proto_dir_user());
  proto_plugin_scan(nd_prefs_get_proto_dir_global());
  proto_plugin_load();
  proto_plugin_setup_gui();
}


const char   *
nd_proto_plugin_get_description(ND_ProtoPlugin *plugin)
{
  if (!plugin)
    return (NULL);

  return (plugin->description());
}


const char   *
nd_proto_plugin_get_author(ND_ProtoPlugin *plugin)
{
  if (!plugin)
    return (NULL);

  return (plugin->author());
}


void    
nd_proto_plugin_show_about(ND_ProtoPlugin *plugin)
{
  char       s[MAXPATHLEN];
  GtkWidget *w;
  GtkWidget *label;

  D_ASSERT_PTR(plugin);
  if (!plugin)
    return;

  w = create_protocol_about_dialog();

  g_snprintf(s, MAXPATHLEN, "%s, %s.", plugin->name(), plugin->version());
  label = gtk_object_get_data(GTK_OBJECT(w), "proto_name_label");
  D_ASSERT_PTR(label);
  gtk_label_set_text(GTK_LABEL(label), s);
  
  g_snprintf(s, MAXPATHLEN, _("Author: %s"), plugin->author());
  label = gtk_object_get_data(GTK_OBJECT(w), "proto_author_label");
  D_ASSERT_PTR(label);
  gtk_label_set_text(GTK_LABEL(label), s);

  label = gtk_object_get_data(GTK_OBJECT(w), "proto_description_label");
  D_ASSERT_PTR(label);
  gtk_label_set_text(GTK_LABEL(label), plugin->description());
  
  gtk_widget_show(w);  
}
