/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_prefs_h
#define __nd_prefs_h

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>

#define ND_DOM_NETDUDE   "Netdude"

typedef enum
{
  ND_PREFS_INT = 1,
  ND_PREFS_FLT = 2,
  ND_PREFS_STR = 3,
  ND_PREFS_UNK = 4
} ND_PrefsType;


typedef struct nd_prefs_entry
{
  const char             *key;
  ND_PrefsType            type;

  int                     int_val;
  float                   flt_val;
  const char             *str_val;

} ND_PrefsEntry;


/**
 * ND_PrefsApplyCallback - signature of preferences-apply functions.
 * @container: container widget from which other widgets can be retrieved.
 * @entries: array of #ND_PrefsEntry structures specifying preferences.
 * @num_entries: length of @entries.
 *
 * This is the signature of functions passed to nd_prefs_add_domain()
 * that get called when the user applies changes in preference settings.
 */
typedef void (*ND_PrefsApplyCallback) (GtkWidget *container,
				       ND_PrefsEntry *entries,
				       int num_entries);

/* You can use this to make statically initialized tables
 * more readable, see nd_prefs.c.
 */
#define ND_UNUSED 0

/**
 * nd_prefs_init - initializes preferences management.
 */
void           nd_prefs_init(void);

/**
 * nd_prefs_load - loads preferences from disk.
 *
 * This initializes the preferences as they're found in the 
 * user's config file, which is created if it doesn't yet
 * exist. Called during Netdude startup. You don't need this.
 */
void           nd_prefs_load(void);


/**
 * nd_prefs_get_netdude_dir - returns netdude config directory.
 *
 * The function returns a pointer to static memory containing
 * the directory to the user's Netdude directory, where
 * preferences and locally-installed plugins are stored.
 *
 * Returns: user's Netdude directory.
 */
const char    *nd_prefs_get_netdude_dir(void);

/**
 * nd_prefs_get_config_file - returns filename of netdude config file.
 *
 * The function returns a pointer to static memory containing
 * the name of the user's Netdude config file.
 *
 * Returns: config file name.
 */
const char    *nd_prefs_get_config_file(void);

/**
 * nd_prefs_get_plugin_dir_global - returns global plugin directory.
 *
 * The function returns a pointer to static memory containing
 * the name of the system-wide feature plugin directory.
 *
 * Returns: global plugin directory.
 */
const char    *nd_prefs_get_plugin_dir_global(void);

/**
 * nd_prefs_get_plugin_dir_user - returns user's plugin directory.
 *
 * The function returns a pointer to static memory containing
 * the name of the user's feature plugin directory.
 *
 * Returns: user's plugin directory.
 */
const char    *nd_prefs_get_plugin_dir_user(void);

/**
 * nd_prefs_get_proto_dir_global - returns global protocol plugin directory.
 *
 * The function returns a pointer to static memory containing
 * the name of the system-wide protocol plugin directory.
 *
 * Returns: global protocol plugin directory.
 */
const char    *nd_prefs_get_proto_dir_global(void);

/**
 * nd_prefs_get_proto_dir_user - returns user's protocol plugin directory.
 *
 * The function returns a pointer to static memory containing
 * the name of the user's protocol plugin directory.
 *
 * Returns: user's plugin directory.
 */
const char    *nd_prefs_get_proto_dir_user(void);



/**
 * nd_prefs_add_domain - adds a preferences domain to the config system.
 * @domain: name of the domain.
 * @container: a widget from which the other widgets can be retrieved.
 * @gui: the GUI widget that contains the preferences widgets.
 * @entries: array of ND_PrefsEntry structures describing preference settings.
 * @num_entries: length of @entries.
 * @apply_cb: callback called when new preferences are applied by user.
 *
 * The function registers a new preferences domain. Each domain gets
 * displayed in its own tab in the preferences dialog and is stored
 * with its own prefix in the config file. @container is a widget from
 * which all other preference widgets can be retrieved via
 * gtk_object_get_data(), such as a toplevel window when using Glade.
 * @gui is the widget containing the actual preferences widgets, usually
 * something like a #GtkVBox. @entries is an array of statically initialized
 * #ND_PrefsEntry items. @apply_cb is called when the user applies new
 * preferences settings and is supposed to contain actions that need
 * to be carried out when preferences for this domain change. As an
 * example of all this, see the TCP plugin in nd_tcp_prefs.c.
 */
void           nd_prefs_add_domain(const char *domain,
				   GtkWidget *container,
				   GtkWidget *gui,
				   ND_PrefsEntry *entries,
				   int num_entries,
				   ND_PrefsApplyCallback apply_cb);

/**
 * nd_prefs_save - saves current preference settings to disk
 *
 * Returns: %TRUE on success, %FALSE otherwise.
 */
int            nd_prefs_save(void);


/**
 * nd_prefs_save_general -  saves generic preferences to disk.
 *
 * The function saves to disk general settings that are not
 * directly manipulated by the user through the preferences
 * dialog, such as load/save directories, recently used files
 * etc.
 */
void           nd_prefs_save_general(void);


/**
 * nd_prefs_set_str_item - set string preference item.
 * @domain: domain of item.
 * @key: name of the item.
 * @data: new value of @key.
 *
 * The function stores @data as the value of @key in the preferences
 * database.
 */
void           nd_prefs_set_str_item(const char *domain, const char *key, const char *data);

/**
 * nd_prefs_set_flt_item - set float preference item.
 * @domain: domain of item.
 * @key: name of the item.
 * @data: new value of @key.
 *
 * The function stores @data as the value of @key in the preferences
 * database.
 */
void           nd_prefs_set_flt_item(const char *domain, const char *key, float data);

/**
 * nd_prefs_set_int_item - set integer preference item.
 * @domain: domain of item.
 * @key: name of the item.
 * @data: new value of @key.
 *
 * The function stores @data as the value of @key in the preferences
 * database.
 */
void           nd_prefs_set_int_item(const char *domain, const char *key, int data);

/**
 * nd_prefs_del_item - deletes a preference item.
 * @domain: domain of item.
 * @key: name of the item.
 *
 * The function removes the preference item from the database.
 */
void           nd_prefs_del_item(const char *domain, const char *key);

/**
 * nd_prefs_get_str_item - retrieves string preference item.
 * @domain: domain of item.
 * @key: name of item.
 * @result: pointer to result.
 * 
 * The function retrieves the specified preference item and stores
 * it in @result.
 *
 * Returns: %TRUE on success, %FALSE otherwise.
 */
gboolean       nd_prefs_get_str_item(const char *domain, const char *key, char **result);

/**
 * nd_prefs_get_flt_item - retrieves float preference item.
 * @domain: domain of item.
 * @key: name of item.
 * @result: pointer to result.
 * 
 * The function retrieves the specified preference item and stores
 * it in @result.
 *
 * Returns: %TRUE on success, %FALSE otherwise.
 */
gboolean       nd_prefs_get_flt_item(const char *domain, const char *key, float *result);

/**
 * nd_prefs_get_int_item - retrieves integer preference item.
 * @domain: domain of item.
 * @key: name of item.
 * @result: pointer to result.
 * 
 * The function retrieves the specified preference item and stores
 * it in @result.
 *
 * Returns: %TRUE on success, %FALSE otherwise.
 */
gboolean       nd_prefs_get_int_item(const char *domain, const char *key, int   *result);


/**
 * nd_prefs_dialog_show - shows preferences dialog.
 */
void           nd_prefs_dialog_show(void);

/* GUI callbacks. */
void           nd_prefs_dialog_ok(void);
void           nd_prefs_dialog_apply(void);
void           nd_prefs_dialog_cancel(void);

void           nd_prefs_fontsel_show(void);

/**
 * nd_dialog_sync - syncs preferences dialog to config settings.
 *
 * The function adjusts the  widgets in the preferences dialog
 * to reflect the current config settings.
 */
void           nd_prefs_dialog_sync(void);

void           nd_prefs_select_tcpdump(void);

#endif
