/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_capture_h
#define __nd_capture_h

#include <gtk/gtk.h>

#include <netdude/nd.h>

/* Structure that contains capture information */
typedef struct nd_capture
{
  /* Name of interface to capture from */
  char        *ifname;

  /* Maximum capture length, 0 means whole packet */
  guint        snap_len;

  /* Filter command as understood by pcap */
  guchar      *filter;

  /* Whether to set the interface in prom. mode */
  gboolean     set_promiscuous;

  /* Whether the capture is finished */
  gboolean     finished;

  /* Dialog to update widgets in, may be null */
  GtkWidget   *dialog;


  /* The trace to which the packets are added */
  ND_Trace    *trace;

  /* Internal -- event monitoring and timeout handling */
  int          gdk_tag;
  int          gtk_timeout;

  /* Limit of packets to capture (passing any filters),
   * and number currently captured.
   */
  guint        packet_limit;
  guint        packet_cur;

  /* Limit of bytes to capture (passing any filters),
   * and number of bytes currently captured.
   */
  guint        size_limit;
  guint        size_cur;

  /* Time at the start of the capture in epoch seconds */
  time_t       time_start;
  /* Duration of capture time in seconds */
  time_t       time_limit;
  /* Current time in seconds */
  time_t       time_cur;

  guint        drop_limit;

} ND_Capture;


/**
 * nd_capture_new - creates a new, blank capture structure.
 *
 * The function allocates and initializes a #ND_Capture
 * structure.
 *
 * Returns: new ND_Capture structure, or NULL when out
 * of memory.
 */
ND_Capture *nd_capture_new(void);


/**
 * nd_capture_free - capture cleanup.
 * @capt: structure to clean up.
 *
 * This function cleans up capture structure -- it is assumed that
 * the interface name and filter strings can be freed,
 * so don't make them point to non-heap data.
 */
void nd_capture_free(ND_Capture *capt);


/**
 * nd_capture_dialog_show - creates and shows a capture dialog.
 *
 * The function creates and shows a packet capture dialog.
 *
 * Returns: new dialog widget.
 */
GtkWidget *nd_capture_dialog_show(void);


/**
 * nd_capture_dialog_close - close-dialog callback.
 * @dialog: dialog to close.
 *
 * The function closes and destroy the dialog,
 * terminating any active captures.
 */
void nd_capture_dialog_close(GtkWidget *dialog);


/**
 * nd_capture_dialog_start - start-capture callback.
 * @dialog: dialog which starts the capture.
 *
 * The function creates a capture structure according to current
 * settings and passes it to nd_capture_start().
 */
void nd_capture_dialog_start(GtkWidget *dialog);


/**
 * nd_capture_choose_filter_file - launches filter file selection dialog.
 * @dialog: capture dialog that needs a filter file.
 *
 * The function shows a file selection dialog and sets the entry
 * field in the capture dialog accordingly when a filter file is selected.
 */
void nd_capture_choose_filter_file(GtkWidget *dialog);


/**
 * nd_capture_start - performs a packet capture.
 * @capture: contains the parameters for the capture.
 *
 * The function starts a new capture accordings to the settings
 * given in @capture. A new trace is added to the GUI that
 * receives the new packets.
 */
void nd_capture_start(ND_Capture *capture);

/**
 * nd_capture_stop - stops packet capture.
 * @capture: capture that is stopped.
 *
 * The function stops the given capture. The structure is not
 * deallocated.
 */
void nd_capture_stop(ND_Capture *capture);


/**
 * nd_capture_possible - checks whether user can perform captures.
 *
 * The function checks whether the user running Netdude can actually
 * perform packet captures and returns the result.
 *
 * Returns: %TRUE if there are currently any up
 * interfaces for which the user can call pcap_open_live(),
 * %FALSE otherwise.
 */
gboolean nd_capture_possible(void);

#endif
