/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file  myx_grt_module.c
 * @brief GRT module related functions.
 * 
 * See also: <a href="../grt.html#Modules">Module</a>
 */


#include "myx_grt_private.h"
#ifdef ENABLE_JAVA_MODULES
#include "myx_grt_java.h"
#endif


/**
 ****************************************************************************
 * @brief Scans a directory for modules and inits them
 * 
 *   Will scan through all files in the directory looking for files with 
 * recognized extensions. Each module will be registered wrt it's functionality.
 *
 * @param grt       the GRT environment
 * @param directory directory where modules are located
 * @param error     pointer to error variable
 * 
 * @return Number of modules that were initialized. -1 on error.
 * @return error will contain the error.
 ****************************************************************************/
int myx_grt_scan_for_modules(MYX_GRT *grt, const char *directory,
                             MYX_GRT_ERROR *error)
{
  GDir *dir;
  const char *entry;
  int count= 0;

  *error= MYX_GRT_NO_ERROR;

  g_return_val_if_fail(grt != NULL, -1);

  dir= g_dir_open(directory, 0, NULL);
  if (!dir)
  {
    *error= MYX_GRT_BAD_PATH;
    return -1;
  }

  while ((entry= g_dir_read_name(dir)) != NULL)
  {
    char *path= g_build_filename(directory, entry, NULL);

    if (myx_grt_module_init(grt, path) == MYX_GRT_NO_ERROR)
    {
      count++;
    }

    g_free(path);
  }

  g_dir_close(dir);

  return count;
}

int myx_grt_scan_for_structs(MYX_GRT *grt, const char *directory, MYX_GRT_ERROR *error)
{
  GDir *dir;
  const char *entry;
  int count= 0;

  *error= MYX_GRT_NO_ERROR;

  g_return_val_if_fail(grt != NULL, -1);

  dir= g_dir_open(directory, 0, NULL);
  if (!dir)
  {
    *error= MYX_GRT_BAD_PATH;
    return -1;
  }

  while ((entry= g_dir_read_name(dir)) != NULL)
  {
    if ( (str_beginswith(entry, "structs.")) && (str_endswith(entry, ".xml")) )
    {
      char *path= g_build_filename(directory, entry, NULL);

      if (myx_grt_struct_load_and_register(grt, path) == MYX_GRT_NO_ERROR)
      {
        count++;
      }

      g_free(path);
    }
  }

  g_dir_close(dir);

  return count;
}


/**
 ****************************************************************************
 * @brief Initializes/registers a module in the file
 * 
 *  This will check the file extension to find out the type of module and
 * will then initialize it through the appropriate loader. The loader will
 * find out the list of protocols the module implements, including which function
 * interfaces in the protocol it has.
 * The created module will then be registered in the GRT.
 *
 * \b NOTE
 *  Modules must implement a function called getModuleInfo(), which 
 *  should return a dictionary/map, containing:
 *     name, a string with the name of the module
 *     implements, a string with the name of the protocol the module implements
 *     functions, a list of strings with the name of functions implemented
 * 
 *  Function names must match the names in the protocol definition.
 *
 * @param grt   the GRT environment
 * @param filename  filename of the module file to be loaded
 * 
 * @return MYX_GRT_NO_ERROR on success
 * @return MYX_GRT_UNKNOWN_MODULE_TYPE if there's no loader that supports it
 * @return MYX_GRT_MODULE_INIT_ERROR if there was an error initializing the module
 ****************************************************************************/
MYX_GRT_ERROR myx_grt_module_init(MYX_GRT *grt, const char *filename)
{
  MYX_GRT_MODULE *module= NULL;
  unsigned int i;
  MYX_GRT_ERROR error;
  
  for (i= 0; i < grt->loaders_num; i++)
  {
    unsigned int j;
    for (j= 0; j < grt->loaders[i]->extensions_num; j++)
    {
      if (g_str_has_suffix(filename, grt->loaders[i]->extensions[j]))
      {
        error= grt->loaders[i]->init_module(grt->loaders[i], filename, &module);
        if (error != MYX_GRT_NO_ERROR)
          return error;
        break;
      }
    }
  }

  if (!module)
    return MYX_GRT_UNKNOWN_MODULE_TYPE;
  else
    return myx_grt_add_module(grt, module);
}

/**
 ****************************************************************************
 * @brief Locates the function descriptor.
 *
 * @param grt           the GRT environment
 * @param module        module name where function is located
 * @param function_name the function name from the module
 * @param search_parent search for the function in the module named in extends
 *       if it's not implemented in the given module.
 *
 * @return The function structure or NULL if it doesn't exist. The value should not
 *   be freed or modified.
 ****************************************************************************/
MYX_GRT_FUNCTION *myx_grt_function_get(MYX_GRT *grt, const char *module, const char *function_name, int search_parent)
{
  unsigned int i;
  
  for (i= 0; i < grt->modules_num; i++)
  {
    unsigned int j;

    if (strcmp(grt->modules[i]->name, module) == 0)
    {
      /* search the function in this module */
      for (j= 0; j < grt->modules[i]->functions_num; j++)
      {
        if (strcmp(grt->modules[i]->functions[j].name, function_name) == 0)
        {
          return grt->modules[i]->functions+j;
        }
      }
      /* if not found, then search in parent */
      if (search_parent && grt->modules[i]->extends)
        return myx_grt_function_get(grt, grt->modules[i]->extends, function_name, search_parent);
    }
  }

  return NULL;
}

/**
 ****************************************************************************
 * @brief Calls a GRT function.
 * 
 *  This will execute the given function, passing the specified argument.
 * The argument will have it's struct type validated before the function
 * is called and its return type is also validated.
 *
 * @param grt the GRT environment
 * @param func the function to be executed
 * @param argument a MYX_GRT_VALUE containing the argument to the called function
 * @param retval where the return value of the function will be placed
 *
 * @return MYX_NO_ERROR for successful invocation. retval will contain the return value
 *      from the function.
 * @return MYX_GRT_BAD_FUNCTION the function or module is not known
 * @return MYX_GRT_VALIDATION_ERROR if there was a validation error of 
 *   parameter or return value
 * @return other errors
 ****************************************************************************/
MYX_GRT_VALUE * myx_grt_function_call(MYX_GRT *grt, MYX_GRT_FUNCTION *func,
                                    MYX_GRT_VALUE *argument, MYX_GRT_ERROR *error)
{
  MYX_GRT_VALUE *retval;

  *error= MYX_GRT_NO_ERROR;
  
  //Asserts
  if ((!grt) || (!func))
  {
    *error= MYX_GRT_INTERNAL_ERROR;
    return NULL;
  }
  //g_return_val_if_fail(grt!=NULL, MYX_GRT_INTERNAL_ERROR);
  //g_return_val_if_fail(func!=NULL, MYX_GRT_INTERNAL_ERROR);

  if (func->param_struct_name && argument)
  {
    if (!myx_grt_dict_struct_validate(grt, argument, func->param_struct_name, 0))
    {
      *error= MYX_GRT_VALIDATION_ERROR;
      return NULL;
    }
  }

  *error= (*func->module->loader->call_function)(func, argument, &retval);
  if (*error != MYX_GRT_NO_ERROR)
    return NULL;
  
  if (retval && func->return_struct_name)
  {
    if (!myx_grt_dict_struct_validate(grt, retval, func->return_struct_name, 0))
    {
      myx_grt_value_release(retval);
      *error= MYX_GRT_VALIDATION_ERROR;
      return NULL;
    }
  }

  return retval;
}

/**
 ****************************************************************************
 * @brief Looks up for the function and calls it.
 * 
 *  Executes the named function from the module, with the given argument.
 *
 * @param grt the GRT environment
 * @param module module name to search the function
 * @param function_name the function name to be executed
 * @param search_parent search for the function in the module named in extends
 *       if it's not implemented in the given module.
 * @param argument a MYX_GRT_VALUE containing the argument to the called function
 * @param retval where the return value of the function will be placed
 *
 * @return MYX_NO_ERROR for successful invocation. retval will contain the return value
 *      from the function.
 * @return MYX_GRT_BAD_FUNCTION if the module or function name given are invalid.
 ****************************************************************************/
MYX_GRT_VALUE * myx_grt_function_get_and_call(MYX_GRT *grt, const char *module, const char *function_name, int search_parent,
                                         MYX_GRT_VALUE *argument, MYX_GRT_ERROR *error)
{
  MYX_GRT_FUNCTION *func= myx_grt_function_get(grt, module, function_name, search_parent);
  if (!func) 
  {
    *error = MYX_GRT_BAD_FUNCTION;
    return NULL;
  }

  return myx_grt_function_call(grt, func, argument, error);
}

/**
 ****************************************************************************
 * @brief Creates a result for a function
 * 
 *  Creates a new MYX_DICT_VALUE that contains the result
 *
 * @param result the result value
 *
 * @return a MYX_DICT_VALUE that can be returned by a function
 ****************************************************************************/
MYX_GRT_VALUE * myx_grt_function_create_result(MYX_GRT_VALUE *result)
{
  MYX_GRT_VALUE *res= NULL;
  
  if (result)
  {
    res= myx_grt_dict_new("");

    myx_grt_dict_item_set_value(res, "value", result);
    myx_grt_value_release(result);
  }

  return res;
}

/**
 ****************************************************************************
 * @brief Extracts the value from a function result
 * 
 *  Returns the MYX_DICT_VALUE that contains the result and releases the result
 *
 * @param result the result value
 *
 * @return a MYX_DICT_VALUE that is the returned value
 ****************************************************************************/
MYX_GRT_VALUE * myx_grt_function_extract_value_from_result(MYX_GRT_VALUE *result)
{
  MYX_GRT_VALUE *value= myx_grt_dict_item_get_value(result, "value");

  myx_grt_value_retain(value);
  myx_grt_value_release(result);

  return value;
}

/**
 ****************************************************************************
 * @brief Creates an error result for a function
 * 
 *  Creates a new MYX_DICT_VALUE that contains the error information
 *
 * @param error the error text
 *
 * @param error_details more detailed information about the error
 *
 * @return a MYX_DICT_VALUE that can be returned by a function to indicate
 *   that an error has occured
 ****************************************************************************/
MYX_GRT_VALUE * myx_grt_function_create_error_result(const char *error, const char *error_details)
{
  MYX_GRT_VALUE *result= myx_grt_dict_new("");
  MYX_GRT_VALUE *val= myx_grt_value_from_string(error);

  myx_grt_dict_item_set_value(result, "error", val);
  myx_grt_value_release(val);

  if (error_details && error_details[0])
  {
    val= myx_grt_value_from_string(error_details);
    myx_grt_dict_item_set_value(result, "detail", val);
    myx_grt_value_release(val);
  }

  return result;
}

/**
 ****************************************************************************
 * @brief Returns the error string of a function call result
 * 
 *  Checks whether a function has returned an error as result.
 *
 * @param result the function result
 *
 * @return NULL if there was no error or a new allocated error string that
 *   has to be freed
 ****************************************************************************/
char * myx_grt_function_check_error(MYX_GRT_VALUE *res, int allow_null_as_result)
{
  MYX_GRT_VALUE *error_val;

  //Check if we have a value
  if ( (res == NULL) && (allow_null_as_result == 0) )
    return g_strdup("The function has not returned a object.");
  else if ( (res == NULL) && (allow_null_as_result == 1) )
    return NULL;

  //Check if we have a dict value
  if (myx_grt_value_get_type(res) != MYX_DICT_VALUE)
    return g_strdup_printf("The function returned a %s instead of a dictionary object.", 
      myx_get_value_type_as_string(myx_grt_value_get_type(res)));

  error_val= myx_grt_dict_item_get_value(res, "error");
  if (error_val)
  {
    return g_strdup_printf("%s" _br 
      "Details: " _br 
      "%s", 
      myx_grt_value_as_string(error_val),
      myx_grt_value_as_string(
        myx_grt_dict_item_get_value(res, "detail"))
      );
  }
  else
    return NULL;
}

/**
 ****************************************************************************
 * @brief Returns the module with the given name
 * 
 *  Searches for the module with the given name and returns it.
 *
 * @param grt the GRT environment
 * @param name name of the wanted module
 *
 * @return The module struct or NULL if the module is not found.
 ****************************************************************************/
MYX_GRT_MODULE *myx_grt_module_get(MYX_GRT *grt, const char *name)
{
  unsigned int i;
  for (i= 0; i < grt->modules_num; i++)
  {
    if (strcmp2(grt->modules[i]->name, name)==0)
      return grt->modules[i];
  }
  return NULL;
}

MYX_GRT_MODULE * myx_grt_module_get_by_index(MYX_GRT *grt, unsigned int index)
{
  if (index < grt->modules_num)
    return grt->modules[index];
  else
    return NULL;
}

MYX_GRT_MODULE_TYPE myx_grt_module_get_type(MYX_GRT_MODULE *module)
{
  return module->loader->loader_type;
}


/**
 ****************************************************************************
 * @brief Get list of modules that extend a given module
 * 
 *   Returns an array with all registered modules that extend the named
 * module.
 *
 * @param grt    the GRT environment
 * @param module name of the module. Can be NULL, in which case it will return 
 *        all registered modules.
 * @param retmodules pointer to array of modules where the result will be returned
 *
 * @returned The number of modules in the array. The module array must be freed.
 ****************************************************************************/
int myx_grt_modules_get_that_extend(MYX_GRT *grt, const char *module, MYX_GRT_MODULE **retmodules[])
{
  unsigned int i;
  int count= 0;

  g_return_val_if_fail(grt!=NULL, 0);
  g_return_val_if_fail(retmodules != NULL, 0);
  
  if (grt->modules_num == 0)
    return 0;

  *retmodules= g_new0(MYX_GRT_MODULE*,grt->modules_num);

  for (i= 0; i < grt->modules_num; i++)
  {
    if (!module || !*module || strcmp(grt->modules[i]->extends, module)==0)
    {
      count++;
      (*retmodules)[count-1]= grt->modules[i];
    }
  }

  if (count == 0)
  {
    g_free(*retmodules);
    *retmodules= NULL;
  }
  else
    *retmodules= g_realloc(*retmodules, sizeof(MYX_GRT_MODULE*)*count);

  return count;
}

int myx_grt_module_get_count(MYX_GRT *grt)
{
  return grt->modules_num;
}

MYX_STRINGLIST * myx_grt_module_get_names(MYX_GRT *grt)
{
  MYX_STRINGLIST *modules= g_new0(MYX_STRINGLIST, 1);
  unsigned int i;

  modules->strings_num= grt->modules_num;
  modules->strings= g_malloc0(sizeof(char *)*modules->strings_num);

  for (i= 0; i<modules->strings_num; i++)
  {
    MYX_GRT_MODULE *module= grt->modules[i];
    modules->strings[i]= g_strdup(module->name);
  }

  return modules;
}

int myx_grt_module_function_get_count(MYX_GRT_MODULE *module)
{
  return module->functions_num;
}

MYX_STRINGLIST * myx_grt_module_function_get_names(MYX_GRT_MODULE *module)
{
  MYX_STRINGLIST *funcs= g_new0(MYX_STRINGLIST, 1);
  unsigned int i;

  funcs->strings_num= module->functions_num;
  funcs->strings= g_malloc0(sizeof(char *)*funcs->strings_num);

  for (i= 0; i<funcs->strings_num; i++)
  {
    MYX_GRT_FUNCTION *func= module->functions+i;
    funcs->strings[i]= g_strdup(func->name);
  }

  return funcs;
}

MYX_GRT_FUNCTION * myx_grt_module_function_get(MYX_GRT_MODULE *module, const char *name)
{
  unsigned int i;
  for (i= 0; i < module->functions_num; i++)
  {
    MYX_GRT_FUNCTION *func= module->functions+i;

    if (strcmp2(func->name, name)==0)
      return func;
  }
  return NULL;
}

MYX_GRT_FUNCTION * myx_grt_module_function_get_by_index(MYX_GRT_MODULE *module, unsigned int index)
{
  if (index < module->functions_num)
    return module->functions+index;
  else
    return NULL;
}

char * myx_grt_module_function_get_params(MYX_GRT_FUNCTION *func)
{
    if (func->param_struct_name)
      return g_strdup(func->param_struct_name);
#ifdef ENABLE_JAVA_MODULES
    else if ((func->module->loader->loader_type==MYX_JAVA_MODULE_TYPE) && 
      (((MYX_JAVA_FUNCTION *)(func->priv))->java_signature))
    {
      char *params= g_strdup(((MYX_JAVA_FUNCTION *)(func->priv))->java_signature);
      params= str_g_replace(params, ";)", ")");
      params= str_g_replace(params, ";", ", ");
      params= str_g_replace(params, "Ljava/lang/String", "string");
      params= str_g_replace(params, "Ljava/lang/Integer", "int");
      params= str_g_replace(params, "Lcom/mysql/grt/GrtList", "list");
      params= str_g_replace(params, "Lcom/mysql/grt/", "");

      return params;
    }
#endif
    else
      return g_strdup("");
}

char * myx_grt_module_function_get_return_type(MYX_GRT_FUNCTION *func)
{
  if (func->return_struct_name)
    return g_strdup(func->return_struct_name);
  else
    return g_strdup("");
}


/* this is private
 ****************************************************************************
 * @brief Checks all modules for new messages to be added to the GRT msg queue.
 *
 * @param grt  
 *
 * @return 
 *****************************************************************************/
MYX_GRT_ERROR myx_grt_fetch_module_messages(MYX_GRT *grt)
{
  unsigned int i;
  MYX_GRT_ERROR error;

  for (i= 0; i < grt->loaders_num; i++)
  {
    MYX_GRT_MSGS msgs;

    msgs.msgs_num= 0;
    msgs.msgs= NULL;
    if (grt->loaders[i]->fetch_messages)
    {
      error= grt->loaders[i]->fetch_messages(grt->loaders[i], &msgs);

      if (msgs.msgs_num > 0)
      {
        if (!grt->msgs)
          grt->msgs= g_new0(MYX_GRT_MSGS, 1);

        grt->msgs->msgs_num+= msgs.msgs_num;
        grt->msgs->msgs= g_realloc(grt->msgs->msgs, grt->msgs->msgs_num*sizeof(MYX_GRT_MSG));
        
        memcpy(grt->msgs->msgs+(grt->msgs->msgs_num-msgs.msgs_num),
               msgs.msgs, msgs.msgs_num*sizeof(MYX_GRT_MSG));
      }
      g_free(msgs.msgs);

      if (error != MYX_GRT_NO_ERROR)
        return error;
    }
  }
  
  return MYX_GRT_NO_ERROR;
}
