/*
  MeCab -- Yet Another Part-of-Speech and Morphological Analyzer
 
  $Id: viterbi.cpp,v 1.18 2004/08/06 18:05:14 taku-ku Exp $;

  Copyright (C) 2001-2004 Taku Kudo <taku-ku@is.aist-nara.ac.jp>
  This is free software with ABSOLUTELY NO WARRANTY.
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/  

#include "viterbi.h"
#include "common.h"
#include <cstring>
#include <algorithm>
#include <stdexcept>

namespace MeCab {

  Viterbi::Viterbi () : alloc_size (MIN_INPUT_BUFFER_SIZE) {};

  Viterbi::Viterbi (Param &param, Tokenizer *t, Connector *c)
     : alloc_size (MIN_INPUT_BUFFER_SIZE)
  {
    if (! open (param, t, c)) throw std::runtime_error (_what);
  }

  Viterbi::~Viterbi () 
  { 
    close ();
  }

  bool Viterbi::open (Param &param, Tokenizer *t, Connector *c)
  {
    try {

      tokenizer = t;
      connector = c; 
     
      if (param.getProfileInt ("build-all-lattice")) {
	pathFreeList = new FreeList<Path> (PATH_FREELIST_SIZE);
	_connect = &Viterbi::connectWithAllPath;
      } else {
	pathFreeList = 0;
	_connect = &Viterbi::connectNormal;
      }

      endNodeList = new Node * [alloc_size];
      lookupCache = new Node * [alloc_size];

      return true;
    }
     
    catch (std::exception &e) {
      _what = std::string ("Viterbi::Viterbi(): ") +  e.what ();
      return false;
    }
  }

  void Viterbi::close () 
  {
    delete [] endNodeList; endNodeList = 0;
    delete [] lookupCache; lookupCache = 0;
    delete pathFreeList;  pathFreeList = 0;
  }

  void Viterbi::clear () 
  {
     tokenizer->clear ();
     if (pathFreeList) pathFreeList->free();
  }
  
  Node *Viterbi::lookup (unsigned int pos)
  {
    Node *resultNode = 0;

    if (lookupCache[pos]) { 
      for (Node *node = lookupCache[pos]; node; node = node->rnext) { 
	Node *newNode = tokenizer->getNewNode ();
	unsigned int id = newNode->id; 
	memcpy (newNode, node, sizeof (Node));
	newNode->rnext = resultNode;
	newNode->id = id;
	resultNode = newNode; 
      }
    } else {
      resultNode = tokenizer->lookup (begin + pos, end);
      lookupCache[pos] = resultNode;
    }

    return resultNode;
  }
  
  Node* Viterbi::analyze (const char *sentence, unsigned int len)
  {
    try {
      if (len >= alloc_size) {
	do { alloc_size *= 2; } while (len > alloc_size);
	delete [] endNodeList;
	delete [] lookupCache;
	endNodeList = new Node * [alloc_size];
	lookupCache = new Node * [alloc_size];
      }

      memset (endNodeList, 0, sizeof (Node *) * (len+1));
      memset (lookupCache, 0, sizeof (Node *) * (len+1));
     
      clear ();     
      begin = const_cast<char *>(sentence);
      end   = begin + len;
      bosNode = tokenizer->getBOSNode ();
      bosNode->surface = begin;
      endNodeList[0] = bosNode;

      for (unsigned int pos = 0; pos < len;  pos++){
	if (endNodeList[pos]) {
	  Node *rNode = lookup (pos);
	  if (! connect (pos, rNode)) return 0;
	}
      }

      eosNode = tokenizer->getEOSNode ();
      eosNode->surface = end; 
      lookupCache[len] = eosNode;
      for (unsigned int pos = len; (int)pos >= 0;  pos--) {
	if (endNodeList[pos]) {
	  if (! connect (pos, eosNode)) return 0;
	  break;
	}
      }
       
      Node *node = eosNode;
      for (Node *prevNode; node->prev; ){
	prevNode = node->prev;
	prevNode->next = node;
	node = prevNode;
      }

      return bosNode;
    }
     
    catch (std::exception &e) {
       _what = std::string ("Viterbi::analyze()") + e.what ();
       return 0;
    }
  }
}

#undef _VITERBI_WITH_ALL_PATH
#include "viterbisub.h"
#define _VITERBI_WITH_ALL_PATH
#include "viterbisub.h"
